(* flowgen.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

signature FLOWGRAPHgen = sig
  
   structure F : FLOWGRAPH
   structure C : CELLS
   structure I : INSTRUCTIONS
   structure P : INSN_PROPERTIES

   sharing I.C = C
   sharing F.I = P.I = I

   val defineLabel   : Label.label -> unit

   val mark : unit -> unit

   (* jmpTable - creates a JMPtable and accumilates it locally  *)
   val jmpTable      : {base:Label.label, targets:Label.label list} -> unit

   (* mlRealConst - creates a REALconst and accumilates it locally *)
   val mlRealConst   : Label.label * string -> unit


   (* mlStringConst - creates a STRINGconst and accumilates it locally *)
   val mlStringConst : Label.label * string -> unit


  (** emitInstr - either creates a new BBLOCK or adds the instruction
   **		  to a BBLOCK that is being built locally.
   **             If the instruction is a branch, then the successor labels
   **		  are noted in a hash table.
   ** Uses: I.branchTargets, I.instrKind
   **)
   val emitInstr     : I.instruction -> unit

  (** exitBlock - associates the list of live registers with the last
   **	code block. The last instruction is usually a branch
   **	with no targets. If not it is assumed to be a label
   **	that will be linked in at some later time. The call
   **	to exitBlock had better reflect the correct list of live 
   **   registers that terminate the branch.
   **)
   val exitBlock  : MLTree.mlrisc list -> unit

  (** endCluster
   **	- cleans up all local references and tables.
   **   - creates the final flowgraph and calls the continuation.
   **)
   val endCluster : int list -> unit

   val beginCluster : unit -> unit

   val cleanUp : unit -> unit
end

functor FlowGraphGen
    	     (structure Flowgraph : FLOWGRAPH
	      structure InsnProps : INSN_PROPERTIES

	      val codegen : Flowgraph.cluster -> unit
	      sharing Flowgraph.I = InsnProps.I) = 
struct


  structure F = Flowgraph
  structure P = InsnProps
  structure I = Flowgraph.I
  structure C = I.C
  
  type label = Label.label

  fun error msg = ErrorMsg.impossible ("FlowGraph." ^ msg)

  val bblkCnt 			   = ref 0 
  fun nextBlkNum () = !bblkCnt before bblkCnt := !bblkCnt + 1
  (** Note - currBlock will always be a reference to a F.BLOCK{..} **)
  fun freshBlk init = 
      F.BBLOCK{blknum=nextBlkNum(),
	       liveIn=ref C.empty,
	       liveOut=ref C.empty,
	       succ=ref [],
	       pred=ref [],
	       insns=ref init}
  val currBlock : F.block      ref = ref (freshBlk [])
  val blockList : F.block list ref = ref []

  fun cleanUp() = (bblkCnt:=0; currBlock:= freshBlk []; blockList:=[])


  local
    fun blockListAdd b = let
      val blocks = !blockList
    in
      case !currBlock 
       of F.BBLOCK{insns = ref [], ...} => blockList := b::blocks
        | blk => 
	   (blockList := b :: blk :: blocks;
	    currBlock := freshBlk [])
    end
    fun padString s = 
	(case ((size s) mod 4)
	   of 0 => (s ^ "\000\000\000\000")
	    | 1 => (s ^ "\000\000\000")
	    | 2 => (s ^ "\000\000")
	    | 3 => (s ^ "\000")
	    | _ => error ""
	(* end case *))
  in
    fun mark ()              = blockListAdd F.MARK
    fun defineLabel lab      = blockListAdd(F.LABEL lab)
    fun mlRealConst arg      = blockListAdd(F.REALconst arg)
    fun mlStringConst(lab,s) = 
      blockListAdd(F.STRINGconst(lab,size s, padString s))
    fun jmpTable {base:label,targets:label list} =
      blockListAdd (F.JMPtable(base,targets))
  end

  (** emitInstr - 
   **	instructions are always added to currBlock.
   **)
  fun emitInstr instr = let
    fun addInstr (F.BBLOCK{insns, ...}) =  insns := instr :: (!insns)
  in
    addInstr(!currBlock);
    case P.instrKind instr
     of P.IK_JUMP => 
         (blockList:= (!currBlock) :: (!blockList);
	  currBlock := freshBlk [])
      | _ => ()
    (*esac*)
  end      

  fun exitBlock liveRegs  = let
    fun liveCells(MLTree.REG r::rest, acc) = 
	  liveCells(rest, C.addReg(r, acc))
      | liveCells(MLTree.FREG f::rest, acc) = 
	  liveCells(rest, C.addFreg(f, acc))
      | liveCells([],acc)      = acc

    fun findCodeBlock(F.BBLOCK{liveOut,...}::_)  = liveOut
      | findCodeBlock(F.LABEL _::blks) = findCodeBlock blks
      | findCodeBlock _                = error "exitBlock.codeBlock"

    val lout = liveCells(liveRegs,C.empty)
  in
    case !currBlock
     of F.BBLOCK{insns = ref [], ...} => let 
	  val outRef = findCodeBlock (!blockList)
        in 
	  outRef := lout
	end
      | F.BBLOCK{liveOut,...} =>
	(liveOut := lout;
	 blockList := (!currBlock) :: (!blockList);
	 currBlock := freshBlk [])
      | _ => error "exitBlock"
   (*esac*)
  end

  fun endCluster regLimits = let
      fun codeBlocks([],all,cblks) = (all, Array.fromList cblks)
	| codeBlocks((blk as F.BBLOCK _)::blks, all, cblks) =
	    codeBlocks(blks, blk::all, blk::cblks)
	| codeBlocks(d::blks, all, cblks) = codeBlocks(blks, d::all, cblks)

      fun mkLabTbl(F.LABEL lab::blks,tbl) = let
	    fun search(F.BBLOCK{blknum,...}::_) = 
	          Intmap.add tbl (Label.id lab,blknum)
	      | search(_::blks) = search blks
	      | search [] = ()
	  in
	    search blks;
	    mkLabTbl(blks, tbl)
	  end
	| mkLabTbl(_::blks,tbl) = mkLabTbl(blks, tbl)
	| mkLabTbl([], tbl) = tbl

      (** update successor and predecessor information **)
      fun graphEdges(F.BBLOCK{blknum,insns,succ,...}::blks,labTbl,arr) = let
	    fun updtPred p = let 
		val F.BBLOCK{pred,...} = Array.sub(arr,p)
	    in
	      pred:=blknum::(!pred)
	    end
	    fun succBlks([],acc) = acc
	      | succBlks(P.FALLTHROUGH::labs, acc) =
	          succBlks(labs, (blknum+1)::acc)
	      | succBlks(P.LABELLED lab::labs, acc) =
		  ((succBlks(labs,(Intmap.map labTbl (Label.id lab))::acc))
		    handle _ => succBlks(labs, acc))
	      | succBlks(_::labs,acc) = succBlks(labs,acc)

	    val lastInstr = ((hd (!insns)) 
		     handle _ => error "endCluster.graphEdges.lastInstr")
	  in
	    case P.instrKind lastInstr
	     of P.IK_JUMP => succ:=succBlks (P.branchTargets lastInstr,[])
	      | _         => succ:=[blknum+1]
	    (*esac*);
	    app updtPred (!succ);
	    graphEdges(blks,labTbl,arr)
	  end
 	| graphEdges(_::blks,labTbl,arr) = graphEdges(blks,labTbl,arr)
	| graphEdges([],labTbl,_) = ()

      val (blocks, blkArr) = codeBlocks(!blockList,[],[])
      exception LabelTable
      val labTbl : int Intmap.intmap = Intmap.new(16,LabelTable)
      val _ = graphEdges(blocks, mkLabTbl(blocks,labTbl), blkArr)
      val regmaps = map (fn size => Array.tabulate (size,fn n => n)) regLimits
    in
	cleanUp();
	codegen (F.CLUSTER{blocks=blocks,regmaps=regmaps}) 
    end

  fun beginCluster _ = ()		(* currently unused *)
end
