(* dot.sml *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Generate a file readable by dot.
 *)

signature DOT = sig
    val prDOTfile : int * 
	            (CPS.lvar -> {known:Limit.kind,words_alloc:int}) *
		    CPS.function list list 
		         -> unit
end

structure Dot : DOT = struct

  fun error msg = ErrorMsg.impossible ("Dot." ^ msg)

  fun calls cexp = let
      fun forall ([],acc)    = acc
	| forall (e::es,acc) = forall(es,f e @ acc)

      and f (CPS.APP(CPS.LABEL l,_))  = [l]
	| f (CPS.APP _)		= []
	| f (CPS.RECORD(_,_,_,e))     = f e
	| f (CPS.SELECT(_,_,_,e))     = f e
	| f (CPS.OFFSET(_,_,_,e))     = f e
	| f (CPS.SWITCH(_,_,es))      = forall(es,[])
	| f (CPS.BRANCH(_,_,_,e1,e2)) = f e1 @ f e2
	| f (CPS.SETTER(_,_,e))       = f e
	| f (CPS.LOOKER(_,_,_,e))     = f e
	| f (CPS.ARITH(_,_,_,e))      = f e
	| f (CPS.PURE(_,_,_,e))       = f e
	| f (CPS.FIX _)               = error "calls.f:FIX"
    in
	f cexp
    end

  fun prDOTfile (cutoff: int,
		 limits : CPS.lvar -> {known:Limit.kind,words_alloc:int},
		 clusters) = let
	val outStrm = IO.open_out "out.dot"
	val pr = IO.outputc outStrm

	fun fName lv = let 
	      fun kind2String () =
		  (case limits lv 
		     of {known=Limit.ESCAPES,...}     => "E"
                      | {known=Limit.KNOWN_CHECK,...} => "C"
		      | {known=Limit.KNOWN,...}       => "K")
	    in
		kind2String() ^ makestring lv
	    end

	fun prHeader() = ( pr "digraph FlowGraph {\n";
			   pr "  rankdir=LR; page=\"8.5,11\";\n";
			   pr "  node [fontsize=6, ht=0.2, wid=0.3];\n";
			   pr "  ratio=compress\n")

	fun doClusters [] = ()
	  | doClusters (cl::clusters) = let
	      fun doCluster [] = ()
		| doCluster ((f,_,body)::fs) = let 
		    val src = fName f
		    val ds = calls body

		    fun edges []  = ()
		      | edges (d::ds) = (pr "\t";  pr src;  pr " -> "; 
					 pr (fName d);  pr "\n";
					 edges ds)
		    fun nodes [] = ()
		      | nodes (n::ns) = 
			(pr (fName n);
			 case limits n
			 of {known=Limit.KNOWN_CHECK,...} => 
			       pr " [shape = box,peripheries=2];\n"
		          | {known=Limit.KNOWN,...} => 
			       pr " [shape = box];\n"
			  | {known=Limit.ESCAPES,...} => 
			       pr " [shape = doublecircle];\n"
			 (* end case *);
			 nodes ns)
		  in
		      nodes (f::ds);
		      edges ds;
		      doCluster fs
		  end 
	    in
		if length cl <= cutoff then () else doCluster cl; 
		doClusters clusters
            end
      in
	  prHeader();
	  doClusters clusters;
	  pr "}\n";
	  IO.close_out outStrm
      end (* prDOTfile *)
end
