(* cluster.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Groups functions that are connected via the call graph.
 *)

(* 
** NOTE:
** To generate out.dot, set System.Control.timings to true
*)

signature CLUSTER = 
  sig
      val cluster : CPS.function list -> CPS.function list list
  end 

structure Cluster : CLUSTER = struct

  fun error msg = ErrorMsg.impossible ("Cluster." ^ msg)

  fun doit funcs = let
      exception ClusterTbl 
      exception FuncTbl
      exception LabelTbl


     (** mapping from lvar to assigned cluster id **)
      val clusterTbl: int Intmap.intmap = Intmap.new(32,ClusterTbl)
      val enterAsgnCluster              = Intmap.add clusterTbl
      val assignedCluster	        = Intmap.map clusterTbl


     (** mapping from cluster id to lvar list **)
      val funcTbl: CPS.lvar list Intmap.intmap = Intmap.new(32,FuncTbl)
      val enterFuncs  			       = Intmap.add funcTbl
      val funcsInCluster		       = Intmap.map funcTbl

      val labelTbl: CPS.function Intmap.intmap = Intmap.new(32,LabelTbl)

      val clusterCnt  = ref 0

      fun calls cexp = let
	  fun forall ([],acc)    = acc
	    | forall (e::es,acc) = forall(es,f e @ acc)

					
	  and f (CPS.APP(CPS.LABEL l,_))  = [l]
	    | f (CPS.APP _)		  = []
	    | f (CPS.RECORD(_,_,_,e))     = f e
	    | f (CPS.SELECT(_,_,_,_,e))   = f e
	    | f (CPS.OFFSET(_,_,_,e))     = f e
	    | f (CPS.SWITCH(_,_,es))      = forall(es,[])
	    | f (CPS.BRANCH(_,_,_,e1,e2)) = f e1 @ f e2
	    | f (CPS.SETTER(_,_,e))       = f e
	    | f (CPS.LOOKER(_,_,_,_,e))   = f e
	    | f (CPS.ARITH(_,_,_,_,e))    = f e
	    | f (CPS.PURE(_,_,_,_,e))     = f e
	    | f (CPS.FIX _)               = error "calls.f:FIX"
	in
	  f cexp
	end

      fun pass1 [] = ()
	| pass1 ((func as (_,f,_,_,body))::rest) = let
	    fun isClusterAsgnd f = 
		  (case assignedCluster f of n => SOME n) handle _ => NONE

	    fun nextClusterNum () = !clusterCnt before (clusterCnt := !clusterCnt+1)

	    fun addFuncToCluster(f,clNum) = let
		  val fs = funcsInCluster clNum handle _ => []
		in
		    enterFuncs (clNum,f::fs)
		end

	    fun addFuncsToCluster(fs,clNum) = let
		  val fs' = funcsInCluster clNum handle _ => []
		in
		    enterFuncs (clNum,fs @ fs')
		end

	    fun remap ([],_) = ()
	      | remap (f::fs,clNum) =
		(case isClusterAsgnd f
		 of NONE =>
			(enterAsgnCluster (f,clNum);
			 addFuncToCluster (f,clNum);
			 remap (fs,clNum))
		  | SOME fClNum => let
		       val fs' = funcsInCluster fClNum
		    in
			  enterFuncs (fClNum,[]);
			  addFuncsToCluster(fs',clNum);
			  app (fn f => enterAsgnCluster (f,clNum)) fs';
			  remap (fs,clNum)
		    end)
	  in
	     remap(f::calls body,nextClusterNum());
	     pass1 rest
	  end

      fun collect (funcs as (_,firstFun,_,_,_)::_) = let
	    fun uniqFuncsInCluster n = SortedList.uniq (funcsInCluster n)
	    fun clusterFuncs fl = map (fn f => Intmap.map labelTbl f) fl
	    fun firstCluster () = let
		fun reorder ([],_) = error "collect.firstCluster.reorder"
		  | reorder (fs as (f::rest),acc) = 
		    if f = firstFun then fs @ acc
		    else reorder (rest,f::acc)

		val firstCluster = assignedCluster firstFun
		val funcs = reorder(uniqFuncsInCluster firstCluster, [])
	      in
		  enterFuncs (firstCluster,[]);
		  clusterFuncs funcs
	      end

	    fun finish (~1,acc) = rev acc
	      | finish (n,acc) = let
		  val funcs = clusterFuncs (uniqFuncsInCluster n)
		  val n1 = n - 1
		in
		    if null funcs then finish(n1,acc)
		    else finish(n1,funcs::acc)
		end
	  in
	      finish(!clusterCnt - 1, [firstCluster()])
	  end
	| collect _ = error "collect"
     in

       app (fn func as (_,f,_,_,_) => Intmap.add labelTbl (f,func)) funcs;
       pass1 funcs;
       collect funcs
     end (* doit *)

  val cluster = Stats.doPhase(Stats.makePhase "Compiler 115 cluster") doit

end (* structure Cluster *)
    
