(* Copyright 1989 by AT&T Bell Laboratories *)
(* closutil.sml *)

(***************************************************************************
 *                                                                         *
 *  Static object environments used in closure conversion phase            *
 *                                                                         *
 ***************************************************************************)

signature CLOSUTIL = 
sig 
  type calleeregs
  type tgt
  type cluster
  type lvar
  type env
  type info

  val closureLvar : unit -> lvar
  val fixhdCSregs : lvar option * calleeregs -> calleeregs
  val freeCSvars : calleeregs -> lvar list
  val mkCSformal : calleeregs * env -> lvar list * CPS.cty list
  val tgtCSregs : tgt -> calleeregs

  val mkClu : int * lvar list * env -> cluster
  val mergeClu : cluster * cluster -> cluster
  val isEmpClu : cluster -> bool
  val delClu : cluster * lvar list -> cluster
  val knownregs : cluster * env * int * int -> lvar list

  val initInfo : CPS.fun_kind * int * calleeregs * lvar list -> info
  val mergeInfo : info * info -> info
  val foldInfo : info list -> info
  val addvInfo : lvar list * info -> info
  val delvInfo : lvar list * info -> info
  val formatInfo : int * env * info -> (cluster * tgt)

  exception NotBound
  val emptyEnv : unit -> env
  val printEnv : env -> unit

  val augValue : (lvar * CPS.cty * int * env) -> env  
  val augCalleeF : (lvar * lvar * calleeregs * env) -> env
  val augEscapeF : (lvar * lvar * lvar * env) -> env
  val augCalleeV : (lvar * lvar * calleeregs * env) -> env
  val augKnownF : (lvar * lvar * lvar list * env) -> env 
  val augSpill : (lvar * lvar list * int * env) -> env

  val get_cty : env -> lvar -> CPS.cty
  val get_depth : env -> lvar -> int

  val augArgs : lvar list * CPS.cty list * int * env
                -> (lvar list * CPS.cty list * env)

  val procAPP : CPS.value * CPS.value list * env
                -> (CPS.cexp * lvar list * calleeregs)

  val freeAnalysis : lvar list * env -> lvar list

  val mkClosure : int * env * cluster * lvar list * lvar list 
                  * CPS.fun_kind 
                  -> (lvar list * (CPS.cexp -> CPS.cexp) 
                      * lvar list * lvar list * env) 

end

functor ClosUtil(MachSpec : MACH_SPEC) : CLOSUTIL = struct

local 
  open CPS Access SortedList
in

type lvar = CPS.lvar 
val pr = Control.Print.say
val error = ErrorMsg.impossible

(*** s1 is a subset of s2 ***)
fun subset(s1,s2) = (difference(s1,s2) = [])

(*** take the first n elements from a list ***)
fun head(n,l) = 
  let fun h(n,[],z) = z
        | h(0,_,z) = z
        | h(n,a::r,z) = h(n-1,r,a::z)
   in h(n,l,[])
  end

(*** return a fresh closure Lvar ***)
val closureLvar = 
  let val save = (!saveLvarNames before saveLvarNames := true)
      val closure = namedLvar(Symbol.varSymbol "closure")
   in (saveLvarNames := save; fn () => dupLvar closure)
  end

fun clean l = 
let fun vars(l, VAR x :: rest) = vars(x::l, rest)
      | vars(l, _::rest) = vars(l,rest)
      | vars(l, nil) = uniq l
 in vars(nil,l)
end

(*****************************************************************************
 *  Utility functions for calleeregs information                             *
 *****************************************************************************)

type ovar = lvar option
type calleeregs = {gpregs : ovar list, fpregs : ovar list}

(* The following should be "val emptyCSregs =" *)
fun emptyCSregs() =
  let fun mklist (n,l) = if n=0 then l else mklist(n-1,NONE::l)
   in {gpregs = mklist(MachSpec.numCalleeSaves,[]), 
       fpregs = mklist(MachSpec.numFloatCalleeSaves,[])}
  end

fun newCSregs () = 
  let fun newlist (n,l,r) = 
        if n <=0 then (rev l,rev r) 
        else (let val v = mkLvar()
               in newlist(n-1,SOME(v)::l,v::r)
              end)

      val (gpregs,gpfree) = newlist(MachSpec.numCalleeSaves,[],[])
      val (fpregs,fpfree) = newlist(MachSpec.numFloatCalleeSaves,[],[])
      val free = gpfree@fpfree
      val cl = (map (fn _ => PTRt) gpfree)@(map (fn _ => FLTt) fpfree)

   in ({gpregs=gpregs,fpregs=fpregs},free,cl)
  end

fun mkCSargs({gpregs,fpregs}) =
  let fun mkgp(SOME v) = VAR v
        | mkgp(NONE) = INT 0          (* blackhole the register *)
      fun mkfp(SOME v) = VAR v
        | mkfp(NONE) = REAL "0.0"     (* should use default real instead *)
   in (map mkgp gpregs)@(map mkfp fpregs)
  end

fun extract l = 
  let fun h(NONE::r,res) = h(r,res)
        | h((SOME v)::r,res) = h(r,enter(v,res))
        | h([],res) = res
   in h(l,[])
  end

fun freeCSvars({gpregs,fpregs}) = merge(extract gpregs,extract fpregs)

fun overCSregs({gpregs=g1,fpregs=f1},{gpregs=g2,fpregs=f2}) =
  let fun overlay(r as (SOME v),_) = r
        | overlay(NONE,r) = NONE

      val ngpregs = List2.map2 overlay (g1,g2)
      val nfpregs = List2.map2 overlay (f1,f2)
   in {gpregs=ngpregs,fpregs=nfpregs}
  end 

fun mergeCSregs({gpregs=g1,fpregs=f1},{gpregs=g2,fpregs=f2}) =
  let fun merge(r as (SOME v),_) = r
        | merge(NONE,r) = r

      val ngpregs = List2.map2 merge (g1,g2)
      val nfpregs = List2.map2 merge (f1,f2)
   in {gpregs=ngpregs,fpregs=nfpregs}
  end 

fun delCSvars(delset,{gpregs,fpregs}) =
  let fun delvar(r as (SOME v)) = if member delset v then NONE else r
        | delvar r = r
      
      val ngpregs = map delvar gpregs
      val nfpregs = map delvar fpregs

   in {gpregs=ngpregs,fpregs=nfpregs}
  end

fun fixhdCSregs(v,{gpregs=NONE::r,fpregs}) = {gpregs=v::r,fpregs=fpregs}
  | fixhdCSregs(v,_) = error "fixhdCSregs in closure phase"

(*****************************************************************************
 *  Utility functions for cluster information                                *
 *****************************************************************************)

type vset = {closures : lvar list , values : lvar list, floats : lvar list}
datatype cluster = CLUSTER of vset list    

val emptyVS = {closures=[],values=[],floats=[]}

fun emptyVset ({closures=[],values=[],floats=[]}) = true
  | emptyVset _ = false

fun varsVset{closures=cs,values=vs,floats=fs} = merge(cs,merge(vs,fs))

fun delVset({closures=c1,values=v1,floats=f1},s) = 
      {closures=difference(c1,s),values=difference(v1,s),
       floats=difference(f1,s)}

fun mergeVset({closures=c1,values=v1,floats=f1},
              {closures=c2,values=v2,floats=f2})
  = {closures=merge(c1,c2),values=merge(v1,v2),floats=merge(f1,f2)}

(*** empty cluster information ***)
fun emptyClu(depth) = 
  let fun h(k,l) = if k < depth then h(k+1,emptyVS::l) else l
   in CLUSTER(h(0,[]))
  end

(*** merge two clusters, may needs branch prediction information ***)
fun mergeClu(CLUSTER c1,CLUSTER c2) = CLUSTER(List2.map2 mergeVset (c1,c2))

(*** delete a set of variables from the current cluster ***)
fun delClu(CLUSTER c,vl) = CLUSTER(map (fn vs => delVset(vs,vl)) c)

fun isEmpClu(CLUSTER l) = 
  let fun h(a::r) = if emptyVset(a) then h r else false
        | h [] = true
   in h l
  end

(*****************************************************************************
 *  Utility functions for register targeting information                     *
 *****************************************************************************)

datatype tgt = CAF of calleeregs | ESF | KNF 

(*** empty register targeting information ***)
fun emptyTgt(CONT) = CAF(emptyCSregs())
  | emptyTgt(ESCAPE) = ESF
  | emptyTgt(NO_INLINE_INTO) = ESF  (* can this occur here? *)
  | emptyTgt _ = KNF  

(*** make a piece of tgtinfo that contains cregs ***)
fun mkTgt(CONT,cregs) = CAF(cregs)
  | mkTgt(ESCAPE,_) = ESF
  | mkTgt(NO_INLINE_INTO,_) = ESF  (* can this occur here? *)
  | mkTgt _ = KNF

(*** merge two pieces of tgtinfo ***)
fun mergeTgt(ESF,ESF) = ESF
  | mergeTgt(KNF,KNF) = KNF
  | mergeTgt(CAF c1,CAF c2) = CAF (mergeCSregs(c1,c2))
  | mergeTgt _ = error "unexpected cases in mergeTgt in closure phase"

(*** overlay the first tgtinf on the second one ***)
fun overTgt(ESF,ESF) = ESF
  | overTgt(KNF,KNF) = KNF
  | overTgt(CAF c1,CAF c2) = CAF(overCSregs(c1,c2))
  | overTgt _ = error "unexpected cases in overTgt in closure phase"

(*** delete certain lvars from the targetting information ***)
fun delTgt(vl,ESF) = ESF
  | delTgt(vl,KNF) = KNF
  | delTgt(vl,CAF c) = CAF(delCSvars(vl,c))

(*** a set of free variables in the tgt ***)
fun varsTgt(CAF c) = freeCSvars(c)
  | varsTgt _ = []

(*** get calleesaveregs info from tgt ***)
fun tgtCSregs(CAF c) = c
  | tgtCSregs _ = error "getCStgt in the closure phase"

(*****************************************************************************
 *  Utility functions for objenv information                                 *
 *****************************************************************************)

datatype object 
  = Value of {ty : cty, depth : int}
  | KnownF of {label : lvar, free : lvar list}
  | EscapeF of {label : lvar, closureVar : lvar}
  | CalleeF of {label : lvar, free : lvar list, calleeregs : calleeregs} 
  | CalleeV of {var : lvar, free : lvar list, calleeregs : calleeregs}
  | Spill of {depth : int, free : lvar list}
  | Closure of {depth : int, vset : vset, other : lvar list}

datatype env = Env of object Intmap.intmap   

exception NotBound
exception Lookup of lvar * env

(*** initialize an empty object environment ***)
fun emptyEnv() = Env(Intmap.new(32,NotBound))

(*** Update an environment ***)
fun augment(m,env as Env(whatMap)) = (Intmap.add whatMap m; env)
fun side_effect(m,env as Env(whatMap)) = (Intmap.add whatMap m; env)

fun augValue(v,t,d,env) = augment((v,Value{ty=t,depth=d}),env)

fun augCalleeF(v,lab,calleeregs,env) =
     augment((v,CalleeF{label=lab,calleeregs=calleeregs,
                        free=freeCSvars(calleeregs)}),env)

fun augCalleeV(v,w,calleeregs,env) = 
     augment((v,CalleeV{var=w,calleeregs=calleeregs,
                        free=enter(w,freeCSvars(calleeregs))}),env)
 
fun augEscapeF(v,lab,cname,env) = 
     augment((v,EscapeF{label=lab,closureVar=cname}),env)

fun augKnownF(v,lab,free,env) = 
     augment((v,KnownF{label=lab,free=free}),env)

fun augSpill(v,free,d,env) = 
     augment((v,Spill{depth=d,free=free}),env)

(**>>
fun augClosure(v,vset,other,depth,env) = 
     augment((v,Closure{vset=vset,other=other,depth=depth}),env)
<<**)

(*** Environment printing, for debugging. ***)
fun printEnv(Env whatMap) =
  let val vp = pr o lvarName
      fun sayv(VAR v) = vp v
        | sayv(LABEL v) = (pr "(L)"; vp v)
        | sayv(INT i) = (pr "(I)"; pr(makestring i))
        | sayv(REAL r) = pr r
        | sayv(STRING s) = (pr "\""; pr s; pr "\"")
        | sayv(OBJECT _) = pr "**OBJECT**"

      fun saygp(SOME v) = (pr "(GPR-"; vp v; pr ")")
        | saygp(NONE) = pr "(GPR-empty-regs)"

      fun sayfp(SOME v) = (pr "(FPR-"; vp v; pr ")")
        | sayfp(NONE) = pr "(FPR-empty-regs)"

      fun plist p l = (app (fn v => (pr " "; p v)) l; pr "\n")
      val ilist = plist vp
      val gplist = plist saygp
      val fplist = plist sayfp

      fun prcalleeregs({gpregs,fpregs}) = (gplist gpregs; fplist fpregs)

      fun pvalue(v,Value _) = (vp v; pr " ")
        | pvalue _ = ()

      fun pknown(v,KnownF{label,free}) =
 	     (vp v; pr "/known "; sayv(LABEL label); pr " -"; ilist free)
        | pknown _ = ()

      fun pfcallee(v,CalleeF{label,calleeregs,...}) = 
             (vp v; pr "/calleef "; sayv(LABEL label); pr " -"; 
                    prcalleeregs calleeregs)
        | pfcallee _ = () 

      fun pvcallee(v,CalleeV{var,calleeregs,...}) =
             (vp v; pr "/calleev "; sayv(VAR var); pr " -";
                    prcalleeregs calleeregs)
        | pvcallee _ = ()

      fun pescape(v,EscapeF{label,closureVar}) =
             (vp v; pr "/escapef "; sayv(LABEL label); pr " -"; 
                    pr "CLOSURE "; vp closureVar)
        | pescape _ = ()

      fun pclosure(v,Closure _) = ()
        | pclosure _ = ()

   in pr "Values:"; Intmap.app pvalue whatMap; pr "\n";
      pr "Known function mapping:\n"; Intmap.app pknown whatMap;
      pr "Callee-save continuation functions:\n"; Intmap.app pfcallee whatMap;
      pr "Callee-save continuation variables:\n"; Intmap.app pvcallee whatMap;
      pr "Escape functions:\n"; Intmap.app pescape whatMap;
      pr "Closures:\n"; Intmap.app pclosure whatMap
  end

(*** whatIs: return type of object bound to an lvar in an environment. ***)
fun whatIs(env as Env(whatMap),v) = 
     Intmap.map whatMap v handle NotBound => 
        (pr "Cannot find the variable "; pr (lvarName v);
         pr "in the following ObjEnv (exception Lookup) : \n";
         printEnv(env); raise Lookup(v,env))

fun get_cty env v = 
     case whatIs(env,v) 
      of Value {ty,...} => ty
       | EscapeF _ => FUNt | KnownF _ => FUNt
       | CalleeF _ => CNTt | CalleeV _ => CNTt
       | _ => PTRt

fun get_depth env v = 
     case whatIs(env,v) 
      of Value {depth=d,...} => d
       | Closure{depth,...} => depth
       | Spill{depth,...} => depth
       | _ => error "get_depth in closure phase 465"

(*** add the extra arguments for continuation variables ***)
fun augArgs(args,cl,depth,env) =
  let fun h((a,CNTt),(x,y,env)) = 
            let val v = dupLvar(a)
                val (new,free,tt) = newCSregs()
                val env = augCalleeV(a,v,new,env)
                val env = fold (fn (w,env) => augValue(w,PTRt,depth,env))
                            free (augValue(v,CNTt,depth,env))
                val nx = v::(free@x)
                val ny = CNTt::(tt@y)
             in (nx,ny,env)
            end
        | h((a,t),(x,y,env)) = (a::x,t::y,augValue(a,t,depth,env))
   in fold h (List2.zip2(args,cl)) ([],[],env)
  end

(*** processing the arguments of APP expressions ***)
fun APPargs(args,env) =
  let fun h(a as VAR v,(l,t)) = 
            (case whatIs(env,v) 
              of Value _ => ((a::l),t)
               | EscapeF _ => ((a::l),t)
               | CalleeF{label,calleeregs,...} => 
                   (let val nargs = mkCSargs(calleeregs)
                     in ((LABEL label)::(nargs@l),calleeregs)
                    end)
               | CalleeV{var,calleeregs,...} =>
                   (let val nargs = mkCSargs(calleeregs)
                     in ((VAR var)::(nargs@l),calleeregs)
                    end)
               | _ => error "unexpected APPargs 212 in the closure phase")
        | h(u,(l,t)) = ((u::l),t)
   in fold h args (nil,emptyCSregs())
  end

(*** procAPP transforms the APP statements to the new form ***)
fun procAPP(f,args,env) =
  let val obj = case f of VAR v => whatIs(env,v) 
                        | _ => error "unexpected situations 231 in closure"
      val (nargs,ntgt) = APPargs(args,env)
   in case obj
       of Value {ty=t,...} => 
            let val l = mkLvar()
                val free = clean (f::nargs)
             in (SELECT(0,f,l,t,APP(VAR l,(VAR l)::f::nargs)), free, ntgt)
            end
        | EscapeF{label,closureVar} =>
            let val free = enter(closureVar,clean(nargs))
             in (APP(LABEL label,(LABEL label)::f::nargs), free, ntgt)
            end
        | CalleeF{label,calleeregs=cregs,...} =>
            let val nargs' = ((mkCSargs cregs)@nargs)
                val free = clean nargs'                
             in (APP(LABEL label,(LABEL label)::nargs'), free, cregs)
            end
        | CalleeV{var=v,calleeregs=cregs,...} =>
            let val nargs' = (VAR v)::((mkCSargs cregs)@nargs)
                val free = clean nargs'
             in (APP(VAR v,nargs'), free, cregs)
            end
        | KnownF{label,free} =>
            let val nargs' = nargs@(map VAR free)
                val free = clean nargs'
             in (APP(LABEL label,nargs'), free, ntgt)
            end
        | _ => error "procAPP in the closure phase 129"
  end   

(****************************************************************************
 * freeAnalysis: Take a list of free variables, and replace each function   *
 *               by its free closure variables.                             *
 ****************************************************************************)

fun freeAnalysis(free,env) =
  fold (fn (v,l) =>
	  case whatIs(env,v)
	   of CalleeF{free,...} => merge(free,l)
	    | CalleeV{free,...} => merge(free,l)
	    | KnownF{free,...} => merge(free,l)
            | EscapeF{closureVar,...} => enter(closureVar,l)
	    | Value _ => enter(v,l)
            | _ => error "closure variables in freeAnalysis")
    free nil


(*****************************************************************************
 * build a cluster from a set of free variables                              *
 *****************************************************************************)

(*** build a new cluster form a set of free variables ***)
fun mkClu(depth,vl,env) = 
  let fun take(d,fr) = 
        let fun g([],cs,vs,fs) = ({closures=cs,values=vs,floats=fs},[])
              | g(z as ((k,v)::r),cs,vs,fs) = 
                    if d <> k then ({closures=cs,values=vs,floats=fs},z)
                    else (case (whatIs(env,v)) 
                           of Closure _ => g(r,enter(v,cs),vs,fs)
                            | Spill _ => g(r,enter(v,cs),vs,fs)
                            | Value{ty=FLTt,...} => g(r,cs,vs,enter(v,fs))
                            | Value _ => g(r,cs,enter(v,vs),fs)
                            | _ => error "mkClu in the closure phase")
          in g(fr,[],[],[])
         end
      fun h(k,res,fr) = if k < depth then let val (r,nfr) = take(k,fr)
                                           in h(k+1,r::res,nfr)
                                          end
                        else res
      val sort2 = Sort.sort (fn ((i,_),(j,_)) => i > j)
      val dl = sort2 (map (fn v => (get_depth env v, v)) vl)
   in CLUSTER (h(0,[],dl))
  end

(*****************************************************************************
 *  Choosing certain candidates to be put in the calleeregs.                 *
 *****************************************************************************)
fun candidates(depth,n,regs,l) = 
  let val vl = extract regs
      fun gather (d,[],s) = s
        | gather (d,x::r,s) = 
            let val nl = map (fn z => if member vl z 
                                      then (d*2+1,z) else (d*2,z)) x
             in if d < 0 then error "candidates in closure phase"
                else gather(d-1,r,s@nl)
            end

      fun compare ((d1 : int,v1 : lvar),(d2,v2)) = 
                 if d1 = d2 then (v1 < v2) else (d1 < d2) 

      val tmp = Sort.sort compare (gather(depth,l,[]))
      val vs = uniq(map #2 (head(n,tmp)))
      val delvs = difference(vl,vs)
      val addvs = difference(vs,vl)
(*
      val _ = app (fn (v:int) => (pr (makestring(v)); pr " ")) vs
      val _ = pr "    ***** vs \n"
      val _ = app (fn (v:int) => (pr (makestring(v)); pr " ")) delvs
      val _ = pr "    ***** delvs \n"
      val _ = app (fn (v:int) => (pr (makestring(v)); pr " ")) addvs
      val _ = pr "    ***** delvs \n"
      val _ = app (fn (d : int,v:int) => (pr (makestring(d)); pr " "; pr (makestring(v)); pr " ")) tmp
      val _ = pr "    ***** all \n"
*)
      val newregs = 
        let fun h(NONE::z,ds,r) = h(z,ds,NONE::r)
              | h((u as (SOME v))::z,ds,r) = 
                   if member ds v then h(z,ds,NONE::r)
                   else h(z,enter(v,ds),u::r)
              | h([],_,r) = rev r
         in h(regs,delvs,[])
        end

      fun scan([],l) = l
        | scan(x::r,NONE::l) = (SOME x)::(scan(r,l))
        | scan(r,(SOME v)::l) = (SOME v)::(scan(r,l))
        | scan _ = error "scan in transCSregs in the closure phase"

   in scan(addvs,newregs)
  end

(*** move some chunk from cluster into the targeting part ***)
fun transTgt(d,CLUSTER l,CAF {gpregs,fpregs},env) = 
     let (*** a temporary hack here ***)
         fun filterCC(a::r,l) = 
               (case whatIs(env,a)
                 of Closure _ => filterCC(r,a::l)
                  | _ => filterCC(r,l))
           | filterCC([],l) = l
         val gvs = map (fn x => (#values x)@(filterCC(#closures x,[]))) l
         val fvs = map #floats l
         val ngpr = candidates(d,MachSpec.numCalleeSaves-1,tl gpregs,gvs)
         val nfpr = candidates(d,MachSpec.numFloatCalleeSaves,fpregs,fvs)
      in CAF {gpregs=(NONE::ngpr),fpregs=nfpr}
     end
  | transTgt(d,_,tgt,env) = tgt

(*** The following is a major gross hack ***)
(*** move some chunk from cluster to the registers ***)
fun knownregs(CLUSTER l,env,gpmax,fpmax) =
  let (*** a temporary hack here ***)
      fun filterCC(a::r,l) = 
            (case whatIs(env,a)
              of Closure _ => filterCC(r,a::l)
               | _ => filterCC(r,l))
        | filterCC([],l) = l

      val gvs = map (fn x => (#values x)@(filterCC(#closures x,[]))) l
      val fvs = map #floats l
      fun emptyRegs(k) = if k <= 0 then [] else NONE::(emptyRegs(k-1))
      val gpregs = emptyRegs(gpmax)
      val fpregs = emptyRegs(fpmax)
      val d = length(l) + 1
      val ngpr = candidates(d,gpmax,gpregs,gvs)
      val nfpr = candidates(d,fpmax,fpregs,fvs)
   in merge(extract ngpr, extract nfpr)
  end
    
(*** make a list of formal arguments according to the CSregs information ***)
fun mkCSformal({gpregs,fpregs},env) =
  let val grabty = get_cty env
      fun mkgpreg(SOME v,(r,z)) = (v::r,(grabty v)::z)
        | mkgpreg(NONE,(r,z)) = ((mkLvar())::r,PTRt::z)

      fun mkfpreg(SOME v,(r,z)) = (v::r,FLTt::z)
        | mkfpreg(NONE,(r,z)) = ((mkLvar())::r,FLTt::z)

   in fold mkgpreg gpregs (fold mkfpreg fpregs ([],[]))
  end

(*****************************************************************************
 *  Utility functions for info data structures                               *
 *****************************************************************************)

(*** info data structure used in the closure analysis ***)
datatype info = INFO of {cluster : cluster, tgt : tgt, 
                         delset : lvar list, addset : lvar list}

(*** initialize info information ***)
fun initInfo(k,d,cregs,free) =
      INFO{cluster=emptyClu(d),tgt=mkTgt(k,cregs),addset=free,delset=[]}

(*** clean-up the current piece of info ***)
fun forceInfo(d,env,INFO{cluster,tgt,addset,delset}) =
  let fun secure(v,r) = 
        case whatIs(env,v)
         of Value _ => (v::r)
          | EscapeF{closureVar,...} => (closureVar::r)
          | Closure _ => (v::r)
          | Spill _ => (v::r) 
          | _ => error "secure in forceInfo in the closure phase"
     
      val newvl = fold secure (difference(addset,delset)) []        
      val newTgt = delTgt(delset,tgt)

      (*** current depth free variables never enter the cluster ***)
      val newClu = mergeClu(mkClu(d,newvl,env),cluster)  
      val newTgt = transTgt(d,newClu,newTgt,env)

   in (newClu,newTgt)
  end  

(*** merging two piece of info datas ***)
fun mergeInfo(INFO{cluster=c1,tgt=t1,addset=a1,delset=d1},
              INFO{cluster=c2,tgt=t2,addset=a2,delset=d2})
    = INFO{cluster=mergeClu(c1,c2),tgt=mergeTgt(t1,t2),
           addset=merge(a1,a2),delset=merge(d1,d2)}

(*** foldmerge several pieces of info datas ***)
fun foldInfo [] = error "foldInfo 942 in the closure phase"
  | foldInfo(a::r) = fold mergeInfo r a

(*** overlaying a piece of info data on another one (do forceInfo) ***)
fun overInfo(d,env,info1,info2) = 
  let val (c1,t1) = forceInfo(d,env,info1)
      val (c2,t2) = forceInfo(d,env,info2)
   in INFO{cluster=mergeClu(c1,c2),tgt=overTgt(t1,t2),
           addset=[],delset=[]}
  end

(*** add a set of free variables into the current info ***)
fun addvInfo(vl,INFO{addset,cluster,tgt,delset}) =
      INFO{addset=merge(vl,addset),cluster=cluster,
           tgt=tgt,delset=delset}

(*** delete a set of free variables from the current info ***)
fun delvInfo(vl,INFO{delset,cluster,tgt,addset}) =
      INFO{delset=merge(vl,delset),cluster=cluster,
           tgt=tgt,addset=addset}

(*** format the info data structure, decide the tgt info ***)
fun formatInfo(d,env,info) =
  let val (clu,tgt) = forceInfo(d,env,info)
      val newClu = delClu(clu,varsTgt tgt)
   in (newClu,tgt)
  end  


(*****************************************************************************
 *  Compacting the current cluster and then decide whethe to build a closure *
 *  now. (build when the depth info of the cluster = the current depth)      *
 *****************************************************************************)

(*** compacting the final cluster, deciding its layout ***)
fun compactClu(depth,env,fixkind,CLUSTER l,csvars) = 
  let val FLAG1 = true
 
(* case fixkind of ESCAPE => false 
                                | _ => true *)

      val FLAG2 = FLAG1
(* case parentKind of KNOWN => false
                                   | _ => FLAG1
*)
      (*** Basic Assumptions: one continuation definition each FIX ***)
      val nl =
        let fun collectFree(c,all) =
              case whatIs(env,c) 
               of Closure{vset=vs,other=other,...} => 
                     merge(varsVset vs,merge(other,all))
                | _ => all

            fun compactVset([],l,accfree) = rev l
              | compactVset(u::r,l,accfree) =
                 let val {closures=cs,values=vs,floats=fs} = u
                     val newfree = fold collectFree cs accfree
                     val nu = {closures=difference(cs,newfree),
                               values=difference(vs,newfree),
                               floats=difference(fs,newfree)}
                  in compactVset(r,nu::l,newfree)
                 end
         in if FLAG1 then compactVset(l,[],fold collectFree csvars [])
                     else l
        end

      val nl = 
        let fun collectSpill (c,all) = 
              case whatIs(env,c) 
               of Spill{free,...} => merge(all,free)
                | Closure{vset={closures=cs,...},...} => 
                       fold collectSpill cs all
                | _ => all       
            val free = fold collectSpill (foldmerge(map #closures l)) csvars
 
            (*** aggressive compactions on Spill closures ***)
            fun compactSpill {closures=cs,values=vs,floats=fs} =
              {closures=difference(cs,free),values=difference(vs,free),
               floats=difference(fs,free)}
         in map compactSpill nl
        end

      val (d,vs,other) = 
        let fun decDepth(a::r,d) = 
                   if emptyVset a then decDepth(r,d-1) else (d,a,r)
              | decDepth _ = (0,emptyVS,[])  (* special case *)

            val (d,topvs,rest)  = decDepth(nl,depth)

            fun h(~1,[]) = []
              | h(_,[]) = error "compactClu 4535 in the closure phase"
              | h(d,a::r) = 
                 let val allgv = merge(#closures a,#values a)
                  in if ((length(#floats a) > 0) orelse
                         ((length(allgv) > 2) andalso FLAG2))
                     then (let val cname = closureLvar()
                               val obj = Closure{vset=a,other=[],depth=d}
                               val env = side_effect((cname,obj),env) 
                            in enter(cname,h(d-1,r))
                           end)
                     else (merge(allgv,h(d-1,r)))
                 end

         in (d,topvs,h(d-1,rest))
        end
   in (d,vs,other,env)
  end

(*** There is a dirty hack on side-effecting env here ***)
fun mkClosure(depth0,env,totalClu,calleeFree,knownFree,fixkind) = let 

val FLAG3 = false
(* case parentKind of KNOWN => true
                             | _ => false *) 
      
val (d,vset,tlfree,env) = 
  compactClu(depth0,env,fixkind,totalClu,calleeFree)

(*** filter out all closures in the calleeregs that should be built now ***)
fun filterCC([],r) = r 
  | filterCC (v::z,r) = 
     (case whatIs(env,v)
       of Closure{depth=k,...} => 
           if (k=depth0) orelse FLAG3 then 
                    filterCC(z,enter(v,r)) else filterCC(z,r)
        | _ => filterCC(z,r))

val cs1 = filterCC(knownFree@calleeFree@tlfree,[])

val fp = closureLvar()   
fun g(a,(header,fvars,kill,free,env)) = 
  if member kill a then (header,fvars,kill,free,env)
  else let val (cs,vs,fs,other,SPILL) =
            case whatIs (env,a) 
             of Closure{vset,other,...} => 
                   (#closures vset,#values vset,#floats vset,other,false)
              | Spill _ =>  ([],[],[],[],true)
              | _ => ([],[],[],[],true)
                     (* error "mkClosure 2435" *)

           val allv = merge(cs,merge(vs,other))
           val (contents,nkill) = 
            case fs of [] => (allv,kill)
                     | _ => (enter(fp,allv),enter(fp,kill))

           val (newhdr,newfvars,newkill,newfree,env) =
            if SPILL then (header,fvars,nkill,enter(a,free),env)
            else 
             (let val nfree = merge(contents,free)
                  val nfvars = merge(fs,fvars)
                  val nkill = enter(a,nkill)
                  val cs1 = filterCC(other,cs)     (*** a temporary hack ***)
               in fold g cs1 (header,nfvars,nkill,nfree,env)
              end)

           val (newheader,newenv) = 
            case contents 
             of [] => (newhdr,env) 
              | _ => (let val ul = map (fn x => (VAR x,OFFp 0)) contents
                          val obj = Value {ty=PTRt,depth=depth0}
                          val env = side_effect((a,obj),env)
                       in (fn ce => newhdr(RECORD(RK_BLOCK,ul,a,ce)),env)
                      end)

        in (newheader,newfvars,newkill,newfree,newenv)
       end

in (* (contents,header,kill,new,env) *)

if (d >= depth0) orelse FLAG3 then 
  (let val (cs,vs,fs) = (#closures vset,#values vset,#floats vset)
       val topcs = merge(cs,cs1) 

       val (header,fvars,kill,free,env) = fold g topcs (fn x => x,fs,[],[],env)
       val (newhdr,env) = 
         case fvars
          of [] => (header,env)
           | _ => (let val ul = map (fn x => (VAR x,OFFp 0)) fvars
                       val obj = Value {ty=PTRt,depth=depth0}
                       val env = side_effect((fp,obj),env)
                    in (fn ce => RECORD(RK_FBLOCK,ul,fp,header(ce)),env)
                   end)
       val (newcs,newkill) = 
         case fs of [] => (cs,kill) 
                  | _ => (enter(fp,cs),enter(fp,kill))

       val contents = merge(newcs,merge(vs,tlfree))
   in (contents,newhdr,newkill,free,env)
  end)
else (* the case d < depth0 *)
  (let val (header,fvars,kill,free,env) = fold g cs1 (fn x => x,[],[],[],env)
       val (newhdr,newkill,env) = 
         case fvars 
          of [] => (header,kill,env)
           | _ => (let val ul = map (fn x => (VAR x,OFFp 0)) fvars
                       val obj = Value {ty=PTRt,depth=depth0}
                       val env = side_effect((fp,obj),env)
                    in (fn ce => RECORD(RK_FBLOCK,ul,fp,header(ce)), 
                        enter(fp,kill),env)
                   end)

       val allvs = (#closures vset)@(#values vset)@tlfree
       val allfs = (#floats vset)
    in case (allvs,allfs)
        of ([],[]) => ([],newhdr,newkill,free,env)
         | ([x],[]) => ([x],newhdr,newkill,free,env)
         | _ => (let val cname = closureLvar()
                     val obj = Closure{depth=d,other=tlfree,vset=vset}
                     val env = side_effect((cname,obj),env)
                  in ([cname],newhdr,newkill,free,env)
                 end)
   end)


end (* function mkClosure *)

end (* local *)

end (* functor ClosUtil *)

