(* viscomp.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

signature PRINTHOOKS =
    sig
	(* all output goes to Control.Print.out *)
      val prAbsyn : StaticEnv.staticEnv -> Absyn.dec -> unit
      val prLambda : Lambda.lexp -> unit
      val prLamFun : Lambda.lexp -> int -> unit
    end

signature VISCOMP = 
sig 
  structure Stats : STATS
  structure Control : CONTROL
  structure Source : SOURCE
  structure ErrorMsg : ERRORMSG
  structure Symbol : SYMBOL
  structure Index : INDEX
  structure StaticEnv : STATICENV
  structure DynamicEnv : DYNENV
  structure BareEnvironment : ENVIRONMENT
  structure Environment : ENVIRONMENT sharing Environment=SCEnv.Env
  structure CoerceEnv : 
      sig val b2e : BareEnvironment.environment->Environment.environment
          val e2b : Environment.environment->BareEnvironment.environment
	  val debugging : bool ref
      end
  structure ModuleId : MODULE_ID
  structure ModuleTable: MODULE_TABLE
  structure SCStaticEnv : SCSTATICENV
  structure PickleEnv : PICKLE_ENV
  structure PersStamps : PERSSTAMPS
  structure PrettyPrint : PRETTYPRINT
  structure PPTable : sig
      val install_pp : string list -> (PrettyPrint.ppstream -> 'a -> unit) -> unit
    end
  structure Ast : AST
  structure Lambda: sig type lexp end
  structure Compile : COMPILE
  structure Interact : INTERACT
  structure Machm : CODEGENERATOR
  structure AllocProf : sig val reset : unit -> unit
			    val print : outstream -> unit
			end
  structure PrintHooks : PRINTHOOKS
  structure Profile : PROFILE
(***
  functor Debugger : DEBUGGERFUN
***)
  structure Boot : sig val coreEnvRef : Environment.environment ref end
  val version : {
          system : string,      	(* the system title *)
	  version_id : int list,	(* the version number *)
          date : string         	(* date of creation *)
	}
  val banner : string
  val architecture: string
end  

functor VisComp(Machm : CODEGENERATOR) : VISCOMP =
struct
  structure Stats = Stats
  structure Control = Control
  structure Source = Source
  structure ErrorMsg = ErrorMsg
  structure Symbol = Symbol
  structure Index = Index
  structure StaticEnv = StaticEnv
  structure DynamicEnv = DynamicEnv
  structure BareEnvironment = Environment
  structure Environment = SCEnv.Env
  structure CoerceEnv =
  struct val b2e = SCEnv.SC
         val e2b = SCEnv.unSC
	 val debugging = SCStaticEnv.debugging
  end
  structure ModuleId = ModuleId
  structure ModuleTable = ModuleTable
  structure SCStaticEnv = SCStaticEnv
  structure PickleEnv = PickleEnv
  structure PersStamps = PersStamps
  structure PrettyPrint = PrettyPrint
  structure PPTable =
    struct
      val install_pp : string list -> (PrettyPrint.ppstream -> 'a -> unit) -> unit
	    = System.Unsafe.cast PPTable.install_pp
    end (* PPTable *)
  structure Ast = Ast
  structure Lambda = Lambda
  structure Compile = CompileF(Machm)
  structure Interact = Interact(EvalLoopF(Compile))
  structure Machm = Machm
  structure AllocProf =
    struct
      val reset = AllocProf.reset
      val print = AllocProf.print_profile_info
    end
  structure PrintHooks : PRINTHOOKS =
    struct fun prAbsyn env d  = 
	       PrettyPrint.with_pp (ErrorMsg.defaultConsumer())
	                 (fn ppstrm => PPAbsyn.ppDec(env,NONE) ppstrm (d,200))
	   fun prLambda lexp = (MCprint.printLexp lexp; print "\n")
	   fun prLamFun lexp v = (MCprint.printFun lexp v; print "\n")
    end
  structure Profile = ProfileFn(ProfEnv(Interact))

(***
  functor Debugger = RealDebugger 
***)
  structure Boot = struct val coreEnvRef = ref(Environment.emptyEnv) end
  val version = Version.version
  val banner = Version.banner
  val architecture = Machm.architecture
end
