(* Copyright 1989,1992 by AT&T Bell Laboratories *)

signature EVALLOOP =
sig
  exception Interrupt

  type envRef = { get: unit -> Environment.environment,
		  set: Environment.environment -> unit }
  type obj = System.Unsafe.object
  type obvec = obj Vector.vector
  type interactParams = {
			 compManagerHook:
			   (Ast.dec * envRef * envRef -> unit) option ref,
			 baseEnvRef: envRef,
			 localEnvRef: envRef,
			 transform: Absyn.dec -> Absyn.dec,
			 instrument: { source: Source.inputSource,
				       compenv: StaticEnv.staticEnv }
			             -> Absyn.dec -> Absyn.dec,
			 perform: (obvec -> obj) -> (obvec -> obj),
			 isolate: (obvec -> obj) -> (obvec -> obj),
			 printer: Environment.environment ->
			          PrettyPrint.ppstream -> Absyn.dec -> unit
			}

  val interact :  interactParams -> unit

  val eval_stream : interactParams -> string * instream -> unit

end

functor EvalLoopF(Compile: COMPILE) : EVALLOOP =
struct
  open ErrorMsg Environment CompUtil 
  structure T = Time

  exception Interrupt

  type envRef = { get: unit -> Environment.environment,
		  set: Environment.environment -> unit }
  type obj = System.Unsafe.object
  type obvec = obj Vector.vector

  type interactParams =
      {
       compManagerHook: (Ast.dec * envRef * envRef -> unit) option ref,
       baseEnvRef: envRef,
       localEnvRef: envRef,
       transform: Absyn.dec -> Absyn.dec,
       instrument: {source: Source.inputSource, compenv: StaticEnv.staticEnv}
                   -> Absyn.dec -> Absyn.dec,
       perform: (obvec -> obj) -> (obvec -> obj),
       isolate: (obvec -> obj) -> (obvec -> obj), 
       printer: Environment.environment -> PrettyPrint.ppstream -> Absyn.dec
                -> unit
      }

  structure U = System.Unsafe

  (* toplevel loop *)

  exception Eof

  fun codegen (arg as { lambda, errors }) =
      let val code = Compile.codegen arg
	  val _ = debugmsg "about to boot\n"
       in Compile.applyCode code
      end

  fun checkErrors({anyErrors,...} : Source.inputSource) = 
      if !anyErrors then raise Error else ()

  fun evalLoop ({compManagerHook,
		 baseEnvRef, localEnvRef, perform, isolate, printer,
		 instrument, transform} : interactParams)
               (source: Source.inputSource) : unit =
     let 

       val parser = Compile.parseOne source
	 
(* The baseEnv and localEnv are purposely refs so that a top-level command
  can re-assign either one of them, and the next iteration of the loop
  will see the new value. It's also important that the toplevelenv
  continuation NOT see the "fetched" environment, but only the ref;
  then, if the user "filters" the environment ref, a smaller image
  can be written. 
  *)

      fun oneUnit () = let
       (* perform one transaction  *)
	  val errors = ErrorMsg.errors source
	  val anyErrors = #anyErrors source

      in
	  case parser () of
	      NONE => raise Eof
	    | SOME ast => let

		  val _ = case !compManagerHook of
		      NONE => ()
		    | SOME cm => cm (ast, baseEnvRef, localEnvRef)

		  val { static = statenv,
		        dynamic = dynenv,
			symbolic = symenv } =
		      Environment.layerEnv (#get localEnvRef (),
					    #get baseEnvRef ())

		  val { absyn, newenv, exportLexp, exportPid, ... } =
		      Compile.elaborate{errors=errors,compenv=statenv,ast=ast,
					corenv= #get Environment.coreEnvRef (),
					transform=transform}
		      before checkErrors source
		      handle Compile.Compile _ => raise Error
		  val _ = Index.report source (absyn,statenv)

		  val absyn =
		      Compile.instrument
		        { compenv=statenv,
			  source=source,
			  corenv= #get Environment.coreEnvRef () }
			(instrument {compenv=statenv,source=source} absyn)

		  val { genLambda, imports } = 
		      Compile.translate{errors=errors,absyn=absyn,
					corenv= #get Environment.coreEnvRef (),
					exportLexp = exportLexp,
					exportPid = exportPid,
					statenv=StaticEnv.atop(newenv,statenv)}
		      before checkErrors source

		  val lambda = Compile.inline { genLambda = genLambda,
					        imports = imports,
						symenv = symenv }

		  val { lambda_e, lambda_i, pid = _ } =
		      Compile.split { lambda = lambda, enable = true }

		  val new_symenv = Compile.symDelta (exportPid, lambda_i)

		  val executable =
		      (if !Control.interp then let
			  val { l0, ln } = lambda_e
		      in
			   foldl (fn (l, r) => (Interp.interp l) o r)
			         (Interp.interp l0)
				 ln
		      end else
			   codegen { errors = errors, lambda = lambda_e })
			   before checkErrors source

		  val executable = isolate (perform executable)

		  val new_dynenv = let
		      val _ = U.Assembly.profCurrent := Profile.otherIndex
		      val result = Compile.execute
			  { executable = executable,
			    imports = imports,
			    exportPid = exportPid,
			    dynenv = dynenv }
		  in
		      U.Assembly.profCurrent := Profile.compileIndex;
		      result
		  end

		  val newEnv = Environment.concatEnv
		      ({ static = newenv,
			 dynamic = new_dynenv,
			 symbolic = new_symenv },
		       #get localEnvRef ())
	      (* refetch localEnvRef because execution may
	       have changed its contents *)

	      in PrettyPrint.with_pp (#errConsumer source)
		  (fn ppstrm =>
		   printer (Environment.layerEnv(newEnv,#get baseEnvRef ()))
		   ppstrm
		   absyn);
		  Index.report source (absyn, Environment.staticPart newEnv);
		  #set localEnvRef newEnv
	      end
	end


       fun loop() = (oneUnit(); loop())

       val oldcont = !U.topLevelCont

     in U.topLevelCont :=
        SMLofNJ.callcc(fn k => (
          SMLofNJ.callcc(fn k' => (SMLofNJ.throw k k'));
          raise Interrupt));
        loop() handle e => (U.topLevelCont := oldcont; raise e)
    end

  (* interactive loop, with error handling *)
  fun interact interactParams : unit =
      let val source = Source.newSource("std_in",1,std_in,true,
					ErrorMsg.defaultConsumer(),NONE);
	  fun flush() = (input(std_in,(can_input std_in)) handle Io _ => "";
			 #anyErrors source := false)
	  val say = Control.Print.say
	  fun loop () =
	      evalLoop interactParams source
	      handle Eof => ()
		   | Interrupt => (say "\nInterrupt\n"; 
				   flush(); loop())
		   | Error => (flush(); loop())
		   | Compile.Compile "syntax error" => (flush(); loop())
		   | exn => let
			fun showhist [s] = say(concat["  raised at: ", s, "\n"])
			  | showhist (s::r) = (
			      showhist r; say (concat["             ", s, "\n"]))
			  | showhist [] = ()
			val exnMsg = (case exn
			       of (Compile.Compile s) =>
				    concat["Compile: \"", s, "\""]
				| _ => General.exnMessage exn
			      (* end case *))
		        in
			  say (concat["\nuncaught exception ", exnMsg, "\n"]);
			  showhist (SMLofNJ.exnHistory exn);
			  flush(); 
			  loop()
			end
       in loop()
      end (* interact *)

  fun eval_stream interactParams (fname:string,stream:instream) : unit =
      let val interactive = is_term_in stream
	  val source = 
	      Source.newSource(fname,1,stream,interactive,
			       ErrorMsg.defaultConsumer(),
			       if not interactive 
			         then Index.openIndexFile fname
			         else NONE)
       in evalLoop interactParams source
	  handle exn =>
	    (Source.closeSource source;
	     case exn
	      of Eof => ()
	       | _ => raise exn)
      end (* eval_stream *)

end (* functor EvalLoop *)

