(* boot.sml
 *
 * Copyright 1992, 1995 by AT&T Bell Laboratories
 *)

signature BOOTENV = sig
    val makePervEnv: unit -> Environment.environment
end

functor BootEnvF (VC: VISCOMP): BOOTENV = struct

    structure Compile = VC.Compile

    open ErrorMsg Modules ModuleUtil Access

    infix //
    val op // = SCStaticEnv.atop

    val say = Control.Print.say
    val flush = Control.Print.flush

    type scsenv = SCStaticEnv.staticEnv
    type pid = PersStamps.persstamp
    type symenv = SymbolicEnv.symenv

    type loadres =
	{ scsenv: scsenv, exportPid: pid option, exportPids: pid list }

    fun load' env fname: loadres = let
	val _ = say (concat ["[Elaborating ", fname, "]\n"])
	val stream = open_in fname
	val source = Source.newSource (fname, 1, stream, false,
				       ErrorMsg.defaultConsumer (),
				       Index.openIndexFile fname)
	val ast = Compile.parse source
	val { absyn, newenv, exportPid, ...} = 
	    Compile.elaborate { errors = ErrorMsg.errors source,
			        compenv = SCStaticEnv.unSC env,
				corenv = #get Environment.coreEnvRef (),
				transform = (fn x=>x), ast=ast }
	val newenv = SCStaticEnv.SC newenv
	val catenv = SCStaticEnv.atop (newenv, env)
	fun show (Absyn.SEQdec decs) = app show decs
	  | show (Absyn.MARKdec (d,_)) = show d
	  | show absyn =
	    PrettyPrint.with_pp (ErrorMsg.defaultConsumer ())
	    (fn ppstrm =>
	     PPDec.ppDec { static = SCStaticEnv.unSC catenv,
			   dynamic = DynamicEnv.empty,
			   symbolic = SymbolicEnv.empty }
	     ppstrm absyn)
    in
	show absyn handle _ => say "ppDec raised exception\n";
	close_in stream;
	{ scsenv = newenv, exportPid = exportPid, exportPids = [] }
    end

    (* necessary? ...
    fun load env fname = let
	val (newenv, _) = load' env fname
    in
	newenv // env
    end
     * ... probably not *)

    datatype runDynEnv =
	NILrde 
      | CONSrde of string * System.Unsafe.object * runDynEnv

    val a_pstruct: runDynEnv ref =
	System.Unsafe.cast System.Unsafe.Assembly.pstruct

    val mkPid = PersStamps.stringToStamp

    infix %
    fun (f % g) e = let
	val e' = f e
	val e'' = g (e' // e)
    in
	e'' // e'
    end

    fun for2list n f = let
	fun g i = if i = n then nil else f i :: g(i+1)
    in
	g 0
    end

    fun readfile fname = let
	val f = open_in fname
	fun g () =
	    case input_line f of
		"" => nil 
	      | line => substring (line, 0, size line - 1) :: g ()
    in
	g () before close_in f
    end

    type compUnit = { (* imports: pid list, *)
		      exportPid: pid option,
		      exportPids: pid list,
		      symenv: symenv,
		      env: PickleEnv.pickledEnv
		    }

    fun targetRead (s: BinIO.instream): compUnit = let
	val << = Word.<<
	infix 4 <<
	val || = Word.orb
	infix 3 ||
	fun getInt () = let
	    val [b3, b2, b1, b0] =
		explode (Byte.bytesToString (BinIO.inputN (s, 4)))
	    fun byte (b, sh) = (Word.fromInt (Char.ord b)) << sh
	in
	  Word.toInt(
	    byte(b3, 0w24) || byte(b2, 0w16) || byte(b1, 0w8) || byte(b0, 0w0))
	end
	val magic = Byte.bytesToString (BinIO.inputN (s, 16))
	(* val _ = if magic = MAGIC then () else raise Compile "bad magic number"*)
	val numImports = getInt ()
	val numExports = getInt ()
	val area1 = getInt ()
	val area2 = getInt ()
	val area3 = getInt ()
	val area4 = getInt ()
	val codeSize = getInt ()
	val envSize = getInt ()
	fun mkPid x = PersStamps.stringToStamp (Byte.bytesToString x)

	fun remaining s = let
	    val last = Position.toInt (BinIO.endPosIn s)
	    val here = Position.toInt (BinIO.getPosIn s)
	    val d = last - here
(*	      val s' = BinIO.getInstream s
	      val (v,_) =BinIO.StreamIO.inputN(s',d)
	      fun showo x = makestring((x div 64)*100 + (x div 8 mod 8)*10+x mod 8)
              fun f i = if i=104 then () else (say (showo(Word8.wordToInt(Word8Vector.sub(v,i))));
					       say " ";
					       f(i+1))
              val _ = (f 0; say "\n")
              val len = String.size(System.Unsafe.cast v) *)
	in
	    (*say ("codesize= " ^ makestring codeSize ^
		   "here = " ^ makestring here ^
		   "   last = " ^ makestring last ^
		   "   d = " ^ makestring d ^ 
		   "   len = " ^ makestring len ^ "\n");*)
	    d
	end

	val bytesPerPid = 16
	fun readPid _ = mkPid (BinIO.inputN (s, bytesPerPid))
	fun blastRead (s, n) = System.Unsafe.blastRead (BinIO.inputN (s, n))

	val imports = for2list numImports readPid

	val (exportPid, exportPids) =
	    if numExports = 0 then (NONE, [])
	    else if numExports = 1 then
		let val p = readPid () in (SOME p, [p]) end
	    else (print "### old-style bin file!\n"; (*raise Match *)
		  (NONE, for2list numExports readPid))

	(* skip area1 *)
	val _ = BinIO.inputN (s, area1)

	(* read inlinable lambda expression, which lives in area2 *)
	val lambda_i =
	    if area2 = 0 then NONE
	    else SOME (blastRead (s, area2))

	(* skip areas 3 & 4 & code *)
	val _ = BinIO.inputN (s, area3 + area4 + codeSize)

	(* finally, read the pickled static env *)
	val remain = remaining s
	(* val _ = if envSize = remain then () else raise Match *)
	val env = blastRead (s, remain)

	val symenv = Compile.symDelta (exportPid, lambda_i)
    in
	{ (* imports = imports, *)
	  exportPid = exportPid,
	  exportPids = exportPids,
	  symenv = symenv,
	  env = env }
    end

    (* read a file from the bin directory *)
    fun readBinFile bindir file = let
	val path = OS.Path.joinDirFile { dir = bindir, file = file }
    in
	readfile path
    end

    (* some standard pathnames (in OS independent syntax) *)
    local
	fun bootFile f = OS.Path.joinDirFile { dir = "boot", file = f }
    in
	val assembly_sig = bootFile "assembly.sig"
	val dummy_sml = bootFile "dummy.sml"
	val core_sml = bootFile "core.sml"
	val pervasives_sml = bootFile "pervasives.sml"
    end (* local *)

    fun newBootEnv (load', bindir) = let
	val files = readBinFile bindir "BOOTSRC"
	val prim = SCStaticEnv.SC Prim.primEnv
	val pids = ref (nil : pid list)
	fun ld fname env = let
	    val { scsenv = env, exportPid = p, exportPids } = load' env fname
	    val _ = pids := foldl (op ::) (!pids) exportPids
		(* case p of
		    NONE => ()
		  | SOME p => pids := p :: (!pids) *)
	in
	    env
	end
	fun many [s] = ld s
	  | many (a::rest) = ld a % many rest
	val asig_env = ld assembly_sig prim
	val sig_prim = asig_env // prim
	val dummy_env = ld dummy_sml sig_prim // sig_prim
	val core_env = ld core_sml dummy_env 
	val _ = #set Environment.coreEnvRef (SCStaticEnv.unSC core_env)
	val _ = VC.Boot.coreEnvRef := { static = core_env // dummy_env,
				        dynamic = DynamicEnv.empty,
					symbolic = SymbolicEnv.empty }
	val env = many files (core_env // sig_prim)
	(* for the time being, we need to support building from both the
	 * PERVSRC list file, and from pervasives.sml.
	 *)
	val resultEnv = let
	    val pervFiles = readBinFile bindir "PERVSRC"
	in
	    many pervFiles env
	end handle _ => ld pervasives_sml env
    in
	(resultEnv, rev (!pids))
    end


    fun sname "mipsel"   = "MipsLittle"
      | sname "mipseb"   = "MipsBig"
      | sname "vax"      = "Vax"
      | sname "m68"      = "M68"
      | sname "sparc"    = "Sparc"
      | sname "hppa"     = "Hppa"
      | sname "rs6000"   = "RS6000"
      | sname "x86"      = "X86"
      | sname "alpha32"  = "Alpha32"
      | sname "bytecode" = "ByteCode"
      | sname a = (say ("Don't Recognize architecture "^a^"\n");
		   raise Match)

    fun ends_with(ab,b) = let
	val abs = size ab and bs = size b
    in
	abs >= bs andalso substring (ab, abs - bs, bs) = b
    end

    fun getVisComp (env0, bindir) getbin = let
	val srcname = Compile.architecture ^ "vis.sml"
	val files = readBinFile bindir "SRCLIST"
	fun f (env, fname :: rest) = let
	    val { scsenv = env', exportPid, exportPids } = getbin env fname
	    val env'' = env' // env
	in
	    if ends_with (fname, srcname) then env'' else f (env'', rest)
	end
    in
	f (env0, files)
    end

    fun elabCompiler (initialEnv, bindir) = let
	val srclist = readBinFile bindir "SRCLIST"
	(* don't elaborate the last file! it's the glue that hasn't
	 * finished executing.
	 *)
	fun allFiles (oldenv, pids, fname :: (rest as _ :: _)) =
	    let
		val { scsenv = newenv, exportPid = newpid, exportPids } =
		    load' oldenv fname
		val pids = case newpid of
		    NONE => pids
		  | SOME p => pids @ [p]
	    in
		allFiles (newenv // oldenv, pids, rest)
	    end
	  | allFiles (oldenv, pids, _) = (oldenv, pids)

    in
	allFiles (initialEnv, [], srclist)
    end handle ex => (say (concat
			   ["\nuncaught exception" ,
			    General.exnMessage ex , "\n"]);
		      flush ();
		      raise ex)

    val bindir = ref ("bin." ^ Compile.architecture)
    val full = ref false

    structure SS = Substring
    val _ = let
	fun bootArg s = let
	    val (s1, s2) = SS.position "@SMLboot=" (SS.all s)
	in
	    if SS.isEmpty s1
	      then SOME (SS.string (SS.triml 9 s2))
	      else NONE
	end
	fun f [] = ()
	  | f ("@SMLfull" :: rest) = (full := true; f rest)
	  | f (arg :: rest) =
	    (case bootArg arg of
		 SOME fname => bindir := fname
	       | NONE => ();
	     f rest)
    in
	f (SMLofNJ.getAllArgs ())
    end

    fun basename s = #file(OS.Path.splitDirFile s)

    fun targetNamer bindir s =
	OS.Path.joinDirFile
	  { dir = bindir,
	    file = OS.Path.joinBaseExt { base= basename s, ext = SOME "bin" } }
  
    fun makePervEnv () = let

	val tnamer = targetNamer (!bindir)

	val theSymEnv = ref SymbolicEnv.empty

	fun getbin (env0: scsenv) sourcename = let
	    val _ =
		say (concat ["Loading static bin for ", sourcename, "\n"])
	    val f = BinIO.openIn (tnamer sourcename)
	    val { exportPid, exportPids, env, symenv } = targetRead f
	    val _ = theSymEnv := SymbolicEnv.atop (symenv, !theSymEnv)
	    val env = PickleEnv.unPickleEnv { env = env, context = env0 }
	in
	    BinIO.closeIn f;
	    { scsenv = SCStaticEnv.SC env,
	      exportPid = exportPid,
	      exportPids = exportPids }
	end

	val ((pervStatEnv, pids), visCompEnv) = 
	    if List.exists (fn s => s="@SMLelab") (SMLofNJ.getAllArgs()) then
	        let
		    val _ = say "\nNow elaborating boot directory\n"
		    val (pSE, pids) = newBootEnv (load', !bindir)
		    val (vSE, morepids) = elabCompiler (pSE, !bindir)
		in
		    ((pSE, pids @ morepids), vSE)
		end
	    else
		let
		    val _ = say "trying bin files\n"
		    val (pSE, pids) = newBootEnv (getbin, !bindir)
		in
		    ((pSE, pids), getVisComp (pSE, !bindir) getbin)
		end
		    
	val pervStatEnv = SCStaticEnv.unSC pervStatEnv
	val visCompEnv = SCStaticEnv.unSC visCompEnv
	val vcSym = Symbol.strSymbol (sname (Compile.architecture) ^ "VisComp")
	val Modules.STRbind(Modules.STRvar{name,access,binding}) = 
	    StaticEnv.look(visCompEnv,vcSym)
	val compSym = Symbol.strSymbol "Compiler"
	val pervStatEnv = 
	    if !full then StaticEnv.atop(visCompEnv,pervStatEnv)
	    else pervStatEnv
	val pervStatEnv =
	    StaticEnv.bind (compSym,
			    Modules.STRbind (Modules.STRvar
					      { name = compSym,
					        access = access,
						binding = binding }),
			    pervStatEnv)

	(* translate run-time system's dynamic env into compiler's dynamic env.
	 * `m' is the map from pids to inlinable lambda expressions. *)
	fun trans_rde NILrde = DynamicEnv.empty
	  | trans_rde (CONSrde (spid, obj, rest)) = let
		val pid = mkPid spid
	    in
		DynamicEnv.bind (pid, obj, trans_rde rest)
	    end

	fun rebindlast (NILrde, pids, env) = (pids, env)
	  | rebindlast (CONSrde (_, a, rde), pids, env) =
	    case rebindlast (rde, pids, env) of
		(pid :: pids', env') => let
		    val _ = (DynamicEnv.look env' pid; ())
			handle DynamicEnv.Unbound => say "%%%% new pid\n"
		    val env'' = DynamicEnv.bind (pid, a, env')
		in
		    case rde of
			CONSrde (_, _, NILrde) =>
			    (* hack for testing new pervasive modules *)
			    VC.Boot.coreEnvRef:= 
			    { static = #static (!VC.Boot.coreEnvRef),
			      dynamic = env'',
			      symbolic = SymbolicEnv.empty }
		      | _ => ();
		    (pids', env'')
		end
	      | z as (nil, env') => z

	val ps = !a_pstruct before a_pstruct := NILrde
	(* val (nil,env) = rebindlast(ps, pids, trans_rde (m, ps)) *)
	val ([], env) = rebindlast (ps, pids, trans_rde ps)

	(* (* hack for testing new pervasive modules *)
        val _ = VC.Boot.coreEnvRef :=
	    { static = #static (!VC.Boot.coreEnvRef),
	      dynamic = env,
	      symbolic = !theSymEnv } *)
	  
    in
	say "Using runtime's dynEnv\n";
	{ static = pervStatEnv, dynamic = env, symbolic = !theSymEnv }
    end handle e  => (say "\nuncaught exception ";
		      say (General.exnMessage e);
		      say "\n";
		      raise e)

end (* structure BootEnv *)
