(* word8.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Word8 : WORD =
  struct

    structure W8 = InlineT.Word8
    structure W31 = InlineT.Word31

    type word = word8  (* 31 bits *)

    val wordSize = 8
    val wordSizeW = 0w8
    val wordShift = InlineT.Word31.-(0w31, wordSizeW)
    fun adapt oper args = W8.andb(oper args, 0wxFF)

    val toLargeWord   : word -> LargestWord.word = W8.toword32
(** NOTE: the following should be a primop **)
    fun toLargeWordX w = toLargeWord(W8.rshift(W8.lshift(w, wordShift), wordShift))
    val fromLargeWord = adapt W8.fromword32

    val toLargeInt   : word -> LargestInt.int = W8.toint
    fun toLargeIntX w = toLargeInt(W8.rshift(W8.lshift(w, wordShift), wordShift))
    val fromLargeInt : LargestInt.int -> word = adapt W8.fromint

    val toInt   : word -> int = W8.toint
    fun toIntX w = toInt(W8.rshift(W8.lshift(w, wordShift), wordShift))
    val fromInt : int -> word = adapt W8.fromint

  (** These should be inline functions **)
    fun << (w : word, k) = if (InlineT.DfltWord.<=(wordSizeW, k))
	  then 0w0
	  else adapt W8.lshift (w, k)
    fun >> (w : word, k) = if (InlineT.DfltWord.<=(wordSizeW, k))
	  then 0w0
	  else W8.rshiftl(w, k)
    fun ~>> (w : word, k) = if (InlineT.DfltWord.<=(wordSizeW, k))
	  then adapt W8.rshift (W8.lshift(w, wordShift), 0w31)
	  else adapt W8.rshift
	    (W8.lshift(w, wordShift), InlineT.DfltWord.+(wordShift, k))

    val orb  : word * word -> word = W8.orb
    val xorb : word * word -> word = W8.xorb
    val andb : word * word -> word = W8.andb
    val notb : word -> word = adapt W8.notb

    val op * : word * word -> word = adapt W8.*
  (* NOTE: waste of time to check for overflow; really should use Word31
   * arithmetic operators; when those get implemented, switch over.
   *)
    val op + : word * word -> word = adapt W8.+
    val op - : word * word -> word = adapt W8.-
    val op div : word * word -> word = W8.div
    fun op mod (a:word, b:word):word = a-(a div b)*b

    val op > : word * word -> bool = W8.>
    val op >= : word * word -> bool = W8.>=
    val op < : word * word -> bool = W8.<
    val op <= : word * word -> bool = W8.<=

    fun w8_to_w32 (w : word8) = InlineT.Word32.fromint(W8.toint w)
    fun w32_to_w8 (w : word32) = W8.fromint(InlineT.Word32.toint w)

    fun fmt radix = (NumFormat.fmtWord radix) o w8_to_w32
    val toString = fmt StringCvt.HEX

    fun scan radix = let
	  val scanLargest = NumScan.scanWord radix
	  fun scan getc cs = (case (scanLargest getc cs)
		 of NONE => NONE
		  | (SOME(w, cs')) => if InlineT.Word32.>(w, 0w255)
		      then raise Overflow
		      else SOME(w32_to_w8 w, cs')
		(* end case *))
	  in
	    scan
	  end
    val fromString = PreBasis.scanString (scan StringCvt.HEX)

  end  (* structure Word8 *)

(* to avoid losing inlining *)
structure Word8 =
  struct
    open Word8
    val toLargeWord   : word -> LargestWord.word = InlineT.Word8.toword32
    val toLargeInt   : word -> LargestInt.int = InlineT.Word8.toint
    val toInt   : word -> int = InlineT.Word8.toint
    val orb  : word * word -> word = InlineT.Word8.orb
    val xorb : word * word -> word = InlineT.Word8.xorb
    val andb : word * word -> word = InlineT.Word8.andb
    val op div : word * word -> word = InlineT.Word8.div
    val op > : word * word -> bool = InlineT.Word8.>
    val op >= : word * word -> bool = InlineT.Word8.>=
    val op < : word * word -> bool = InlineT.Word8.<
    val op <= : word * word -> bool = InlineT.Word8.<=
  end;

