(* word8vector.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Word8Vector : MONO_VECTOR =
  struct

    val (op <)  = InlineT.DfltInt.<
    val (op >=) = InlineT.DfltInt.>=
    val (op +)  = InlineT.DfltInt.+

  (* unchecked access operations *)
    val unsafeUpdate = InlineT.Word8Vector.update
    val unsafeSub = InlineT.Word8Vector.sub

    type vector = CharVector.vector
    type elem = Word8.word

    val maxLen = Core.max_length

    val fromList : elem list -> vector
	  = InlineT.cast CharVector.fromList
    val tabulate : (int * (int -> elem)) -> vector
	  = InlineT.cast CharVector.tabulate

    val length   = InlineT.Word8Vector.length
    val sub      = InlineT.Word8Vector.chkSub
    val extract : (vector * int * int option) -> vector
	  = InlineT.cast CharVector.extract
    val concat : vector list -> vector
	  = InlineT.cast CharVector.concat

    fun app f arr = let
	  val len = length arr
	  fun app i = if (i < len)
		then (f (unsafeSub(arr, i)); app(i+1))
		else ()
	  in
	    app 0
	  end

    fun foldl f init arr = let
	  val len = length arr
	  fun fold (i, accum) = if (i < len)
		then fold (i+1, f (unsafeSub(arr, i), accum))
		else accum
	  in
	    fold (0, init)
	  end

    fun foldr f init arr = let
	  fun fold (i, accum) = if (i >= 0)
		then fold (i-1, f (unsafeSub(arr, i), accum))
		else accum
	  in
	    fold (length arr - 1, init)
	  end

    fun chkSlice (arr, i, NONE) = let val len = length arr
	  in
	    if (InlineT.DfltInt.ltu(len, i))
	      then raise Subscript
	      else (arr, i, len)
	  end
      | chkSlice (arr, i, SOME n) = let val len = length arr
	  in
	    if ((0 <= i) andalso (0 <= n) andalso (i+n <= len))
	      then (arr, i, i+n)
	      else raise Subscript
	  end

    fun appi f slice = let
	  val (arr, start, stop) = chkSlice slice
	  fun app i = if (i < stop)
		then (f (i, unsafeSub(arr, i)); app(i+1))
		else ()
	  in
	    app start
	  end

    fun foldli f init slice = let
	  val (arr, start, stop) = chkSlice slice
	  fun fold (i, accum) = if (i < stop)
		then fold (i+1, f (i, unsafeSub(arr, i), accum))
		else accum
	  in
	    fold (start, init)
	  end

    fun foldri f init slice = let
	  val (arr, start, stop) = chkSlice slice
	  fun fold (i, accum) = if (i >= start)
		then fold (i-1, f (i, unsafeSub(arr, i), accum))
		else accum
	  in
	    fold (stop - 1, init)
	  end

  end

(* rebind the structure to preserve inlining *)
structure Word8Vector =
  struct
    open Word8Vector
    val length   = InlineT.Word8Vector.length
    val sub      = InlineT.Word8Vector.chkSub
  end
