(* word32.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Word32 : WORD =
  struct
    infix 7 * div mod
    infix 6 + -
    infix 4 > < >= <=

    structure W32 = InlineT.Word32

    type word = word32

    val wordSize = 32

    val toLargeWord   : word -> LargestWord.word = (fn x => x)
    val toLargeWordX  : word -> LargestWord.word = (fn x => x)
    val fromLargeWord : LargestWord.word -> word = (fn x => x)

    val toLargeInt   : word -> LargestInt.int = W32.toint
    val toLargeIntX  : word -> LargestInt.int = W32.toint
    val fromLargeInt : LargestInt.int -> word = W32.fromint

    val toInt   : word -> int = W32.toint
    val toIntX  : word -> int = W32.toint
    val fromInt : int -> word = W32.fromint

    val orb  : word * word -> word = W32.orb
    val xorb : word * word -> word = W32.xorb
    val andb : word * word -> word = W32.andb
    val notb : word -> word = W32.notb

  (** These should be inline functions **)
 (*   fun << (w : word, k) = if (InlineT.DfltWord.<=(0w32, k))
	  then 0w0
	  else W32.lshift(w, k)
    fun >> (w : word, k) = if (InlineT.DfltWord.<=(0w32, k))
	  then 0w0
	  else W32.rshiftl(w, k) *)

    val << : word * Word.word -> word = W32.lshift
    val >> : word * Word.word -> word = W32.rshiftl
    fun ~>> (w : word, k) = if (InlineT.DfltWord.<=(0w32, k))
	  then W32.rshift(w, 0w31)
	  else W32.rshift(w, k)

    val op * : word * word -> word = W32.*
    val op + : word * word -> word = W32.+
    val op - : word * word -> word = W32.-
    val op div : word * word -> word = W32.div
    fun op mod(a:word,b:word):word = a-(a div b)*b

    val op > : word * word -> bool = W32.>
    val op >= : word * word -> bool = W32.>=
    val op < : word * word -> bool = W32.<
    val op <= : word * word -> bool = W32.<=

    val fmt = NumFormat.fmtWord
    val toString = fmt StringCvt.HEX

    val scan = NumScan.scanWord
    val fromString = PreBasis.scanString (scan StringCvt.HEX)

  end  (* structure Word32 *)

(* to avoid losing inlining *)
structure Word32 =
  struct
    open Word32
    val toLargeWord   : word -> LargestWord.word = (fn x => x) (* INLINE? *)
    val toLargeWordX  : word -> LargestWord.word = (fn x => x)
    val fromLargeWord : LargestWord.word -> word = (fn x => x)
    val toLargeInt   : word -> LargestInt.int = InlineT.Word32.toint
    val toLargeIntX  : word -> LargestInt.int = InlineT.Word32.toint
    val fromLargeInt : LargestInt.int -> word = InlineT.Word32.fromint
    val toInt   : word -> int = InlineT.Word32.toint
    val toIntX  : word -> int = InlineT.Word32.toint
    val fromInt : int -> word = InlineT.Word32.fromint
    val notb : word -> word =InlineT.Word32.notb
    val orb  : word * word -> word = InlineT.Word32.orb
    val xorb : word * word -> word = InlineT.Word32.xorb
    val andb : word * word -> word = InlineT.Word32.andb
    val << : word * Word.word -> word = InlineT.Word32.lshift
    val >> : word * Word.word -> word = InlineT.Word32.rshiftl
    val op * : word * word -> word = InlineT.Word32.*
    val op + : word * word -> word = InlineT.Word32.+
    val op - : word * word -> word = InlineT.Word32.-
    val op div : word * word -> word = InlineT.Word32.div
    val op > : word * word -> bool = InlineT.Word32.>
    val op >= : word * word -> bool = InlineT.Word32.>=
    val op < : word * word -> bool = InlineT.Word32.<
    val op <= : word * word -> bool = InlineT.Word32.<=
  end;

