(* system.sml
 *
 * COPYRIGHT (c) 1994 AT&T Bell Laboratories.
 *)

(* no signature because of inlining *)
structure System = 
struct
  open Overloads
  structure Runtime : RUNTIMECONTROL = 
    struct
      val gcCtl : ((string * int ref) list -> unit) =
	    CInterface.c_function "SMLNJ-RunT" "gcControl"
      fun gc n = gcCtl [("DoGC", ref n)]
    end

  structure Tags : TAGS = Tags

  structure Poll : POLL = 
    struct
      exception BadPollFreq

      val defaultHandler = fn k => k

      val _ = Assembly.pollHandler := defaultHandler
      val handlerValid = ref false

      val pollEvent = Assembly.pollEvent
      val pollFreq = Assembly.pollFreq

      fun setHandler NONE = (Assembly.pollHandler := defaultHandler;
			     handlerValid := false)
	| setHandler (SOME h) = (Assembly.pollHandler := h;
				 handlerValid := true)

      fun inqHandler () = if !handlerValid then SOME (!Assembly.pollHandler)
			  else NONE

      fun setFreq NONE = pollFreq := 0
	| setFreq (SOME x) = if x <= 0 then raise BadPollFreq
			     else pollFreq := x

      fun inqFreq () = let val x = !pollFreq
		       in
			   if x = 0 then NONE
			   else SOME x
		       end
    end (* structure Poll *)

  structure CleanUp : CLEAN_UP = CleanUp
  structure Weak : WEAK = Weak

  structure Unsafe (* no signature, inlining *) =
  struct
    structure Assembly : ASSEMBLY = Assembly
    structure CInterface : CINTERFACE = CInterface
    structure Susp : SUSP = Susp
    structure PolyCont =
      struct
	type 'a cont = 'a cont
	type 'a control_cont = 'a InlineT.control_cont
	val callcc = InlineT.unsafecallcc
	val throw = InlineT.throw
	val capture = InlineT.unsafecapture
	val escape = InlineT.escape
      end
    type object = Assembly.object
    type instream = IO.instream
    type outstream = IO.outstream
    val boxed : 'a -> bool = InlineT.boxed
    val ordof : string * int -> char = InlineT.CharVector.sub
    val slength : string -> int = InlineT.CharVector.length
    val objLength : 'a -> int = InlineT.objlength
    val getObjTag : 'a -> int = InlineT.gettag
    val special : (int * 'a) -> 'b = InlineT.mkspecial
    val setSpecial : ('a * int) -> unit = InlineT.setspecial
    val getSpecial : 'a -> int = InlineT.getspecial
    val store : string * int * char -> unit = InlineT.CharVector.update
    val subscript : 'a Array.array * int -> 'a = InlineT.PolyArray.sub
    val update : 'a Array.array * int * 'a -> unit = InlineT.PolyArray.update
    val subscriptv : 'a Vector.vector * int -> 'a = InlineT.PolyVector.sub
    val subscriptf : RealArray.realarray * int -> real = InlineT.Float64Array.sub
    val updatef : RealArray.realarray * int * real -> unit = InlineT.Float64Array.update
    val getvar : unit -> 'a = InlineT.getvar
    val setvar : 'a -> unit = InlineT.setvar
    val getpseudo : int -> 'a = InlineT.getpseudo
    val setpseudo : 'a * int -> unit = InlineT.setpseudo
    val setmark : 'a -> unit = InlineT.setmark
    val dispose : 'a -> unit = InlineT.dispose
    val gethdlr : unit -> 'a cont = InlineT.gethdlr
    val sethdlr : 'a cont -> unit = InlineT.sethdlr
    val cast = InlineT.cast
    val blastWrite = Blast.blastWrite
    val blastRead = Blast.blastRead
    val create_s = Assembly.A.create_s
    val create_b = Assembly.A.create_b
    val store_s : string * int * char -> unit = InlineT.CharVector.update
    local
      exception UNDEFINED
      val defaultCont : unit cont =
	    InlineT.callcc (fn k1 => (
	      InlineT.callcc (fn k2 => (InlineT.throw k1 k2));
	      raise UNDEFINED))
    in
      val topLevelCont = ref defaultCont
    end (* local *)
    local
	datatype A = unboxed | boxed of object
	val cast = InlineT.cast
    in  exception Boxity
	val tuple : object -> object vector
		    = cast(fn unboxed => raise Boxity
			    | x as boxed _ => x)
	val string : object -> string = cast (fn x=>x)
	val real : object -> real = cast (fn x=>x)
	val int : object -> int
		    = cast(fn x as unboxed => x
			    | boxed _ => raise Boxity)
	val word : object -> word = cast (fn x=>x)
	val word8 : object -> word8 = cast (fn x=>x)
	val word32 : object -> word32 = cast (fn x=>x)
    end (* local datatype A ... *)

    val profiling = ref false
    val sprofiling = ref false
    val profile_register : (string -> int * int array * int ref) ref =
       Core.profile_register
   val profile_sregister : (Assembly.object * string -> Assembly.object) ref = 
       Core.profile_sregister

  end (* Unsafe *)

  val runtimeStamp = ref ""
  val errorMatch =  Core.errorMatch

  val interactive = ref true

end
