(* unix.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Unix : UNIX =
  struct

    structure P = Posix.Process
    structure PE = Posix.ProcEnv
    structure PF = Posix.FileSys
    structure IO = Posix.IO
    structure SS = Substring

    fun protect f x = let
          val _ = Signals.maskSignals Signals.MASKALL
          val y = (f x) handle ex => 
                    (Signals.unmaskSignals Signals.MASKALL; raise ex)
          in
            Signals.unmaskSignals Signals.MASKALL; y
          end

    fun fdReader (name : string, fd : IO.file_desc) =
	  PosixTextPrimIO.mkReader {
              initBlkMode = true,
              initPos = Position.fromInt 0,
              name = name,
              fd = fd
            }

    fun fdWriter (name, fd) =
          PosixTextPrimIO.mkWriter {
	      appendMode = false,
              initBlkMode = true,
              name = name,
              lineBuf=NONE,
              chunkSize=2048,  (* see os-unix/posix-bin-prim-io.sml *)
              fd = fd
            }

    val openOutFD = TextIO.mkOutstream o TextIO.StreamIO.mkOutstream o fdWriter
    val openInFD = TextIO.mkInstream o TextIO.StreamIO.mkInstream o fdReader

    datatype proc = PROC of {
        pid : P.pid,
        ins : TextIO.instream,
        outs : TextIO.outstream
      }

    fun executeInEnv (cmd, argv, env) = let
          val p1 = IO.pipe ()
          val p2 = IO.pipe ()
          fun closep () = (
                IO.close (#outfd p1); 
                IO.close (#infd p1);
                IO.close (#outfd p2); 
                IO.close (#infd p2)
              )
          val base = SS.string(SS.taker (fn c => c <> #"/") (SS.all cmd))
          fun startChild () =
                case protect P.fork () of
                  SOME pid =>  pid           (* parent *)
                | NONE => let
                    val oldin = #infd p1
                    val newin = Posix.FileSys.wordToFD 0w0
                    val oldout = #outfd p2
                    val newout = Posix.FileSys.wordToFD 0w1
                    in
                      if oldin = newin then ()
                      else (
                        IO.dup2{old = oldin, new = newin};
                        IO.close oldin
                      );
                      if oldout = newout then ()
                      else (
                        IO.dup2{old = oldout, new = newout};
                        IO.close oldout
                      );
                      P.exece (cmd, base::argv, env)
                    end
          val _ = TextIO.flushOut TextIO.stdOut
          val pid = (startChild ()) handle ex => (closep(); raise ex)
          val ins = openInFD (base^"_exec_in", #infd p2)
          val outs = openOutFD (base^"_exec_out", #outfd p1)
          in
              (* close the child-side fds *)
            IO.close (#outfd p2);
            IO.close (#infd p1);
              (* set the fds close on exec *)
            IO.setfd (#infd p2, IO.FD.flags [IO.FD.cloexec]);
            IO.setfd (#outfd p1, IO.FD.flags [IO.FD.cloexec]);
            PROC {
              pid = pid,
              ins = ins,
              outs = outs
            }
          end

    fun execute (cmd, argv) = executeInEnv (cmd, argv, PE.environ())

    fun streamsOf (PROC{ins,outs,...}) = (ins, outs)

    fun kill (PROC{pid,...},signal) = P.kill (P.K_PROC pid, signal)

    fun reap (PROC{pid,ins,outs}) = let
        (* protect is probably too much; typically, one
         * would only mask SIGINT, SIGQUIT and SIGHUP
         *)
          fun waitProc () = #2(protect P.waitpid (P.W_CHILD pid,[]))
          in
            TextIO.closeIn ins;
            TextIO.closeOut outs;
            waitProc ()
          end

  end (* structure Unix *)
