(* posix-prim-io.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * This implements the UNIX version of the OS specific text primitive
 * IO structure.  It is implemented by a trivial translation of the
 * binary operations (see posix-bin-prim-io.sml).
 *)

structure PosixTextPrimIO : sig

    include OS_PRIM_IO

    val stdIn  : unit -> PrimIO.reader
    val stdOut : unit -> PrimIO.writer
    val stdErr : unit -> PrimIO.writer

  end = struct

    structure PF = Posix.FileSys
    structure BinPrimIO = PosixBinPrimIO
    structure PrimIO = TextPrimIO

    type file_desc = PF.file_desc

    val toFPI = Position.fromInt  (* temporary *)

  (* The buffer size is chosen as the largest string allocated in the
   * nursery.
   *)
    val bufferSzB = 2048

(* A clean version of translateIn, but slow:
 fun translateIn (BinPrimIO.Rd{readBlock, readNoBlock, readaBlock,
			       readaNoBlock, block, canInput, name,
			       chunkSize, close, getPos, findPos,
			       setPos, endPos}) =
   let val transv : Word8Vector.vector -> CharVector.vector = Byte.string

       fun opt f (SOME x) = SOME(f x)
	 | opt f NONE = NONE

    in TextPrimIO.Rd
        {readBlock=opt (fn f => fn i => transv(f i)) readBlock, 
	 readNoBlock=opt (fn f => fn x=> opt transv (f x)) readNoBlock,
	 readaBlock=NONE,
	 readaNoBlock=NONE,
	 block=block, 
	 canInput=canInput, 
	 name=name,
	 chunkSize=chunkSize,
	 close=close,
	 getPos=getPos, 
	 findPos=findPos,
	 setPos=setPos, 
	 endPos=endPos}
   end

*)


 (* If Char.char is really Word8.word, then very efficient versions of
  * translateIn and translateOut are possible:
  *)
    val translateIn : BinPrimIO.PrimIO.reader -> PrimIO.reader = InLine.cast
    val translateOut : BinPrimIO.PrimIO.writer -> PrimIO.writer = InLine.cast

    fun openRd fname = translateIn(BinPrimIO.openRd fname)
    fun openWr fname = translateOut(BinPrimIO.openWr fname)
    fun openApp fname = translateOut(BinPrimIO.openApp fname)

    fun mkReader args = translateIn(BinPrimIO.mkReader args)
    fun mkWriter args = translateOut(BinPrimIO.mkWriter(InLine.cast args))
	(* we need the case because of the lineBuf argument *)

    fun stdIn () = mkReader{
	    fd = PF.stdin,
	    name = "<stdIn>",
	    initPos = toFPI 0,
	    initBlkMode = true (* Bug!  Should check! *)
	  }

    fun stdOut() = mkWriter{
	    fd= PF.stdout,
	    name="<stdOut>",
	    initBlkMode=true (* Bug!  Should check! *),
	    appendMode=false (* Bug!  Should check! *),
	    lineBuf=SOME(fn 10 => true | _ => false),
	    chunkSize=bufferSzB
	  }

    fun stdErr() = mkWriter{
	    fd		= PF.stderr,
	    name	= "<stdErr>",
	    initBlkMode	= true, (* Bug!  Should check! *)
	    appendMode	= false, (* Bug!  Should check! *)
	    lineBuf	= NONE,
	    chunkSize	= 1	(* don't want to buffer this streams!! *)
	  }

  end; (* PosixPrimIO *)

