(* os-io.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * NOTE: this interface has been proposed, but not yet adopted by the
 * Standard basis committee.
 *)

structure OS_IO : OS_IO =
  struct

  (* an io_desc is an abstract descriptor for an OS object that
   * supports I/O (e.g., file, tty device, socket, ...).
   *)
    type io_desc = OS.IO.io_desc

  (* return an integer ID for the I/O descriptor; note that IDs may
   * be reused during an execution, but that no two active I/O
   * devices will have the same ID.
   *)
    fun id (OS.IO.IODesc fd) = fd

  (* return the kind of I/O descriptor; values include: "FILE",
   * "PIPE", "SOCK", and "TTY".
   *)
    fun kind (OS.IO.IODesc fd) = let
	  val fd = Posix.FileSys.wordToFD(SysWord.fromInt fd)
	  val ftype = Posix.FileSys.ST.fileType(Posix.FileSys.fstat fd)
	  in
	    if      (Posix.FileSys.isReg ftype) then "FILE"
	    else if (Posix.FileSys.isDir ftype) then "DIR"
	    else if (Posix.FileSys.isChr ftype) then "TTY"
	    else if (Posix.FileSys.isBlk ftype) then "??"
	    else if (Posix.FileSys.isLink ftype) then "LINK"
	    else if (Posix.FileSys.isFIFO ftype) then "PIPE"
	    else if (Posix.FileSys.isSock ftype) then "SOCK"
	    else "UNKNOWN"
	  end

    type poll_flags = {rd : bool, wr : bool, err : bool}
    datatype poll_desc = PollDesc of (io_desc * poll_flags)
    datatype poll_info = PollInfo of (io_desc * poll_flags)

  (* create a polling operation on the given descriptor; note that
   * not all I/O devices support polling, but for the time being, we
   * don't test for this.
   *)
    fun pollDesc iod = SOME(PollDesc(iod, {rd=false, wr=false, err=false}))

  (* return the I/O descriptor that is being polled *)
    fun pollToIODesc (PollDesc(iod, _)) = iod

    exception Poll

  (* set polling events; if the polling operation is not appropriate
   * for the underlying I/O device, then the Poll exception is raised.
   *)
    fun pollIn (PollDesc(iod, {rd, wr, err})) =
	  PollDesc(iod, {rd=true, wr=wr, err=err})
    fun pollOut (PollDesc(iod, {rd, wr, err})) =
	  PollDesc(iod, {rd=rd, wr=true, err=err})
    fun pollErr (PollDesc(iod, {rd, wr, err})) =
	  PollDesc(iod, {rd=rd, wr=wr, err=true})

  (* polling function *)
    local
      val poll' : ((int * word) list * (int * int) option) -> (int * word) list =
	    CInterface.c_function "POSIX-OS" "poll"
      fun join (false, _, w) = w
        | join (true, b, w) = Word.orb(w, b)
      fun test (w, b) = (Word.andb(w, b) <> 0w0)
      val rdBit = 0w1 and wrBit = 0w2 and errBit = 0w4
      fun fromPollDesc (PollDesc(OS.IO.IODesc fd, {rd, wr, err})) =
	    ( fd,
	      join (rd, rdBit, join (wr, wrBit, join (err, errBit, 0w0)))
	    )
      fun toPollInfo (fd, w) = PollInfo(OS.IO.IODesc fd, {
	      rd = test(w, rdBit), wr = test(w, wrBit), err = test(w, errBit)
	    })
    in
    fun poll (pds, timeOut) = let
	  val timeOut = (case timeOut
		 of SOME(PreBasis.TIME{sec, usec}) => SOME(sec, usec)
		  | NONE => NONE
		(* end case *))
	  val info = poll' (List.map fromPollDesc pds, timeOut)
	  in
	    List.map toPollInfo info
	  end
    end (* local *)

  (* check for conditions *)
    fun isIn (PollInfo(_, flgs)) = #rd flgs
    fun isOut (PollInfo(_, flgs)) = #wr flgs
    fun isErr (PollInfo(_, flgs)) = #err flgs
    fun infoToPollDesc  (PollInfo arg) = PollDesc arg

  end (* OS_IO *)

