(* socket-sig.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

signature SOCKET =
  sig

  (* sockets are polymorphic; the instantiation of the type variables
   * provides a way to distinguish between different kinds of sockets.
   *)
    type ('af, 'sock) sock
    type 'af sock_addr

  (* witness types for the socket parameter *)
    type dgram
    type 'a stream
    type passive	(* for passive streams *)
    type active		(* for active (connected) streams *)

    eqtype addr_family
      sharing type addr_family = NetHostDB.addr_family
    eqtype sock_type

  (* address families *)
    structure AF : sig
        val unix : addr_family   (* 4.3BSD internal protocols *)
        val inet : addr_family   (* DARPA internet protocols *)
	val list : unit -> (string * addr_family) list
	    (* list known address families *)
        val toString : addr_family -> string
	val fromString : string -> addr_family option
      end

  (* socket types *)
    structure SOCK : sig
	val stream : sock_type		(* stream sockets *)
	val dgram : sock_type		(* datagram sockets *)
	val raw : sock_type		(* raw-protocol sockets *)
	val rdm : sock_type		(* reliably-delivered message *)
	val seqPacket : sock_type	(* sequenced packet stream *)

	val list : unit -> (string * sock_type) list
	    (* list known socket types *)
	val toString : sock_type -> string
	val fromString : string -> sock_type option
      end

  (* socket control operations *)
    structure Ctl : sig

      (* get/set socket options *)
        val getDEBUG		: ('a, 'b) sock -> bool
        val setDEBUG		: (('a, 'b) sock * bool) -> unit
        val getREUSEADDR	: ('a, 'b) sock -> bool
        val setREUSEADDR	: (('a, 'b) sock * bool) -> unit
        val getKEEPALIVE	: ('a, 'b) sock -> bool
        val setKEEPALIVE	: (('a, 'b) sock * bool) -> unit
        val getDONTROUTE	: ('a, 'b) sock -> bool
        val setDONTROUTE	: (('a, 'b) sock * bool) -> unit
        val getLINGER		: ('a, 'b) sock -> Time.time option
        val setLINGER		: (('a, 'b) sock * Time.time option) -> unit
        val getBROADCAST	: ('a, 'b) sock -> bool
        val setBROADCAST	: (('a, 'b) sock * bool) -> unit
        val getOOBINLINE	: ('a, 'b) sock -> bool
        val setOOBINLINE	: (('a, 'b) sock * bool) -> unit
        val getSNDBUF		: ('a, 'b) sock -> int
        val setSNDBUF		: (('a, 'b) sock * int) -> unit
        val getRCVBUF		: ('a, 'b) sock -> int
        val setRCVBUF		: (('a, 'b) sock * int) -> unit
        val getTYPE		: ('a, 'b) sock -> sock_type
        val getERROR		: ('a, 'b) sock -> bool

	val getPeerName		: ('a, 'b) sock -> 'b sock_addr
	val getSockName		: ('a, 'b) sock -> 'b sock_addr

(** NOTE: getSockType is the same as getTYPE.  Also, the other two
 ** operations are not supported by the OS.  The implementation will
 ** have to keep track of the information.
 **)
	val getSockAF		: ('a, 'b) sock -> addr_family
	val getSockType		: ('a, 'b) sock -> sock_type
	val getSockProto	: ('a, 'b) sock -> int

	val setNBIO		: (('a, 'b) sock * bool) -> unit
	val getNREAD		: ('a, 'b) sock -> int
	val getATMARK		: ('a, active stream) sock -> bool
      end (* Ctl *)

  (* socket address operations *)
    val sameAddr     : ('a sock_addr * 'a sock_addr) -> bool
    val addrToString : 'a sock_addr -> string

  (* socket management *)
    val accept  : ('a, passive stream) sock
		    -> (('a, active stream) sock * 'a sock_addr)
    val bind    : (('a, 'b) sock * 'a sock_addr) -> unit
    val connect : (('a, 'b) sock * 'a sock_addr) -> unit
    val listen  : (('a, passive stream) sock * int) -> unit
    val close   : ('a, 'b) sock -> unit
    datatype shutdown_mode = NO_RECVS | NO_SENDS | NO_RECVS_OR_SENDS
    val shutdown : (('a, 'b stream) sock * shutdown_mode) -> unit

    val pollDesc : ('a, 'b) sock -> OS.IO.poll_desc

  (* Sock I/O option types *)
    type out_flags = {don't_route : bool, oob : bool}
    type in_flags = {peek : bool, oob : bool}

  (* Sock output operations *)
    val sendVec	   : (('a, 'b stream) sock * {buf: Word8Vector.vector, i: int, sz: int option})
			-> int
    val sendArr	   : (('a, 'b stream) sock * {buf: Word8Array.array, i: int, sz: int option})
			-> int
    val sendVec'   : (('a, 'b stream) sock * {buf: Word8Vector.vector, i: int, sz: int option} * out_flags)
			-> int
    val sendArr'   : (('a, 'b stream) sock * {buf: Word8Array.array, i: int, sz: int option} * out_flags)
			-> int
    val sendVecTo  : (('a, dgram) sock * 'a sock_addr * {buf: Word8Vector.vector, i: int, sz: int option})
			-> int
    val sendArrTo  : (('a, dgram) sock * 'a sock_addr * {buf: Word8Array.array, i: int, sz: int option})
			-> int
    val sendVecTo' : (('a, dgram) sock * 'a sock_addr * {buf: Word8Vector.vector, i: int, sz: int option} * out_flags)
			-> int
    val sendArrTo' : (('a, dgram) sock * 'a sock_addr * {buf: Word8Array.array, i: int, sz: int option} * out_flags)
			-> int

  (* Sock input operations *)
    val recvVec      : (('a, 'b stream) sock * int)
			-> Word8Vector.vector
    val recvArr	     : (('a, 'b stream) sock * {buf: Word8Array.array, i: int, sz: int option})
			-> int
    val recvVec'     : (('a, 'b stream) sock * int * in_flags)
			-> Word8Vector.vector
    val recvArr'     : (('a, 'b stream) sock * {buf: Word8Array.array, i: int, sz: int option} * in_flags)
			-> int
    val recvVecFrom  : (('a, dgram) sock * int)
			-> (Word8Vector.vector * 'b sock_addr)
    val recvArrFrom  : (('a, dgram) sock * {buf: Word8Array.array, i: int})
			-> (int * 'a sock_addr)
    val recvVecFrom' : (('a, dgram) sock * int * in_flags)
			-> (Word8Vector.vector * 'b sock_addr)
    val recvArrFrom' : (('a, dgram) sock * {buf: Word8Array.array, i: int} * in_flags)
			-> (int * 'a sock_addr)

  end
