(* pre-sock.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * These are some common type definitions used in the sockets library.  This
 * structure is called Sock, so that the signatures will compile.
 *)

structure PreSock =
  struct

  (* the raw representation address data *)
    type addr = Word8Vector.vector

  (* the raw representation of an address family *)
    type af = CInterface.system_const

  (* the raw representation of a socket (a file descriptor for now) *)
    type socket = int

  (* an internet address; this is here because it is abstract in the
   * NetHostDB and IP structures.
   *)
    datatype in_addr = INADDR of addr

  (* an address family *)
    datatype addr_family = AF of af

  (* socket types *)
    datatype sock_type = SOCKTY of CInterface.system_const

  (* sockets are polymorphic; the instantiation of the type variables
   * provides a way to distinguish between different kinds of sockets.
   *)
    datatype ('sock, 'af) sock = SOCK of socket
    datatype 'af sock_addr = ADDR of addr

  (** Utility functions for parsing/unparsing network addresses **)
    local
      structure SysW = SysWord
      structure SS = Substring
      structure SCvt = StringCvt
      exception BadAddress
      fun toW (ss, bits) = let
	    fun scan radix ss = (case (SysW.scan radix SS.getc ss)
		   of NONE => raise BadAddress
		    | (SOME(w, _)) => w
		  (* end case *))
(** Note: the scan function may take care of the "0x" **)
	    val w = (case SS.getc ss
		   of NONE => raise BadAddress
		    | (SOME(#"0", ss')) => (case (SS.getc ss')
		         of NONE => 0w0
			  | (SOME((#"x" | #"X"), _)) => scan SCvt.HEX ss
			  | _ => scan SCvt.OCT ss
		        (* end case *))
		    | (SOME(c, _)) => scan SCvt.DEC ss
		  (* end case *))
		    handle _ => raise BadAddress
	    in
	      if (SysW.>= (SysW.>>(0wxffffffff, Word.-(0w32, bits)), w))
	        then w
	        else raise BadAddress
	    end
    in
    fun toWords s = ((
	  case (SS.fields (fn #"." => true | _ => false) (SS.all s))
	   of [a, b, c, d] => 
		[toW(a, 0w8), toW(b, 0w8), toW(c, 0w8), toW(d, 0w8)]
	    | [a, b, c] => [toW(a, 0w8), toW(b, 0w8), toW(c, 0w16)]
	    | [a, b] => [toW(a, 0w8), toW(b, 0w24)]
	    | [a] => [toW(a, 0w32)]
	    | _ => []
	  (* end case *))
	    handle BadAddress => [])
    fun fromBytes (a, b, c, d) = let
	  val fmt = Word8.fmt StringCvt.DEC
	  in
	    concat [fmt a, ".", fmt b, ".", fmt c, ".", fmt d]
	  end
    end

  end; (* PreSock *)

(* We alias this structure to Sock so that the signature files will compile.
 * We also need to keep the PreSock structure visible, so that structures
 * compiled after the real Sock structure still have access to the representation
 * types.
 *)
structure Sock = PreSock;

