(* os-io-sig.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * NOTE: this interface has been proposed, but not yet adopted by the
 * Standard basis committee.
 *)

signature OS_IO =
  sig
    eqtype io_desc
	(* an io_desc is an abstract descriptor for an OS object that
	 * supports I/O (e.g., file, tty device, socket, ...).
	 *)

    val id : io_desc -> int
	(* return an integer ID for the I/O descriptor; note that IDs may
	 * be reused during an execution, but that no two active I/O
	 * devices will have the same ID.
	 *)

    val kind : io_desc -> string
	(* return the kind of I/O descriptor; values include: "FILE",
	 * "PIPE", "SOCK", and "TTY".
	 *)

    type poll_desc
	(* this is an abstract representation of a polling operation on
	 * an I/O descriptor.
	 *)
    type poll_info
	(* this is an abstract representation of the per-descriptor
	 * information returned by the poll operation.
	 *)

    val pollDesc : io_desc -> poll_desc option
	(* create a polling operation on the given descriptor; note that
	 * not all I/O devices support polling.
	 *)
    val pollToIODesc : poll_desc -> io_desc
	(* return the I/O descriptor that is being polled *)

    exception Poll

  (* set polling events; if the polling operation is not appropriate
   * for the underlying I/O device, then the Poll exception is raised.
   *)
    val pollIn  : poll_desc -> poll_desc
    val pollOut : poll_desc -> poll_desc
    val pollErr : poll_desc -> poll_desc

  (* polling function *)
    val poll : (poll_desc list * Time.time option) -> poll_info list
	(* a timeout of NONE means wait indefinitely; a timeout of
	 * (SOME Time.zeroTime) means do not block.
	 *)

  (* check for conditions *)
    val isIn 		: poll_info -> bool
    val isOut		: poll_info -> bool
    val isErr		: poll_info -> bool
    val infoToPollDesc  : poll_info -> poll_desc

  end (* OS_IO *)

