(* Copyright 1989 by AT&T Bell Laboratories *)
(* primop.sig *)

signature PRIMOP =
  sig

    (* numkind includes kind and number of bits *)
    datatype numkind = INT of int | UINT of int | FLOAT of int
 
    datatype arithop
      = + | - | * | / | ~		(* int or float *)
      | ABS				(* floating point only *)
      | LSHIFT | RSHIFT | RSHIFTL	(* int only *)
      | ANDB | ORB | XORB | NOTB	(* int only *)

    datatype cmpop = > | >= | < | <= | LEU | LTU | GEU | GTU | EQL | NEQ

  (* Various primitive operations.  Those that are designated "inline" are
   * expanded into lambda code in terms of other operators,
   * as is the "checked=true" version of NUMSUBSCRIPT or NUMUPDATE.
   *)

    datatype primop
      = ARITH of {oper: arithop, 
	          overflow: bool, (* raises overflow exception? *)
	          kind: numkind}
      | CMP of {oper: cmpop, kind: numkind}
      | ROUND of {floor: bool, fromkind: numkind, tokind: numkind}
      | REAL of {fromkind: numkind, tokind: numkind}

    (* NUMSUBSCRIPT and NUMUPDATE are for arrays of floats or integers
     * stored WITHOUT boxing or tags.
     *)
      | NUMSUBSCRIPT of {kind: numkind, checked: bool, immutable: bool}
      | NUMUPDATE of {kind: numkind, checked: bool}

      | SUBSCRIPT			(* polymorphic array subscript *)
      | SUBSCRIPTV			(* poly vector subscript *)
      | INLSUBSCRIPT			(* inline poly array subscript *)
      | INLSUBSCRIPTV			(* inline poly vector subscript *)

      | PTREQL | PTRNEQ		        (* pointer equality *)
      | POLYEQL | POLYNEQ		(* polymorphic equality *)
      | BOXED | UNBOXED			(* boxity tests *)
      | LENGTH				(* vector, string, array, ... length *)
      | OBJLENGTH			(* length of arbitrary heap object *)
      | CAST
      | GETRUNVEC			(* get the pointer to the run-vector *)
      | MARKEXN				(* mark an exception value with a string *)
      | GETHDLR | SETHDLR		(* get/set exn handler pointer *)
      | GETVAR | SETVAR			(* get/set var register *)
      | GETPSEUDO | SETPSEUDO           (* get/set pseudo registers *)
      | SETMARK | DISPOSE               (* capture/dispose frames *)
      | MAKEREF				(* allocate a ref cell *)
      | CALLCC | CAPTURE | THROW	(* continuation operations *)
      | DEREF				(* dereferencing *)
      | ASSIGN				(* assignment; this is short for *)
					(* update(a, 0, v) *)
      | UPDATE			(* array update (maybe boxed) *)
      | INLUPDATE		(* inline array update (maybe boxed) *)
      | BOXEDUPDATE		(* boxed array update *)
	
      | UNBOXEDUPDATE		(* update array of integers WITH tags *)
      | GETTAG				(* extract the tag portion of an *)
					(* object's descriptor as an ML int *)
      | MKSPECIAL			(* make a special object *)
      | SETSPECIAL			(* set the state of a special object *)
      | GETSPECIAL			(* get the state of a special object *)
      | USELVAR | DEFLVAR
      | INLDIV | INLMOD | INLREM        (* inline interger arithmetic *)
      | INLMIN |INLMAX | INLABS         (* inline interger arithmetic *) 
      | INLNOT                          (* inline bool not operator *)
      | INLCOMPOSE                      (* inline compose "op o"  operator *)
      | INLBEFORE                       (* inline "before" operator *) 
      | INL_ARRAY			(* inline polymorphic array allocation *)
      | INL_VECTOR			(* inline polymorphic vector allocation *)
      | INL_MONOARRAY of numkind	(* inline monomorphic array allocation *)
      | INL_MONOVECTOR of numkind	(* inline monomorphic vector allocation *)

    val IADD : primop  (* default integer addition *)
    val ISUB : primop  (* default integer subtraction *)
    val IMUL : primop
    val IDIV : primop
    val INEG : primop

    val FEQLd : primop
    val IEQL : primop
    val INEQ : primop
    val IGT : primop
    val ILT : primop
    val ILE : primop
    val IGE : primop

    val pr_primop: primop -> string

  end (* PRIMOP *)
