(* Copyright 1989 by AT&T Bell Laboratories *)
(* lambda.sml *)

structure Lambda = struct 

    local 
	open Access
	type lty = LambdaType.lty
    in

	type dataconstr = Symbol.symbol * conrep * lty

	datatype con =
	    DATAcon of Symbol.symbol * conrep * lty
	  | INTcon of int
	  | WORDcon of word
	  | WORD32con of Word32.word
	  | REALcon of string
	  | STRINGcon of string
	  | VLENcon of int 

	datatype lexp =
	    VAR of lvar
	  | INT of int
	  | WORD of word
	  | WORD32 of Word32.word
	  | REAL of string
	  | STRING of string
	  | EXNF of lexp * lty                 
	  | EXNC of lexp              
	  | PRIM of primop * lty 
	  | FN of lvar * lty * lexp
	  | FIX of lvar list * lty list * lexp list * lexp
	  | APP of lexp * lexp
	  | SWITCH of lexp * conrep list * (con * lexp) list * lexp option
	  | CON of dataconstr * lexp
	  | DECON of dataconstr * lexp
	  | VECTOR of lexp list
	  | RECORD of lexp list
	  | SRECORD of lexp list    (* record used to represent structures *)
	  | SELECT of int * lexp
	  | RAISE of lexp * lty 
	  | HANDLE of lexp * lexp
	  | WRAP of lty * lexp
	  | UNWRAP of lty * lexp

	fun hashlambda l = let
	    val crc = CRC.new ()
	    val lty = LambdaType.crc crc
	    val { word, word32, int, string, primop, conrep, app = c,
		  list, option, ... } =
		CRCUtil.mkUtils (crc, true)

	    fun dataconstr (s, cr, t) = (string (Symbol.symbolToString s);
					 conrep cr; lty t)

	    fun con (DATAcon dc) = (c 0; dataconstr dc)
	      | con (INTcon i) = (c 1; int i)
	      | con (WORDcon w) = (c 2; word w)
	      | con (WORD32con w32) = (c 3; word32 w32)
	      | con (REALcon s) = (c 4; string s)
	      | con (STRINGcon s) = (c 5; string s)
	      | con (VLENcon i) = (c 6; int i)

	    fun lexp (VAR v) = (c 0; int v)
	      | lexp (INT i) = (c 1; int i)
	      | lexp (WORD w) = (c 2; word w)
	      | lexp (WORD32 w32) = (c 3; word32 w32)
	      | lexp (REAL s) = (c 4; string s)
	      | lexp (STRING s) = (c 5; string s)
	      | lexp (EXNF (e, t)) = (c 6; lexp e; lty t)
	      | lexp (EXNC e) = (c 7; lexp e)
	      | lexp (PRIM (p, t)) = (c 8; primop p; lty t)
	      | lexp (FN (v, t, e)) = (c 9; int v; lty t; lexp e)
	      | lexp (FIX (vl, tl, el, e)) =
		(c 10; list int vl; list lty tl; list lexp el; lexp e)
	      | lexp (APP (e1, e2)) = (c 11; lexp e1; lexp e2)
	      | lexp (SWITCH (e, crl, cel, eo)) =
		(c 12; list conrep crl;
		 list (fn (c, e) => (con c; lexp e)) cel;
		 option lexp eo)
	      | lexp (CON (dc, e)) = (c 13; dataconstr dc; lexp e)
	      | lexp (DECON (dc, e)) = (c 14; dataconstr dc; lexp e)
	      | lexp (VECTOR el) = (c 15; list lexp el)
	      | lexp (RECORD el) = (c 16; list lexp el)
	      | lexp (SRECORD el) = (c 17; list lexp el)
	      | lexp (SELECT (i, e)) = (c 18; int i; lexp e)
	      | lexp (RAISE (e, t)) = (c 19; lexp e; lty t)
	      | lexp (HANDLE (e1, e2)) = (c 20; lexp e1; lexp e2)
	      | lexp (WRAP (t, e)) = (c 21; lty t; lexp e)
	      | lexp (UNWRAP (t, e)) = (c 22; lty t; lexp e)
	in
	    option lexp l; CRC.extract crc
	end

	local
	    val trh = LambdaType.rehashcons
	    fun dcrh (s, cr, t) = (s, cr, trh t)
	    fun cerh (DATAcon dc, e) = (DATAcon (dcrh dc), erh e)
	      | cerh (c, e) = (c, erh e)
	    and erh (EXNF (e, t)) = EXNF (erh e, trh t)
	      | erh (EXNC e) = EXNC (erh e)
	      | erh (PRIM (p, t)) = PRIM (p, trh t)
	      | erh (FN (v, t, e)) = FN (v, trh t, erh e)
	      | erh (FIX (vl, tl, el, e)) =
		FIX (vl, map trh tl, map erh el, erh e)
	      | erh (APP (e1, e2)) = APP (erh e1, erh e2)
	      | erh (SWITCH (e, crl, cel, eo)) =
		SWITCH (erh e, crl, map cerh cel,
			case eo of NONE => NONE | SOME e => SOME (erh e))
	      | erh (CON (dc, e)) = CON (dcrh dc, erh e)
	      | erh (DECON (dc, e)) = DECON (dcrh dc, erh e)
	      | erh (VECTOR el) = VECTOR (map erh el)
	      | erh (RECORD el) = RECORD (map erh el)
	      | erh (SRECORD el) = SRECORD (map erh el)
	      | erh (SELECT (i, e)) = SELECT (i, erh e)
	      | erh (RAISE (e, t)) = RAISE (erh e, trh t)
	      | erh (HANDLE (e1, e2)) = HANDLE (erh e1, erh e2)
	      | erh (WRAP (t, e)) = WRAP (trh t, erh e)
	      | erh (UNWRAP (t, e)) = UNWRAP (trh t, erh e)
	      | erh e = e
	in
	    val rehashcons = erh
	end

	fun CON' ((_, REF, lt), e) = APP (PRIM (P.MAKEREF, lt), e)
	  | CON' x = CON x

	fun DECON' ((_, REF, lt), e) = APP (PRIM (P.DEREF, lt), e)
	  | DECON' x = DECON x

	(* general alpha-conversion on lexp
	 * val copy: (unit -> lvar) -> lexp -> lexp *)
	fun copy mkLvar = let

	    val look = IntmapF.lookup

	    fun rename (lv, m) = let
		val lv' = mkLvar ()
		val m' = IntmapF.add (m, lv, lv')
	    in
		(lv', m')
	    end

	    fun c (VAR lv, m) = VAR (look m lv)
	      | c (x as INT _, _) = x
	      | c (x as WORD _, _) = x
	      | c (x as WORD32 _, _) = x
	      | c (x as REAL _, _) = x
	      | c (x as STRING _, _) = x
	      | c (EXNF (e, t), m) = EXNF (c (e, m), t)
	      | c (EXNC e, m) = EXNC (c (e, m))
	      | c (x as PRIM _, _) = x
	      | c (x as FN (lv, t, e), m) = let
		    val (lv', m') = rename (lv, m)
		in
		    FN (lv', t, c (e, m'))
		end
	      | c (FIX (lvl, ltl, el, e), m) = let
		    fun ren1 (lv, (lvl, m)) = let
			val (lv', m') = rename (lv, m)
		    in
			(lv' :: lvl, m')
		    end
		    val (lvl', m') = foldr ren1 ([], m) lvl
		    fun c' x = c (x, m')
		in
		    FIX (lvl', ltl, map c' el, c' e)
		end
	      | c (APP (e1, e2), m) = APP (c (e1, m), c (e2, m))
	      | c (SWITCH (e, crl, cel, eo), m) = let
		    fun cc (cr, x) = (cr, c (x, m))
		    fun co NONE = NONE
		      | co (SOME x) = SOME (c (x, m))
		in
		    SWITCH (c (e, m), crl, map cc cel, co eo)
		end
	      | c (CON (dc, e), m) = CON (dc, c (e, m))
	      | c (DECON (dc, e), m) = DECON (dc, c (e, m))
	      | c (VECTOR lel, m) = VECTOR (map (fn x => c (x, m)) lel)
	      | c (RECORD lel, m) = RECORD (map (fn x => c (x, m)) lel)
	      | c (SRECORD lel, m) = SRECORD (map (fn x => c (x, m)) lel)
	      | c (SELECT (i, e), m) = SELECT (i, c (e, m))
	      | c (RAISE (e, t), m) = RAISE (c (e, m), t)
	      | c (HANDLE (e1, e2), m) = HANDLE (c (e1, m), c (e2, m))
	      | c (WRAP (t, e), m) = WRAP (t, c (e, m))
	      | c (UNWRAP (t, e), m) = UNWRAP (t, c (e, m))
	in
	    fn e => c (e, IntmapF.empty)
	end

    end (* local *)

end
