(* access.sml
 *
 * COPYRIGHT (c) 1989 by AT&T Bell Laboratories
 *)

structure Access : ACCESS =
struct

  fun inc r = r := !r + 1

  structure P = 
    struct

    datatype numkind = INT of int | UINT of int | FLOAT of int
 
    datatype arithop
      = + | - | * | / | ~		(* int or float *)
      | ABS				(* floating point only *)
      | LSHIFT | RSHIFT | RSHIFTL	(* int only *)
      | ANDB | ORB | XORB | NOTB	(* int only *)

    datatype cmpop = > | >= | < | <= | LEU | LTU | GEU | GTU | EQL | NEQ

    (* Various primitive operations.  Those that are designated "inline" 
     * are expanded into lambda representation in the InlineOps structure.
     *)
    datatype primop
      = ARITH of {oper: arithop, 
	          overflow: bool, (* raises overflow exception? *)
	          kind: numkind}
      | CMP of {oper: cmpop, kind: numkind}
      | ROUND of {floor: bool, fromkind: numkind, tokind: numkind}
      | REAL of {fromkind: numkind, tokind: numkind}

      | NUMSUBSCRIPT of {kind: numkind, checked: bool, immutable: bool}
      | NUMUPDATE of {kind: numkind, checked: bool}

      | SUBSCRIPT		     (* polymorphic array subscript *)
      | SUBSCRIPTV		     (* poly vector subscript *)
      | INLSUBSCRIPT		     (* inline poly array subscript *)
      | INLSUBSCRIPTV		     (* inline poly vector subscript *)
				    
      | PTREQL | PTRNEQ		     (* pointer equality *)
      | POLYEQL | POLYNEQ	     (* polymorphic equality *)
      | BOXED | UNBOXED		     (* boxity tests *)
      | LENGTH			     (* vector, string, array, ... length *)
      | OBJLENGTH		     (* length of arbitrary heap object *)
      | CAST			    
      | GETRUNVEC		     (* get the pointer to the run-vector *)
      | MARKEXN                      (* mark an exception value with a string *)
      | GETHDLR | SETHDLR	     (* get/set exn handler pointer *)
      | GETVAR | SETVAR		     (* get/set var register *)
      | GETPSEUDO | SETPSEUDO        (* get/set pseudo registers *)
      | SETMARK | DISPOSE            (* capture/dispose frames *)
      | MAKEREF			     (* allocate a ref cell *)
      | CALLCC | CAPTURE | THROW     (* continuation operations *)
      | DEREF			     (* dereferencing *)
      | ASSIGN			     (* assignment; this is short for 
				        an update operation *)
      | UPDATE			     (* array update (maybe boxed) *)
      | INLUPDATE		     (* inline array update (maybe boxed) *)
      | BOXEDUPDATE		     (* boxed array update *)
      | UNBOXEDUPDATE		     (* unboxed array update *)
      | GETTAG			     (* extract the tag portion of an 
				        object's descriptor as an ML int *)
      | MKSPECIAL		     (* make a special object *)
      | SETSPECIAL		     (* set the state of a special object *)
      | GETSPECIAL		     (* get the state of a special object *)
      | USELVAR | DEFLVAR	    
      | INLDIV | INLMOD | INLREM     (* inline interger arithmetic *)
      | INLMIN |INLMAX | INLABS      (* inline interger arithmetic *) 
      | INLNOT                       (* inline bool not operator *)
      | INLCOMPOSE                   (* inline compose "op o"  operator *)
      | INLBEFORE                    (* inline "before" operator *) 
      | INL_ARRAY		     (* inline polymorphic array allocation *)
      | INL_VECTOR		     (* inline polymorphic vector allocation *)
      | INL_MONOARRAY of numkind     (* inline monomorphic array allocation *)
      | INL_MONOVECTOR of numkind    (* inline monomorphic vector allocation *)


    val IADD = ARITH{oper=op +, overflow=true, kind=INT 31}
    val ISUB = ARITH{oper=op -, overflow=true, kind=INT 31}
    val IMUL = ARITH{oper=op *, overflow=true, kind=INT 31}
    val IDIV = ARITH{oper=op /, overflow=true, kind=INT 31}
    val INEG = ARITH{oper=op ~, overflow=true, kind=INT 31}

    val FEQLd = CMP{oper=EQL, kind=FLOAT 64}
    val IEQL = CMP{oper=EQL, kind=INT 31}
    val INEQ = CMP{oper=NEQ, kind=INT 31}
    val IGT = CMP{oper=op >, kind=INT 31}
    val ILT = CMP{oper=op <, kind=INT 31}
    val IGE = CMP{oper=op >=, kind=INT 31}
    val ILE = CMP{oper=op <=, kind=INT 31}

    fun pr_numkind (INT 31)      = ""
      | pr_numkind (INT bits)    = makestring bits
      | pr_numkind (UINT 32)     = "u"
      | pr_numkind (UINT bits)   = "u" ^ makestring bits
      | pr_numkind (FLOAT 64)    = "f"
      | pr_numkind (FLOAT  bits) = "f" ^ makestring bits

    fun pr_primop(ARITH{oper,overflow,kind}) =
          ((case oper 
             of op + => "+" |  op - => "-" |  op * => "*"
              | op / => "/" |  op ~ => "~" | LSHIFT => "lshift" 
              | RSHIFT => "rshift" | RSHIFTL => "rshift_l" | ABS => "abs"
              | ANDB => "andb" | ORB => "orb" | XORB => "xorb" 
              | NOTB => "notb")
            ^ (if overflow then "" else "n")
            ^ pr_numkind kind)
      | pr_primop(CMP{oper,kind}) =
	  ((case oper 
             of op > => ">" |  op < => "<" | op >= => ">=" | op <= => "<="
	      | GEU => ">=U" | GTU => ">U" | LEU => "<=U" | LTU => "<U"
              | EQL => "=" | NEQ => "<>" )
            ^ pr_numkind kind)

      | pr_primop(ROUND{floor=true,fromkind=FLOAT 64,tokind=INT 31}) = "floor"
      | pr_primop(ROUND{floor=false,fromkind=FLOAT 64,tokind=INT 31}) = "round"
      | pr_primop(ROUND{floor,fromkind,tokind}) =
          ((if floor then "floor" else "round")
           ^ pr_numkind fromkind ^ "_" ^ pr_numkind tokind)

      | pr_primop(REAL{fromkind=INT 31,tokind=FLOAT 64}) = "real"
      | pr_primop(REAL{fromkind,tokind}) =
          ("real" ^ pr_numkind fromkind ^ "_" ^ pr_numkind tokind)
		   
      | pr_primop(NUMSUBSCRIPT{kind,checked,immutable}) = 
	  ("numsubscript" ^ pr_numkind kind
	   ^ (if checked then "c" else "")
	   ^ (if immutable then "v" else ""))

      | pr_primop (NUMUPDATE{kind,checked}) = 
          ("numupdate" ^ pr_numkind kind ^ (if checked then  "c" else ""))

      | pr_primop DEREF = "!"
      | pr_primop ASSIGN = ":="
      | pr_primop BOXED = "boxed"
      | pr_primop UNBOXED = "unboxed"
      | pr_primop CAST = "cast"
      | pr_primop PTREQL = "ptreql"
      | pr_primop PTRNEQ = "ptrneq"  
      | pr_primop POLYEQL = "polyeql"
      | pr_primop POLYNEQ = "polyneq"  
      | pr_primop GETHDLR = "gethdlr"
      | pr_primop MAKEREF = "makeref"
      | pr_primop SETHDLR = "sethdlr"
      | pr_primop LENGTH = "length"
      | pr_primop OBJLENGTH = "objlength"
      | pr_primop CALLCC = "callcc"
      | pr_primop CAPTURE = "capture"
      | pr_primop THROW = "throw"
      | pr_primop SUBSCRIPT = "subscript"
      | pr_primop UNBOXEDUPDATE = "unboxedupdate"
      | pr_primop BOXEDUPDATE = "boxedupdate"
      | pr_primop UPDATE = "update"
      | pr_primop INLSUBSCRIPT = "inlsubscript"
      | pr_primop INLSUBSCRIPTV = "inlsubscriptv"
      | pr_primop INLUPDATE = "inlupdate"
      | pr_primop SUBSCRIPTV = "subscriptv"
      | pr_primop GETRUNVEC = "getrunvec"
      | pr_primop GETVAR = "getvar"
      | pr_primop SETVAR = "setvar"
      | pr_primop GETPSEUDO = "getpseudo"
      | pr_primop SETPSEUDO = "setpseudo"
      | pr_primop SETMARK = "setmark"
      | pr_primop DISPOSE = "dispose"
      | pr_primop GETTAG = "gettag"
      | pr_primop MKSPECIAL = "mkspecial"
      | pr_primop SETSPECIAL = "setspecial"
      | pr_primop GETSPECIAL = "getspecial"
      | pr_primop USELVAR = "uselvar"
      | pr_primop DEFLVAR = "deflvar"
      | pr_primop INLDIV = "inldiv"
      | pr_primop INLMOD = "inlmod"
      | pr_primop INLREM = "inlrem"
      | pr_primop INLMIN = "inlmin"
      | pr_primop INLMAX = "inlmax"
      | pr_primop INLABS = "inlabs"
      | pr_primop INLNOT = "inlnot"
      | pr_primop INLCOMPOSE = "inlcompose"
      | pr_primop INLBEFORE = "inlbefore"
      | pr_primop (INL_ARRAY) = "inl_array"
      | pr_primop (INL_VECTOR) = "inl_vector"
      | pr_primop (INL_MONOARRAY kind) =
          concat ["inl_monoarray(", pr_numkind kind, ")"]
      | pr_primop (INL_MONOVECTOR kind) =
          concat ["inl_monovector(", pr_numkind kind, ")"]
      | pr_primop MARKEXN = "markexn"
    end

  type lvar = int      (* lambda variable id number *)
  type slot = int      (* position in structure record *)
  type path = int list (* slot chain terminated by lambda variable id number *)
  type primop = P.primop

  (* access: how to find the dynamic value corresponding to a variable.
    A PATH is an absolute address from a lambda-bound variable (i.e. we find
    the value of the lambda-bound variable, and then do selects from that).
    PATH's are kept in reverse order.   A SLOT is a position in a structure,
    and is relative to the address of the lambda-bound variable for the
    structure.   INLINE means that there is no dynamic value for the variable,
    which is a closed function: instead the compiler will generate "inline"
    code for the variable.  If we need a dynamic value, we must eta-expand
    the function.

    See modules.sig for the invariants of access paths in environments *)

  datatype access 
    = SLOT of slot
    | LVAR of int
    | EXTERN of PersStamps.persstamp
    | INLINE of primop
    | PATH of int * access
    | NO_ACCESS

  datatype conrep
      = UNTAGGED
      | TAGGED of int
      | TAGGEDREC of int * int
      | UNTAGGEDREC of int
      | CONSTANT of int
      | TRANSPARENT
      | REF
      | LISTCONS
      | LISTNIL
      | VARIABLE of access (* exception constructor *)
      | VARIABLEc of access (* exception constructor with no argument *)

  (* the different kinds of records *)
  datatype record_kind
    = RK_VECTOR
    | RK_RECORD
    | RK_SPILL
    | RK_ESCAPE
    | RK_EXN
    | RK_CONT
    | RK_FCONT
    | RK_KNOWN
    | RK_BLOCK
    | RK_FBLOCK
    | RK_I32BLOCK

  fun newLvar r () = (inc r; !r)

  (* local *)
  val varcount = ref 0
  exception NoLvarName
  val lvarNames : string Intmap.intmap = Intmap.new(32, NoLvarName)
  val name = Intmap.map lvarNames
  val giveLvarName = Intmap.add lvarNames

  val saveLvarNames = Control.saveLvarNames
  val mkLvar = newLvar varcount
  fun newCanonicalLvars () = newLvar (ref 0)
  fun sameName(v,w) =
      if !saveLvarNames
      then giveLvarName(v,name w)
	     handle NoLvarName => (giveLvarName(w, name v)
				      handle NoLvarName => ())
      else ()
  fun dupLvar v =
      (inc varcount;
       if !saveLvarNames
       then giveLvarName(!varcount,name v) handle NoLvarName => ()
       else ();
       !varcount)
  fun namedLvar(id: Symbol.symbol) =
      (inc varcount;
       if !saveLvarNames then giveLvarName(!varcount,Symbol.name id) else ();
       !varcount)
  fun lvarName(lv : lvar) : string =
      (name lv ^ makestring lv) handle NoLvarName => makestring lv
  fun lvarIsNamed lv = (name lv; true) handle NoLvarName => false

  fun pr_lvar(lvar:lvar) = makestring(lvar)
  fun pr_slot(slot:slot) = makestring(slot)
  fun pr_path'[] = "]"
    | pr_path'[x:int] = makestring x ^ "]"
    | pr_path'((x:int)::rest)= makestring x ^ "," ^ pr_path' rest
  fun pr_path path = "[" ^ pr_path' path
  fun pr_access (SLOT slot) = "SLOT(" ^ pr_slot slot ^ ")"
    | pr_access (LVAR i) = "LVAR(" ^ pr_lvar i ^ ")"
    | pr_access (PATH(i,a)) = "PATH(" ^ makestring i ^ ","^ pr_access a ^ ")"
    | pr_access (INLINE po) = "INLINE(" ^ P.pr_primop po ^ ")"
    | pr_access (EXTERN pid) = "EXTERN(" ^ PersStamps.stampToString pid ^ ")"
    | pr_access NO_ACCESS = "NO_ACCESS"

end  (* structure Access *)
