(* alpha32Instr.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure Alpha32Instr = struct
    structure C = Alpha32Cells

    datatype labexp = POSLAB of Label.label * int  
      		    | NEGLAB of Label.label * int

					(* Instruction formats *)
    datatype operand =
      REGop of int
    | IMMop of int
    | HIMASKop of (int list)
    | LOMASKop of (int list)
    | HILABop of labexp
    | LOLABop of labexp

    datatype cond_code =
	CC_EQ | CC_NEQ
      | CC_LT | CC_LE | CC_GT | CC_GE
      | CC_LTU | CC_LEU | CC_GTU | CC_GEU

    type MemoryFormat = int * (int * int) (* register, (basereg, disp) *)
					(* I need to pass the label because 
                                         * it is helpful when scheduling *)
    type BranchFormat = int * Label.label 
    type OperateFormat = int * operand * int (* src, src, dst *)
    type FPOperateFormat = int * int * int   (* src, src, dst *)
      

    datatype instruction =
      DEFFREG of int     	(* define a floating point register *)
			        (* Memory Integer Load/Store *)
      
    | LDA of int * int * operand	(* Load Address *)
    | LDAH of int * int * operand
    | LDL of MemoryFormat		(* Load memory into Int. register *)
    | LDQ of MemoryFormat
    | LDQ_U of MemoryFormat		(* Allign addr. and Load quadword  *)
    | STL of MemoryFormat		(* Store Int. register into memory *)
    | STQ of MemoryFormat		(* Store Quadword *)
    | STQ_U of MemoryFormat		(* Allign addr. and Store Quadword *)
      
			        (* Control Instructions *)
      
    | BR of BranchFormat		(* Unconditional Branch *)
    | JMPL of MemoryFormat * Label.label list
                                        (* Jump to a computed address. I 
                                         * pass along the possible labels 
                                         * also because they are helpful 
                                         * when scheduling  *)
    | JSR of MemoryFormat * C.cellset * C.cellset
      
    | BEQ of BranchFormat	(* Integer Conditional Branches *)
    | BGE of BranchFormat
    | BGT of BranchFormat
    | BLE of BranchFormat
    | BLT of BranchFormat
    | BNE of BranchFormat
    | BLBC of BranchFormat
    | BLBS of BranchFormat
    | FBEQ of BranchFormat	(* Floating Point Conditional Branches *)
    | FBGE of BranchFormat
    | FBGT of BranchFormat
    | FBLE of BranchFormat
    | FBLT of BranchFormat
    | FBNE of BranchFormat
      
				(* Integer Arithmetic *)

    | ZAP of OperateFormat  
    | ADDL of OperateFormat  
    | ADDLV of OperateFormat		(* Longword Add with overflow*)
    | ADDQ of OperateFormat		(* Quadword Add *)
    | SUBL of OperateFormat		(* Longword substract *)
    | SUBLV of OperateFormat		(* Longword Substract with overflow*)
    | SUBQ of OperateFormat		(* Quadword substract *)
    | MULL of OperateFormat		(* Longword multiply *)
    | MULLV of OperateFormat		(* Longword multiply with overflow *)
    | CMPULE of OperateFormat
    | CMPULT of OperateFormat
    | CMPEQ of OperateFormat
    | CMPLE of OperateFormat
    | CMPLT of OperateFormat
       (* SGNXL (s,d) generates ADDL (s, REGop 31, d).  It serves to sign
          extend its longword argument *)
    | SGNXL of int * int (* source, destination *)
      
				(* Logical/Shift Instructions *)
      
    | AND of OperateFormat		(* AND *)
    | BIS of OperateFormat		(* OR *)
    | XOR of OperateFormat		(* XOR *)
    | SRA of OperateFormat		
    | SRL of OperateFormat		
    | SLL of OperateFormat

				(* Byte operations *)

    | INSBL of OperateFormat		(* Insert byte *)
    | EXTBL of OperateFormat		(* Extract Byte *)
    | EXTQH of OperateFormat		(* Extract Quadword *)
    | MSKBL of OperateFormat		(* Mask byte *)
    | MSKLH of OperateFormat		(* Mask long *)
				(* Floating Point Load/Store *)
      
    | LDT of MemoryFormat		(* Load T-floating point *)
    | STT of MemoryFormat		(* Store T-floating point *)
      
				(* Floating Point Operate *)
    | CPYS of FPOperateFormat		(* Copy Sign *)
    | CPYSN of FPOperateFormat
    | CVTQT of FPOperateFormat
    | CVTTQ of FPOperateFormat
    | CMPTEQ of FPOperateFormat
    | CMPTLT of FPOperateFormat
    | CMPTLE of FPOperateFormat
    | ADDT of FPOperateFormat 
    | SUBT of FPOperateFormat 
    | MULT of FPOperateFormat
    | DIVT of FPOperateFormat
				(* Misc *)
      
    | TRAPB				(* Trap barrier *)

		                (* Span Dependent Instructions *)
      
					(* This are macros that have different 
                                         * expansions depending on the value 
                                         * of the labels *)
    | LADDR of int * labexp * int	(* Load Address of a label *)
    | BRANCH of Label.label * Label.label(* Unconditional Branch *)

      
      
end


