(* alpha32Cells.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure Alpha32Cells = struct
  structure T = MLTree
  structure S = SortedList

  fun error msg = ErrorMsg.impossible ("Alpha32Cells."^msg)

  (** the list of integers [from, from+1, ..., to-1, to] **)
  fun upto (from,to) = if from>to then [] else from::(upto (from+1,to))
  infix upto

 (** these registers are always live **)
  val allocptr as T.REG allocptrR	= T.REG 13
  val limitptr as T.REG limitptrR	= T.REG 9
  val storeptr 				= T.REG 12
  val stackptr as T.REG stackptrR	= T.REG 30
  val exnptr as T.REG exnptrR		= T.REG 14
  val varptr  				= T.REG 10
  val baseptr as T.REG baseptrR		= T.REG 4

  (** these registers may be used as misc regs whenever possible **)
  val gclinkreg	as T.REG gclinkregR	= T.REG 27
  val maskreg as T.REG maskRegR		= T.REG 5
  val exhausted as T.REG exhaustedR	= T.REG 11
  val stdlink				= T.REG 3
  val stdclos				= T.REG 2
  val stdarg 				= T.REG 0
  val stdcont                   	= T.REG 1

  (* Note: Using varptr as the instruction count register is stricly
   * 	incorrect. This register is never used in the compiler itself, and 
   *  so its use here, prevents the need to recompile the compiler when
   *  turning on instruction counting. 
   * A correct implementation would use a miscreg instead.
   *)

  type cell = int

  val availFregs = 0 upto 28 (* reserve f29 and f30 for compiler temporaries *)
  val floatregs   = map T.FREG availFregs
  val savedfpregs = []  

  val icountR         		= 15	 (* instruction counting *)  
  val globalptrR		= 29	 (* C global pointer *)

  val asmTmpR			= 28
  val zeroR		 	= 31 (* always 0 *)

  val miscregs = map T.REG ((6 upto 8)@(16 upto 26))
  val calleesave = Array.fromList(miscregs)

  local
    val avail = [gclinkreg, maskreg, exhausted,
                 stdlink, stdclos, stdarg, stdcont] @ miscregs
    val availR = (map (fn T.REG r => r | _ => error "availRegs") avail)
  in
    val availRegs = S.uniq availR
  end


  val nrRegs =  32  (* 32 integer registers *)
  val nrFregs = 32  (* 32 floating point registers *)

  local 
      val regCnt = ref nrRegs           
      val fregCnt = ref nrFregs          
      fun bump (r as ref c) = (r := c+1; c)
  in
      fun newReg ()   = bump regCnt
      fun newFreg()   = bump fregCnt
      fun maxReg()    = !regCnt
      fun maxFreg()   = !fregCnt
      fun resetRegs() = (regCnt:= nrRegs; fregCnt:= nrFregs)
  end


  fun regMap(T.REG r)   = r 
  fun fregMap(T.FREG f) = f

  type cellset = int list * int list * int list
  fun cellset2string _ = ""
  val empty = ([],[],[])
  fun addReg(r, (rc,fc,mc)) = (S.enter(r,rc), fc, mc)
  fun addFreg(f, (rc,fc,mc)) = (rc, S.enter(f,fc), mc)

  val dedicated = (S.remove (availRegs, 0 upto 31), S.uniq [zeroR,29,30], [])

end








