/* unix-timers.c
 *
 * COPYRIGHT (c) 1995 by AT&T Bell Laboratories.
 *
 * A UNIX specific interface to the system timers.
 */

#include "ml-unixdep.h"
#include "ml-base.h"
#include "ml-state.h"
#include "ml-timer.h"

#ifdef HAS_GETRUSAGE

#include <sys/time.h>
#include <sys/resource.h>
typedef struct rusage time_struct_t;
#ifndef OPSYS_NETBSD
#define GET_TIME(t)		getrusage(RUSAGE_SELF, &(t))
#else /* NetBSD */
/* getrusage under NetBSd occasionally returns decreasing values for
   system time usage.  This breaks some sml code, which assumes time
   is non-decreasing.  The work-around below, remembers the maximum
   values for both system and user time, and returns the previous
   maximum when getrusage reports a smaller value.  */
static struct timeval stime={0,0}, utime={0,0};
#define MAX_TIME(tv, ref) \
	      if (timercmp(&(tv), &(ref), <= )) (tv) = (ref); else (ref) = (tv)
#define GET_TIME(t)		{ getrusage(RUSAGE_SELF, &(t)); \
				  MAX_TIME((t).ru_stime,stime); \
				  MAX_TIME((t).ru_utime,utime); }
#endif
#define SYS_TIME(t)		((t).ru_stime)
#define USR_TIME(t)		((t).ru_utime)
#define SET_TIME(tp, t)		{ *((struct timeval *)(tp)) = (t); }

#else /* !HAS_GETRUSAGE */

#include <sys/times.h>
PVT long	ClksPerSec = 0;
typedef struct tms time_struct_t;
#define GET_TIME(t)		times(&(t))
#define SYS_TIME(t)		((t).tms_stime)
#define USR_TIME(t)		((t).tms_utime)
#define SET_TIME(tp, t)		{ 					\
	Time_t		*__tp = (tp);					\
	clock_t		__t = (t);					\
	__tp->uSeconds = ((__t % ClksPerSec) * 1000000) / ClksPerSec;	\
	__tp->seconds  = (__t / ClksPerSec);				\
    }

#endif /* HAS_GETRUSAGE */



/* InitTimers:
 *
 * Do any system specific timer initialization.
 */
void InitTimers ()
{
#ifndef HAS_GETRUSAGE
    if (ClksPerSec == 0)
	ClksPerSec = sysconf(_SC_CLK_TCK);
#endif

} /* end of InitTimers */


/* GetCPUTime:
 *
 * Get the user and/or system cpu times in a system independent way.
 */
void GetCPUTime (Time_t *user_t, Time_t *sys_t)
{
    time_struct_t    ts;

    GET_TIME(ts);
    if (user_t != NIL(Time_t *)) {
	SET_TIME(user_t, USR_TIME(ts));
    }
    if (sys_t != NIL(Time_t *)) {
	SET_TIME(sys_t, SYS_TIME(ts));
    }

} /* end of GetCPUTime. */
