/* tags.h
 *
 * COPYRIGHT (c) 1992 by AT&T Bell Laboratories.
 *
 * These are the macros for object tags and descriptors.
 */

#ifndef _TAGS_
#define _TAGS_

#define MAJOR_MASK	0x3		/* bits 0-1 are the major tag */

#ifdef BOXED1
					/* Major tag: */
#define TAG_boxed	0x1		/*   01 - pointers */
#define TAG_desc	0x3		/*   11 - descriptors */
#define TAG_unboxed_b0	0x0		/*   00, 10 - unboxed (bit 0 is 0) */
					/* High bit of descriptor for objects with */
					/* a length operation is unboxed bit-0: */
#define HASLEN		0x0		/*   0xxx - objects with length */
#define NOLEN		0x8		/*   1xxx - objects without a length */

/* mark/unmark an ML pointer to make it look like an unboxed object */
#define MARK_PTR(p)	((ml_val_t)((Addr_t)(p) & ~0x1))
#define UNMARK_PTR(p)	((ml_val_t)((Addr_t)(p) | 0x1))

#else /* BOXED1 */
					/* Major tag: */
#define TAG_boxed	0x0		/*   00 - pointers */
#define TAG_desc	0x2		/*   10 - descriptors */
#define TAG_unboxed_b0	0x1		/*   01, 11 - unboxed (bit 0 is 1) */
					/* High bit of descriptor for objects with */
					/* a length operation is unboxed bit-0: */
#define HASLEN		0x8		/*   1xxx - objects with length */
#define NOLEN		0x0		/*   0xxx - objects without length */

/* mark/unmark an ML pointer to make it look like an unboxed object */
#define MARK_PTR(p)	((ml_val_t)((Addr_t)(p) | 0x1))
#define UNMARK_PTR(p)	((ml_val_t)((Addr_t)(p) & ~0x1))

#endif /* BOXED1 */

/* Descriptors have four more tag bits (defined below).  For objects that
 * have a ML length operation, the high bit is the unboxed bit-0 value. */
#define DTAG_SHIFTW	2
#define DTAG_WID	4
#define DTAG_MASK	(((1 << DTAG_WID)-1) << DTAG_SHIFTW)
#define TAG_SHIFTW	(DTAG_SHIFTW+DTAG_WID)

#define DTAG_record	(HASLEN|0x0)	/* records (len != 2) and vectors */
#define DTAG_vector	DTAG_record
#define DTAG_array	(HASLEN|0x1)	/* arrays and refs */
#define DTAG_string	(HASLEN|0x2)	/* strings */
/*			(HASLEN|0x3)	unused */
#define DTAG_bytearray	(HASLEN|0x4)	/* bytearrays */
#define DTAG_realdarray	(HASLEN|0x5)	/* real array (double precision) */
/*			(HASLEN|0x6)	unused */
/*			(HASLEN|0x7)	unused */
#define DTAG_pair	(NOLEN|0x0)	/* pairs */
#define DTAG_reald	(NOLEN|0x1)	/* reals (double precision) */
/*			(NOLEN|0x2)	unused */
#define DTAG_variant	(NOLEN|0x3)	/* tagged variant record (see below) */
#define DTAG_special	(NOLEN|0x4)	/* special (susp, weak ptr, ...) */
#define DTAG_backptr	(NOLEN|0x5)	/* a back pointer (obsolete) */
#define DTAG_cont	(NOLEN|0x5)	/* a quasi-stack frame */
#define DTAG_extern	(NOLEN|0x6)	/* an external symbol reference (used in */
					/* exported heap images) */
#define DTAG_forwarded	(NOLEN|0x7)	/* a forwarded object */

/* Variant tags have a small length field, plus other bits */
#define VARIANT_LEN_BITS	4
#define VARIANT_OTHER_BITS	(BITS_PER_WORD-(DTAG_SHIFTW+VARIANT_LEN_BITS))

/* Build a descriptor from a descriptor tag and a length */
#ifndef _ASM_
#define MAKE_TAG(t)	(((t) << DTAG_SHIFTW) | TAG_desc)
#define MAKE_DESC(l,t)	((ml_val_t)(((l) << TAG_SHIFTW) | MAKE_TAG(t)))
#else
#define MAKE_TAG(t)	(((t)*4) + TAG_desc)
#define MAKE_DESC(l,t)	(((l)*64) + MAKE_TAG(t))
#endif

#define DESC_pair	MAKE_DESC(2, DTAG_pair)
#define DESC_exn	MAKE_DESC(3, DTAG_record)
#define DESC_ref	MAKE_DESC(1, DTAG_array)
#define DESC_reald	MAKE_DESC(2, DTAG_reald)

#define DESC_forwarded	MAKE_DESC(0, DTAG_forwarded)

/* There are two kinds of special objects: suspensions and weak pointers
 * The length field of these defines the state and kind of special object:
 */
#define SPCL_evaled_susp	0	/* unevaluated suspension */
#define SPCL_unevaled_susp	1	/* evaluated suspension */
#define SPCL_weak		2	/* weak pointer */
#define SPCL_null_weak		3	/* nulled weak pointer */

#define DESC_evaled_susp	MAKE_DESC(SPECIAL_evaled_susp, DTAG_special)
#define DESC_unevaled_susp	MAKE_DESC(SPCL_unevaled_susp, DTAG_special)
#define DESC_weak		MAKE_DESC(SPCL_weak, DTAG_special)
#define DESC_null_weak		MAKE_DESC(SPCL_null_weak, DTAG_special)

/* tests on words:
 *   isBOXED(W)   -- true if W is tagged as an boxed value
 *   isUNBOXED(W) -- true if W is tagged as an unboxed value
 *   isDESC(W)    -- true if W is tagged as descriptor
 */
#define isBOXED(W)	(((Word_t)(W) & MAJOR_MASK) == TAG_boxed)
#define isUNBOXED(W)	(((Word_t)(W) & 0x1) == TAG_unboxed_b0)
#define isDESC(W)	(((Word_t)(W) & MAJOR_MASK) == TAG_desc)

/* extract descriptor fields */
#define GET_LEN(D)		(((Word_t)(D)) >> TAG_SHIFTW)
#define GET_STR_LEN(D)		((((Word_t)(D))+(3<<TAG_SHIFTW)) >> (TAG_SHIFTW+2))
#define GET_REALDARR_LEN(D)	(GET_LEN(D)*(REALD_SZB/WORD_SZB))
#define GET_TAG(D)		((((Word_t)(D)) & DTAG_MASK) >> DTAG_SHIFTW)
#define GET_VARIANT_LEN(D)	(GET_LEN(D) & ((1 << VARIANT_LEN_BITS)-1))

#endif /* !_TAGS_ */
