/*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edoardo Biagioni (Edoardo.Biagioni@cs.cmu.edu)
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Ken Cline (Kenneth.Cline@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

		i.	Abstract

	Functions to interface with the Berkeley packetfilter pseudo-device



		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	iii.	RCS Log
	1.	includes and extern declarations
	2.	ml_bpfopen: string -> int
	3.	ml_bpfget_address: int -> bytearray
	4.	ml_bpfset_filter: int * bytearray -> unit
	5.	ml_bpfnread: int -> int
	6.	ml_select

		iii.	RCS Log

$Log: header.c,v $

*/

/*
	1.	includes and extern declarations
 */

#include "ml-osdep.h"
#include "ml-unixdep.h"
#include INCLUDE_FCNTL_H
#include "ml-base.h"
#include "ml-values.h"
#include "ml-state.h"
#include "ml-objects.h"
#include "ml-c.h"
#include "ml-globals.h"
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <net/bpf.h>
#include <stdio.h>
#include <string.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_types.h>
#include <unistd.h>

/*
	2.	ml_bpfopen: string -> int
 */

ml_val_t ml_bpfopen (ml_state_t *msp, ml_val_t arg)
{
    char	*interface = PTR_MLtoC(char, arg);
    int		i;
    u_int	blen=8192;
    struct ifreq ifreq;

    /* find an available packetfilter */
    for (i = 0; i < 256; i++)
      {
	char devname[20];
	int fd, immed=1;

	sprintf (devname, "/dev/bpf%d", i);
	if ( (fd = open(devname, O_RDWR, 0)) < 0 )
	  {
	    if (errno = EBUSY) continue;
	    else
	      {
		perror ("bpfopen: open");
		return RaiseSysError (msp, NIL(char *));
	      }
	  }
	if (ioctl(fd, BIOCSBLEN, &blen) < 0)
	  {
	    close(fd);
	    perror ("bpfopen: BIOCSBLEN");
	    return RaiseSysError (msp, NIL(char *));
	  }
	if (interface && (interface[0] != 0) &&
	    strncpy(ifreq.ifr_name, interface, IFNAMSIZ) &&
	    ioctl(fd, BIOCSETIF, &ifreq) < 0)
	  {
	    close(fd);
	    perror ("bpfopen: BIOCSETIF");
	    return RaiseSysError (msp, NIL(char *));
	  }
	if (ioctl(fd, BIOCIMMEDIATE, &immed)<0)
	  {
	    close(fd);
	    perror ("bpfopen: BIOCMBIS");
	    return RaiseSysError (msp, NIL(char *));
	  }
	return INT_CtoML (fd);
      }
    return RaiseSysError (msp, NIL(char *));
}

/*
	3.	ml_bpfget_address: int -> bytearray
 */

ml_val_t ml_bpfget_address (ml_state_t *msp, ml_val_t arg)
{
    int			pf = INT_MLtoC(arg);
    ml_val_t		result;

#if 0
    /* NetBSD documentation claims this should work, but it doesn't */

    struct endevp	p;

    if (ioctl(pf, BIOCDEVP, &p) < 0)
      return RaiseSysError (msp, NIL(char *));

    result = ML_AllocBytearray (msp, 6);
    ((char*)result)[0] = p.end_addr[0];
    ((char*)result)[1] = p.end_addr[1];
    ((char*)result)[2] = p.end_addr[2];
    ((char*)result)[3] = p.end_addr[3];
    ((char*)result)[4] = p.end_addr[4];
    ((char*)result)[5] = p.end_addr[5];
#endif

/* This is from NetBSD's usr/src/usr.sbin/bootpd/getether.c */
    int fd;
    register int n;
    struct ifreq ibuf[16], ifr, name_ifr;
    struct ifconf ifc;
    register struct ifreq *ifrp, *ifend;

    /* get interface name for this packetfilter */
    if (ioctl(pf, BIOCGETIF, &name_ifr) < 0)
      return RaiseSysError (msp, NIL(char *));
    /* Fetch the interface configuration */
    fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (fd < 0) return RaiseSysError (msp, NIL(char *));
    ifc.ifc_len = sizeof(ibuf);
    ifc.ifc_buf = (caddr_t) ibuf;
    if (ioctl(fd, SIOCGIFCONF, (char *) &ifc) < 0 ||
	ifc.ifc_len < sizeof(struct ifreq))
      {
	close (fd);
	return RaiseSysError (msp, NIL(char *));
      }
    close (fd);
    /* Search interface configuration list for link layer address. */
    ifrp = ibuf;
    ifend = (struct ifreq *) ((char *) ibuf + ifc.ifc_len);
    while (ifrp < ifend) {
      /* Look for interface */
      int l = strlen(name_ifr.ifr_name);
      if (strncmp(name_ifr.ifr_name, ifrp->ifr_name, l) == 0 &&
	  strcmp(name_ifr.ifr_name+l+1, ifrp->ifr_name+l) == 0 &&
	  ifrp->ifr_addr.sa_family == AF_LINK &&
	  ((struct sockaddr_dl *) &ifrp->ifr_addr)->sdl_type == IFT_ETHER) {
	result = ML_AllocBytearray (msp, 6);
	bcopy(LLADDR((struct sockaddr_dl *) &ifrp->ifr_addr), result, 6);
	return result;
      }
      /* Bump interface config pointer */
      n = ifrp->ifr_addr.sa_len + sizeof(ifrp->ifr_name);
      if (n < sizeof(*ifrp))
	n = sizeof(*ifrp);
      ifrp = (struct ifreq *) ((char *) ifrp + n);
    }

    return RaiseSysError (msp, NIL(char *));
}

/*
	4.	ml_bpfset_filter: int * bytearray -> unit
 */

ml_val_t ml_bpfset_filter (ml_state_t *msp, ml_val_t arg)
{
    int			fd      = REC_SELINT(arg,0);
    struct bpf_program	*filter	= (struct bpf_program *)REC_SELPTR(char,arg,1);

    if (ioctl(fd, BIOCSETF, filter))
      return RaiseSysError (msp, NIL(char *));

    return ML_false;
}

/*
	5.	ml_bpfnread: int -> int
		Returns the number of bytes available for reading
 */

ml_val_t ml_bpfnread (ml_state_t *msp, ml_val_t arg)
{
    int n, fd =  INT_MLtoC(arg);

    if (ioctl(fd, FIONREAD, &n) < 0)
      return RaiseSysError (msp, NIL(char *));

    if (n<0) n=0;

    return INT_CtoML (n);
}

/*
	6.	ml_select
		int * int list * int list * int list * (int * int) option
		-> int
		The last parameter indicates how long to wait
		before returning: if the option is NONE, then don't return
		until the FDs are ready for I/O, otherwise use the
		two ints as the time in seconds and microseconds.
 */

ml_val_t ml_select (ml_state_t *msp, ml_val_t arg)
{
    int			nfds		= REC_SELINT(arg, 0);
    ml_val_t		read_fds	= REC_SEL(arg, 1);
    ml_val_t		write_fds	= REC_SEL(arg, 2);
    ml_val_t		except_fds	= REC_SEL(arg, 3);
    ml_val_t		time_spec	= REC_SEL(arg, 4);
/*
    int			*time_sec	= REC_SELPTR(int, arg, 5);
    int			*time_usec	= REC_SELPTR(int, arg, 6);
*/
    int			result;
    struct timeval	timeout;
    int			select_value;
    fd_set		rd, wr, ex;

    FD_ZERO (&rd);
    while (read_fds != ML_nil) {
      FD_SET ((REC_SELINT ((ml_val_t *)read_fds, 0)), &rd);
      read_fds = REC_SEL ((ml_val_t *)read_fds, 1);
    }
    FD_ZERO (&wr);
    while (write_fds != ML_nil) {
      FD_SET ((REC_SELINT ((ml_val_t *)write_fds, 0)), &wr);
      write_fds = REC_SEL ((ml_val_t *)write_fds, 1);
    }
    FD_ZERO (&ex);
    while (except_fds != ML_nil) {
      FD_SET ((REC_SELINT ((ml_val_t *)except_fds, 0)), &ex);
      except_fds = REC_SEL ((ml_val_t *)except_fds, 1);
    }
/*
    printf ("first word of bitsets is 0x%x, 0x%x, 0x%x\n",
	    rd.fds_bits[0], wr.fds_bits[0], ex.fds_bits[0]);
*/

    if (time_spec != ML_nil) {
      ml_val_t * time_pair = REC_SELPTR(ml_val_t, time_spec, 0);
      int time_sec  = REC_SELINT(time_pair, 0);
      int time_usec = REC_SELINT(time_pair, 1);
/*
      printf ("time_spec is 0x%x, time_pair is 0x%x, *time_pair is 0x%x\n",
	      (int) time_spec, (int) time_pair, *(int *)time_pair);
      printf ("time_spec is 0x%x, ML_nil is 0x%x, time is %d.%d\n",
	      time_spec, ML_nil, time_sec, time_usec);
*/
      timeout.tv_sec = time_sec;
      timeout.tv_usec = time_usec;
      result = select (nfds, &rd, &wr, &ex, &timeout);
    }
    else {
/*
      printf ("time_spec is %x, ML_nil is %x\n", time_spec, ML_nil);
*/
      result = select (nfds, &rd, &wr, &ex, NULL);
    }

    return INT_CtoML (result);
}
