abstraction AbsPath: ABSPATH = struct

    structure P = OS.Path
    structure F = OS.FileSys

    type id = string

    datatype t =
	C				(* cwd *)
      | T of {
	      context: t,
	      spec: string,		(* as specified *)
	      elab: string option ref,	(* lazy context + spec *)
	      id: id option ref		(* lazy unique file id *)
	     }

    val cur = C
    val dummy = T { context = C, spec = "<nofile>",
		    elab = ref NONE, id = ref NONE }

    (* cwd caching *)
    val cwd_cache: string option ref = ref NONE

    (* query current wd and load cache if necessary *)
    fun cwd () =
	case !cwd_cache of
	    SOME s => s
	  | NONE => let
		val s = F.getDir ()
	    in
		cwd_cache := SOME s;
		s
	    end

    (* verify cache contents; tell whether content has changed *)
    fun newcwd () = let
	val s' = F.getDir ()
    in
	case !cwd_cache of
	    SOME s =>
		if s = s' then false else (cwd_cache := SOME s'; true)
	  | NONE => (cwd_cache := SOME s'; true)
    end

    (* pathname as specified (disregarding context) *)
    fun spec (T { spec, ... }) = spec
      | spec C = P.currentArc

    (* elaborate pathname (set spec into context) *)
    fun elab (T { elab = ref (SOME e), ... }) = e
      | elab (T { context, spec, elab = el, ... }) = let
	    val e =
		if P.isAbsolute spec then spec
		else
		    case context of
			C => spec	(* relative to CWD *)
		      | c => P.mkCanonical (P.concat (elab c, spec))
	in
	    el := SOME e;
	    e
	end
      | elab C = P.currentArc

    (* pathname `id' (for telling if two names refer to the same file) *)
    fun id (T { id = ref (SOME i), ... }) = i
      | id C = cwd ()
      | id (p as T { id = i as ref NONE, spec, context, ... }) = let
	    val p =
		if P.isAbsolute spec then
		    F.fullPath spec handle _ => P.mkCanonical spec
		else let
		    val d = cwd ()
		    val ic = id context
		in
		    (F.chDir ic;
		     (F.fullPath spec before F.chDir d)
		     handle _ =>
			 (F.chDir d;
			  P.mkCanonical (P.concat (ic, spec))))
		    handle _ => let
			val ec = elab context
		    in
			P.mkCanonical (P.concat (ec, spec))
		    end
		end
	in
	    i := SOME p;
	    p
	end

    (* do two paths refer to the same file *)
    fun sameFile (p1, p2) = (id p1) = (id p2)

    fun bare (context, spec) =
	T { spec = spec, context = context, elab = ref NONE, id = ref NONE }

    (* converting from strings using native pathname syntax *)
    fun native { context, spec } =
	bare (context, P.mkCanonical spec)

    (* converting from UNIX-style pathnames (the CM `standard') *)
    fun standard { context, spec } = let
	fun delim c = c = #"/"

	fun transl ".." = OS.Path.parentArc
	  | transl "." = OS.Path.currentArc
	  | transl arc = arc

	fun mk x = native { context = context, spec = P.toString x }
    in
	case String.fields delim spec of
	    "" :: arcs =>
		mk { isAbs = true, vol = "", arcs = map transl arcs }
	  | arcs =>
		mk { isAbs = false, vol = "", arcs = map transl arcs }
    end

    fun current name = native { context = C, spec = name }

    fun splitDirFile C = { dir = bare (C, P.parentArc),
			   file = P.file (cwd ()) }
      | splitDirFile (T { spec, context, ... }) = let
	val { dir, file } = P.splitDirFile spec
	val dir = if dir = "" then P.currentArc else dir
    in
	{ dir = bare (context, dir), file = file }
    end

    fun joinDirFile { dir = C, file } = bare (C, file)
      | joinDirFile { dir = T { spec, context, ... }, file } = let
	val j = P.mkCanonical (P.joinDirFile { dir = spec, file = file })
    in
	bare (context, j)
    end

    val dir = #dir o splitDirFile
    val file = #file o splitDirFile

    fun splitBaseExt C = { base = C, ext = NONE }
      | splitBaseExt (T { spec, context, ... }) = let
	val { base, ext } = P.splitBaseExt spec
    in
	{ base = bare (context, base), ext = ext }
    end

    fun joinBaseExt { base = C, ext = NONE } = C
      | joinBaseExt { base = C, ext = SOME e } = let
	    val f = P.file (cwd ())
	    val fe = P.joinBaseExt { base = f, ext = SOME e }
	in
	    bare (C, P.concat (P.parentArc, fe))
	end
      | joinBaseExt { base = T { spec, context, ... }, ext } = let
	    val j = P.joinBaseExt { base = spec, ext = ext }
	in
	    bare (context, j)
	end

    val base = #base o splitBaseExt
    val ext = #ext o splitBaseExt

    fun extendExt { path = p, ext = e, sep = dot } = let
	val { base, ext } = splitBaseExt p
    in
	case ext of
	    NONE => joinBaseExt { base = base, ext = SOME e }
	  | SOME e' => joinBaseExt { base = base,
				     ext = SOME (concat [e', dot, e]) }
    end

    fun exists p =
	OS.FileSys.access (elab p, []) handle _ => false

    fun modTime p =
	OS.FileSys.modTime (elab p) handle _ => Time.zeroTime

end
