(*
 * srctypes/fname-rules.sml:
 *   Rules for how to make up names for CM-managed files
 *
 *   Copyright (c) 1995 by AT&T Bell Laboratories
 *
 * author: Matthias Blume (blume@cs.princeton.edu)
 *)
functor FnameRulesFun (structure Compiler: COMPILER
		       val singlebindir: AbsPath.t option
		       val namelength_limited: bool): FNAME_RULES =
  struct

    type textfile = { open_in: unit -> TextIO.instream,
		      open_out: unit -> TextIO.outstream,
		      name: AbsPath.t }

    type binfile = { open_in: unit -> BinIO.instream,
		     open_out: unit -> BinIO.outstream,
		     name: AbsPath.t }

    val arch = Compiler.architecture
    val sep = "."

    fun add_ext NONE f = f
      | add_ext (SOME e) f = let
	val { base, ext } = AbsPath.splitBaseExt f
	val new_ext =
	    case ext of
		NONE => SOME e
	      | SOME old_ext => SOME (concat [old_ext, sep, e])
    in
	AbsPath.joinBaseExt { base = base, ext = new_ext }
    end

    val bin_ext = if namelength_limited then NONE else SOME "bin"
    val stable_ext = if namelength_limited then NONE else SOME "stable"

    val archdirname = "." ^ arch
    val decldirname = ".depend"

    val mkdir = OS.FileSys.mkDir o AbsPath.elab
    fun isDir path =
	(OS.FileSys.access (path, []) andalso OS.FileSys.isDir path)
    val isDir = isDir o AbsPath.elab

    fun checkdir dir = if isDir dir then () else mkdir dir

    fun subdir (dir, "") = (dir, fn () => ())
      | subdir (dir, arc) = let
	    val r = AbsPath.joinDirFile { dir = dir, file = arc }
	in
	    (r, fn () => checkdir r)
	end

    val textopen = (TextIO.openIn, TextIO.openOut)
    val binopen = (BinIO.openIn, BinIO.openOut)

    fun genericFileFor (gendir, suf, (open_in, open_out)) path = let
	val { dir, file } = AbsPath.splitDirFile path
	val (dir, check) = subdir (dir, gendir)
	val base = AbsPath.joinDirFile { dir = dir, file = file }
	val path = add_ext suf base
	fun oi () = open_in (AbsPath.elab path)
	fun oo () = (check (); open_out (AbsPath.elab path))
    in
	{ open_in = oi, open_out = oo, name = path }
    end

    val declFileFor = genericFileFor (decldirname, NONE, textopen)
    val stableFileFor = genericFileFor (archdirname, stable_ext, textopen)

    val binFileFor =
	case singlebindir of
	    NONE => genericFileFor (archdirname, bin_ext, binopen)
	  | SOME d => let
		(*
		 * `Batch' compilation drops all the binfiles into
		 * one single directory -- let's pray there are no duplicate
		 * source names in different source directories...
		 *)
		fun binFileFor name = let
		    val base = AbsPath.joinDirFile { dir = d,
						     file = AbsPath.file name }
		    val path = add_ext bin_ext base
		    fun oi () = BinIO.openIn (AbsPath.elab path)
		    fun oo () = BinIO.openOut (AbsPath.elab path)
		in
		    { open_in = oi, open_out = oo, name = path }
		end
	    in
		binFileFor
	    end


    fun errorTextFile exn = let
	fun err () = raise exn
    in
	{ open_in = err, open_out = err, name = AbsPath.dummy }
    end

  end
