(*
 * sched/recompile.sml: selective recompilation
 *
 *   Copyright (c) 1995 by AT&T Bell Laboratories
 *
 * author: Matthias Blume (blume@cs.princeton.edu)
 *)
functor RecompileFun (structure DagTraversal: DAG_TRAVERSAL
		      structure Driver: DRIVER
		      structure SysEnv:SYS_ENV
		      structure Control: CONTROL
		      structure LE: LAZY_ENV
		      structure Prune: PRUNE
		      sharing
			  DagTraversal.SysDag.ModuleName.Compiler =
			  SysEnv.Compiler =
			  Driver.Compiler
		      and
			  Prune.SD =
			  DagTraversal.SysDag =
			  Driver.SysDag): RECOMPILE = struct

    structure SysDag = DagTraversal.SysDag
    structure ModuleName = SysDag.ModuleName
    structure GroupDag = SysDag.GroupDag
    structure SmlSource = GroupDag.SmlSource
    structure Compiler = ModuleName.Compiler
    structure CUnit = SmlSource.CUnit
    structure Iid = CUnit.Iid
    structure Env = Compiler.Environment
    structure Pid = Compiler.PersStamps

    exception RecompileInternalError
    and WrongConfiguration of string * string
    and CompilationErrors of exn list

    type iid = Iid.t
    type senv = Env.staticEnv

    type iinfo = SmlSource.cu Futures.future
    datatype 'env enviid =
	ENVIID of 'env * iid
      | ENVIID_FAIL of exn
    datatype 'env cinfo =
	CI of { iinfo: iinfo,
	        env_iid: 'env enviid Futures.future,
		lib: bool }

    fun recomp
	{ getenv, filterBy, senvOf, runEnvOf, symenvOf,
	  genEnvD, emptyEnv, get, final }
	(SysDag.AE { roots, namemap, ... }, baseEnv) = let

	val lBaseEnv = LE.make baseEnv
	val lEmptyEnv = LE.make emptyEnv

	fun d2sml (GroupDag.DAG { smlsource, ... }) = smlsource

	fun fetch cuf () =
	    case Futures.get cuf of
		SmlSource.SUCCESS cu => ENVIID (getenv cu, CUnit.iid cu)
	      | SmlSource.FAILURE exn => ENVIID_FAIL exn

	val seen_before = SmlSource.cunit_again o d2sml

	fun pre _ = ()

	fun extern mn = ModuleName.symbolOf mn

	fun cross (SysDag.CE { dag, filter = NONE, ... }, iinfo) =
	    CI { iinfo = iinfo,
		 env_iid = Futures.future (fetch iinfo),
		 lib = false }
	  | cross (SysDag.CE { dag, filter = SOME filter, lib }, iinfo) = let
		val fsyms =
		    map ModuleName.symbolOf (ModuleName.makelist filter)
		fun filterenv () =
		    case fetch iinfo () of
			ENVIID (e, iid) => let
			    val f_e = filterBy fsyms e
			    val f_iid = Iid.aug (iid, senvOf f_e)
			in
			    ENVIID (f_e, f_iid)
			end
		      | fail => fail
	    in
		CI { iinfo = iinfo,
		     env_iid = Futures.future filterenv,
		     lib = lib }
	    end

	fun intern (_, ii) = ii

	fun subcombine (e, c, i) = let

	    fun i_combine (iinfo, (e, pl, exl)) =
		case fetch iinfo () of
		    ENVIID (e', p') =>
			(LE.layer (LE.make e', e), p' :: pl, exl)
		  | ENVIID_FAIL exn => (e, [], exn :: exl)

	    val (all_i_env, all_i_iids, all_i_exns) =
		foldr i_combine (lEmptyEnv, [], []) i

	    fun c_combine (CI { lib, env_iid, ... },
			   (export_e, other_e, pl, exl)) =
		case Futures.get env_iid of
		    ENVIID (e', p') => let
			val le' = LE.make e'
		    in
			if lib then
			    (export_e, LE.layer (le', other_e), p' :: pl, exl)
			else
			    (LE.layer (le', export_e), other_e, p' :: pl, exl)
		    end
		  | ENVIID_FAIL exn => (export_e, other_e, [], exn :: exl)

	    val (export_c, other_c, all_c_iids, all_c_exns) =
		foldr c_combine (lEmptyEnv, lEmptyEnv, [], []) c

	    val _ =
		case all_i_exns @ all_c_exns of
		    [] => ()
		  | exl => raise CompilationErrors exl

	    val expenv = LE.layer (all_i_env, export_c)
	    val compenv = LE.layer (LE.layer (expenv, other_c), lBaseEnv)

	    val iids =
		Iid.union (Iid.makeset all_c_iids,
			   Iid.makeset all_i_iids)
	    
	    (* val iids =
	           Iid.union (Iid.makeset (basePid ::
		                                       (map c_iid c)),
		                    Iid.makeset (map i_iid i)) *)

	in
	    (compenv, expenv, iids)
	end

	fun combine (_, e, c, i, dag) = let
	    val sml = d2sml dag
	    fun compenv'n'iids () = let
		val (compenv, expenv, iids) = subcombine (e, c, i)
	    in
		(fn () => get compenv, iids)
	    end
	    val cuf = SmlSource.cunit { smlsource = sml,
				        mkcompenv = compenv'n'iids,
					senvOf = senvOf,
					runEnvOf = runEnvOf,
					symenvOf = symenvOf }
	in
	    if Control.keep_going NONE then
		cuf
	    else
		case Futures.get cuf of
		    SmlSource.SUCCESS _ => cuf
		  | SmlSource.FAILURE exn => raise exn
	end

	val tr = DagTraversal.traversal
	    { seen_before = seen_before, pre = pre, extern = extern,
	      cross = cross, intern = intern, combine = combine }

	fun ce2dag (SysDag.CE { dag, ... }) = dag

	fun check ([], []) = ()
	  | check ([], exns) = raise CompilationErrors exns
	  | check (iinfo :: t, exns) =
	    case Futures.get iinfo of
		SmlSource.SUCCESS _ => check (t, exns)
	      | SmlSource.FAILURE exn => check (t, exn :: exns)
    in
	check (tr roots, []);
	final (getenv, filterBy, lEmptyEnv, namemap)
    end

    fun make_topenv (getenv, filterBy, lEmptyEnv, namemap) = let	    

	fun mk_top_env ((nl, SysDag.CE { dag = GroupDag.DAG { smlsource,
							      ... },
					 filter,
					 ... }), e0) = let
	    val cu = Futures.get (SmlSource.cunit_again smlsource)
	    val env =
		case cu of
		    SmlSource.SUCCESS u => getenv u
		  | SmlSource.FAILURE exn => raise RecompileInternalError
	    val env = case filter of
		NONE => env
	      | SOME f => let
		    val f = map ModuleName.symbolOf (ModuleName.makelist f)
		in
		    filterBy f env
		end
	in
	    LE.layer (LE.make env, e0)
	end

    in
	foldl mk_top_env lEmptyEnv namemap
    end

    fun cvtFilter f = fn s => fn e => f (e, s)

    val statsymGet = let
	fun layer ((st, sy), (st', sy')) =
	    (Env.layerStatic (st, st'),
	     Env.layerSymbolic (sy, sy'))
	fun cons (st, sy) = (Env.consolidateStatic st,
			     Env.consolidateSymbolic sy)
    in
	LE.gen (layer, cons)
    end
    val envGet = LE.gen (Env.layerEnv, Env.consolidateEnv)

    val recomp_only = recomp {
			      getenv = fn u => (CUnit.senv u, CUnit.symenv u),
			      filterBy =
			        cvtFilter (fn ((st, sy), s) =>
					   (Env.filterStaticEnv (st, s), sy)),
			      senvOf = fn (st, _) => st,
			      symenvOf = fn (_, sy) => sy,
			      runEnvOf = NONE,
			      genEnvD = (fn (cu, _) =>
					 (CUnit.senv cu, CUnit.symenv cu)),
			      emptyEnv =
			        (Env.staticPart Env.emptyEnv,
				 Env.symbolicPart Env.emptyEnv),
			      get = statsymGet,
			      final = fn _ => ()
			     }

    fun getStatSymBaseEnv () = let
	val e = SysEnv.getBaseEnv ()
    in
	(Env.staticPart e, Env.symbolicPart e)
    end

    fun only descfile = let
	val e as (senv, _) = getStatSymBaseEnv ()
    in
	Driver.driver (fn ae => (recomp_only (ae, e); ()), senv) descfile
    end

    fun and'stabilize recursive descfile = let
	val e as (senv, _) = getStatSymBaseEnv ()
    in
	Driver.driver (fn ae as SysDag.AE { stabilizer, ... } =>
		       (recomp_only (ae, e);
			stabilizer only recursive),
		       senv)
	  descfile
    end

    fun exec_genEnvD (cu, compenv) =
	CUnit.execute (cu, Env.dynamicPart (envGet compenv))

    fun getEnv cu =
	case CUnit.env cu of
	    NONE => raise RecompileInternalError
	  | SOME e => e

    val recomp_and_run = recomp {
				 getenv = getEnv,
				 filterBy = cvtFilter Env.filterEnv,
				 senvOf = Env.staticPart,
				 symenvOf = Env.symbolicPart,
				 runEnvOf = SOME (fn x => x),
				 genEnvD = exec_genEnvD,
				 emptyEnv = Env.emptyEnv,
				 get = envGet,
				 final = make_topenv
				}

    fun and'run'regardless (pruning, cpusym) descfile = let
	val env = SysEnv.getBaseEnv ()
	val senv = Env.staticPart env
	val delta =
	    Driver.driver
	      (fn ae => let
		  val ae = case pruning of
		      NONE => ae
		    | SOME files => let
			  val smn = ModuleName.structMN
			  val fmn = ModuleName.functMN
			  val syms = [smn (cpusym ^ "VisComp"), fmn "IntShare"]
			  val syms = ModuleName.makeset syms
			  val ae = case SysDag.select_roots (ae, syms) of
			      { ae = SOME ae, ... } => ae
			    | _ => raise RecompileInternalError
			  val files = map AbsPath.current files
			  fun apriori (GroupDag.DAG { smlsource, ... }) = let
			      val f = SmlSource.name smlsource
			  in
			      List.exists (fn f' => AbsPath.sameFile (f, f'))
			                  files
			  end
		      in
			  Prune.prune apriori ae
		      end

	      in
		  envGet (recomp_and_run (ae, env))
	      end,
	       senv)
	      descfile
    in
	Control.vsay
	  "[introducing new bindings into toplevel environment...]\n";
	SysEnv.addToInteractiveEnv delta
    end

    fun and'run (hostconf, targetconf, pruning) = let
	fun don't _ =
	    raise WrongConfiguration (Arch.confname hostconf,
				      Arch.confname targetconf)
    in
	if hostconf = targetconf then
	    and'run'regardless (pruning, Arch.cpusym (#cpu hostconf))
	else
	    don't
    end
	    
    fun exec'once_genEnvD (cu, compenv) =
	case CUnit.env cu of
	    SOME e => e
	  | NONE => 
		CUnit.execute (cu, Env.dynamicPart (envGet compenv))

    val and'run'once = 
	envGet o (recomp {
			  getenv = getEnv,
			  filterBy = cvtFilter Env.filterEnv,
			  senvOf = Env.staticPart,
			  symenvOf = Env.symbolicPart,
			  runEnvOf = SOME (fn x => x),
			  genEnvD = exec'once_genEnvD,
			  emptyEnv = Env.emptyEnv,
			  get = envGet,
			  final = make_topenv
			 })

    fun withAe f = Driver.driver (f, Env.staticPart (SysEnv.getBaseEnv ()))

end
