(*
 * cm/retarget.sml: Constructing cross-batch-compilers
 *
 *   Copyright (c) 1995 by AT&T Bell Laboratories
 *
 * author: Matthias Blume (blume@cs.princeton.edu)
 *)
structure CMR: sig

    val retarget: { bindir: string, target: string } -> unit
    functor CMBFun: CMB_FUN

end = struct

    functor CMBFun = CMBFun

    structure Env = Compiler.Environment
    structure Symbol = Compiler.Symbol

    structure CM = CMB.CM
    structure CU = CM.CUnit

    fun addarchsuffix (name, cpun) = concat [name, ".", cpun]

    val cpusym = Arch.cpusym

    fun load_binfiles (bindir, listfile) = let

	val layer = Env.layerEnv
	fun binname name = AbsPath.joinDirFile { dir = bindir, file = name }
	val is = TextIO.openIn (AbsPath.elab (binname listfile))

	fun all_files () = let

	    val perv = #get Env.pervasiveEnvRef ()

	    fun one_file env = let
		val line = TextIO.inputLine is
		val binfile = substring (line, 0, size line - 1)
		val binpath = binname binfile
		val _ = CM.Control.vsay (concat ["[reading ",
						 AbsPath.elab binpath,
						 "]\n"])
		val full = layer (env, perv)
		val cu = CU.fetchUnit (binpath, Env.staticPart full)
	    in
		CU.execute (cu, Env.dynamicPart full)
	    end

	    fun loop env =
		if end_of_stream is then
		    env
		else
		    loop (layer (one_file env, env))
	in
	    loop (Env.emptyEnv)
	end

	val env = all_files () handle exn => (TextIO.closeIn is; raise exn)
    in
	TextIO.closeIn is; env
    end

    fun filtered_load (bindir, listfile, sym) = let
	val lenv = load_binfiles (bindir, listfile)
	val fenv = Env.filterEnv (lenv, [sym])
	val nenv = Env.concatEnv (fenv, #get Env.topLevelEnvRef ())
    in
	#set Env.topLevelEnvRef nenv
    end

    fun load_compiler (bindir, cpun, compname) = let
	val listfile = addarchsuffix ("BINLIST", cpun)
	val sym = Symbol.strSymbol compname
    in
	filtered_load (bindir, listfile, sym)
    end

    val hostcpu = Arch.cpu Compiler.architecture
    val hostos = SMLofNJ.SysInfo.getOSKind ()
    val hostconf = { cpu = hostcpu, os = hostos }
    val hostconfn = Arch.confname hostconf

    fun new_compiler (compname, targetosn) =
	concat
	["local\n",
	 "structure Old = Compiler.Environment\n",
	 "structure New = ", compname, ".Environment\n",
	 "val _ = #set New.topLevelEnvRef (#get Old.topLevelEnvRef ())\n",
	 "val _ = #set New.pervasiveEnvRef (#get Old.pervasiveEnvRef ())\n",
	 "val _ = #set New.coreEnvRef (#get Old.coreEnvRef ())\n",
	 "in structure CMB = CMR.CMBFun (structure Compiler = ",
	 compname,
	 " val version = \"",
	 CMB.version,
	 " (retarget)\" val hostconfn = \"", hostconfn, "\"\n",
	 " val targetosn = \"", targetosn, "\") end\n"]
	
    fun retarget { bindir, target } = let
	val bindir = AbsPath.current bindir
	val targetconfn = target
	val targetconf as { cpu = targetcpu, os = targetos } =
	    Arch.conf targetconfn
	val targetosn = Arch.osname targetos
	val targetcpus = Arch.cpusym targetcpu
	val targetcpun = Arch.cpuname targetcpu
	val compname = targetcpus ^ "VisComp"
    in
	load_compiler (bindir, targetcpun, compname);
	Compiler.Interact.use_stream 
	  (open_string (new_compiler (compname, targetosn)))
    end

    val retarget = CM.Complain.complaining retarget

end
