(*
 * cm/complain.sml: Guard functions with exception handling/error reporting
 *
 *   Copyright (c) 1995 by AT&T Bell Laboratories
 *
 * author: Matthias Blume (blume@cs.princeton.edu)
 *)
functor ComplainFun (structure GroupDag: GROUP_DAG
		     structure ImpExp: IMP_EXP
		     structure SysDag: SYS_DAG
		     structure CUnit: CUNIT
		     structure Decl: DECL
		     structure Driver: DRIVER
		     structure ED: ENTITY_DESC
		     structure Recompile: RECOMPILE
		     structure BurgSource: TOOL_SOURCE
		     structure LexSource: TOOL_SOURCE
		     structure YaccSource: TOOL_SOURCE
		     structure SmlSource: SML_SOURCE
		     structure Source: SOURCE
		     structure Control: CONTROL): COMPLAIN =
struct

    structure MN = ImpExp.ModuleName

    fun msg lab l = Control.say (concat ["\n!* ", lab, concat l, "\n"])

    val err = msg "CM error: "
    val bug = msg "CM bug: "
    val warnerr = msg "CM warning: "
    val warnbug = msg "CM bug warning: "

    fun names [] = ""
      | names [n] = n
      | names [n1, n2] = concat [n1, " and ", n2]
      | names [n1, n2, n3] = concat [n1, ", ", n2, ", and ", n3]
      | names (hd :: tl) = concat [hd, ", ", names tl]

    fun waswere [_] = "was"
      | waswere _ = "were"

    fun moan (err, bug) = let
      fun moan' exn =
	case exn of
	    GroupDag.MultipleDefinitions (n, f1, f2) =>
		err [f1, ", ", f2, ": multiple definitions for ", n]
	  | GroupDag.Cycle (s, f) =>
		err [f, ": cycle among definitions involving ", s]
	  | GroupDag.IllegalToplevelOpen s =>
		err [s, ": toplevel `open'"]
	  | GroupDag.GroupDagInternalError =>
		bug ["internal error in GroupDag"]
	  | ImpExp.Undefined n =>
		bug ["undefined ", MN.makestring n, " not handled in ImpExp"]
	  | ImpExp.IllegalToplevelOpen =>
		bug ["IllegalToplevelOpen from ImpExp"]
	  | ImpExp.InternalError s =>
		bug ["internal error in ImpExp: ", s]
	  | SysDag.EntityCycle s =>
		err ["cycle among entities involving ", s]
	  | SysDag.MultipleDefinitions (nl, e, sf1, sf2) =>
		err [e, ": ", names nl, " ", waswere nl,
		     " imported from both ", sf1, " and ", sf2]
	  | SysDag.ExportedNamesNotDefined (nl, e) =>
		err [e, ": ", names nl, " ", waswere nl,
		     " not defined anywhere"]
	  | SysDag.Stabilize f =>
		err ["missing prerequisite for stabilization: ", f]
	  | SysDag.SysDagInternalError =>
		bug ["internal error in SysDag"]
	  | CUnit.FormatError =>
		err ["binfile doesn't have the required format"]
	  | CUnit.Outdated =>
		bug ["cannot recover from an outdated binary file"]
	  | CUnit.Compile s =>
		err ["compile: ", s]
	  | CUnit.NoCodeBug =>
		bug ["executable code not available"]
	  | Decl.InternalError =>
		bug ["internal error in Decl"]
	  | Decl.FormatError =>
		err ["declfile doesn't have the required format"]
	  | Classify.UnknownExtension s =>
		err ["don't know filename extension ", s]
	  | ED.BadEntityDescription (f, s) =>
		err [f, ": syntax error: ", s]
	  | ED.FileNotFound f =>
		err [f, ": description file not found"]
	  | ED.AliasNestingTooDeep f =>
		err [f, ": too many nested aliases (cycle?)"]
	  | MN.ModuleNameError =>
		bug ["internal error in ModuleName"]
	  | MN.PathError =>
		bug ["internal error in ModuleName (PathError)"]
	  | Recompile.RecompileInternalError =>
		bug ["internal error in Recompile"]
	  | Recompile.WrongConfiguration (comp, host) =>
		err ["code compiled for ", comp,
		     " cannot run on ", host]
	  | Recompile.CompilationErrors exnlist =>
		app moan' exnlist
	  | LexSource.ToolError s =>
		err ["ML-Lex failed: ", s]
	  | YaccSource.ToolError s =>
		err ["ML-Yacc failed: ", s]
	  | BurgSource.ToolError s =>
		err ["ML-Burg failed: ", s]
	  | SmlSource.SourceFileDoesNotExist s =>
		err [s, ": SML source file not found"]
	  | SmlSource.UserCodeExn (f, e) =>
		(err [f, ": exception raised in user code"]; moan' e)
	  | SmlSource.SmlSourceInternalError =>
		bug ["internal error in SmlSource"]
	  | Source.ToolNotImplemented s =>
		err ["the tool `", s, "' is not implemented"]
	  | OS.SysErr (s, _) =>
		err ["a system-call has failed: ", exnMessage exn]
	  | Io _ =>
		err [exnMessage exn]
	  | Lexer.LexicalError (f, s) =>
		err [f, ": lexical error: ", s]
	  | Lexer.SyntaxError (f, s) =>
		err [f, ": syntax error (preprocessor): ", s]
	  | Lexer.UserError (f, msg) =>
		err [f, ": ", msg]
	  | Arch.BadConf s =>
		err ["unrecognized target configuration name: ", s]
	  | Arch.BadCpu s =>
		err ["unrecognized cpu type: ", s]
	  | Arch.BadOS s =>
		err ["unrecognized os type: ", s]
	  | _ => let
		val name = exnMessage exn
	    in
		err [name, " (sorry, can't be more specific)"]
	    end
    in
	moan'
    end

    val complain = moan (err, bug)
    val warn = moan (warnerr, warnbug)

    fun complaining f arg = f arg handle exn => (complain exn; raise exn)
    fun warning f arg = f arg handle exn => warn exn

end
