(*
 * arch/arch.sml: CPU architectures and OS configuration for CM
 *
 *   Copyright (c) 1995 by AT&T Bell Laboratories
 *
 * author: Matthias Blume (blume@cs.princeton.edu)
 *)
abstraction Arch: ARCH = struct

    exception  BadConf of string and BadCpu of string and BadOS of string

    structure S = SMLofNJ.SysInfo

    type cpu = string
    type os = SMLofNJ.SysInfo.os_kind
    type conf = { cpu: cpu, os: os }

    fun confname { cpu, os = S.UNIX } = cpu ^ "-unix"
      | confname { cpu, os = S.WIN32 } = cpu ^ "-win32"
      | confname { cpu, os = S.MACOS } = cpu ^ "-macos"
      | confname { cpu, os = S.OS2 } = cpu ^ "-os2"

    fun os "unix" = S.UNIX
      | os "win32" = S.WIN32
      | os "macos" = S.MACOS
      | os "os2" = S.OS2
      | os s = raise BadOS s

    fun cpu s =
	case s of
	    "alpha32" => s
	  | "sparc" => s
	  | "mipsel" => s
	  | "mipseb" => s
	  | "hppa" => s
	  | "rs6000" => s
	  | "x86" => s
	  | "m68" => s
	  | "bytecode" => s
	  | _ => raise BadCpu s

    fun conf s = let

	fun fieldsep #"-" = true
	  | fieldsep _ = false
	val fields = String.fields fieldsep s

    in
	case fields of
	    [s1, s2] => { cpu = cpu s1, os = os s2 }
	  | _ => raise BadConf s
    end

    fun cpuname cpu = cpu

    fun cpusym "alpha32"   = "Alpha32"
      | cpusym "sparc"     = "Sparc"
      | cpusym "mipsel"    = "MipsLittle"
      | cpusym "mipseb"    = "MipsBig"
      | cpusym "m68"       = "M68"
      | cpusym "hppa"      = "Hppa"
      | cpusym "rs6000"    = "RS6000"
      | cpusym "x86"       = "X86"
      | cpusym "bytecode"  = "ByteCode"
      | cpusym cpu = raise BadCpu cpu

    fun osname S.UNIX = "unix"
      | osname S.WIN32 = "win32"
      | osname S.MACOS = "macos"
      | osname S.OS2 = "os2"

end
