;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                                  TUT                                  ;;
;;                       Tree Unification Toolkit                        ;;
;;                                                                       ;;
;;                         Version 1.0 15-Aug-91                         ;;
;;                                                                       ;;
;;                          John R. R. Leavitt                           ;;
;;                              Alex Franz                               ;;
;;                    Center for Machine Translation                     ;;
;;                      Carnegie-Mellon University                       ;;
;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;; Copyright (c) 1991                                                    ;;
;; Carnegie Mellon University. All Rights Reserved.                      ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;
;;                                                                       ;;
;; File Created:   15-Aug-91 by JRRL                                     ;;
;; Last Edit Date: 15-Aug-91 by JRRL                                     ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;
;; File Description                                                      ;;
;;                                                                       ;;
;; This file contains all of the code used solely for the unification    ;;
;; checker component of TUT.                                             ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;



;;-----------------------------------------------------------------------;;
;; Change Log                                                            ;;
;;                                                                       ;;
;; 15-Aug-91 by JRRL: Separated out the unification checker code into    ;;
;;                      this file.                                       ;;
;;                                                                       ;;
;;; 26-Aug-91 by amf: fixed unify-not-p to deal with double :NOT case.
;;;
;;; 24-Sep-91 by amf: fixed tree-unify-p to handle fs case correctly.
;;;
;;; 11-Jun-92 by amf: changed names of or-p, not-p to tut-or-p, tut-not-p
;;;                   because of clash with
;;;                   good ole' pseudo-unify.lisp
;;;
;;; 12-Oct-92 by amf: changed more names to avoid name clashes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;-----------------------------------------------------------------------;
;;; Package Statements:                                                   ;
;;;-----------------------------------------------------------------------;

(in-package 'user)
;;; (use-package 'meister)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                        Customization Variables                        ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;-----------------------------------------------------------------------;;
;; Variable *TUT-STRING-UNIFY-PRED*
;;
;; Specifes a user defined unification checking predicate for strings.

(defvar *tut-string-unify-pred* nil
  "Specifes a user defined unification checking predicate for strings.
   If used, this variable should specify a predicate that will take two
   strings and determine if they are unifiable.  If so, a non-NIL value
   should be returned; if not, NIL should be returned.")


;;-----------------------------------------------------------------------;;
;; Variable *TUT-NUMBER-UNIFY-PRED*
;;
;; Specifes a user defined unification checking predicate for numbers.

(defvar *tut-number-unify-pred* nil
  "Specifes a user defined unification checking predicate for numbers.
   If used, this variable should specify a predicate that will take two
   numbers and determine if they are unifiable.  If so, a non-NIL value
   should be returned; if not, NIL should be returned.")


;;-----------------------------------------------------------------------;;
;; Variable *TUT-SYMBOL-UNIFY-PRED*
;;
;; Specifes a user defined unification checking predicate for symbols.

(defvar *tut-symbol-unify-pred* nil
  "Specifes a user defined unification checking predicate for symbols.
   If used, this variable should specify a predicate that will take two
   symbols and determine if they are unifiable.  If so, a non-NIL value
   should be returned; if not, NIL should be returned.")


;;-----------------------------------------------------------------------;;
;; Variable *TUT-OTHER-UNIFY-PRED*
;;
;; Specifes a user defined unification checking predicate for miscellaneous
;; data types.

(defvar *tut-other-unify-pred* nil
  "Specifes a user defined unification checking predicate for miscellaneous
   data types.  If used, this variable should specify a predicate that will
   take two values, which may be of any type other than string, number, or
   symbol, and determine if they are unifiable.  If so, a non-NIL value
   should be returned; if not, NIL should be returned.")



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                     Unification Checking Functions                    ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+(and :cmu :new-compiler)
(proclaim '(inline tree-unify-fs-p
		   tree-unify-or-p
		   tree-unify-multiple-p
		   tree-unify-not-p
		   tree-unify-op-p))

#-(and :cmu :new-compiler)
(proclaim '(notinline tree-unify-fs-p
		      tree-unify-or-p
		      tree-unify-multiple-p
		      tree-unify-not-p
		      tree-unify-op-p))

(proclaim '(inline tree-unify-symbol-p
		   tree-unify-string-p
		   tree-unify-number-p
		   tree-unify-other-p))

;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-FS-P
;;
;; Predicate for FS unification checking

(defun tree-unify-fs-p (fs1 fs2)
  (let (failed assoc-result)
    (tut-trace-progn
     (format *standard-output* "~&---------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-fs-p]: checking f-structures~%")     
     (pprint fs1)
     (pprint fs2)
     (force-output *standard-output*))
    (do ((slots fs1 (cdr slots)))
	((or failed
	     (null slots))
	 (not failed))
      (setq assoc-result (assoc (caar slots) fs2 :test #'eq))
      (if assoc-result
	  (setq failed (not (tree-unify-p (cadar slots)
					  (second assoc-result))))))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-OR-P
;;
;; Predicate for :OR unification checking

(defun tree-unify-or-p (fs1 fs2)
  (let ((succeeded nil))
    (tut-trace-progn
     (format *standard-output* "~&---------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-or-p]: checking f-structures~%")     
     (pprint fs1)
     (pprint fs2))
    (do ((fss (cdr fs1) (cdr fss)))
	((or succeeded (null fss))
	 succeeded)
      (setq succeeded (tree-unify-p (car fss) fs2)))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-MULTIPLE-P
;;
;; Predicate for :MULTIPLE unification checking

(defun tree-unify-multiple-p (fs1 fs2)
  (let ((failed nil))
    (tut-trace-progn
     (format *standard-output* "~&---------------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-multiple-p]: checking f-structures~%")
     (pprint fs1)
     (pprint fs2))
    (do ((fss (cdr fs1) (cdr fss)))
	((or failed (null fss))
	 (not failed))
      (setq failed (not (tree-unify-p (car fss) fs2))))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-NOT-P
;;
;; Predicate for :NOT unification checking

;; 26-Aug-91 by amf: meistered this up to deal with double :NOT

(defun tree-unify-not-p (fs1 fs2)
  (tut-trace-progn
   (format *standard-output* "~&----------------------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-not-p]: checking f-structures~%")
   (pprint fs1)
   (pprint fs2))
  (if (tut-not-p fs2)
      t                                         ; then
      (not (tree-unify-p (second fs1) fs2))))   ; else


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-SYMBOL-P
;;
;; Predicate for symbol unification checking

(defun tree-unify-symbol-p (fs1 fs2)
  (declare (symbol fs1))
  (tut-trace-progn
   (format *standard-output* "~&--------------------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-symbol-p]: checking symbols~%")
   (pprint fs1)
   (pprint fs2))
  (cond ((not (symbolp fs2))
	 nil)
	(*tut-symbol-unify-pred*
	 (funcall *tut-symbol-unify-pred* fs1 fs2))
	(*tut-symbol-unify-fn*
	 (not (eq (funcall *tut-symbol-unify-fn* fs1 fs2)
		  *tut-fail-value*)))
	(t
	 (eq fs1 fs2))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-STRING-P
;;
;; Predicate for string unification checking

(defun tree-unify-string-p (fs1 fs2)
  (declare (string fs1))
  (tut-trace-progn
   (format *standard-output* "~&--------------------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-string-p]: checking strings~%")
   (pprint fs1)
   (pprint fs2))
  (cond ((not (stringp fs2))
	 nil)
	(*tut-string-unify-pred*
	 (funcall *tut-string-unify-pred* fs1 fs2))
	(*tut-string-unify-fn*
	 (not (eq (funcall *tut-string-unify-fn* fs1 fs2)
		  *tut-fail-value*)))
	(t
	 (string-equal fs1 fs2))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-NUMBER-P
;;
;; Predicate for number unification checking

(defun tree-unify-number-p (fs1 fs2)
  (declare (number fs1))
  (tut-trace-progn
   (format *standard-output* "~&TUT [tree-unify-number-p]: checking numbers~%")
   (format *standard-output* "~&--------------------------------------~%")   
   (pprint fs1)
   (pprint fs2))
  (cond ((not (numberp fs2))
	 nil)
	(*tut-number-unify-pred*
	 (funcall *tut-number-unify-pred* fs1 fs2))
	(*tut-number-unify-fn*
	 (not (eq (funcall *tut-number-unify-fn* fs1 fs2)
		  *tut-fail-value*)))
	(t 
	 (= fs1 fs2))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-OTHER-P
;;
;; Predicate for miscellaneous data type unification

(defun tree-unify-other-p (fs1 fs2)
  (tut-trace-progn
   (format *standard-output* "~&-----------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-other-p]: checking~%")
   (pprint fs1)
   (pprint fs2))
  (cond (*tut-other-unify-pred*
	 (funcall *tut-other-unify-pred* fs1 fs2))
	(*tut-other-unify-fn*
	 (not (eq (funcall *tut-other-unify-fn* fs1 fs2)
		  *tut-fail-value*)))
	(t 
	 (equalp fs1 fs2))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-OP-P
;;
;; Predicate for handling user defined operator unification checking

(defun tree-unify-op-p (fs1 fs2)
  (let ((op (car fs1)))
    (tut-trace-progn
     (format *standard-output* "~&-------------------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-op-p]: checking user-defined operators~%")
     (pprint fs1)
     (pprint fs2))   
    (cond ((fboundp (tut-op-pred-name op))
	   (funcall (tut-op-pred-name op) fs1 fs2))
	  ((fboundp (tut-op-fn-name op))
	   (not (eq (funcall (tut-op-fn-name op) fs1 fs2) *tut-fail-value*)))
	  (t
	   (error "~s has not been defined as a TUT operator.~%" op)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                     Unification Checker Interface                     ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-P
;;
;; Top-level unification checking predicate

(defun tree-unify-p (fs1 fs2)
  (cond ((null fs1)        fs2)
	((null fs2)        fs1)
	((and (tut-fs-p fs1)
	      (tut-fs-p fs2)   (tree-unify-fs-p        fs1 fs2)))
	((tut-multiple-p fs1)  (tree-unify-multiple-p  fs1 fs2))
	((tut-multiple-p fs2)  (tree-unify-multiple-p  fs2 fs1))
	((tut-or-p fs1)        (tree-unify-or-p        fs1 fs2))
	((tut-or-p fs2)        (tree-unify-or-p        fs2 fs1))
	((tut-not-p fs1)       (tree-unify-not-p       fs1 fs2))
	((tut-not-p fs2)       (tree-unify-not-p       fs2 fs1))
	((tut-op-p fs1)        (tree-unify-op-p        fs1 fs2))
	((tut-op-p fs2)        (tree-unify-op-p        fs2 fs1))
	((symbolp fs1)         (tree-unify-symbol-p    fs1 fs2))
	((symbolp fs2)         *tut-fail-value*)
	((stringp fs1)         (tree-unify-string-p    fs1 fs2))
	((stringp fs2)         *tut-fail-value*)
	((numberp fs1)         (tree-unify-number-p    fs1 fs2))
	((numberp fs2)         (tree-unify-number-p    fs2 fs1))
	(t                     (tree-unify-other-p     fs1 fs2))))




