;;;==================================================================;
;;; -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;==================================================================;
;;;
;;;               Center for Machine Translation
;;;                 Carnegie-Mellon University
;;;                                                                       
;;;------------------------------------------------------------------;
;;;                                                                       
;;; Copyright (c) 1995
;;; Carnegie Mellon University. All Rights Reserved.                      
;;;                                                                       
;;;------------------------------------------------------------------;
;;;
;;;          File: cache.lisp
;;;  File created:  2-May-95 by ndb
;;;        Author: Nicholas Brownlow <ndb@cs.cmu.edu>
;;; Last Modified: 20-Jun-95 at 13:54
;;;
;;;------------------------------------------------------------------;
;;; Description
;;;
;;; Code to support a fixed-size cache which, when full, replaces least
;;; recently used entries.


;;;==================================================================;

;;; Documentation
;;;
;;; This code implements a cache with the following features:
;;;
;;; - A hash table maps keys onto cached values.
;;; - The cache has a fixed size.
;;; - Cache entries are queued according to recency of usage.
;;; - When the cache is full and a new entry is added, the least recently used
;;;   entry is removed.
;;;
;;; Usage Example
;;; -------------
;;; 
;;; Create a new cache that can hold 2 entries in an EQ hash table:
;;;
;;; USER> (setf *cache* (create-cache :size 2 :test #'eq)) 
;;;
;;; Clear the cache hit/miss statistics:
;;;
;;; USER> (clear-cache-statistics *cache*)
;;;
;;; Add three items to the cache.  Since the cache is limited to two entries,
;;; when the :C entry is added the :A entry is automatically removed:
;;;
;;; USER> (cache-add *cache* :a 1)
;;; USER> (cache-add *cache* :b 2)
;;; USER> (cache-add *cache* :c 3)
;;;
;;; Print the cache to confirm its contents.  Entries are listed in decreasing
;;; order of recency of usage:
;;;
;;; USER> (print-cache *cache*) 
;;; :C -=> 3
;;; :B =-> 2
;;;
;;; Access the :B entry (present) and the :A entry (absent):
;;; 
;;; USER> (cache-get *cache* :b) 
;;; 2
;;; USER> (cache-get *cache* :a)
;;; NIL
;;;
;;; Now, when the cache is listed, the :B entry comes first:
;;;
;;; USER> (print-cache *cache*)
;;; :B -=> 2
;;; :C =-> 3
;;;
;;; Of the two cache accesses, one was a hit and one was a miss:
;;;
;;; USER> (print-cache-statistics *cache*)
;;; Total:  2
;;; Hits:   1
;;; Misses: 1
;;; Hit %:  0.5
;;; Miss %: 0.5
;;;
;;; Finally, remove the :C entry:
;;;
;;; USER> (cache-remove *cache* :c) 


;;;==================================================================;

;;; Package Statements

(in-package :user)


;;;==================================================================;

;;; Structures

(defstruct cache
  "A CACHE contains the following fields:
TABLE  -- hash table of CACHEITEMs
SIZE   -- maximum size of cache
FIRST  -- first CACHEITEM in usage queue (most recently used)
LAST   -- last CACHEITEM in usage queue (least recently used)
HITS   -- number of CACHE-GET hits
MISSES -- number of CACHE-GET misses"
  table
  (size 0 :type integer)
  (first nil :type list)
  (last nil :type list)
  (hits 0 :type integer)
  (misses 0 :type integer)) 

(defstruct cacheitem
  "A CACHEITEM contains the following fields:
KEY   -- hash table key
VALUE -- value corresponding to KEY
PREV  -- previous CACHEITEM in usage queue
NEXT  -- next CACHEITEM in usage queue"
  key
  value
  (prev nil :type list)
  (next nil :type list)) 


;;;==================================================================;

;;; Create, clear, and print CACHEs

(defun create-cache (&key (size 10) (test #'eql))
  "Creates and returns a new CACHE of the given SIZE with a hash table that uses TEST."
  (make-cache :table (make-hash-table :size size :test test)
	      :size size)) 


(defun clear-cache-statistics (cache)
  "Clears the CACHE hit/miss statistics."
  (setf (cache-hits cache) 0
	(cache-misses cache) 0))

(defun clear-cache (cache)
  "Clears the CACHE.  Since the cache purges least recently used entries, it
should never be necessary to clear the cache."
  (clrhash (cache-table cache))
  (setf (cache-first cache) nil
	(cache-last cache) nil)
  (clear-cache-statistics cache))


(defun print-cache-statistics (cache &optional (stream *standard-output*))
  "Prints the CACHE hit/miss statistics."
  (let ((total (+ (cache-hits cache) (cache-misses cache))))
    (format stream "Total:  ~D~%Hits:   ~D~%Misses: ~D~%Hit %:  ~F~%Miss %: ~F~%"
	    total (cache-hits cache) (cache-misses cache)
	    (/ (cache-hits cache) (float total))
	    (/ (cache-misses cache) (float total)))))

(defun print-cache (cache &optional (stream *standard-output*))
  "Prints the CACHE.  Entries are listed in decreasing order of recency of
usage.  Each entry is shown as KEY ==> VALUE."
  (do ((cacheitem (cache-first cache) (cacheitem-next cacheitem)))
      ((null cacheitem))
    (format stream "~S ~A~A> ~S~%"
	    (cacheitem-key cacheitem)
	    (if (cacheitem-prev cacheitem) #\= #\-)
	    (if (cacheitem-next cacheitem) #\= #\-)
	    (cacheitem-value cacheitem))))


;;;==================================================================;

;;; Manage the CACHE usage queue of CACHEITEMS

(defun cache-insert-item-front (cache cacheitem)
  "Inserts CACHEITEM first in CACHE usage queue.  Returns CACHEITEM."
  (when (cache-first cache)
    (setf (cacheitem-prev (cache-first cache)) cacheitem))
  (unless (cache-last cache)
    (setf (cache-last cache) cacheitem))
  (setf (cacheitem-next cacheitem) (cache-first cache)
	(cacheitem-prev cacheitem) nil
	(cache-first cache) cacheitem))

(defun cache-remove-item (cache cacheitem)
  "Removes CACHEITEM from CACHE usage queue."
  (if (cacheitem-prev cacheitem)
      (setf (cacheitem-next (cacheitem-prev cacheitem))
	    (cacheitem-next cacheitem))
    (setf (cache-first cache) (cacheitem-next cacheitem)))
  (if (cacheitem-next cacheitem)
      (setf (cacheitem-prev (cacheitem-next cacheitem))
	    (cacheitem-prev cacheitem))
    (setf (cache-last cache) (cacheitem-prev cacheitem))))

(defun cache-move-item-front (cache cacheitem)
  "Moves CACHEITEM to the front of the CACHE usage queue.  Returns CACHEITEM."
  (when (cacheitem-prev cacheitem)
    ;; CACHEITEM is not already first
    (cache-remove-item cache cacheitem)
    (cache-insert-item-front cache cacheitem))
  cacheitem)


;;;==================================================================;

;;; Add/get/remove items to/from CACHEs

(defun cache-add (cache key value)
  "Adds/replaces a cache entry with VALUE for KEY in CACHE.  Moves entry to the
front of the usage queue for CACHE.  Returns VALUE."
  (let ((cacheitem (gethash key (cache-table cache))))
    (cond (cacheitem
	   ;; KEY is in CACHE
	   (setf (cacheitem-value cacheitem) value)
	   (cache-move-item-front cache cacheitem))
	  ((>= (hash-table-count (cache-table cache)) (cache-size cache))
	   ;; CACHE is full
	   (unless (cache-last cache)
	     (error "No room in cache"))
	   ;; Repurpose the last CACHEITEM on the usage queue
	   (setf cacheitem (cache-last cache))
	   (remhash (cacheitem-key cacheitem) (cache-table cache))
	   (setf (cacheitem-key cacheitem) key
		 (cacheitem-value cacheitem) value
		 (gethash key (cache-table cache)) cacheitem)
	   (cache-move-item-front cache cacheitem))
	  (t
	   ;; Add new CACHEITEM at front of usage queue
	   (setf cacheitem (make-cacheitem :key key
					   :value value)
		 (gethash key (cache-table cache)) cacheitem)
	   (cache-insert-item-front cache cacheitem)))
    value)) 


(defun cache-get (cache key)
  "Returns the value for KEY in CACHE, if present.  Moves any existing cache
entry to the front of the usage queue for CACHE.  Updates the CACHE
statistics."
  (let ((cacheitem (gethash key (cache-table cache))))
    (cond (cacheitem
	   (incf (cache-hits cache))
	   (cache-move-item-front cache cacheitem)
	   (cacheitem-value cacheitem))
	  (t
	   (incf (cache-misses cache))
	   nil))))


(defun cache-remove (cache key)
  "Removes the cache entry for KEY in CACHE.  Returns cache entry structure if
present."
  (let ((cacheitem (gethash key (cache-table cache))))
    (when cacheitem
      (remhash (cacheitem-key cacheitem) (cache-table cache))
      (cache-remove-item cache cacheitem))
    cacheitem))

