(format t "******************************************************~%")
(format t "**                LCFLEX Parser                     **~%")
(format t "**                 version 1-0                      **~%")
(format t "**                                                  **~%")
(format t "**        Written by Carolyn Penstein Rose          **~%")
(format t "**     Learning Research and Development Center     **~%")
(format t "**          University of Pittsburgh                **~%")
(format t "**                                                  **~%")
(format t "**        in collaboration with Alon Lavie          **~%")
(format t "**        Language Technologies Institute           **~%")
(format t "**          Carnegie Mellon University              **~%")
(format t "**                                                  **~%")
(format t "**        (c) 1998 All rights reserved              **~%")
(format t "******************************************************~%")

(defstruct edge cat p1 p2 needed children fstruct rule used skip insert unif)
(defvar *active-hold* nil)
(defvar *gram* nil)

(defvar *ambiguity-packing* t)
(defvar *twolevel* nil)
(defvar *lc-prediction* t) 
(defvar *skip-beam* nil) 
(defvar *pruning* t) ; You can't have pruning without ambiguity packing
(defvar *restarts* nil)
(defvar *poss-inserts* nil)
(defvar *unif-flex* nil)
(defvar *num-unifs* nil)
(defvar *no-dups* nil)
(defvar *trace-rules* nil)
(defvar *skip-limit* 0)
(defvar *goal* '(<start>))
(defvar *token-position* 0)
(defvar *lexmode* 'standard)

(defvar *inactive-hold* (make-array 100))
(defvar *optimize* t)
(defvar *hold-inactives* t)
(defvar *rule-bookkeeping* nil)
(defvar *dummies* (make-array 1000))
(defvar *insert-equal* nil)

(defvar edge-array (make-array 1000 :adjustable t :fill-pointer t))
(defvar active-edges (make-array 100))
(defvar inactive-edges-start (make-array 100))
(defvar inactive-edges-end (make-array 100))
(defvar amb-array (make-array 100))
(defvar *startfun*)
(defvar *tfuns* nil)

(defvar *loaded* nil)

(when (null *loaded*)

(defun set-first-flex ()
)

(defun set-second-flex ()
)

(defun unif-okay (list)
(if *num-unifs*
   (not (> (length list) *num-unifs*))
   t))

)

(setf *loaded* t)

;-------------------------------------------------------------------------
;                       Getting Lexical Entries
;-------------------------------------------------------------------------

(defun get-lex-ents (word)
 (if (equal *lexmode* 'standard)
   (list (list word (list '(lex +) (list 'value word))))
  (let ((ents (cadr (assoc word *lex*))))
     (if (equal (car ents) '*or*)
         (mapcar #'(lambda (e) (get-lex-aux e)) (cdr ents))
         (list (get-lex-aux ents))))))

(defun get-lex-aux (ent)
  (list (lexsym ent) ent))

(defun lexsym (ent)
  (with-input-from-string (st (format nil "<~a>" (cadr (assoc 'cat ent))))
     (read st)))


;------------------------------------------------------------------------
; Top Level Parsing Functions
;------------------------------------------------------------------------

; This function converts the rule list in *grammar-table* (from GLR*) to
; a format that can be used for left corner prediction.  Rules are indexed
; by the leftmost daughter category.  For each rule, the RHS category, 
; daughter categories to the left of the leftmost daughter (nil),
; the daughter categories to the right of the leftmost daughter, and the name
; of the associated unification rule are listed.

(defun init-grammar (grafile)
  (setf *insert-equal* nil)
  (compgra grafile)
  (loadgra grafile)
  (make-lc-table)
  (convert-lc)
  (write-lc)
  (read-lc)
  (setup-grammar)
  (make-ord-table)
  (convert-ord)
  (write-ord)
  (read-ord)
  (if *poss-inserts*
     (progn
       (get-insert-equ-table)
       (make-lc-table)
       (convert-lc)
       (write-lc)
       (read-lc)
       (setup-grammar)
       (make-ord-table)
       (convert-ord)
       (write-ord)
       (read-ord))))

(defun init-grammar-short (grafile)
  (compgra grafile)
  (read-lc)
  (setup-grammar))

(defun setup-grammar ()
(let ((ent nil))
  (setf *gram* (make-hash-table))
  (setf *gg* nil)
  (dotimes (x (length *grammar-table*))
    (setf *gg* (push (aref *grammar-table* x) *gg*)))
  (setf *gg* (reverse *gg*))
  (setf *startfun* (caddr (aref *grammar-table* 0)))
  (dotimes (x (length *grammar-table*))
      (setq ent (aref *grammar-table* x))
      (dolist (e (rule-insert-sets (cadr ent)))
        (if (not (and (equal 1 (length (cadr e)))
                      (equal (car (cadr e)) (car ent))))
            (push (list (car ent) nil (cadr e) (caddr ent) (caddr e))
	        (gethash (car e) *gram*)))))))

(defun setup-tfuns ()
 (let ((temp nil))
  (setf *tfuns* 
       (remove nil (mapcar #'(lambda (rule) 
            (setf temp (find-rule-fun (car rule) (cadr rule)))
            (if temp (list temp rule))) *trace-rules*)))))

(defun rule-insert-sets (lhs)
  (do ((pre (list (car lhs)))
       (set (list (list (car lhs) lhs nil))))
     ((null (can-insert-cats-ext pre)) set)
     (pop lhs)
     (push (list (car lhs) lhs (reverse pre)) set)
     (setf pre (push (car lhs) pre))))
     
; This is the top level parsing function.  No $ is required at the end of
; the sentence as it is in GLR*.
(defun parse (sentence)
  (setf *token-position* 0)
  (setup-tfuns)
  (setf sentence (ap-sep sentence))
  (initialize-chart sentence) 
  (build-edge nil nil nil nil nil '((dummy +)(counter 1)) nil nil nil nil)
  (setup-dummy-cats (length sentence))
  (if *twolevel*
      (twolevel-lc-parse *goal* sentence)
      (lc-parse *goal* sentence))
  (disable-rule-bookkeeping)
  (if *restarts*
      (all-analyses (ap-sep sentence))
      (combine (mapcar #'(lambda (g)
      (intersection (get-inactive-start 0 g)
                (get-inactive-end (length sentence) g))) *goal*))))

(defun parse-result (sentence)
  (if *restarts*
      (all-analyses (ap-sep sentence))
      (combine (mapcar #'(lambda (g)
      (intersection (get-inactive-start 0 g)
                (get-inactive-end (length sentence) g))) *goal*))))

(defun all-analyses (sent)
(let ((edges nil))
  (dotimes (x (length sent))
    (dolist (g *goal*)
     (setf edges (append edges 
         (cdr (assoc g (aref inactive-edges-start x)))))))
  (sort edges #'edspan)))

(defun edspan (ed1 ed2)
  (if (> (- (edge-p2 (aref edge-array ed1)) (edge-p1 (aref edge-array ed1)))
         (- (edge-p2 (aref edge-array ed2)) (edge-p1 (aref edge-array ed2))))
      t nil))


; Clear off all chart data structures.
(defun initialize-chart (sentence)
  (setf *active-hold* nil)
  (dotimes (x 100) (setf (aref *inactive-hold* x) nil))
  (setf (fill-pointer edge-array) 0)
  (dotimes (x (+ (length sentence) 2))
     (setf (aref active-edges x) nil)
     (setf (aref inactive-edges-start x) nil)
     (setf (aref inactive-edges-end x) nil)))

; return the fstructure from the first inactive edge spanning q0 to q that has
; category cat.
(defun return-analyses (cat q0 q)
  (mapcar #'(lambda (ed) (edge-fstruct (aref edge-array ed)))
	  (remove nil (list (car (intersection (get-inactive-start q0 cat)
			(get-inactive-end q cat)))))))



; This function allows you to see how edge creation was distributed over
; a particular sentence
(defun anal-edges (sent)
  (let ((count (make-array (+ 1 (length sent)) :initial-element 0)))
    (dotimes (x (fill-pointer edge-array))
       (setf (aref count (edge-p2 (aref edge-array x)))
	     (+ 1 (aref count (edge-p2 (aref edge-array x))))))
    (dotimes (x (+ 1 (length sent)))
	(print (aref count x)))))

(defun twolevel-lc-parse (cat sent)
  (setf sent (ap-sep sent))
  (setf *input-list* (make-input-list sent))
  (setf *active-hold* nil)
  (set-first-flex)
  ; Parsing: first stage (restarts only)
  (dotimes (x (length sent))
     (setf *token-position* (+ *token-position* 1))
     (init-rule-bookkeeping)
     ; insert inactive '<start> edges at each non-zero vertex in order to
     ; allow for restarts.
	     (dotimes (y x) ; copy references to active edges from previous 
                            ; vertices until beam is filled up.
	     	(if (or (not (or *skip-beam* *skip-limit*))
                        (and *skip-beam* (null *skip-limit*)
                             (< (length (get-active x)) *skip-beam*))
                        (and *skip-limit* (null *skip-beam*)
                             (< y *skip-limit*))
                        (and (< y *skip-limit*) (< (length (get-active x)) *skip-beam*)))
	     	         (progn 
                                (copy-actives (- (- x y) 1) x)
                                )))

             ; pack active edges ending after the current word.
	     (packactives-pos x) 
	     (print (elt sent x))
     ; Loop once for each lexical entry for the current word
     (reset-inactives)
     (dolist (ent (get-lex-ents (elt sent x)))
             
             ; insert an inactive edge for the current lexical entry
             (insert-inactive (car ent) x (+ x 1) nil
		  (cadr ent) nil nil nil))
             (process-inactive-hold (+ x 1))
             (ambiguity-packing x (+ x 1))
             (do ()
                ((null *active-hold*) t)
                (process-active-hold (+ x 1))
                (if (null *active-hold*)
                    (process-inactive-hold (+ x 1)))))

  (let ((curp1 nil)
        (curp2 nil))
  (setf *active-hold* nil)
  (set-second-flex)
  ; in get-rangeelts, the sentence will be segmented into contiguous
  ; portions acording to the restarts analysis done in the first stage.
  ; each rangeelt covers one span of text and includes all of the
  ; inactive edges (possibly packed) that span the associated range and are
  ; not a parent of any other inactive edge spanning that range.
  (dolist (rangeelt (get-rangeelts sent))
     (init-rule-bookkeeping)
     (setf curp1 (edge-p1 (aref edge-array (car rangeelt))))
     (setf *token-position* curp1)
     (setf curp2 (edge-p2 (aref edge-array (car rangeelt))))
     (dotimes (x curp1)
	     	(if (or (not (or *skip-beam* *skip-limit*))
                        (and *skip-beam* (null *skip-limit*)
                             (< (length (get-active curp1)) *skip-beam*))
                        (and *skip-limit* (null *skip-beam*)
                             (< x *skip-limit*))
                        (and (< x *skip-limit*) (< (length (get-active curp1)) *skip-beam*)))
                    (copy-actives (- (- curp1 x) 1) curp1)))
        (packactives-pos curp1)
        ; this is very similar to process inactive edge, except that a list
        ; of inactive edges spanning the same range are inserted instead of
        ; just one.  Also, a new inactive edge does not have to be created
        ; for each since they already exist.  The appropriate reference simply 
        ; needs to be inserted in the appropariate place in the chart.
        (process-rangeelt rangeelt)
        (process-inactive-hold curp2)
        (ambiguity-packing curp1 curp2)
        (do ()
            ((null *active-hold*) t)
            (process-active-hold curp2)
            (if (null *active-hold*)
                (process-inactive-hold curp2))))))

;------------------------------------------------------------------------
; Intermediate parsing functions
;------------------------------------------------------------------------

(defun get-insert-equ-table ()
  (parse nil)
  (setf *input-list* '((1 dummy)))
  (get-insert-equ-table-aux)
  (setf *insert-equal* 
      (mapcar #'(lambda (nt) (list (car nt) 
           (mapcar #'(lambda (ed) 
             (edge-insert (aref edge-array ed))) (cdr nt)))) 
                (aref inactive-edges-start 0))))

(defun get-insert-equ-table-aux ()
 (dolist (rule (gethash nil *gram*))
     (insert-inactive-ext 
        (elt rule 0) 0 0 nil (do-unification-test (getargs (length (car (last rule)))) (elt rule 3)) 
             nil (elt rule 4) nil))
 (if *hold-inactives* (process-inactive-hold 0))
 (do ((hold nil))
   ((null *active-hold*) t)
      (process-active-hold 0)
      (process-inactive-hold 0)))
    

(defun getargs (n)
  (let ((res nil))
    (dotimes (x n) (push 0 res))
    res))

(defun insert-inactive-ext (cat p1 p2 children unif-res skip insert unif)
   (cond ((equal t (car unif-res))
          (insert-inactive cat p1 p2 children (cadr unif-res) skip insert unif))
         ((not (null (cadr unif-res)))
          (insert-inactive cat p1 p2 children (cadr unif-res) skip insert (append unif (list (caddr unif-res)))))
         (t nil)))

(defun headed-rules-short-lex (lex)
  (mapcar #'(lambda (rule) (list (car rule) (cadr rule) (list lex) (cadddr rule) (caddddr rule))) (gethash '% *gram*)))

(defun caddddr (list) (cadr (cdddr list)))

(defun insert-inactive-old (cat p1 p2 children struct skip insert unif)
  (if struct
      (let ((edgenum (build-edge cat p1 p2 nil children struct nil skip 
                           insert unif))
	    (predicted (alt-predicted p1))
            (extendhold nil))
           (insert-inactive-start p1 edgenum cat)
           (insert-inactive-end p2 edgenum cat)

           ; loop once for each rule that has the category of the current
           ; edge as its leftmost daughter category
           (dolist (rule (append (headed-rules-short cat)
			         (if (lex-p struct)
			             ;(headed-rules-short '%)
                                     (headed-rules-short-lex cat)
                                     )))

                   ; if top-down filtering is turned on and the RHS 
                   ; category of the current rule is not predicted,
                   ; don't do anything.
             (cond ((and *lc-prediction* 
                         (not (head-link-short (car rule) predicted)))
	             nil)
                   ((not (can-insert-cats-ext (append (elt rule 4) insert))) nil)
                   ; if the current rule has more than one daughter, create
                   ; a new active edge, but put in on hold until all of the
                   ; edges that will become inactive after the current word
                   ; is processed have been processed.
	           ((> (length (elt rule 2)) 1)
	            (hold-active (car rule) p1 p1 (elt rule 2) 
                         (children-from-inserts (elt rule 4) p1)
			  	 (elt rule 3) nil (elt rule 4) nil))

                   ; if the current rule has only one daughter, create a new
                   ; inactive edge from that rule and insert it into the chart.
	           (t (insert-inactive-ext (car rule) p1 p2 
                          (append (children-from-inserts (elt rule 4) p1) 
                                  (list edgenum))
				 (do-unification-ext 
                                      (append (children-from-inserts
                                          (elt rule 4) p1) (list edgenum))
						 (elt rule 3)) skip 
                                      (append insert (elt rule 4)) unif))))

           ; extend the active edges that end where the current edge begins
           (dolist (edge (get-active p1))
               (if (and
                    (member cat (edge-needed (aref edge-array edge)))
                    (setf extendhold
                     (can-extend cat (edge-needed (aref edge-array edge))
                        (edge-insert (aref edge-array edge)) p1)))
	           (extend-active (aref edge-array edge) edgenum (cadr extendhold)))))))

(defun can-extend (cat needed inserted pos)
  (let ((insert nil))
    (do ((c (pop needed) (pop needed)))
      ((equal c cat) t)
      (push c insert))
    (if (can-insert-cats-ext (append insert inserted))
        (list t (children-from-inserts (reverse insert) pos))
         nil)))

(defun shorten-by-len (list1 list2)
  (if list2
      (shorten-by-len (cdr list1) (cdr list2))
      list1))

; This function extends the current active edge over the new inactive child 
; edge
(defun extend-active (edge newchild newinsert)
  (let ((children nil))
             ; If the new child edge is not enough to make this edge inactive,
             ; create a new active edge that will eventually be extended over 
             ; the child edge, but put it on hold for now.
       (cond ((and *skip-limit*
                 (> (length (append (edge-skip edge)
                            (addskip (edge-p2 edge) 
                              (edge-p1 (aref edge-array newchild)))))
                      *skip-limit*)) nil)
             ((> (length (edge-needed edge)) (+ 1 (length newinsert)))
              (hold-active (edge-cat edge)
		   (edge-p1 edge)
                   ;(edge-p2 edge)
		   (edge-p1 (aref edge-array newchild)) ; to make p2 reliable
		   (shorten-by-len (edge-needed edge) newinsert)
		   (append (edge-children edge) newinsert)
		   (edge-rule edge)
                   (append (edge-skip edge)
                      (addskip (edge-p2 edge)
                        (edge-p1 (aref edge-array newchild))))
                   ;(append (edge-skip edge) 
                   ;        (addskip 
                   ;           (edge-p2 edge) 
                   ;           (edge-p1 (aref edge-array newchild))))
                   (edge-insert-edge)
                   ;(append 
                   ;   (append (edge-insert edge) 
                   ;        (edge-insert (aref edge-array newchild)))
                   ;   (mapcar #'(lambda (newch) 
                   ;       (edge-cat (aref edge-array newch))) newinsert))
                   (edge-unif edge)
                   ;(append (edge-unif edge)
                   ;        (edge-unif (aref edge-array newchild)))
                  ))

              ; if the new child is enough to make the current active edge 
              ; inactive, and the current active edge is packed, unpack the
              ; edge and extend each of the edges packed inside the current
              ; active edge and then the edge itself.
	      ((listp (edge-rule edge))
	       (let ((hold nil)
	             (chil nil)
	             (need nil)
	             (p2 nil))

                  ; Loop for each packed edge       
	          (dolist (ed (cdr (edge-rule edge)))
	             (setf hold (aref edge-array ed))
	             (setf chil (edge-children hold))
	             (setf need (edge-needed hold))
	             (setf p2 (edge-p2 hold))
	             (setf (edge-p2 hold) (edge-p2 edge))
	             (setf (edge-children hold)
		           (append (edge-children hold)
			           (lastn (- (length (edge-needed hold))
				             (length (edge-needed edge)))
				          (edge-children edge))))
	             (setf (edge-needed hold) 
                           (rembeg (edge-needed hold)
                               (- (length (edge-needed hold))
                                  (length (edge-needed edge)))))
	             (extend-active hold newchild newinsert)
	             (setf (edge-p2 hold) p2)
	             (setf (edge-needed hold) need)
	             (setf (edge-children hold) chil))

                   ; extend the current edge
	           (extend-active (aref edge-array
				    (build-edge (edge-cat edge) (edge-p1 edge)
				      (edge-p2 edge) (edge-needed edge)
				      (edge-children edge) nil
				      (car (edge-rule edge))
                                      (edge-skip edge)
                                      (edge-insert edge)
                                      (edge-unif edge)))
			            newchild newinsert)))

                 ; if the current active edge is not packed and only needs
                 ; the new child in order to become inactive, create an
                 ; inactive edge and place it in the chart.
	      (t (insert-inactive-ext (edge-cat edge) (edge-p1 edge)
		                  (edge-p2 (aref edge-array newchild))
		                  (setf children
			                (append (append (edge-children edge) 
                                                   newinsert)
                                                (list newchild)))
		                  (do-unification-ext children (edge-rule edge))
                                  (append (edge-skip edge)
                                          (append (addskip 
                                                     (edge-p2 edge)
                                                     (edge-p1 
                                                       (aref edge-array 
                                                             newchild)))
                                                  (edge-skip (aref edge-array 
                                                                   newchild))))
                                   (append (append (edge-insert edge)
                                         (mapcar #'(lambda (newch) (edge-cat (aref edge-array newch))) newinsert))
                                           (edge-insert 
                                              (aref edge-array newchild)))
                                   (append (edge-unif edge)
                                           (edge-unif
                                              (aref edge-array newchild))))))))



(defun rembeg (list n)
  (let ((res list))
    (dotimes (x n)
      (setf res (cdr res)))
    res))

; returns the last n elements of list
(defun lastn (n list)
  (let ((res nil))
    (dotimes (x n)
       (setf res (cons (car (last list)) res))
       (setf list (butlast list)))
  res))

; as long as there isn't already an active edge on hold with all of the
; same information, one will be created and placed in the *active-hold* list
(defun hold-active (cat p1 p2 needed children rule skip insert unif)
  (if (not (duplicate-active cat p1 p2 needed children rule skip insert unif))
      (push (build-edge cat p1 p2 needed children nil rule skip insert unif) 
            *active-hold*)))

(defun duplicate-active (cat p1 p2 needed children rule skip insert unif)
  (let ((stat nil)
	(edge nil))
    (dolist (ed *active-hold*)
       (setf edge (aref edge-array ed))
       (if (and (equal rule (edge-rule edge))
		(equal p1 (edge-p1 edge))
		(equal p2 (edge-p2 edge))
		(equal children (edge-children edge))
		(equal needed (edge-needed edge))
                (not (insert-subsume insert (edge-insert edge)))
                (not (unif-subsume unif (edge-unif edge)))
                (not (skip-subsume skip (edge-skip edge))))
	   (progn
	     (setf stat t))))
    stat))

; Extend the active edges that are on hold
(defun process-active-hold (pos)
(let ((hold *active-hold*))
(setf *active-hold* nil)
  ; loop for each active edge on hold
  (dolist (act hold)
     (let* ((edge (aref edge-array act))
            (skip nil)
            (unif nil)
            (insert nil)
	    (new-children (get-inactive-start (edge-p2 edge)
					      (car (edge-needed edge))))
	    (new-children2 (get-inactive-end pos (car (edge-needed edge))))

            ; childnums is a list of references to inactive edges of the
            ; category needed by the current active edge
	    (childnums (intersection new-children new-children2))
	    (child (aref edge-array (car childnums))))

       ; the current active edge can be modified once, so for all of the
       ; new child inactive edges except for the first one, a copy of the
       ; current active edge must be made first before extending the active
       ; edge over the associated inactive child edge.  This happens inside
       ; process-active-hold-aux
       (process-active-hold-aux edge (cdr childnums))
       ; for the first child inactive edge, modify the active edge so that
       ; it includes the child inactive edge as it's next child.  Make
       ; necessary adjustments to the edge in terms of span, skips, children,
       ; and needed.
       
       (setf skip (append (edge-skip edge)
                    (append (addskip (edge-p2 edge) (edge-p1 child)) 
                          (edge-skip child))))
       (setf (edge-p2 edge) (edge-p2 child))
       (setf insert (append (edge-insert edge) (edge-insert child)))
       (setf unif (append (edge-unif edge) (edge-unif child)))
       (if (and (can-insert-cats-ext insert)
                (unif-okay unif)
                (or (null *skip-limit*)
                    (not (> (length skip) *skip-limit*))))
          (progn
       (setf (edge-children edge) (append (edge-children edge)
					  (list (car childnums))))
       (setf (edge-needed edge) (cdr (edge-needed edge)))
       (setf (edge-skip edge) skip)
       (setf (edge-insert edge) insert)
       (setf (edge-unif edge) unif)
       ; insert the modified active edge into the chart
       (insert-active (edge-p2 edge) act)
       ))))))

(defun process-active-hold-aux (edge childnums)
  (let ((cat (edge-cat edge))
        (skip nil)
        (unif nil)
        (insert nil)
	(p1 (edge-p1 edge))
	(needed (cdr (edge-needed edge)))
	(children (edge-children edge))
	(fstruct nil)
	(rule (edge-rule edge))
	(child nil))
    (dolist (ch childnums)
	    (setf child (aref edge-array ch))
            (setf skip          (append (edge-skip edge)
                                        (append
                                            (addskip (edge-p2 edge) 
                                                     (edge-p1 child))
                                            (edge-skip child))))

            (setf insert (append (edge-insert edge) (edge-insert child)))
            (setf unif (append (edge-unif edge) (edge-unif child)))
            (if (and (can-insert-cats-ext insert)
                     (unif-okay unif)
                     (or (null *skip-limit*)
                         (not (> (length skip) *skip-limit*))))
	       (insert-active (edge-p2 child)
			   (build-edge cat p1 (edge-p2 child) needed
			        (append children (list ch))
			        fstruct rule skip insert unif))))))

; Copy forward references to active edges ending at previous vertices.
; This is how skipping takes place.  All of the words in between the
; previous vertex and the current vertex are skipped.
(defun copy-actives (frompos topos)
 (if (get-active frompos)
  (progn
    (dolist (act (get-active frompos))
      (if (and (equal (edge-p2 (aref edge-array act)) frompos)
               (or (null *skip-limit*)
                   (not (> (length (edge-skip (aref edge-array act))) 
                       *skip-limit*))))
          	  (insert-active topos act))))))

; this function returns true if the fstructure passed in is a lexical entry
; This is sort of specific to our grammar
(defun lex-p (struct)
  (and (listp struct)
       (not (member (car struct) '(*multiple* *or*)))
       (equal (cadr (assoc 'lex struct)) '+)))

; Returns the list of categories that are the next needed child in the set
; of active edges ending at vertex number pos.
(defun alt-predicted (pos)
  (cond ((null *lc-prediction*) nil)
        ((equal pos 0) *goal*)
        (*poss-inserts*
         (let ((res nil)
	         (cat nil))
            (dolist (ed (get-active pos))
                (dolist (cat (lc-w-inserts (edge-needed (aref edge-array ed))))
                  (if (not (member cat res))
	             (push cat res))))
            (if *restarts* (append *goal* res)
             res)))
        (t
           (let ((res nil)
	         (cat nil))
            (dolist (ed (get-active pos))
                (setf cat (car (edge-needed (aref edge-array ed))))
                (if (not (member cat res))
	           (push cat res)))
            (if *restarts* (append *goal* res)
             res)))))

(defun lc-w-inserts (catlist)
 (do ((res nil)
      (cur nil))
    ((not (can-insert-cats-ext res)) res)
    (push (pop catlist) res)))

(defun build-edge (cat p1 p2 needed children fstruct rule skip insert unif)
  (vector-push-extend (make-edge :cat cat
				 :p1 p1
				 :p2 p2
				 :needed (copy-list needed)
				 :children (copy-list children)
				 :fstruct fstruct
				 :rule (if (listp rule)
					  (copy-list rule)
					  rule)
				 :skip skip
                                 :insert insert
                                 :unif unif)
		      edge-array)
  (- (fill-pointer edge-array) 1))

(defun do-unification (children rule)
(if (check-rule-bookkeeping t children rule)
 (let ((res
  (apply rule (mapcar #'(lambda (ch)
		  (setf (edge-used (aref edge-array ch)) t)
		  (edge-fstruct (aref edge-array ch)))
		      children)))
       (rule-info nil))
  (if (setf rule-info (assoc rule *tfuns*))
      (progn
        (format t "~%------------------------------------------------~%")
        (format t "RULE: ~a~%" (cadr rule-info))
        (format t "CHILDREN:~%")
        (mapcar #'(lambda (ch) (format t "~a ~a~%~%" 
                      (edge-cat (aref edge-array ch)) 
                      (edge-fstruct (aref edge-array ch)))) children)
        (format t "RESULT: ~a ~a~%" (car (cadr rule-info)) 
           res)
        (format t "------------------------------------------------~%")
        ))
 (if *no-dups*
    (remove-dup-ilts res)
    res))))


; This function computes the top down filtering.  If cat is in the transitive
; or reflexive closure of the left corner relation for goal return t, otherwise
; return nil.
(defun head-link-short (cat goal)
  (setf goal (remove nil goal))
  (do ((stat nil)
       (cur nil))
      ((or (equal stat t) (null (setf cur (pop goal)))) stat)
      (if (or (equal cat cur)
	      (member cat (cadr (assoc cur *lc2*))))
	    (setf stat t))))

; returns a list of rules with cat as the category of the leftmost daughter
(defun headed-rules-short (cat)
  (gethash cat *gram*))

(defun make-input-list (sent)
 (let ((result nil))
   (dotimes (x (length sent))
     (push (list (+ x 1) (elt sent x)) result))
   result))

;---------------------------------------------------------------------------
; Ambiguity Packing Functions
;---------------------------------------------------------------------------

; if pruning is turned on, it takes place during the packing
(defun ambiguity-packing (p1 p2)
  (if *ambiguity-packing*
    (progn
      ; pack the active edges ending at p1
      (packactives p1)

      ; Pack inactive edges ending at p2.
      ; The edges are already divided into groups acording to their
      ; category.  Loop for each group.
      (dolist (group (aref inactive-edges-end p2))

        ; use amb-array to divide the edges into piles according to where
        ; the edges start
	(dotimes (x p2) (setf (aref amb-array x) nil))
	(dolist (ednum (cdr group))
	   (push ednum (aref amb-array (edge-p1 (aref edge-array ednum)))))

        ; pack the edges in each pile
	(dotimes (x p2) (if (> (length (aref amb-array x)) 1)
                            (dolist (pile (skip-piles (prune-inactive-list
			                                 (aref amb-array x))))
			     (ambiguity-packing-aux pile))))))))
                           

; Returns a list of piles that have more than one edge in them
; each pile corresponds to a set of words skipped
(defun skip-piles (edges)
  (let ((piles nil))
    (dolist (ed edges)
      (setf piles (list-by-skip-insert-unif
                      (list (edge-skip (aref edge-array ed))
                            (edge-insert (aref edge-array ed))
                            (edge-unif (aref edge-array ed))) ed piles)))
    (remove nil (mapcar #'(lambda (pile) (if (> (length (cdr pile)) 1)
                                             (cdr pile)))
                          piles))))


(defun list-by-skip-insert-unif (error ed list)
  (let ((orig (assoc error list :test #'error-equal)))
    (if orig
	(rplacd (assoc error list :test #'error-equal) (push ed (cdr orig)))
        (push (list error ed) list))
    list))

; makes sure you skipped and inserted exactly the same things
(defun error-equal (error1 error2)
 (if (and (equal (car error1)
              (car error2))
          (insert-error-equal (cadr error1) (cadr error2))
          (unif-error-equal (caddr error1) (caddr error2)))
      t nil))

(defun insert-error-equal (insert1 insert2)
 (let ((ins1 nil)
       (ins2 nil))
     (dolist (ins insert1)
         (setf ins1 (insert-needed ins ins ins1)))
     (dolist (ins insert2)
         (setf ins2 (insert-needed ins ins ins2)))
     (if (and (equal (length ins1) (length ins2))
              (null (set-difference ins1 ins2 :test #'equal)))
         t nil)))

(defun unif-error-equal (unif1 unif2)
   (if (and (equal (length unif1) (length unif2))
            (null (set-difference unif1 unif2 :test #'equal)))
       t nil))

(defun ambiguity-packing-aux (edges)
  (let ((fnum nil)
	(fst nil)
	(father nil)
	(hold (aref edge-array (car edges))))

    ; Find a father edge to pack the other edges into
    (dolist (ed edges)
	(if (and (null father)
		 (not (edge-used (aref edge-array ed))))
	    (progn
	      (setf father (aref edge-array ed))
	      (setf fnum ed))))

    ; if no suitable father edge could be found among the edges (shouldn't
    ; happen) create one.
    (if (null father)
	(progn
	  (setf fnum
	      (build-edge (edge-cat hold) (edge-p1 hold) (edge-p2 hold) nil
			  nil nil nil (edge-skip 
                                         (aref edge-array (car edges)))
                                      (edge-insert 
                                         (aref edge-array (car edges)))
                                      (edge-unif 
                                         (aref edge-array (car edges)))))
	  (setf father (aref edge-array fnum))
	  (insert-inactive-start (edge-p1 father) fnum (edge-cat father))
	  (insert-inactive-end (edge-p2 father) fnum (edge-cat father))))


    (setf (edge-children father) (if (and (edge-children father)
                                          (atom (car (edge-children father))))
				       (list (edge-children father))
                                       (edge-children father)))
    (setf fst (edge-fstruct father))
    (cond ((null fst) (setf (edge-fstruct father) (list '*or*)))
	  ((equal (car fst) '*or*) t)
	  (t (setf (edge-fstruct father) (list '*or* fst))))
    ; for each edge you pack into the father edge, add its fstruct to the
    ; accumulating list of fstructs in the father edge, and add its children
    ; to the accumulating list of children in the father edge - then 
    ; remove its reference from the chart (it will be implicitly represented in
    ; the chart by its father edge).
    (dolist (ed edges)
       (if (not (equal ed fnum))
	   (progn
	     (remove-inactive-start (edge-p1 father) ed (edge-cat father))
	     (remove-inactive-end (edge-p2 father) ed (edge-cat father))
             (if (atom (car (edge-children (aref edge-array ed))))
	         (setf (edge-children father)
		    (append (edge-children father)
			   (list (edge-children (aref edge-array ed)))))
                 (setf (edge-children father)
                       (append (edge-children father)
                               (edge-children (aref edge-array ed)))))
	     (setf fst (edge-fstruct (aref edge-array ed)))
	        (if (equal (car fst) '*or*)
		   (setf (edge-fstruct father) (append (edge-fstruct father)
						     (cdr fst)))
	           (setf (edge-fstruct father) (append (edge-fstruct father)
						     (list fst)))))))
           (setf (edge-children father) (remove nil (edge-children father)))
           (if *no-dups*
              (setf (edge-fstruct father) (remove-dup-ilts (edge-fstruct father))))))

(defun remove-dup-ilts (fstruct)
  (cond ((atom fstruct) fstruct)
        ((not (equal '*or* (car fstruct))) fstruct)
        (t
  (let ((res nil))
    (dolist (fst (cdr fstruct))
        (if (not (member fst res :test #'comp-ilt))
            (setf res (push fst res))))
    (if (> (length res) 1)
        (cons '*or* (reverse res))
        (car res))))))

(defun packactives-pos (pos)
 (aref active-edges pos))

; (let ((list nil))
;     (dolist (ed (get-active pos))
;	 (setf list (insert-needed (edge-needed (aref edge-array ed)) ed
;				   list)))
;     (setf (aref active-edges pos)
;	   (combine (mapcar #'(lambda (group) 
;                                      (packactives-aux (cdr group) pos))
;			    list)))))

(defun packactives (pos)
*active-hold*)

;  (let ((list nil))
;     (dolist (ed *active-hold*)
;	 (setf list (insert-needed (pa ed) ed list)))
;     (setf *active-hold*
;	   (combine (mapcar #'(lambda (group) 
;                                      (packactives-aux (cdr group) pos))
;			    list)))))

(defun pa (ed)
  (if *skip-limit*
      (list (edge-needed (aref edge-array ed))
            (edge-skip (aref edge-array ed)))
      (edge-needed (aref edge-array ed))))

(defun packactives-aux (list pos)
list)

;  (prune-active-list list pos))

;  (let ((edges nil)
;	(others nil)
;        (others2 nil)
;	(ed nil))
;    (dolist (edge (prune-active-list list pos))
;       (setf ed (aref edge-array edge))
;       (cond ((not (equal (edge-p2 ed) pos))
;	      (push edge others))
;             ((listp (edge-rule ed))
;              (push edge others2))
;	     (t (push edge edges))))
;    (if (> (length edges) 1)
;	(progn
;	   (setf (edge-rule (aref edge-array (car edges)))
;	              (append (cons (edge-rule (aref edge-array (car edges)))
;			    (cdr edges)) others2))
;           (cons (car edges) others))
;        (append (append edges others) others2))))
  

(defun insert-needed (needed ed list)
  (let ((orig (assoc needed list :test #'equal)))
    (if orig
	(rplacd (assoc needed list :test #'equal) (push ed (cdr orig)))
        (push (list needed ed) list))
    list))

;---------------------------------------------------------------------------
; Pruning Functions
;---------------------------------------------------------------------------

(defun prune-actives-pos (pos) 
   (if (and nil *pruning*)  ; disabled because it is not really safe to
                            ; prune active edges.  The one with less skipping
                            ; may not unify correctly.
       (let ((list nil))
            (dolist (ed (get-active pos))
	            (setf list (insert-needed 
                                   (edge-needed (aref edge-array ed))
                                 ed
	                         list)))
            (setf (aref active-edges pos)
	             (combine (mapcar #'(lambda (group) 
                                         (prune-active-list (cdr group) pos))
				   list))))))


(defun addskip (end start)
  (if (not (equal end start))
      (poslist (+ end 1) start)))

(defun poslist (from to)
   (let ((res nil))
       (dotimes (x (+ (- to from) 1))
          (push (+ x from) res))
       (reverse res)))

(defun skip-sort-edge (ed1 ed2)
  (if (skip-sort (edge-skip (aref edge-array ed1))
                 (edge-skip (aref edge-array ed2)))
       t nil))

(defun skip-sort (skip1 skip2)
  (or (< (length skip1)
         (length skip2))
      (skip-subsume skip1 skip2)))

(defun skip-subsume (skiplist1 skiplist2)
   (if (and skiplist2
            (not (equal skiplist1 skiplist2))
            (null (set-difference skiplist1 skiplist2)))
       t nil))

(defun insert-subsume (insert1 insert2)
 (let ((ins1 nil)
       (ins2 nil))
     (dolist (ins insert1)
         (setf ins1 (insert-needed ins ins ins1)))
     (dolist (ins insert2)
         (setf ins2 (insert-needed ins ins ins2)))
     (if (and ins2
              (null (and (equal (length ins1) (length ins2))
                         (null (set-difference ins1 ins2 :test #'equal))))
              (null (set-difference ins1 ins2 :test #'lsub)))
         t nil)))

(defun lsub (l1 l2)
  (if (and (equal (car l1) (car l2))
           (not (< (length l1) (length l2))))
      t nil))

(defun unif-subsume (unif1 unif2)
   (insert-subsume unif1 unif2))

;   (if (and unif2
;            (not (null (set-difference unif2 unif1 :test #'equal)))
;            (null (set-difference unif1 unif2 :test #'equal)))
;       t nil))

(defun skip-subsume-edge (ed1 ed2)
  (skip-subsume 
       (edge-skip (aref edge-array ed1))
       (edge-skip (aref edge-array ed2))))

(defun skip-more (ed1 ed2)
  (> (length (edge-skip (aref edge-array ed1)))
     (length (edge-skip (aref edge-array ed2)))))

(defun skip-more-pos (ed1 ed2 pos)
  (> (+ (length (edge-skip (aref edge-array ed1)))
        (length (poslist (+ (edge-p2 (aref edge-array ed1)) 1) pos)))
     (+ (length (edge-skip (aref edge-array ed2)))
        (length (poslist (+ (edge-p2 (aref edge-array ed2)) 1) pos)))))

(defun prune-inactive-list (edges)
 (let ((res nil)
       (edge nil))
  (cond ((null *pruning*) (setf res edges))
        ((or *poss-inserts* *unif-flex*) (setf res (basic-prune-ext edges)))
        (t (setf res (basic-prune edges))))
  (dolist (ed (set-difference edges res))
    (setf edge (aref edge-array ed))
    (remove-inactive-start (edge-p1 edge) ed (edge-cat edge))
    (remove-inactive-end (edge-p2 edge) ed (edge-cat edge)))
  res))


; edges is a list of indexes into edge-array
(defun basic-prune (edges)
  (basic-prune-aux (sort edges #'skip-sort-edge)))

(defun basic-prune-ext (edges)
  ; first stage
  (setf edges (combine (mapcar #'(lambda (cluster) (basic-prune (cdr cluster)))
      (make-insert-unif-list edges))))
  ; second stage
  (combine (mapcar #'(lambda (edlist) (prune-unif (prune-insert edlist)))
               (make-skip-list edges))))

(defun make-insert-unif-list (edges)
   (let ((list nil))
     (dolist (ed edges)
        (setf list (list-by-insert-unif (list (edge-insert (aref edge-array ed))
                                              (edge-unif (aref edge-array ed))) ed list)))
      list))

(defun list-by-insert-unif (error ed list)
  (let ((orig (assoc error list :test #'short-error-equal)))
    (if orig
	(rplacd (assoc error list :test #'short-error-equal) (push ed (cdr orig)))
        (push (list error ed) list))
    list))

(defun short-error-equal (error1 error2)
 (if (and (insert-error-equal (car error1) (cadr error2))
          (unif-error-equal (cadr error1) (caddr error2)))
      t nil))

(defun make-skip-list (edges)
  (let ((list nil))
     (dolist (ed edges)
        (setf list (list-by-skip (edge-skip (aref edge-array ed)) ed list)))
     (mapcar #'(lambda (ed) (cdr ed)) list)))

(defun list-by-skip (skip ed list)
  (let ((orig (assoc skip list :test #'equal)))
    (if orig
	(rplacd (assoc skip list :test #'equal) (push ed (cdr orig)))
        (push (list skip ed) list))
    list))

(defun prune-unif (edgelist)
  (do* ((edges (sort edgelist #'unif-sort))
        (res (list (pop edges)))
        (cur (pop edges) (pop edges)))
       ((or (null cur)
           (member cur res :test #'rev-unif-subsume-edge)) res) 
      (push cur res)))

(defun unif-sort (ed1 ed2)
  (or (< (length (edge-unif (aref edge-array ed1)))
         (length (edge-unif (aref edge-array ed2))))
      (unif-subsume (edge-unif (aref edge-array ed1))
                    (edge-unif (aref edge-array ed2)))))

(defun rev-unif-subsume-edge (ed1 ed2)
  (unif-subsume (edge-unif (aref edge-array ed2))
                (edge-unif (aref edge-array ed1))))

(defun prune-insert (edgelist)
  (do* ((edges (sort edgelist #'insert-sort))
        (res (list (pop edges)))
        (cur (pop edges) (pop edges)))
       ((or (null cur)
           (member cur res :test #'rev-insert-subsume-edge)) res) 
      (push cur res)))

(defun insert-sort (ed1 ed2)
  (or (< (length (edge-insert (aref edge-array ed1)))
         (length (edge-insert (aref edge-array ed2))))
      (unif-subsume (edge-insert (aref edge-array ed1))
                    (edge-insert (aref edge-array ed2)))))

(defun rev-insert-subsume-edge (ed1 ed2)
  (insert-subsume (edge-insert (aref edge-array ed2))
                  (edge-insert (aref edge-array ed1))))

(defun basic-prune-pos (edges pos)
  (if (or *poss-inserts* *unif-flex*)
      (basic-prune-pos2 edges pos)
      (basic-prune-pos1 edges pos)))

; This basic prune function is only used when insertions are possible
(defun basic-prune-pos2 (edges pos)
  ; first stage
  (setf edges (combine (mapcar #'(lambda (cluster) 
                                     (basic-prune-pos1 (cdr cluster) pos))
      (make-insert-unif-list edges))))
  ; second stage
  (combine (mapcar #'(lambda (edlist) (prune-unif (prune-insert edlist)))
               (make-skip-pos-list edges pos))))

(defun make-skip-pos-list (edges pos)
  (let ((list nil))
     (dolist (ed edges)
        (setf list (list-by-skip (append (edge-skip (aref edge-array ed))
                                         (poslist (+ (edge-p2 
                                             (aref edge-array ed)) 1) pos))
                                 ed list)))
     (mapcar #'(lambda (ed) (cdr ed)) list)))

(defun basic-prune-pos1 (edges pos)
(let ((res
  (basic-prune-pos-aux 
    (sort edges #'(lambda (ed1 ed2)
        (skip-sort    (append (edge-skip (aref edge-array ed1)) 
                           (poslist (+ (edge-p2 (aref edge-array ed1)) 1) pos))
                      (append (edge-skip (aref edge-array ed2)) 
                           (poslist (+ (edge-p2 (aref edge-array ed2)) 1) pos))
     )))
     pos)))
  res))

(defun basic-prune-aux (edges)
  (do ((res (list (pop edges)))
       (cur (pop edges) (pop edges)))
      ((or (null cur)
           (member cur res :test #'rev-skip-subsume-edge)) res) 
      (push cur res)))

(defun rev-skip-subsume-edge (ed1 ed2)
  (skip-subsume-edge ed2 ed1))

(defun basic-prune-pos-aux-old (edges pos)
  (do ((res (list (pop edges)))
       (cur (pop edges) (pop edges)))
      ((or (null cur)
           (skip-more-pos cur (car res) pos)) res)
      (if (not (skip-subsume
                   (append (edge-skip (aref edge-array cur))
                           (poslist (+ (edge-p2 (aref edge-array cur)) 1) pos))
                   (append (edge-skip (aref edge-array (car res)))
                           (poslist (+ (edge-p2 (aref edge-array (car res))) 1)
                                    pos))))
          (push cur res))))

(defun basic-prune-pos-aux (edges pos) 
  (do ((res (list (pop edges)))
       (cur (pop edges) (pop edges)))
      ((or (null cur)
           (member cur res :test #'(lambda (e1 e2)
               (skip-subsume (append (edge-skip (aref edge-array e2))
                              (poslist (+ (edge-p2 (aref edge-array e2)) 1) pos))
                             (append (edge-skip (aref edge-array e1))
                               (poslist (+ (edge-p2 (aref edge-array e1)) 1) pos))
                     ))))
               res) 
      (push cur res)))

(defun prune-active-list (edges pos)
  (cond ((and edges *pruning*)
         (combine (mapcar #'(lambda (clus) (basic-prune-pos clus pos))
                          (cluster-edges edges))))
        (t edges)))

(defun cluster-edges (edges)
   (do* ((sorted (sort edges #'active-comp))
         (res nil)
         (curlist (list (pop sorted)))
         (cur (pop sorted) (pop sorted)))
        ((null cur) (cons curlist res))
        (if (active-equ cur (car curlist))
            (push cur curlist)
            (progn
              (push curlist res)
              (setf curlist (list cur))))))

(defun active-equ (ed1 ed2)
   (and (equal (edge-p1 (aref edge-array ed1))
               (edge-p1 (aref edge-array ed2)))
        (equal (edge-cat (aref edge-array ed1))
               (edge-cat (aref edge-array ed2)))))

(defun active-comp (ed1 ed2)
   (if (or (< (get (edge-cat (aref edge-array ed1)) 'ntval)
              (get (edge-cat (aref edge-array ed2)) 'ntval))
           (and (equal (edge-cat (aref edge-array ed1))
                       (edge-cat (aref edge-array ed2)))
                (< (edge-p1 (aref edge-array ed1))
                   (edge-p1 (aref edge-array ed2)))))
       t nil))

;---------------------------------------------------------------------------
; Chart Maintanance Functions
;---------------------------------------------------------------------------

(defun listunion (list)
  (cond ((null list) nil)
	(t (union (car list) (listunion (cdr list))))))

(defun combine (list)
  (cond ((null list) nil)
	(t (append (car list) (combine (cdr list))))))

(defun general-insert (pos num cat array)
  (let ((orig (assoc cat (aref array pos))))
    (if orig
	(rplacd (assoc cat (aref array pos)) (push num (cdr orig)))
	(push (list cat num) (aref array pos)))))

(defun general-remove (pos num cat array)
  (let ((new (set-difference (cdr (assoc cat (aref array pos))) (list num))))
    (rplacd (assoc cat (aref array pos)) new)))

(defun general-get (pos cat array)
  (cdr (assoc cat (aref array pos))))

(defun insert-inactive-start (pos edgenum cat)
  (general-insert pos edgenum cat inactive-edges-start))

(defun remove-inactive-start (pos edgenum cat)
  (general-remove pos edgenum cat inactive-edges-start))

(defun return-inactives (pos1 pos2 cat)
  (intersection (get-inactive-start pos1 cat)
		(get-inactive-end pos2 cat)))

(defun get-inactive-start (pos cat)
  (general-get pos cat inactive-edges-start))

(defun insert-inactive-end (pos edgenum cat)
  (general-insert pos edgenum cat inactive-edges-end))

(defun remove-inactive-end (pos edgenum cat)
  (general-remove pos edgenum cat inactive-edges-end))

(defun get-inactive-end (pos cat)
  (general-get pos cat inactive-edges-end))

(defun insert-active (pos edgenum)
(let ((ch (edge-children (aref edge-array edgenum))))
  (if (and
        (can-insert-cats-ext (append (edge-insert (aref edge-array edgenum))
                               (edge-needed (aref edge-array edgenum))))
        (equal pos (edge-p2 (aref edge-array edgenum)))
        (not (and (equal 1 (length ch))
                  (equal (edge-cat (aref edge-array edgenum))
                         (edge-cat (aref edge-array (car ch)))))))
      (convert-active edgenum))
  (push edgenum (aref active-edges pos))))

(defun preprune (cat p1 p2 skip insert unif)
  (let ((edgenums (intersection 
                     (cdr (assoc cat (aref inactive-edges-start p1)))
                     (cdr (assoc cat (aref inactive-edges-end p2)))))
        (edge nil)
        (stat nil))
    
    (dolist (ed edgenums)
       (setf edge (aref edge-array ed))
       (if (insert-subsume insert (edge-insert edge))
           (setf stat t)))
    stat))

(defun convert-active (edgenum)
 (let* ((edge (aref edge-array edgenum))
       (p2 (edge-p2 edge)))
         (insert-inactive-ext (edge-cat edge)
                          (edge-p1 edge)
                          (edge-p2 edge)
                          (append (edge-children edge)
                                  (children-from-inserts 
                                      (edge-needed edge) p2))
                          (do-unification-ext 
                             (append (edge-children edge)
                                     (children-from-inserts 
                                         (edge-needed edge) p2)) 
                             (edge-rule edge)) 
                           (edge-skip edge)
                           (append (edge-insert edge) (edge-needed edge))
                           (edge-unif edge))))

(defun remove-active (pos edgenum)
  (setf (aref active-edges pos)
	(set-difference (aref active-edges pos) (list edgenum))))

(defun get-active (pos)
  (aref active-edges pos))

(defun get-active-cat (pos cat)
  (remove nil (mapcar #'(lambda (ed) 
                          (if (equal cat (edge-cat (aref edge-array ed))) 
                              ed nil)) 
                      (get-active pos))))


;-----------------------------------------------------------------------
; Top Down Filtering Resource Creation Functions
;-----------------------------------------------------------------------

(defvar *lc* nil)
(defvar *lc2* nil)

(defun write-lc ()
  (with-open-file (file "lcfile" :direction :output :if-exists :supersede
                     :if-does-not-exist :create)
     (format file "~a~%" *lc2*))
  t)

(defun read-lc ()
  (with-open-file (file "lcfile" :direction :input)
     (setf *lc2* (read file)))
  (dotimes (x (length *lc2*))
     (setf (get (car (elt *lc2* x)) 'ntval) x))
  t)

(defun convert-lc ()
  (let ((lc nil)
	(rule nil)
	(syms nil)
	(ent nil))
    (dotimes (x (length *grammar-table*))
       (setf rule (aref *grammar-table* x))
       (if (not (member (car rule) syms))
	   (push (car rule) syms))
       (dolist (c (poss-pred-cats rule))
            (if (not (member c syms))
	        (push c syms))))
    (dotimes (x (length syms))
	(setf (aref *lc* x x) 1))
    (dolist (sym syms)
       (setf ent nil)
       (dotimes (x (length syms))
	  (if (equal (aref *lc* (get sym 'num) x) 1)
	      (push (elt syms x) ent)))
       (push (list sym ent) lc))
    (setf *lc2* lc)
    t))

(defun get-all-syms ()
(let ((syms nil)
      (rule nil))
    (dotimes (x (length *grammar-table*))
       (setf rule (aref *grammar-table* x))
       (if (not (member (car rule) syms))
	   (push (car rule) syms))
       (dolist (c (cadr rule))
            (if (not (member c syms))
	        (push c syms))))
    syms))

(defun make-lc-table ()
  (let ((syms nil)
	(rule nil)
	(table nil))

    (dotimes (x (length *grammar-table*))
       (setf rule (aref *grammar-table* x))
       (if (not (member (car rule) syms))
	   (push (car rule) syms))
       (dolist (c (poss-pred-cats rule))
            (if (not (member c syms))
	        (push c syms))))
    (setf table (make-array (list (length syms) (length syms))
			    :initial-element 0))
    (dotimes (x (length syms))
	(setf (aref table x x) 1))
    (dotimes (x (length syms))
       (setf (get (elt syms x) 'num) x))

    (dotimes (x (length *grammar-table*))
       (setf rule (aref *grammar-table* x))
       (dolist (c (poss-pred-cats rule))
          (setf (aref table (get (car rule) 'num)
		   (get c 'num)) 1)))
    (dotimes (i (length syms))
       (dotimes (j (length syms))
          (if (equal 1 (aref table j i))
	     (dotimes (k (length syms))
		 (if (equal 1 (aref table i k))
		     (setf (aref table j k) 1))))))

    (setf *lc* table)
    t))

(defun poss-pred-cats (rule)
  (poss-pred-cats-aux (cadr rule)))

(defun poss-pred-cats-aux (catlist)
  (do ((cats nil))
    ((or (null (can-insert-cats-ext cats))
         (null catlist)) cats)
    (push (pop catlist) cats)))

; will be replaced by a function that will look at *poss-inserts* to
; determine whether a list of inserted categories is allowed.
(defun can-insert-cats (cats)
  (cond ((null cats) t)
        ((null *poss-inserts*) nil)
        ((equal '*or* (car *poss-inserts*))
         (do* ((poss (cdr *poss-inserts*))
               (stat nil)
               (cur (pop poss) (pop poss)))
              ((or stat (null cur)) stat)
              (setf stat (can-insert-cats-aux cats cur))))
        (t (can-insert-cats-aux cats *poss-inserts*))))

(defun can-insert-cats-ext (cats)
  (if (null (can-insert-cats cats))
      (do* ((catlist (cat-combos cats))
            (stat nil))
           ((or stat (null catlist)) stat)
         (setf stat (can-insert-cats (pop catlist))))
      t))

(defun cat-combos (cats)
  (all-combos (mapcar #'(lambda (cat) (cons (list cat) (cadr (assoc cat *insert-equal*)))) cats)))

(defun all-combos (list)
  (cond ((null list) nil)
        ((equal (length list) 1) (car list))
        (t (let ((hold (all-combos (cdr list)))
                 (res nil))
             (dolist (x hold)
               (dolist (y (car list))
                  (push (append y x) res)))
              res))))

(defun can-insert-cats-aux (cats poss)
  (let ((ins nil)
        (num nil)
        (stat t))
    (dolist (cat cats)
      (setf ins (insert-needed cat cat ins)))
    (dolist (set ins)
       (setf num (cadr (assoc (car set) poss)))
       (cond ((null stat) nil)
             ((null num) (setf stat nil))
             ((equal num '*any*) t)
             ((not (> (length (cdr set)) num))
              t)
             (t (setf stat nil))))
    stat))

(defun insert-cats ()
  (insert-cats-aux *poss-inserts*))

(defun insert-cats-aux (catlist)
  (let ((cats nil))
     (cond ((null catlist) nil)
           ((equal (car catlist) '*or*)
            (dolist (ins (cdr catlist))
               (setf cats (union cats (insert-cats-aux ins)))))
           (t (dolist (pair catlist)
                (if (not (member (cadr pair) cats))
                    (push (car pair) cats)))))
     cats))

(defun setup-dummy-cats (num)
 (dotimes (x (+ num 1)) (setf (aref *dummies* x) nil)))

; (setf *dummies* (make-array (+ num 1)))
; (let ((cats (cats-from-poss)))
;   (dotimes (x (+ num 1))
;      (dolist (cat cats)
;         (setf (aref *dummies* x)
;               (push (list cat 
;                    (build-edge cat x x nil nil '((dummy +)(counter 1)) nil nil nil nil))
;               (aref *dummies* x)))))))

(defun cats-from-poss ()
  (let ((cats nil))
    (cond ((equal *poss-inserts* t) (setf cats (get-all-syms)))
          ((equal (car *poss-inserts*) '*or*)
            (dolist (poss (cdr *poss-inserts*))
               (setf cats (union cats (cats-from-poss-aux poss)))))
           (t (setf cats (cats-from-poss-aux *poss-inserts*))))
    cats))

(defun cats-from-poss-aux (poss)
  (mapcar #'(lambda (p) (car p)) poss))

(defun get-dummy (cat pos)
   (cadr (assoc cat (aref *dummies* pos))))

(defun dummyfs-p (fs)
  (if (member '(dummy +) fs :test #'equal)
      t nil))

(defun children-from-inserts (cats pos)
  (mapcar #'(lambda (cat) 
     (if (assoc cat (aref *dummies* pos))
         (cadr (assoc cat (aref *dummies* pos)))
         (progn 
            (push (list cat (build-edge cat pos pos nil nil '((dummy +)(counter 1)) nil nil nil nil)) (aref *dummies* pos))
            (cadr (assoc cat (aref *dummies* pos)))))
     ) cats))

;-----------------------------------------------------------------------
; Two Level Parsing Resource Functions
;-----------------------------------------------------------------------

(defun get-rangeelts (sent)
  (let ((skip-positions (find-shortest-skip-path 0 (length sent)))
        (edges nil)
        (rangeelts nil))
  (dotimes (x (length edge-array))
    (if (and
           (null (edge-needed (aref edge-array x)))
           (member (list (edge-p1 (aref edge-array x)) 
                         (edge-p2 (aref edge-array x)))
                   skip-positions
                   :test #'equal))
        (push x edges)))
   (setf rangeelts (remove nil 
       (mapcar #'(lambda (ed) 
                      (if (not (member ed edges :test #'parentof))
                                  ed nil)) edges)))
   (clean-indices)
   (print (group-rangeelts rangeelts))))

(defun childof (ed1 ed2)
  (let ((edge2 (aref edge-array ed2)))
      (cond ((listp (car (edge-children edge2)))
             (if (member ed1 (combine (edge-children edge2)))
                 t nil))
            (t (if (member ed1 (edge-children edge2))
                   t nil)))))


(defun parentof (ed2 ed1)
  (let ((edge2 (aref edge-array ed2)))
      (cond ((listp (car (edge-children edge2)))
             (if (member ed1 (combine (edge-children edge2)))
                 t nil))
            (t (if (member ed1 (edge-children edge2))
                   t nil)))))

(defun group-rangeelts (rangeelts)
 (do* ((cur (pop rangeelts) (pop rangeelts))
       (curedge (if cur (aref edge-array cur))
                (if cur (aref edge-array cur)))
       (curp1 (if cur (edge-p1 curedge)))
       (curp2 (if cur (edge-p2 curedge)))
       (curgroup nil)
       (finished nil))
      ((null cur) (append (list curgroup) finished))
      (cond ((null curgroup) (setf curgroup (list cur)))
            ((and (equal curp1 (edge-p1 curedge))
                  (equal curp2 (edge-p2 curedge)))
               (push cur curgroup))
            (t (push curgroup finished)
               (setf curgroup (list cur))
               (setf curp1 (edge-p1 curedge))
               (setf curp2 (edge-p2 curedge))))))
             
; temporary until I can find a better algorithm somewhere
(defun find-shortest-skip-path (p1 p2)
  (let ((ranges nil)
        (edge nil)
        (range nil)
        (best nil))
    (dolist (inact-cat (aref inactive-edges-start p1))
       (dolist (ed (cdr inact-cat))
           (setf edge (aref edge-array ed))
           (setf range (list (edge-p1 edge) (edge-p2 edge)))
           (if (not (member range ranges :test #'equal))
               (push range ranges))))
    (setf best (car (sort ranges #'(lambda (r1 r2) (> (cadr r1) (cadr r2))))))
    (cond ((null best)
           (if (< p1 (- p2 1))
               (find-shortest-skip-path (+ p1 1) p2)
               nil))
          ((not (equal p2 (cadr best)))
           (cons best (find-shortest-skip-path (cadr best) p2)))
          (t (list best)))))

(defun coverage (len)
  (let ((path (find-shortest-skip-path-fp 0 len))
        (seg 0)
        (cov 0)
        (eds nil)
        (skip nil)
        (sk nil))
     (dolist (range path)
        (setf skip (- (cadr range) (car range)))
        (dolist (g *goal*)
        (setf eds (append eds (intersection 
              (cdr (assoc g (aref inactive-edges-start (car range))))
              (cdr (assoc g (aref inactive-edges-end (cadr range))))))))
        (if eds (setf seg (+ seg 1)))
        (dolist (ed eds)
          (if (< (setf sk (length (edge-skip (aref edge-array ed)))) skip)
              (setf skip sk)))
        (setf cov (+ cov (- (- (cadr range) (car range)) skip))))
     (list seg cov)))

(defun find-shortest-skip-path-fp (p1 p2)
  (let ((ranges nil)
        (inact-cat nil)
        (edge nil)
        (range nil)
        (best nil))
       (dolist (g *goal*)
          (setf inact-cat (append inact-cat (cdr (assoc g (aref inactive-edges-start p1))))))
       (cond ((equal p1 p2) nil)
             ((null inact-cat) (find-shortest-skip-path-fp (+ p1 1) p2))
             (t
       (dolist (ed inact-cat)
           (setf edge (aref edge-array ed))
           (setf range (list (edge-p1 edge) (edge-p2 edge)))
           (if (and (not (member range ranges :test #'equal))
                    (not (equal (car range) (cadr range))))
               (push range ranges)))
       (if (null ranges)
           (find-shortest-skip-path-fp (+ p1 1) p2)
           (progn
             (setf best (car (sort ranges #'(lambda (r1 r2) (> (cadr r1) 
                           (cadr r2))))))
             (if (not (equal p2 (cadr best)))
                 (cons best (find-shortest-skip-path-fp (cadr best) p2))
                 (list best))))))))


(defun process-rangeelt (rangeelt)
 (let ((edge nil)
       (cat nil)
       (p1 nil)
       (p2 nil)
       (struct nil)
       (extendhold nil)
       (predicted (alt-predicted (edge-p1 (aref edge-array (car rangeelt))))))
  (dolist (range-onecat rangeelt)
     (setf edge (aref edge-array range-onecat))
     (setf cat (edge-cat edge))
     (setf struct (edge-fstruct edge))
     (insert-inactive-start (setf p1 (edge-p1 edge)) 
                            range-onecat (edge-cat edge))
     (insert-inactive-end (setf p2 (edge-p2 edge)) 
                          range-onecat (edge-cat edge))
     (dolist (rule (append (headed-rules-short cat)
			  (if (lex-p struct)
			      (headed-rules-short '%))))

       (cond ((and *lc-prediction* 
                   (not (head-link-short (car rule) predicted)))
	      nil)
             ((not (can-insert-cats-ext (append (elt rule 4) 
                                            (edge-insert edge)))) nil)
	     ((> (length (elt rule 2)) 1)
	      (hold-active (car rule) p1 p1 (elt rule 2)
                                (children-from-inserts (elt rule 4) p1)
				(elt rule 3) (edge-skip edge) (elt rule 4) 
                                     (edge-unif edge)))
	     (t (insert-inactive-ext (car rule) p1 p2 
                     (append (children-from-inserts (elt rule 4) p1)
                                            (list range-onecat))
		      (do-unification-ext 
                           (append (children-from-inserts (elt rule 4) p1)
                                            (list range-onecat))
						 (elt rule 3)) 
                                 (edge-skip edge)
                                 (append (elt rule 4) (edge-insert edge))
                                 (edge-unif edge)))))
     (dolist (act (get-active (edge-p1 edge)))
               (if (and
                    (member cat (edge-needed (aref edge-array act)))
                    (setf extendhold
                     (can-extend cat (edge-needed (aref edge-array act))
                        (edge-insert (aref edge-array act)) p1)))
	           (extend-active (aref edge-array act) 
                                  range-onecat (cadr extendhold)))))))


(defun clean-indices ()
   (dotimes (x (length inactive-edges-start))
     (setf (aref inactive-edges-start x) nil))
   (dotimes (x (length inactive-edges-end))
     (setf (aref inactive-edges-end x) nil))
   (dotimes (x (length active-edges))
     (setf (aref active-edges x) nil)))

;-----------------------------------------------------------------------
; Unification for Insertions
;-----------------------------------------------------------------------

(defun unify (fs1 fs2) ; returns a fs or *fail*
  (cond ((equal fs1 '*dummy*) fs2)
        ((equal fs2 '*dummy*) fs1)
        ((null fs1) fs2)
	((null fs2) fs1)
 	((multi-p fs1) (unify*multiple* fs1 fs2))
 	((multi-p fs2) (unify*multiple* fs2 fs1))
	((or-p fs1) (unify*OR* fs1 fs2))
	((or-p fs2) (unify*OR* fs2 fs1))
	((not-p fs1) (unify*NOT* fs1 fs2))
	((not-p fs2) (unify*NOT* fs2 fs1))
     	((user-p fs1)(unify-user-values fs1 fs2))
	((user-p fs2)(unify-user-values fs2 fs1))
	((atom fs1) (if (equal fs1 fs2) fs2 '*fail*))
	((atom fs2) '*fail*)
	(t (unify-fs fs1 fs2))))

(defun getvalue (fs path)   ; returns a fs; or nil if undefined
 (cond ((equal '*dummy* fs) '*dummy*)
       ((dummyfs-p fs) '*dummy*)
       ((not (listp fs)) nil)
       (t (let ((assoc-result (second (assoc (car path) fs))))
           (cond 
        ;; if assoc-result is nil, return nil
             ((null assoc-result) nil)
        ;; if path length is 1, return assoc-result
             ((eq (length path) 1) assoc-result)
        ;; otherwise call getvalue recursively
             (t (getvalue assoc-result (cdr path))))))))

(defun setvalue (fs path val &key (mode 'OVERWRITE))  ; returns a fs
  (cond
      ((equal '*dummy* fs) (setvalue nil path val :mode mode))
      ((null val) fs)
      ((null path)
 	 (case mode
	  (OVERWRITE val)
   	  (UNIFY (unify fs val))
	  (C-UNIFY (if fs (unify fs val) '*FAIL*))
	  (DEFINED (if fs fs '*FAIL*)) 
	  (UNDEFINED (if fs '*FAIL* nil))
	  (PUSH (append-fs-to-fs fs val))
	  (REMOVE nil)
	  (POP (pop-multi-value fs))))
         ;; if fs is atom and path is not nil, error.
      ((and (atom fs) fs)(terpri)
       (princ "### Path ")(princ path)(princ " cannot access ")(princ fs)
       nil)
         ;; if fs is *OR*, do each of the disjunction.
         ;; Return *fail* when all of the disjunction fail.
      ((eq (car fs) '*OR*)
       (or
        (list-to-value
	 (map-dolist+ (fs1 (cdr fs))
 	  (setvalue fs1 path val :mode mode)))
        '*FAIL*))

         ;; if fs is *MULTIPLE*, do each of the conjunction.
         ;; Return *fail* when any one of the disjunction fails.
      ((eq (car fs) '*MULTIPLE*)
       (let ((newvalue nil))
         (setq newvalue
           (map-dolist (fs1 (cdr fs))
 	   (setvalue fs1 path val :mode mode)))
         (if (member '*FAIL* newvalue)
             '*FAIL*
             (cons '*MULTIPLE* newvalue))))
          
         ;; else take ASSOC
      (t (let ((assoc-result (or (assoc (car path) fs)
                                 (and (dummyfs-p fs)
                                      (null (cdr path))
                                      (list (car path) '*dummy*))
                                 (and (dummyfs-p fs)
                                      (list (car path) '((dummy +)(counter 1)))))))
	  (cond
            
            ((null (cdr assoc-result))
	      ;; If the path doesn't exist	
	    (case mode
	      ((OVERWRITE UNIFY PUSH) (append (create-path path val) fs))
	      ((C-UNIFY DEFINED) '*FAIL*)
	      ((UNDEFINED REMOVE) fs)))
	      
	    (t   ;; If the path exists, call setvalue recursively
	    (let ((rec-result (setvalue (second assoc-result)
					(cdr path) val :mode mode)))
              (cond ((eq rec-result '*FAIL*) '*FAIL*)
		    ((null rec-result)(remove assoc-result fs))
	            ((cons (list (car path) rec-result)
			(remove assoc-result fs)))))))))))

(defun ap-sep (sent)
 (decontraction  sent))


;--------------------------------------------------------------------------
;                 Compiling Flexible Unification
;--------------------------------------------------------------------------

(defvar *level* 0)

(defun compgra (grafilename)
  (let ((temp
   (lfg-compiler
       (gra-preprocess (read-file-list (append-str grafilename ".gra")))
       :gra-file grafilename)))
   (with-open-file (file (append-str grafilename ".lisp") :direction :output
                       :if-exists :supersede :if-does-not-exist :create)
       (dolist (fun (cadr temp))
         (pprint fun file)
         (format file "~%")))
   (with-open-file (file (append-str grafilename ".rul") :direction :output
                       :if-exists :supersede :if-does-not-exist :create)
       (format file "(")
       (dolist (rule (car temp))
         (format file "~a~%" rule))
       (format file ")~%"))
   (compile-file (append-str grafilename ".lisp"))
   (loadgra grafilename)))

(defun loadgra (grafilename)
  (load grafilename)
  (let ((rules (with-open-file (file (append-str grafilename ".rul") 
                                     :direction :input) 
                   (read file))))
     (setf *grammar-table* (make-array (length rules)))
     (dotimes (x (length rules))
        (setf (aref *grammar-table* x) (elt rules x))))
  t)
   


(defun lfg-compiler (gra &key gra-file)		;; used in function names

 (let ((fun-list nil)				;;function definitions
       (acfg nil)				;; acfg rule list
       (count 0))				;; counter
  (format t "*** LFG Compiler started~%")
  (gensym 0)
  (setf *gensym-counter* 1)
  (dolist (rule gra)
   (let* ((lhs (first rule))			;; left hand side symbol
	  (rhs (second rule))			;; right hand side symbols
	  (aug (third rule))			;; augmentation
	  (lisp-func				;; generated lisp function
	      (compile-selective-unif #'compile-augmentation (length rhs) aug))
 	  (func-name
	       (intern (string-upcase
	 		 (append-str gra-file
				    (symbol-name (gensym "F-")))))))
    (incf count)
    (when (zerop (mod count 20))
	  (format t "LFG [~3D]~%" count))

	;; Push final function definition to FUN-LIST.
	;;
    (push `(defun ,func-name ,@(rest lisp-func))
	    fun-list )
	
	;; Push rule with function name into ACFG.
	;;
    (push (list lhs rhs func-name) acfg)

 ;; (format t "~A~%" (symbol-name func-name))
   ))

	;; Return a list of ACFG and FUN-LIST.
  (setq acfg (nreverse acfg))
  (setq fun-list (nreverse fun-list))
  (format t "*** LFG Compiler done~%")
  (list acfg fun-list))
)




(defun compile-selective-unif-old (fun len statements)
  (let ((dummy '((x0 = x40)))
	(part nil)
	(parms nil)
        (select1 nil)
	(select2 nil)
        (comp-sel nil)
        (comp-reg nil)
	(comp-full nil))
    (setf select1 (get-selective-statements statements))
    (if (empty-res select1) (setf select1 (append select1 dummy)) (setf part t))
    (setf select2 (get-complement-statements statements part))
    (setf comp-sel (fix-fun (apply fun (list len select1))))
    (setf comp-reg (fix-fun (apply fun (list len select2))))
    (setf comp-full (fix-fun (apply fun (list len statements))))
    (setf parms (append (cadr comp-reg) (list '&key (list 'x40  (quote '((select-unif +)))))))
    `(lambda ,parms (cond ((null *do-parse-time-feats*) ,(caddr comp-full))
			   (*parse-time-flag* ,(caddr comp-sel))
			   (t ,(caddr comp-reg))))))







(defun compile-selective-unif (fun len statements)
  (let* ((piles (seperate-statements statements))
         (funs (mapcar #'(lambda (sts) (fix-fun (apply fun (list len sts))))
                             piles))
         (parms (append (cadr (car funs)) 
                        (list '&key (list 'x40 (quote '((select-unif +)))))))
         (cond nil))
       
       (dotimes (x (length funs))
          (push (list (list 'equal '*level* x) (caddr (elt funs x))) cond))
       (setf cond (cons 'cond (reverse cond)))
       `(lambda ,parms ,cond)))


(defun seperate-statements (statements)
 (if (simple-states-p statements)
     (transform-simple statements)
     (transform-complex statements)))

(defun simple-states-p (statements)
  (let ((stat t))
    (dolist (st statements)
       (if (not (or (equal (car st) '*test*)
                    (not (member (car st) '(*or* *eor* *case*)))))
           (setf stat nil)))
     stat))

(defun transform-simple (statements)
   (let ((piles nil))
     (dotimes (x (+ (length *unif-flex*) 1))
        (push (remove nil (mapcar #'(lambda (st) (level-st st x)) statements))
           piles))
     (setf piles (reverse piles))
     (append (mapcar #'(lambda (pile) (append '((x0 = x40)) pile))
                               (butlast piles))
             (list (append '((x0 = x40))
                      (car (last piles))
                      '(((x0 select-unif) = *remove*)))))))

(defun level-st (statement level)
  (cond ((state-on-level statement level)
         statement)
        ((and (equal level 0) (equal (length statement) 3))
         (list (car statement) '= '*dummy*))
        (t nil)))

(defun state-on-level (statement level)
  (let ((stat t))
    (dotimes (x level)
       (if (and stat (state-on-level-complex statement x))
           (setf stat nil)))
     (if (and stat (not (state-on-level-complex statement level)))
        (setf stat nil))
     stat))

(defun state-on-level-complex (statement level)
  (if (null (intersection (get-feats statement) (higher-feats level)))
      t nil))

(defun higher-feats (level)
   (let ((feats *unif-flex*))
       (dotimes (x level)
          (pop feats))
       (combine feats)))

(defun get-feats (statement)
  (cond  ((and (equal (car statement) '*test*)
               (atom (cadr statement))) nil)
         ((equal (car statement) '*test*)
         (combine (mapcar #'(lambda (path) (if (and (listp path)
                                                    (atom (car (last path)))
                                                    (not (member (car path) 
                                                       '(*not* *and* *or*)))) 
                                                     (cdr path))) 
                                                    (cdr (cadr statement)))))
        (t (feats-aux (elt statement 0)))))

(defun feats-aux (feat)
  (if (and (listp feat)
           (atom (car (last feat)))
           (not (member (car feat) '(*not* *and* *or*))))
      (cdr feat)))
(defun fix-fun (fun)
  (if (equal (length fun) 3)
      fun
      (list (car fun) (cadr fun) (car (last fun)))))


; if there's a possibility for ambiguity, you do the whole unification each
; time
(defun transform-complex (statements)
  (append (transform-complex-aux (modify-complex-statements statements))
          (list (append statements '(((x0 select-unif) = *remove*))))))

(defun modify-complex-statements (statements)
;(print "modify")
;(pprint statements)
  (mapcar #'(lambda (st) 
       (cond ((and (atom (car st)) (atom (cadr st))) st)
             ((member (car st) '(*or* *eor*))
              (cons '*or* (mapcar #'(lambda (s)
                         (modify-complex-statements s)) (cdr st))))
             ;((equal '*case* (car st))
             ; (cons '*or*
             ;      (mapcar #'(lambda (s) (modify-complex-statements (cdr s)))
             ;                (cddr st))))
            (t st)))
         statements))

(defun transform-complex-aux (statements)
 (let ((piles nil))
  (dotimes (x (length *unif-flex*))
     (push (get-selective-statements statements x) piles))
  (reverse piles)))

(defun get-selective-statements (statements level)
  (remove nil (mapcar #'(lambda (state)
			  (cond ((equal '*or* (car state))
				 (let ((ors (remove nil
					  (mapcar #'(lambda (list)
						      (get-selective-statements list level))
						  (cdr state)))))
				   (if ors (cons '*or* ors))))
				 ((state-on-level-complex state level) state)
				 (t nil))) statements)))

(defun do-unification-ext (children rule)
(let ((res (do-unification-ext2 t children rule))
      (rule-info nil))

  (if (setf rule-info (assoc rule *tfuns*))
      (progn
        (format t "~%------------------------------------------------~%")
        (format t "RULE: ~a~%" (cadr rule-info))
        (format t "CHILDREN:~%")
        (mapcar #'(lambda (ch) (format t "~a ~a~%~%" 
                      (edge-cat (aref edge-array ch)) 
                      (edge-fstruct (aref edge-array ch)))) children)
        (format t "RESULT: ~a ~a~%" (car (cadr rule-info)) 
           (cadr res))
        (if (null (car res)) 
            (format t "FLEX-UNIF: ~a~%" (caddr res)))
        (format t "------------------------------------------------~%")
        ))
   res))


(defun do-unification-test (children rule)
(let ((res (do-unification-ext2 nil children rule))
      (rule-info nil))

  (if (setf rule-info (assoc rule *tfuns*))
      (progn
        (format t "~%------------------------------------------------~%")
        (format t "RULE: ~a~%" (cadr rule-info))
        (format t "CHILDREN:~%")
        (mapcar #'(lambda (ch) (format t "~a ~a~%~%" 
                      (edge-cat (aref edge-array ch)) 
                      (edge-fstruct (aref edge-array ch)))) children)
        (format t "RESULT: ~a ~a~%" (car (cadr rule-info)) 
           (cadr res))
        (if (null (car res)) 
            (format t "FLEX-UNIF: ~a~%" (caddr res)))
        (format t "------------------------------------------------~%")
        ))
   res))

(defun init-rule-bookkeeping ()
 (setf *rule-bookkeeping* nil))

(defun disable-rule-bookkeeping ()
 (setf *rule-bookkeeping* t))

(defun check-rule-bookkeeping (flag children rule)
  (cond ((equal t *rule-bookkeeping*) t)
        ((member children (assoc rule *rule-bookkeeping*) :test #'equal) nil)
        ((null flag) t)
        (t (setf *rule-bookkeeping* (insert-needed rule children *rule-bookkeeping*))
            t)))

(defun do-unification-ext2 (test children rule)
(if (check-rule-bookkeeping test children rule)
  (do ((stat t)
        (res nil)
        (old nil)
        (level 0))
       ((or (null stat)
            (> level (length *unif-flex*))) 
               (progn (if *no-dups* (setf old (remove-dup-ilts old)))
                      (list stat old level)))
       (setf *level* level)
       (setf res 
            (apply rule 
              (if old
              (append 
                (mapcar #'(lambda (ch)
		  (setf (edge-used (aref edge-array ch)) t)
		  (edge-fstruct (aref edge-array ch)))
		      children)
                (list ':x40 old))
               (mapcar #'(lambda (ch)
		  (setf (edge-used (aref edge-array ch)) t)
		  (edge-fstruct (aref edge-array ch)))
		      children))))
        (if res
            (progn
              (setf old res)
              (setf level (+ level 1))
             )
            (progn
              (setf stat nil)
             )))))

(defun do-reunification-ext (children rule)
  (do ((stat t)
        (res nil)
        (old nil)
        (level 0))
       ((or (null stat)
            (> level (length *unif-flex*))) 
               (progn (if *no-dups* (setf old (remove-dup-ilts old)))
                      (list stat old level)))
       (setf *level* level)
       (setf res 
            (apply rule 
              (if old
              (append 
                (mapcar #'(lambda (ch)
                  (or (second (assoc ch *new-value-list*))
                      (edge-fstruct (aref edge-array ch))))
		      children)
                (list ':x40 old))
               (mapcar #'(lambda (ch)
                  (or (second (assoc ch *new-value-list*))
                      (edge-fstruct (aref edge-array ch))))
		      children))))
        (if res
            (progn
              (setf old res)
              (setf level (+ level 1))
             )
            (progn
              (setf stat nil)
             ))))


;--------------------------------------------------------------------------
;                    Statistical Disambiguation
;--------------------------------------------------------------------------

; This statistical disambiguation code is based on the code developed
; by Alon Lavie for disambiguation in GLR*.

;----------------------------------------------------------------------------
; PART 1: Train Statistics For Statistical Disambiguation
;----------------------------------------------------------------------------

(defvar *transition-bigrams* nil)
(defvar *start* 0)

(setq *NULL-OUT* (open "/dev/null" :direction :output :if-exists :overwrite))
(setq *ignored-list* nil)

(defun train-transition-bigrams (trainfile bigramfile)
     (setf *restarts* nil)
     (setf *unif-flex* nil)
     (setf *skip-beam* nil)
     (setf *skip-limit* 0)
     (setf *poss-inserts* nil)
     (initialize-transition-bigrams)
     (with-open-file (infile trainfile :direction :input)
           (do ((sent nil)
                (ilt nil))
               ((equal (setf sent (read infile nil 'eof)) 'eof) t)
               (setf sent (remove '$ sent))
               (setf ilt (read infile nil 'eof))
               (train-transition-bigrams-aux sent ilt)))
     (save-transition-bigrams bigramfile))

(defun setup-stats (bigramfile)
  (read-transition-bigrams bigramfile)
  (normalize-transition-bigrams))

(defun initialize-transition-bigrams ()
  (let ((lexemes (get-all-lexemes))
        (num-rules (length *grammar-table*)))
    (dotimes (x (length lexemes))
       (setf (get (elt lexemes x) 'lex-ind) (+ x num-rules)))
    (setf *transition-bigrams* 
          (make-array (list (+ (length *grammar-table*) (length lexemes))
                            (+ (length *grammar-table*) (length lexemes)))
                      :initial-element 0))
   t))

(defun get-all-lexemes ()
  (let ((lexemes nil))
    (dotimes (x (length *grammar-table*))
      (dolist (lex (cadr (aref *grammar-table* x)))
        (if (and (lexical-p lex)
                 (not (member lex lexemes)))
            (push lex lexemes))))
    lexemes))

(defun get-all-lexemes-old ()
  (let ((lexemes nil))
    (dolist (lc *lc2*)
      (dolist (lex (cadr lc))
        (if (and (lexical-p lex)
                 (not (member lex lexemes)))
            (push lex lexemes))))
    lexemes))

(defun lexical-p (lex)
   (if (not (equal #\< (char (format nil "~A" lex) 0)))
       t nil))

(defun train-transition-bigrams-aux (sent ilt)
  (let ((edge nil))
     (parse sent)
     (setf edge (find-correct-edge (ap-sep sent) ilt))
     (if edge
         (update-transition-counts 
            (transitions-from-parse-tree
               (tree-from-edge edge)))
         (format t "failed~%"))))

(defun amb-and-cor (sent ilt)
  (parse sent)
  (let ((edges (combine (mapcar #'(lambda (g)
           (return-inactives 0 (length sent) g)) *goal*))))
     (if (and edges
              (or (< 1 (length edges))
                  (listp (car (edge-children (aref edge-array (car edges))))))
              (find-correct-edge sent ilt))
         t nil)))
  
(defun tree-from-edge (edge)
(if (edge-children edge)
    (let* ((rule-num (second (find-rule-number 
                      (edge-cat edge) 
                      (mapcar #'(lambda (ch) (edge-cat (aref edge-array ch))) 
                               (edge-children edge)))))
           (rule (aref *grammar-table* (- rule-num 1))))

          (list rule-num 
                (mapcar #'(lambda (ch cat) 
                             (if (equal cat '%)
                                 (get '% 'lex-ind)
                                 (tree-from-edge (aref edge-array ch))))
                             (edge-children edge)
                             (cadr rule))))
    (get (edge-cat edge) 'lex-ind)))

(defun transitions-from-parse-tree (tree)
     (if (listp tree)
         (append (combine 
                (mapcar #'(lambda (ch) (transitions-from-parse-tree ch))
                       (cadr tree)))
              (list (car tree)))
         (list tree)))

(defun update-transition-counts (action-sequence)
  (let ((bigrams (pairlis (butlast action-sequence) (cdr action-sequence))))
      (dolist (bi bigrams)
          (setf (aref *transition-bigrams* (car bi) (cdr bi))
                (+ (aref *transition-bigrams* (car bi) (cdr bi)) 1)))))


;; Updated Function from Noah (modified by Alon)
; find-rule-number determines the rule number based on the lhs and rhs

(defun find-rule-number (lhs rhs)
   (let ((found-rule nil))
      (setq found-rule                            ; first pass no wild card
         (do* ((rule-list  *gg* (cdr rule-list))
	       (rule (car rule-list) (car rule-list)))
	      ((or (and (equal (car rule) lhs) (equal (cadr rule) rhs))
		   (eq rule nil)) rule)))

      (when (null found-rule)                   ; second pass allow wild cards
         (setq found-rule
            (do* ((rule-list  *gg* (cdr rule-list))
		  (rule (car rule-list) (car rule-list)))
		 ((or (and (equal (car rule) lhs) (match-rhs (cadr rule) rhs))
		      (eq rule nil)) rule))))

      (if found-rule                                                ;; result
         (list (car found-rule) 
	       (read-from-string (string-left-trim 
				  "ABCDEFGHIJKLMNOPQRSTUVWXYZ-_." 
				  (write-to-string (third found-rule)))))
	 nil)))




(defun match-rhs (rule-rhs actual-rhs)
   (cond ((and (null rule-rhs)
               (null actual-rhs)) t)
         ((and (or (equal (car rule-rhs) (car actual-rhs))
                   (equal (car rule-rhs) '%))
               (match-rhs (cdr rule-rhs) (cdr actual-rhs))) t)
         (t nil)))



(defun find-rule-fun (lhs rhs)
   (let ((found-rule nil))
      (setq found-rule                            ; first pass no wild card
         (do* ((rule-list  *gg* (cdr rule-list))
	       (rule (car rule-list) (car rule-list)))
	      ((or (and (equal (car rule) lhs) (equal (cadr rule) rhs))
		   (eq rule nil)) rule)))

      (when (null found-rule)                   ; second pass allow wild cards
         (setq found-rule
            (do* ((rule-list  *gg* (cdr rule-list))
		  (rule (car rule-list) (car rule-list)))
		 ((or (and (equal (car rule) lhs) (match-rhs (cadr rule) rhs))
		      (eq rule nil)) rule))))

      (if found-rule
          (caddr found-rule))))

(defun find-correct-edge (sent ilt)
 (let ((edge nil))
   (dolist (ed (combine (mapcar #'(lambda (g)
                           (intersection (get-inactive-start 0 g)
                             (get-inactive-end (length sent) g))) *goal*)))
      (if (and (null edge)
               (correct-edge-p (aref edge-array ed) ilt))
          (setf edge (unpack-train ed ilt))))
   edge))

(defun unpack-train (ed ilt)
 (if *ambiguity-packing*
     (progn
        (setf *new-value-list* nil)
        (setf *node-parent-list* nil)
        (unpack-train-aux ed ilt)))
 (aref edge-array ed))

(defun unpack-train-aux (ed ilt)
 (let* ((edge (aref edge-array ed))
        (children (edge-children edge))
        (correct nil))
   (if (listp (car children))
       (dolist (ch children)
          (if (null correct)
              (progn
                 (setf (edge-children edge) ch)
                 (if (train-check-unify ed ilt)
                     (setf correct t))))))
   (dolist (child (edge-children edge))
      (setf *node-parent-list* (append (list (list child ed)) 
                                       *node-parent-list*))
      (unpack-train-aux child ilt))))

; modification on consistant function found below
(defun train-consistent (edgenum ilt)
   (let ((cat (edge-cat (aref edge-array edgenum)))
         (parent (second (assoc edgenum *node-parent-list*))))
      (if (and (member cat *goal*)
               (null parent))
          (return-from train-consistent (correct-edge-p (aref edge-array edgenum) ilt))
          (train-check-unify parent ilt))))
                   
; modification on check-unify function found below
(defun train-check-unify (edgenum ilt)
   (let ((lhs (edge-cat (aref edge-array edgenum)))
         (rhs nil)
         (unif nil)
         (rule-number nil)
         (fun nil)
         (son-list nil)
         (new-value nil))
      (setq son-list (edge-children (aref edge-array edgenum)))
      (setq rhs 
         (mapcar #'(lambda (edge)
                     (let ((cat (edge-cat (aref edge-array edge))))
                        (when (listp cat)
                           (setq cat (car cat)))
                        cat))
                  son-list))
      (when (listp lhs) (setq lhs (car lhs)))
      (setq rule-number (second (find-rule-number lhs rhs)))
      (setq fun (third (get-rule-by-fun-number rule-number)))
      (setq unif (get-new-value fun son-list))
      (setq new-value (cadr unif))
      (when (listp new-value)
         (setq new-value (fix-parsed-words-slot edgenum new-value)))
      (cond ((null new-value) (return-from train-check-unify nil))
            (t (setf (edge-fstruct (aref edge-array edgenum)) new-value)
               (setf (edge-unif (aref edge-array edgenum))
                     (combine (mapcar #'(lambda (e) 
                         (edge-unif (aref edge-array e))) son-list)))
               (if (null (car unif))
                   (push (car (last unif)) 
                         (edge-unif (aref edge-array edgenum))))
               (return-from train-check-unify (train-consistent edgenum ilt))))
))

; modified from Alon's find-correct-sons in disambig-sched.lisp
(defun correct-edge-p (edge correct-val)
   (let ((ilt (edge-fstruct edge))
         (son-vals nil)
         (ilt-val-list nil)
         (correct-sons nil))

         (if (equal (car ilt) '*OR*)
            (setq son-vals (cdr ilt))
            (setq son-vals (list ilt)))

         (dolist (a-val son-vals nil)
            (if (find 'utterances a-val :key #'car)
                (setq ilt-val-list
                    (if (equal (caadr (find 'utterances a-val :key #'car)) 
                          '*MULTIPLE*)
	                (reverse (cdadr (find 'utterances a-val :key #'car)))
	                (reverse (cdr (find 'utterances a-val :key #'car)))))
                 (setq ilt-val-list (list a-val)))

            (dolist (ilt-val ilt-val-list nil)
               (when (comp-ilt ilt-val correct-val)
                  (return-from correct-edge-p t)))))
   nil)


(defun correct-ilt-p (ilt correct-val)
   (let ((son-vals nil)
         (ilt-val-list nil)
         (correct-sons nil))

         (if (equal (car ilt) '*OR*)
            (setq son-vals (cdr ilt))
            (setq son-vals (list ilt)))

         (dolist (a-val son-vals nil)
            (setq ilt-val-list
               (if (equal (caadr (find 'utterances a-val :key #'car)) 
                          '*MULTIPLE*)
	       (reverse (cdadr (find 'utterances a-val :key #'car)))
	       (reverse (cdr (find 'utterances a-val :key #'car)))))

            (dolist (ilt-val ilt-val-list nil)
               (when (comp-ilt ilt-val correct-val)
                  (return-from correct-ilt-p t)))))
   nil)

(defun correct-edge-p-old (edge correct-val)
   (let ((son-list (edge-children edge))
         (son-vals nil)
         (ilt-val-list nil)
         (correct-sons nil))
      (dolist (a-son son-list nil)
         (setq son-vals (edge-fstruct (aref edge-array a-son)))
         (if (equal (car son-vals) '*OR*)
            (setq son-vals (cdr son-vals))
            (setq son-vals (list son-vals)))
         (dolist (a-val son-vals nil)
            (setq ilt-val-list
               (if (equal (caadr (find 'utterances a-val :key #'car)) 
                          '*MULTIPLE*)
	       (reverse (cdadr (find 'utterances a-val :key #'car)))
	       (reverse (cdr (find 'utterances a-val :key #'car)))))

            (dolist (ilt-val ilt-val-list nil)
               (when (comp-ilt ilt-val correct-val)
                  (return-from correct-edge-p-old t))))))
   nil)


; uses the Good Turing method of smoothing
(defun normalize-transition-bigrams ()
 (let ((size (array-dimension *transition-bigrams* 0))
       (sum 0))
    (dotimes (x size)
       (setf sum 0)
       (dotimes (y size)
         (if (equal (aref *transition-bigrams* x y) 0)
             (setf (aref *transition-bigrams* x y) 1))
         (setf sum (+ sum (aref *transition-bigrams* x y))))
       (dotimes (y size)
         (setf (aref *transition-bigrams* x y) 
               (* 1.0 (/ (aref *transition-bigrams* x y) sum)))))))

(defun save-transition-bigrams (bifile)
 (let ((size (array-dimension *transition-bigrams* 0)))
    (with-open-file (file bifile :direction :output :if-exists :supersede :if-does-not-exist :create)
      (format file "~a " size)
      (dotimes (x size)
        (dotimes (y size)
           (if (> (aref *transition-bigrams* x y) 0)
               (format file "~a " (list x y (aref *transition-bigrams* x y)))))
        (format file "~%")))))

(defun initialize-lexemes ()
  (let ((lexemes (get-all-lexemes))
        (num-rules (length *grammar-table*)))
    (dotimes (x (length lexemes))
       (setf (get (elt lexemes x) 'lex-ind) (+ x num-rules)))
   t))


(defun read-transition-bigrams (bifile)
  (initialize-lexemes)
  (let ((size nil))
    (with-open-file (file bifile :direction :input)
       (setf size (read file nil 'eof))
       (setf *transition-bigrams* (make-array (list size size) :initial-element 0))
	   (do ((ent nil))
	       ((equal (setf ent (read file nil 'eof)) 'eof) t)
	       (setf (aref *transition-bigrams* (car ent)(cadr ent)) (caddr ent))))))

;----------------------------------------------------------------------------
; PART 2: Assign Estimated Statistical Scores to Inactive Edges
;----------------------------------------------------------------------------

(defvar *node-score-list* nil)

(defun assign-edge-scores ()
 (let ((edge nil))
     (setf *node-score-list* nil)
     (dotimes (x (length edge-array))
        (setf edge (aref edge-array x))
        (if (null (edge-needed edge))
            (score-edge x edge)))))

; adapted from Alon's score-node found in disambig-run.lisp
(defun score-edge (edgenum edge)
   (let ((packed-list (edge-children edge))
         (lhs (edge-cat edge))
         (rhs nil)
         (rule-number nil)
         (last-action-score nil)
         (max-score nil)
         (final-score nil)
         (son-score nil)
         (son-score-list nil)
         (ambig-score-list nil)
         (score nil)
         (count nil)
         (score-list nil)
         (hold nil)
         (first-action nil)
         (first-action-list nil)
         (last-action nil))
      (when (not (null (assoc edgenum *node-score-list*)))
         (return-from score-edge (list (second (assoc edgenum *node-score-list*))
                                       (fourth (assoc edgenum *node-score-list*)))))
      (when (listp lhs) (setq lhs (car lhs)))
      (cond ((null packed-list) 
                (setq final-score 
                   (list 1.0 1))) ; don't know what the shift score is until
                                  ; you know what the rule was
            (t  (setq score-list nil)
                (if (not (listp (car packed-list)))
                    (setf packed-list (list packed-list)))
                (dolist (ch (combine packed-list))
                    (if (null (assoc ch *node-score-list*))
                        (score-edge ch (aref edge-array ch))))
                (dolist (son-list packed-list nil)
		   
                   (setf first-action 
                         (car (fourth (assoc (car son-list) 
                         *node-score-list*))))
                   (setq son-score-list nil)
                   (dolist (a-son son-list nil)
                      (setq son-score (list (second (assoc a-son *node-score-list*))
                                            (fourth (assoc a-son *node-score-list*))))
                      (when (null son-score)
                         (setf son-score
                               (score-edge a-son (aref edge-array a-son))))
                      (setq son-score-list
                         (cons son-score son-score-list)))
                   (setq rhs 
                      (mapcar #'(lambda (ed)
                         (let ((cat (edge-cat (aref edge-array ed))))
                            (when (listp cat)
                               (setq cat (car cat)))
                            cat))
                         son-list))
                   (setq rule-number (second (find-rule-number lhs rhs)))
                   (setq last-action rule-number)
                   (setf hold 
                      (compute-rule-score edgenum rule-number (reverse son-score-list) first-action))
                   (setf score (car (cadr hold)))
                   (setf count (car (cadr hold)))
                   (setf first-action (car hold))
                   (setq first-action-list 
                      (append first-action-list 
                              (list (list score first-action))))
                   (setq score-list 
                      (append score-list (list (list score count)))))
                (setq ambig-score-list (pairlis packed-list score-list))
                (setq max-score (apply #'max (mapcar #'car score-list)))
                (setq first-action (cadr (assoc max-score first-action-list)))
                (setq final-score (assoc max-score score-list))))
      (setq *node-score-list* 
         (cons (list edgenum final-score ambig-score-list (list first-action
                                                                last-action))
               *node-score-list*))
      (return-from score-edge (list final-score (list first-action last-action)))))

; return (list first-action (list score count))
; for now we compute transitions for (ch1 ch2 ... rule)

(defun compute-rule-score (edgenum rule-number son-score-list first-action)
 (setf son-scores son-score-list)
 (let* ((count (+ (length son-score-list)
                  (apply #'+ (mapcar #'second 
                                       (mapcar #'car son-score-list)))))
        (rhs (cadr (aref *grammar-table* (- rule-number 1))))
        (actions (remove nil (mapcar #'(lambda (son-score elt)
                             (cond ((null (car (cadr son-score)))
                                    (if (get elt 'lex-ind)
                                        (list 1.0 (get elt 'lex-ind)
                                                  (get elt 'lex-ind))
                                        nil))
                                    (t (cons (caar son-score) 
                                             (cadr son-score)))))
                       son-score-list rhs)))
        (score (apply #'* (mapcar #'car actions))))
        ;(print actions)
       ; now just multiply in children transitions
       (mapcar #'(lambda (ch1 ch2) 
                    (setf score (* score (bigram (third ch1) (second ch2)))))
                 (butlast actions) (cdr actions))
       (setf score (* score (bigram (third (car (last actions))) 
                                    rule-number)))
    (list (cadr (car actions)) (list score count))))

(defun bigram (el1 el2)
  (let ((res (aref *transition-bigrams* el1 el2)))
    (if (not (> res 0))
        (print (list el1 el2)))
    res))

;----------------------------------------------------------------------------
; PART 3: Unpack Best Scoring Analysis
;----------------------------------------------------------------------------

(defvar *unpack-fun-ors* t)
(defvar *disambig-flag* t)
(defvar *node-son-list* nil)
(defvar *node-parent-list* nil)
(defvar *new-value-list* nil)
(defvar *unpack-right* nil)

;; Adapted from Alon's unpack-top in disambig-run.lisp
;; unpack-top starts the unpacking from a top <START> edge, then computes
;;            the actual final prob-score and stores it in *node-score-list*

(defun unpack-top (edgenum)
   (let ((cat (edge-cat (aref edge-array edgenum)))
	 (new-value nil)
         (true-score nil))
      (when (listp cat) (setq cat (car cat)))
      (when (member cat *goal*)
         (setq *node-parent-list* nil)
         (setq *node-son-list* nil)
         (setq *new-value-list* nil)
         (unpack-edge edgenum)
         (when *unpack-fun-ors*
	    (setq new-value (second (assoc edgenum *new-value-list*)))
            (setq new-value 
               (list (car (select-by-counter (tree-explode-ors new-value)))))
	    (rplacd (assoc edgenum *new-value-list*) new-value))
         (setq true-score (find-true-score edgenum))
         (rplacd (assoc edgenum *node-score-list*)
            (list true-score 
                  (third (assoc edgenum *node-score-list*))
                  (fourth (assoc edgenum *node-score-list*)))))))

; Adapted from Alon's unpack-node function in disambig-run.lisp
;; unpack-edge is the recursive ambiguity unpacking function
(defun unpack-edge (edgenum)
   (let ((packed-list (edge-children (aref edge-array edgenum)))
         (lhs (edge-cat (aref edge-array edgenum)))
         (best-son-list nil)
         (best-score nil)
         (best-value nil)
         (best-fun nil)
         (rhs nil)
         (rule-number nil)
         (fun nil)
         (cur-value nil)
         (cur-score nil)
         (new-value nil))
      (if (not (listp (car packed-list)))
          (progn
            (setf packed-list (list packed-list))
            ))
      (when (listp lhs) (setq lhs (car lhs)))
      (dolist (son-list packed-list new-value)
         (setq rhs 
            (mapcar #'(lambda (a-node)
                        (let ((cat (edge-cat (aref edge-array a-node))))
                           (when (listp cat)
                              (setq cat (car cat)))
                           cat))
                     son-list))
         (setq rule-number (second (find-rule-number lhs rhs)))
         (setq fun (third (get-rule-by-fun-number rule-number)))
         (setq cur-value (cadr (do-unification-ext son-list fun)))
         (setq cur-score (score-ambig edgenum son-list cur-value))
         (setq *new-value-list* 
            (cons (list edgenum cur-value) *new-value-list*))
         (when (consistent edgenum)
            (when (or (null best-score)
                      (< cur-score best-score))
               (setq best-son-list son-list)
               (setq best-score cur-score)
               (setq best-value cur-value)
               (setq best-fun fun))))
      (when (not (null best-son-list))
         (update-parents edgenum best-son-list)
         (update-sons edgenum best-son-list)
         (setq *new-value-list* 
            (cons (list edgenum best-value) *new-value-list*))
         (if *unpack-right*
            (dolist (a-son (reverse best-son-list) nil)   ; unpack right first
               (unpack-edge a-son))
            (dolist (a-son best-son-list nil)             ; unpack left first
               (unpack-edge a-son)))
         (setq new-value (cadr (get-new-value best-fun best-son-list)))
         (when (listp new-value)
            (setq new-value (fix-parsed-words-slot edgenum new-value)))
         (setq *new-value-list* 
            (cons (list edgenum new-value) *new-value-list*))
         (return-from unpack-edge new-value))))


; from disambig-run.lisp
; get-rule-by-fun-number fetches the proper grammar rule from *gg* given the 
;                        grammar function number
(defun get-rule-by-fun-number (fun-num)
   (let ((prefix-name nil)
         (rule-id nil))
      (setq prefix-name 
         (read-from-string 
            (string-right-trim "1234567890"
			       (write-to-string (third (car *gg*))))))
      (setq rule-id 
         (read-from-string
            (concatenate 'string (write-to-string prefix-name)
                                 (write-to-string fun-num))))
      (dolist (a-rule *gg* nil)
         (when (eq (third a-rule) rule-id)
            (return-from get-rule-by-fun-number a-rule)))))


; update-parents updates the node-parent-list
; from disambig-run.lisp
(defun update-parents (mother son-list)
   (dolist (a-son son-list nil)
      (setq *node-parent-list* (cons (list a-son mother) *node-parent-list*))))

; update-sons updates the node-son-list
; from disambig-rub.lisp
(defun update-sons (mother son-list)
   (setq *node-son-list* (cons (list mother son-list) *node-son-list*)))

; get-new-value gets the f-struct value using the new-value-list
; adapted from Alon's get-new-value from disambig-run.lisp
(defun get-new-value (func sons)
   (if (null func) 
          'no-function                                     ; then
           (do-reunification-ext sons func)))

;          (apply func (mapcar #'(lambda (a-son)            ; else
;                                   (let ((son-val nil))
;                                      (setq son-val 
;                                         (second (assoc a-son 
;                                                        *new-value-list*)))
;                                      (when (null son-val)
;                                         (setq son-val 
;                                            (edge-fstruct 
;                                               (aref edge-array a-son))))
;                                      son-val))
;                               sons))))

; fix-parsed-words-slot fixes the "word-parsed-to-this-point" slot
; adapted from fix-parsed-words-slot from disambig-run.lisp
(defun fix-parsed-words-slot (edgenum new-value)
   (let ((new-value-list nil)
         (new-parsed-words nil)
         (old-value nil)
         (old-parsed-words nil)
         (correct-value nil))
      (if (equal (car new-value) '*or*)
          (setq new-value-list (cdr new-value))
          (setq new-value-list (list new-value)))
      (dolist (a-value new-value-list nil)
         (setq new-parsed-words 
            (find 'words-parsed-to-this-point a-value :key #'car))
         (when new-parsed-words
            (setq old-value (edge-fstruct (aref edge-array edgenum)))
            (if (equal (car old-value) '*or*)
                (setq old-value (cadr old-value)))
            (setq old-parsed-words
               (find 'words-parsed-to-this-point old-value :key #'car))
            (when old-parsed-words
               (rplacd (assoc 'words-parsed-to-this-point a-value)
                       (cdr old-parsed-words))))
         (setq correct-value (cons a-value correct-value)))
      (if (> (length correct-value) 1)
          (setq correct-value (cons '*or* (reverse correct-value)))
          (setq correct-value (car correct-value)))))

; consistent re-does the unifications to verify consistency all the way up
; adapted from Alon's consistent function found in disambig-run.lisp
(defun consistent (edgenum)
   (let ((cat (edge-cat (aref edge-array edgenum)))
         (parent (second (assoc edgenum *node-parent-list*))))
      (if (and (member cat *goal*)
               (null parent))
          (return-from consistent t)                 ; then
          (check-unify parent))))                    ; else

; check-unify checks the unification itself
; adapted from Alon's check-unify found in disambig-run.lisp
(defun check-unify (edgenum)
   (let ((lhs (edge-cat (aref edge-array edgenum)))
         (rhs nil)
         (rule-number nil)
         (fun nil)
         (son-list nil)
         (new-value nil))
      (setq son-list (second (assoc edgenum *node-son-list*)))
      (setq rhs 
         (mapcar #'(lambda (edge)
                     (let ((cat (edge-cat (aref edge-array edge))))
                        (when (listp cat)
                           (setq cat (car cat)))
                        cat))
                  son-list))
      (when (listp lhs) (setq lhs (car lhs)))
      (setq rule-number (second (find-rule-number lhs rhs)))
      (setq fun (third (get-rule-by-fun-number rule-number)))
      (setq new-value (cadr (get-new-value fun son-list)))
      (when (listp new-value)
         (setq new-value (fix-parsed-words-slot edgenum new-value)))
      (cond ((null new-value) (return-from check-unify nil))
            (t (setq *new-value-list* 
                  (cons (list edgenum new-value) *new-value-list*))
               (return-from check-unify (consistent edgenum))))))

(defun find-true-score (edgenum)
  (let ((actions (transitions-from-parse-tree 
                     (final-tree-from-edge edgenum)))
        (score 1.0))
    (list (progn
             (mapcar #'(lambda (ac1 ac2) (setf score (* score (bigram ac1 ac2))))
                    (butlast actions) (cdr actions))
             score)
          (length actions))))

(defun final-tree-from-edge (edgenum)
(if (assoc edgenum *node-son-list*)
    (let* ((rule-num (second (find-rule-number 
                      (edge-cat (aref edge-array edgenum))
                      (mapcar #'(lambda (ch) (edge-cat (aref edge-array ch))) 
                               (cadr (assoc edgenum *node-son-list*))))))
           (rule (aref *grammar-table* (- rule-num 1))))

          (list rule-num 
                (mapcar #'(lambda (ch cat) 
                             (if (equal cat '%)
                                 (get '% 'lex-ind)
                                 (final-tree-from-edge ch)))
                             (cadr (assoc edgenum *node-son-list*))
                             (cadr rule))))
    (get (edge-cat (aref edge-array edgenum)) 'lex-ind)))

;; score-ambig computes the combined score of a particular ambiguity
;; adapted from Alon's score-ambig from disambig-run.lisp
(defun score-ambig (edgenum son-list cur-value)
    (let ((prob-entry nil)
         (prob-score nil)
         (prob-count nil)
         (adj-score nil)
         (ambig-vals nil)
         (best-vals nil)
         (best-count nil)
         (final-score nil))
      (setq prob-entry
         (assoc son-list (third (assoc edgenum *node-score-list*))
                         :test #'equal))
      (setq prob-score (second prob-entry))
      (setq prob-count (third prob-entry))
      (if (= prob-score 0)
          (setq adj-score (* 0.1 prob-count))
          (setq adj-score (* 0.1 (- (log prob-score 10)))))

      (if (listp cur-value)
         (progn                                                 ; then
            (if (and *new-unify* (eq (car cur-value) '*OR*))
	        (setq ambig-vals (cdr cur-value))   ; then
	        (setq ambig-vals (list cur-value))) ; else

            (setq ambig-vals (no-duplicate ambig-vals))
            (setq best-vals (select-by-counter ambig-vals))
            (setq best-count (second (assoc 'counter (car best-vals)))))

         (setq best-count 0))                  ; else - cur-value is "T"

      (setq final-score (+ (* 1.1 best-count) 
                           adj-score))

      (return-from score-ambig final-score)))

;----------------------------------------------------------------------------
; PART 4: Top Level Disambiguation Functions
;----------------------------------------------------------------------------

(defun set-input-list (sent)
  (let ((result nil))
    (dotimes (x (length sent))
       (push (list (+ x 1) (elt sent x)) result))
    (setf *input-list* result)))

; disp-parses displays the final disambiguated parse
; adapted from Alon's disp-parses in disambig-run.lisp
(defun disp-parses (sent)
   (set-input-list sent)
   (let ((value-list nil) 
         (score-list nil)
         (topedge-list nil)
         (display-cnt 0)
         (a-sym nil)
         (sym-node-ptr nil)
         (sym-score nil)
         (sym-span nil)
         (sym-subst nil)
         (sym-skip nil)
         (skip-struct-list nil)
         (sym-sent nil)
         (sym-quality nil)
         (sym-value nil)
         (top-node-score nil)
         (true-score nil)
         (disp-values nil) 
         (big nil) 
         (small nil))

;; setup for disambiguation

      (when *disambig-flag*
         (assign-edge-scores)
         (setq *new-value-list* nil))

      (setq topedge-list (parse-result sent))

      (setq score-list (score-parses topedge-list))
      (setq *parse-score-list* score-list)

      (dolist (symbol-pair score-list value-list)

          (when (= display-cnt *display-limit*)
                (setq *parse-value* value-list)
                (return-from disp-parses (car value-list)))

          (setq a-sym (car symbol-pair))
          (setq sym-score (second symbol-pair))
          (setq sym-skip (edge-skip (aref edge-array a-sym)))
          (setq sym-subst nil)
          (setq sym-span (list (edge-p1 (aref edge-array a-sym))
                               (edge-p2 (aref edge-array a-sym))))
          ;(setq sym-skip (full-skip sym-skip sym-span))
          (setq skip-struct-list nil) ;(anal-skip sym-skip))
          (setq sym-sent nil) ;(get-parsed-sent sym-skip sym-subst))
          (setq *skipped-words* sym-skip)
          (setq *subst-words* sym-subst)

          (setq sym-quality (qual-parse sym-score))

          (setq *start-node* a-sym)

	  (setq sym-value (edge-fstruct (aref edge-array a-sym)))

	  (when (not (listp sym-value))
	     (setq sym-value (list (list 'value sym-value))))

	  (if (and *new-unify* (eq (car sym-value) '*OR*))
	      (setq sym-value (cdr sym-value))   ; then
	      (setq sym-value (list sym-value))) ; else

;; AL - now find the best ambiguity 

          (if (and *disambig-flag* 
                   (or (> (length sym-value) 1)
                       (is-ambiguous (car sym-value))))
             (progn   ; unpack only if ambiguous
                (unpack-top a-sym)
                (setq top-node-score 
                   (second (assoc a-sym *node-score-list*)))
                (setq *parse-prob* top-node-score)
                (setq true-score (second (car (score-parses (list a-sym)))))
                (rplacd (assoc a-sym *parse-score-list*) (list true-score))

                (setq disp-values 
                   (second (assoc a-sym *new-value-list*)))
	        (if (and *new-unify* (eq (car disp-values) '*OR*))
	            (setq disp-values (cdr disp-values))   ; then
	            (setq disp-values (list disp-values)))) ; else

             (progn                      ;; else clause (no disambiguation)
                (setq true-score sym-score)
                (setq disp-values (select-by-counter sym-value))))


          (setq *parse-score* true-score)
          (setq disp-values (no-duplicate disp-values))
	  (setq value-list (append value-list disp-values))

          (setq display-cnt (+ display-cnt 1))

          (format *out* "~2& Parse of input sentence :")
          (format *out* "~&   ~A " sym-sent)
          (format *out* "~2& Parse score is : ~D " true-score)
          (format *out* "    Parse quality is : ~A " sym-quality)
          (format *out* "~2& Words skipped : ~A " sym-skip)
	  (cond 
	        ((> (length sym-value) 1)
	           (format *out* "~2& ~D (~D) ambiguities found and"
		           (length sym-value) (length disp-values)))
	        (t
	           (format *out* "~2& ~D (~D) ambiguity found and"
			   (length sym-value) (length disp-values))))
	       )
      (setq *parse-value* value-list)
      (pprint (car value-list))
      (return-from disp-parses (car value-list))))


; full-skip creates the full list of skipped words, including edges
(defun full-skip (skip-list span)
   (let ((temp-skip (append (append (poslist 1 (- (car span) 1)) skip-list)
                            (poslist (+ (cadr span) 1) 
                                     (length *input-list*))))
         (word))
      (mapcar #'(lambda (sk) (setf word (assoc sk *input-list*))
                             (list (second word) (first word)))
            temp-skip)))

;; score-parses creates an sorted association list of symbol nodes and their 
;;              ranked scores. This list is then used in disp-parses to 
;;              display the best parse result(s)
;; adapted from Alon's score-parses found in disambig-run.lisp
(defun score-parses (edge-list)
   (let ((cur-sym-pos nil)
         (cur-sym-skip nil)
         (cur-sym-subst nil)
         (full-sym-skip nil)
         (best-count nil)
         (sym-node-ptr nil)
         (ambig-res nil)
         (ambig-score 0)
         (parse-score nil)
         (score-list nil))

; go over all parses and score them, building the assoc list

        (dolist (a-sym edge-list score-list)
           (setq cur-sym-pos (list (edge-p1 (aref edge-array a-sym))
                                   (edge-p2 (aref edge-array a-sym))))
           (setq cur-sym-skip (edge-skip (aref edge-array a-sym)))
           (setq full-sym-skip (full-skip cur-sym-skip cur-sym-pos))
           (setq best-count (get-best-count a-sym))

           (when (null best-count)
              (setq best-count 1))
           (when *disambig-flag*
              (setq ambig-res (second (assoc a-sym *node-score-list*)))
              (if (= (car ambig-res) 0)
                 (setq ambig-score (* 0.1 (second ambig-res)))
                 (setq ambig-score (* 0.1 (- (log (car ambig-res) 10))))))
           (setq parse-score (+ (skip-penalty full-sym-skip)
                                (* 0.9 (length cur-sym-subst))
                                (* 1.1 best-count)
                                ambig-score))
           (setq score-list 
             (cons (list a-sym parse-score) score-list)))
        (setq score-list 
           (sort score-list #'(lambda (x y)
                                 (< (second x) (second y)))))))

;; get-best-count determines the sentence counter of the best ambiguity in 
;; a parse.
;; adapted from Alon's get-best-count in v9-0.lisp
(defun get-best-count (a-sym)
   (let ((sym-node-ptr nil)
         (sym-value nil)
         (best-values nil)
         (best-count 1))

      (when *disambig-flag*
         (setq sym-value (second (assoc a-sym *new-value-list*))))
      (if (null sym-value)
         (setq sym-value (edge-fstruct (aref edge-array a-sym))))

      (when (not (listp sym-value))
         (setq sym-value (list (list 'value sym-value))))
;	 (return-from get-best-count best-count))

      (if (and *new-unify* (eq (car sym-value) '*OR*))
	  (setq sym-value (cdr sym-value))   ; then
	  (setq sym-value (list sym-value))) ; else

      (setq best-values (no-duplicate sym-value))
      (setq best-values (select-by-counter best-values))
      (setq best-count (second (assoc 'counter (car best-values))))))

;; qual-parse determines the quality of a parse - GOOD/BAD
; from disambig-run.lisp
(defun qual-parse (score)
   (cond
      ((>= score 8) 'BAD)
      ((and (> score 3)
            (>= (/ score (- (length *input-list*) 1)) '1/2)) 'BAD)
       (t 'GOOD)))

;; skip-penalty computes the penalty score for the skipped words
; from v9-0.lisp
(defun skip-penalty (skip-list)
   (let ((word-pen nil)
         (sent-factor (/ (length *input-list*)))
         (a-pen 0)
         (total 0))
      (dolist (a-skip skip-list total)
         (if (null *semantic-list*)
             (if (is-noise (car a-skip))  ; no semantic penalty
                 (setq word-pen 0.0)      ; noise word gets penalty 0
                 (setq word-pen 1.0))     ; non-noise word gets penalty 1
             (progn               ; penalty according to semantic value
                (setq word-pen (second (assoc (car a-skip) *semantic-list*)))
                (when (null word-pen) 
                   (setq word-pen 
                      (second (assoc 'skip-default *semantic-list*)))
                   (when (null word-pen) (setq word-pen 0)))))
         (setq a-pen (* word-pen (+ 0.95 (* 0.1 sent-factor (second a-skip)))))
         (setq total (+ total a-pen)))))


;----------------------------------------------------------------------------
; PART 5: Utility Functions for Disambiguation
;----------------------------------------------------------------------------


;; is-ambiguous returns T when the argument f-struct contains an *OR*
;; from disambig-run.lisp
(defun is-ambiguous (f-struct)
   (let ((f-struct-list nil)
         (temp-res nil))

      (cond ((null f-struct) nil)
            ((atom f-struct) nil)
            ((equal (car f-struct) '*OR*) t)
            ((equal (car f-struct) '*MULTIPLE*)
                (setq f-struct-list (cdr f-struct))
                (dolist (an-f-struct f-struct-list nil)
                   (setq temp-res (is-ambiguous an-f-struct))
                   (when temp-res (return-from is-ambiguous t))))
            ((listp f-struct)
                (dolist (a-slot f-struct nil)
                   (setq temp-res (is-ambiguous (second a-slot)))
                   (when temp-res (return-from is-ambiguous t))))
            (t t))))


;;; comp-ilt compares two ILTs to determine their equivalence
;;; adapted from Noah's f-struct-diff.lisp code
;;; from disambig-sched.lisp

(defun comp-ilt (ilt1 ilt2)
   (setq *num-mismatches* 0)
   (if (diff ilt1 ilt2 :outfile *NULL-OUT*
                       :ignore '(words-parsed-to-this-point allow-break
                                 counter speech-act a-speech-act))
       (return-from comp-ilt t)
       (return-from comp-ilt nil)))
		   
; from disambig-sched.lisp
(defun diff (a b &key (outfile t) (ignore nil) (consider nil))
  (fstruct-equal a b :outfile outfile :ignore ignore :consider consider))

;from disambig-sched.lisp
(defun fstruct-equal (fstruct1 fstruct2 &key (outfile t) (slothist nil) (ignore nil) (consider nil))
  (cond ((and (null fstruct1) (null fstruct2)) t)
        ((and (atom fstruct1) (atom fstruct2)) 
	 (if (equal fstruct1 fstruct2)
	     t
	   (progn (format outfile "--------~%Slot fillers differ:~% CONTEXT: ~S~% Filler for f-structure 1: ~S~% Filler for f-structure 2: ~S~%" slothist fstruct1 fstruct2)
		  (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*)))
		  nil)))

;; at least one MULTIPLE VALUE  OR ;; at least one *OR* value
	((or
	  (and (or (eql (or-atom-car fstruct1) '*OR*) (eql (or-atom-car fstruct2) '*OR*))
	       (not  (or (eql (or-atom-car fstruct1) '*MULTIPLE*) (eql (or-atom-car fstruct2) '*MULTIPLE*))))
	  (and (or (eql (or-atom-car fstruct1) '*MULTIPLE*) (eql (or-atom-car fstruct2) '*MULTIPLE*))
	       (not  (or (eql (or-atom-car fstruct1) '*OR*) (eql (or-atom-car fstruct2) '*OR*)))))
	 (MULT-COMPARE fstruct1 fstruct2 :outfile outfile :slothist slothist :ignore ignore :consider consider))
	
;; one *OR* and one *MULTIPLE*
	((or (and (eql (or-atom-car fstruct1) '*OR*) (eql (or-atom-car fstruct2) '*MULTIPLE*))
	     (and (eql (or-atom-car fstruct1) '*MULTIPLE*) (eql (or-atom-car fstruct2) '*OR*)))
	 (format outfile "--------~%One slot has *MULTIPLE* values and the other slot has ambiguities *OR* ~%Filler for f-structure 1:~%")
	 (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*)))
	 (pprint fstruct1 outfile)
	 (format outfile "~%Filler for f-structure 2: ~%")
	 (pprint fstruct2 outfile)
	 (setq ok nil))
	
        ((and (listp fstruct1) (listp fstruct2))
         (let ((ok t))
	   (dolist (slot1 fstruct1)
		   (if (or (and consider (member (car slot1) consider))  ; we consider on consider list
			   (and ignore (not (member (car slot1) ignore))) ; we consider when not ignore
			   (and (not consider) (not ignore)))            ;we consider everything when no list
		       
		       (let ((slot2 (assoc (car slot1) fstruct2)))
			 (if (null slot2) 
			     (progn
			       (setq ok nil)
			       (format outfile "--------~%The following slot is in the first fstructre but not the second")
			       (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*)))
			       (pprint slot1 outfile) 
			       (if slothist (format outfile "~%With the context of ~S~%" slothist)(format outfile "~%"))
			       nil)
			   (if (not (fstruct-equal (cadr slot1) (cadr slot2) :outfile outfile :slothist (append slothist (list (car slot1))) :ignore ignore :consider consider))
			       (setq ok nil))))
		     (if (not (member (car slot1) *ignored-list*))
			 (setq *ignored-list* (cons (car slot1) *ignored-list*)))))
	   
	   
	   (dolist (slot2 fstruct2)
		   (if (or (and consider (member (car slot2) consider))  ; we consider on consider list
			   (and ignore (not (member (car slot2) ignore))) ; we consider when not ignore
			   (and (not consider) (not ignore)))
		       (let ((slot1 (assoc (car slot2) fstruct1)))
			 (if (null slot1)
			     (progn
			       (setq ok nil)
			       (format outfile "--------~%The following slot is in the second fstructre but not the first")
			       (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*)))
			       (pprint slot2 outfile)
			       (if slothist (format outfile "~%With the context of ~S~%" slothist)(format outfile "~%")))))
		     (if (not (member (car slot2) *ignored-list*))
			 (setq *ignored-list* (cons (car slot2) *ignored-list*)))))
	   ok))
	(t (format outfile "--------~%Slot fillers differ -- one is a simple filler the other is a sub-structure.~%The context is ~S ~%The filler for f-structure 1:" slothist)
	   (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*)))
	   (pprint fstruct1 outfile)
	   (format outfile "~%The filler for f-structure2:")
	   (pprint fstruct2 outfile)
	   nil)))

; from disambig-sched.lisp
(defun mult-compare (fstruct1 fstruct2 &key (outfile t) (slothist nil) (ignore nil) (consider nil))
  (let ((multlist1 (if (or (eql (or-atom-car fstruct1) '*OR*) (eql (or-atom-car fstruct1) '*MULTIPLE*)) (cdr fstruct1) (list fstruct1)))
	(multlist2 (if (or (eql (or-atom-car fstruct2) '*OR*) (eql (or-atom-car fstruct2) '*MULTIPLE*)) (cdr fstruct2) (list fstruct2)))
	(count1 -1)
	(mult-ambig (if (or (eql (or-atom-car fstruct1) '*MULTIPLE*) (eql (or-atom-car fstruct2 ) '*MULTIPLE*))
			'*multiple*
		      '*OR*))
	(ok t))
    
    (if (and (Not (eql (or-atom-car fstruct1) '*OR*)) (Not (eql (or-atom-car fstruct1) '*MULTIPLE*)))
	(progn (format outfile "--------~%F-structure 2 has ~S values and f-structure 1 does not.~%" mult-ambig) (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*))))
      (if (and (Not (eql (or-atom-car fstruct2) '*OR*)) (Not (eql (or-atom-car fstruct2) '*MULTIPLE*)))
	  (progn (format outfile "--------~%F-structure 1 has ~S values and f-structure 2 does not.~%" mult-ambig) (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*))))
	(setq slothist (append slothist (list mult-ambig)))))
    
    (dolist (item1 multlist1)
	    (setq count1 (1+ count1))
	    (let ((current-multlist2 multlist2))
	      (if (not
		   (do* ((item2-list current-multlist2 (cdr item2-list))
			 (count2 0 (1+ count2))
			 (item2 (car item2-list) (car item2-list))
			 (equal-bool (fstruct-equal item1 item2 :outfile *null-out*
						    :slothist slothist :ignore ignore :consider consider)
				     (fstruct-equal item1 item2 :outfile *null-out*
						    :slothist slothist :ignore ignore :consider consider)))
			((or equal-bool (null item2-list))  ;; exit clause
			 (progn                          ;; exit result
			   (if equal-bool (setq multlist2 (subst-nth multlist2 count2)))
			   equal-bool))))
		  (progn 
		    (setq ok nil)
		    (format outfile "--------~%The following ~S value is in the first fstructure but not the second" mult-ambig)
		    (if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*)))
		    (pprint item1 outfile)
		    (format outfile "~%With the context of ~S~%" slothist)))))
    (setq multlist2 (remove nil multlist2))
    (if multlist2
	(dolist (missing multlist2)
		(setq ok nil)
		(format outfile "--------~%The following ~S value is in the second fstructure but not the first" mult-ambig)
		(if (not (equal outfile *null-out*)) (setq *num-mismatches* (1+ *num-mismatches*)))
		(pprint missing outfile)
		(format outfile "~%With the context of ~S~%" slothist)))
    ok))


;    (setf multlist2 (remove-nth multlist2 count2))))
; from disambig-sched.lisp
(defun subst-nth (item-list index)
  (if (eql index 0)
      (cons nil (cdr item-list))
    (if (> index (- (length item-list) 2))
	(append (subseq item-list 0 index)
		(list nil))
      (append (subseq item-list 0 index)
	      (list nil)
	      (subseq item-list (1+ index))))))

;from disambig-sched.lisp
(defun or-atom-car (x)
  (if (atom x) 
      nil
    (car x)))


;--------------------------------------------------------------------------
; REORDERING code
;--------------------------------------------------------------------------


(defun reset-inactives ()
(dotimes (x 100) (setf (aref *inactive-hold* x) nil)))


(defun insert-inactive-hold (pos cat ed)
  (let ((orig (assoc cat (aref *inactive-hold* pos))))
    (if orig
	(rplacd (assoc cat (aref *inactive-hold* pos)) (push ed (cdr orig)))
        (push (list cat ed) (aref *inactive-hold* pos)))))

(defun hold-inactive (after cat p1 p2 needed children rule skip insert unif)
(let ((num nil))
  (if (and (not (duplicate-inactive cat p1 p2 needed children rule skip insert unif))
          (cadr (do-unification-test (append children (list after)) rule)))
      (progn
        (insert-inactive-hold p1 cat 
           (setf num (build-edge cat p1 p2 needed children nil rule skip insert unif)))))))

(defun duplicate-inactive (cat p1 p2 needed children rule skip insert unif)
  (let ((stat nil)
	(edge nil))
    (dolist (ed (cdr (assoc cat (aref *inactive-hold* p1))))
       (setf edge (aref edge-array ed))
       (if (and (equal rule (edge-rule edge))
		(equal p1 (edge-p1 edge))
		(equal p2 (edge-p2 edge))
		(equal children (edge-children edge))
		(equal needed (edge-needed edge))
                (not (insert-subsume insert (edge-insert edge)))
                (not (unif-subsume unif (edge-unif edge)))
                (not (skip-subsume skip (edge-skip edge))))

	   (progn
	     (setf stat t))))
    stat))

(defun get-next-inactives ()
  (do ((x 99)
       (inacts nil))
      ((or inacts (equal -1 x)) (if inacts
                                    (progn
                                      
                                      (setf (aref *inactive-hold* x)
                                            (sortnts (aref *inactive-hold* x)))
                                      (setf inacts (aref *inactive-hold* x))
                                      (setf (aref *inactive-hold* x) (cdr inacts))
                                      (cons x (car inacts)))
                                    nil))
      (setf inacts (aref *inactive-hold* x))
      (if (null inacts) (setf x (- x 1)))))

(defun sortnts (inactlist)
(if *optimize*
     (sort inactlist #'nt<ed)
     inactlist))

(defun nt<ed (ed1 ed2)
   (if (nt< (car ed1) (car ed2))
       t nil))

(defun get-spec-inactives (cat pos)
  (let ((inacts (aref *inactive-hold* pos))
        (new nil)
        (res nil))
    (dolist (in inacts)
      (if (equal (car in) cat)
          (setf res (cons pos in))
          (push in new)))
   (setf (aref *inactive-hold* pos) new)
   res))

; Extend the inactive edges that are on hold
(defun process-inactive-hold (pos)
  (do* ((inacts (get-next-inactives) (get-next-inactives))
        (cat (cadr inacts) (cadr inacts))
        (p1 (car inacts) (car inacts)))
     ((null inacts) t)
     (dolist (ed (cddr inacts))
        (process-inactive-hold-aux pos p1 cat ed))
     (do ((inacts2 (get-spec-inactives cat p1) (get-spec-inactives cat p1)))
         ((null inacts2) t)
         (dolist (ed (cddr inacts2))
            (process-inactive-hold-aux pos p1 cat ed)))
     (setf packedges 
          (intersection (cdr (assoc cat (aref inactive-edges-start p1)))
                        (cdr (assoc cat (aref inactive-edges-end pos)))))
     (if packedges
        (dolist (pile (skip-piles (prune-inactive-list  packedges)))
              (setf pile (remove nil 
                  (mapcar #'(lambda (ed) (if (edge-used (aref edge-array ed))
                                             nil ed)) pile)))
              (if pile (ambiguity-packing-aux pile))))
     ))

(defun noloop (cat list)
  list)

; (let ((children nil))
;  (remove nil 
;      (mapcar #'(lambda (ed) 
;          (setf children (edge-children (aref edge-array ed)))
;          (if (and children (listp (car children)))
;              (setf children (car children)))
;          (if    (and 
;                   (equal (length children) 1) 
;                   (equal cat (edge-cat (aref edge-array (car children)))))
;              nil ed)) list))))

(defun process-inactive-hold-aux (pos p1 cat act)
     (let* ((edge (aref edge-array act))
            (skip nil)
            (unif nil)
            (insert nil)
	    (new-children (get-inactive-start (edge-p2 edge)
					      (car (edge-needed edge))))
	    (new-children2 (get-inactive-end pos (car (edge-needed edge))))

            ; childnums is a list of references to inactive edges of the
            ; category needed by the current active edge
	    (childnums (noloop (car (edge-needed edge)) (intersection new-children new-children2)))
	    (child (if childnums (aref edge-array (car childnums)))))
       (if child
          (progn
       ; the current active edge can be modified once, so for all of the
       ; new child inactive edges except for the first one, a copy of the
       ; current active edge must be made first before extending the active
       ; edge over the associated inactive child edge.  This happens inside
       ; process-active-hold-aux
       (process-inactive-hold-aux2 edge (cdr childnums))
       ; for the first child inactive edge, modify the active edge so that
       ; it includes the child inactive edge as it's next child.  Make
       ; necessary adjustments to the edge in terms of span, skips, children,
       ; and needed.
       (setf skip (append (edge-skip edge)
                  (append (addskip (edge-p2 edge) (edge-p1 child)) 
                          (edge-skip child))))
       (setf (edge-p2 edge) (edge-p2 child))
       (setf insert  (append (edge-insert edge) (edge-insert child)))
       (setf unif 
             (append (edge-unif edge) (edge-unif child)))
       (if (and (can-insert-cats-ext insert)
                (unif-okay unif)
                (or (null *skip-limit*)
                    (not (> (length skip) *skip-limit*))))
           (progn
       (setf (edge-children edge) (append (edge-children edge)
					  (list (car childnums))))
       (setf (edge-needed edge) (cdr (edge-needed edge)))
       (setf (edge-skip edge) skip)
       (setf (edge-insert edge) insert)
       (setf (edge-unif edge) unif)
       (setf (edge-fstruct edge) 
             (do-unification-ext (edge-children edge) (edge-rule edge)))
       (if (equal t (car (edge-fstruct edge)))
           (progn
             (setf (edge-fstruct edge) (cadr (edge-fstruct edge)))
             (insert-inactive-simple act))
           (if (not (null (edge-fstruct edge)))
               (progn
                (setf (edge-unif edge)
                      (append (edge-unif edge) 
                         (list (caddr (edge-fstruct edge)))))
                (setf (edge-fstruct edge) (cadr (edge-fstruct edge)))
                (insert-inactive-simple act)
                 )))))))))

(defun process-inactive-hold-aux2 (edge childnums)
  (let ((cat (edge-cat edge))
	(p1 (edge-p1 edge))
        (skip nil)
        (unif nil)
        (insert nil)
	(needed (cdr (edge-needed edge)))
	(children (edge-children edge))
	(fstruct nil)
	(rule (edge-rule edge))
	(child nil))
    (dolist (ch childnums)
	    (setf child (aref edge-array ch))
            (setf skip          (append (edge-skip edge)
                                        (append
                                            (addskip (edge-p2 edge) 
                                                     (edge-p1 child))
                                            (edge-skip child))))
            (setf insert (append (edge-insert edge) 
                                        (edge-insert child)))
            (setf unif (append (edge-unif edge)
                                        (edge-unif child)))
            (if (and (can-insert-cats-ext insert)
                     (unif-okay unif)
                     (or (null *skip-limit*)
                         (not (> (length skip) *skip-limit*))))
                 (insert-inactive-ext cat p1 
			        (edge-p2 child) 
			        (append children (list ch))
			        (do-unification-ext (append children (list ch))
                                       rule)
                                skip
                                insert unif
                                )))))


(defun insert-inactive-simple (ed)
   (let* ((edge (aref edge-array ed))
          (cat (edge-cat edge))
          (p1 (edge-p1 edge))
          (p2 (edge-p2 edge))
          (children (edge-children edge))
          (struct (edge-fstruct edge))
          (skip (edge-skip edge))
          (insert (edge-insert edge))
          (unif (edge-unif edge)))
      (insert-inactive-simple-aux 
          ed cat p1 p2 children struct skip insert unif)))

(defun insert-inactive (cat p1 p2 children struct skip insert unif)
  (if struct
      (insert-inactive-simple-aux 
          (build-edge cat p1 p2 nil children struct nil skip insert unif)
          cat p1 p2 children struct skip insert unif)))

; we basically need a new way of inserting inactive edges
; then we need a new way of doing ambiguity packing

(defun headed-rules-short-lex (lex)
  (mapcar #'(lambda (rule) (list (car rule) (cadr rule) (list lex) (cadddr rule) (caddddr rule))) (gethash '% *gram*)))

(defun caddddr (list) (cadr (cdddr list)))

(defun insert-inactive-simple-aux (edgenum cat p1 p2 children struct skip insert unif)
  (if struct
      (let ((predicted (alt-predicted p1))
            (extendhold nil))
           (insert-inactive-start p1 edgenum cat)
           (insert-inactive-end p2 edgenum cat)

           ; loop once for each rule that has the category of the current
           ; edge as its leftmost daughter category
           (dolist (rule (append (headed-rules-short cat)
			         (if (lex-p struct)
			             (headed-rules-short-lex cat))))

                   ; if top-down filtering is turned on and the RHS 
                   ; category of the current rule is not predicted,
                   ; don't do anything.
             (cond ((and *lc-prediction* 
                         (not (head-link-short (car rule) predicted)))
	             nil)
                   ((not (can-insert-cats-ext (append (elt rule 4) insert))) nil)
                   ; if the current rule has more than one daughter, create
                   ; a new active edge, but put in on hold until all of the
                   ; edges that will become inactive after the current word
                   ; is processed have been processed.
	           ((> (length (elt rule 2)) 1)
	            (hold-active (car rule) p1 p1 (elt rule 2) 
                         (children-from-inserts (elt rule 4) p1)
			  	 (elt rule 3) nil (elt rule 4) nil))

                   ; if the current rule has only one daughter, create a new
                   ; inactive edge from that rule and insert it into the chart.
                   (*hold-inactives*
                       (hold-inactive edgenum (car rule) p1 p1 (elt rule 2)
                           (children-from-inserts (elt rule 4) p1)
                                 (elt rule 3) nil (elt rule 4) nil))
	           (t (insert-inactive-ext (car rule) p1 p2 
                          (append (children-from-inserts (elt rule 4) p1) 
                                  (list edgenum))
				 (do-unification-ext 
                                      (append (children-from-inserts
                                          (elt rule 4) p1) (list edgenum))
						 (elt rule 3)) skip 
                                      (append insert (elt rule 4)) unif))))

           ; extend the active edges that end where the current edge begins
           (dolist (edge (get-active p1))
               (if (and
                    (member cat (edge-needed (aref edge-array edge)))
                    (setf extendhold
                     (can-extend cat (edge-needed (aref edge-array edge))
                        (edge-insert (aref edge-array edge)) p1)))
	           (extend-active (aref edge-array edge) edgenum (cadr extendhold)))))))



; This function extends the current active edge over the new inactive child 
; edge
(defun extend-active (edge newchild newinsert)
  (let ((children nil))
             ; If the new child edge is not enough to make this edge inactive,
             ; create a new active edge that will eventually be extended over 
             ; the child edge, but put it on hold for now.
       (cond ((and *skip-limit*
                  (> (length (append (edge-skip edge)
                             (addskip (edge-p2 edge) 
                               (edge-p1 (aref edge-array newchild)))))
                       *skip-limit*)) nil)
             ((> (length (edge-needed edge)) (+ 1 (length newinsert)))
              (hold-active (edge-cat edge)
		   (edge-p1 edge)
		   (edge-p1 (aref edge-array newchild)) ; to make p2 reliable
		   (shorten-by-len (edge-needed edge) newinsert)
		   (append (edge-children edge) newinsert)
		   (edge-rule edge)
                   (append (edge-skip edge)
                       (addskip (edge-p2 edge)
                           (edge-p1 (aref edge-array newchild))))
                   ;(append (edge-skip edge) 
                   ;        (addskip 
                   ;           (edge-p2 edge) 
                   ;           (edge-p1 (aref edge-array newchild))))
                   (edge-insert edge)
                   ;(append 
                   ;   (append (edge-insert edge) 
                   ;        (edge-insert (aref edge-array newchild)))
                   ;   (mapcar #'(lambda (newch) 
                   ;       (edge-cat (aref edge-array newch))) newinsert))
                   (edge-unif edge)
                   ;(append (edge-unif edge)
                   ;        (edge-unif (aref edge-array newchild)))
                   ))
              ; if the new child is enough to make the current active edge 
              ; inactive, and the current active edge is packed, unpack the
              ; edge and extend each of the edges packed inside the current
              ; active edge and then the edge itself.
	      ((listp (edge-rule edge))
	       (let ((hold nil)
	             (chil nil)
	             (need nil)
	             (p2 nil))

                  ; Loop for each packed edge       
	          (dolist (ed (cdr (edge-rule edge)))
	             (setf hold (aref edge-array ed))
	             (setf chil (edge-children hold))
	             (setf need (edge-needed hold))
	             (setf p2 (edge-p2 hold))
	             (setf (edge-p2 hold) (edge-p2 edge))
	             (setf (edge-children hold)
		           (append (edge-children hold)
			           (lastn (- (length (edge-needed hold))
				             (length (edge-needed edge)))
				          (edge-children edge))))
	             (setf (edge-needed hold) (edge-needed edge))
	             (extend-active hold newchild newinsert)
	             (setf (edge-p2 hold) p2)
	             (setf (edge-needed hold) need)
	             (setf (edge-children hold) chil))

                   ; extend the current edge
	           (extend-active (aref edge-array
				    (build-edge (edge-cat edge) (edge-p1 edge)
				      (edge-p2 edge) (edge-needed edge)
				      (edge-children edge) nil
				      (car (edge-rule edge))
                                      (edge-skip edge)
                                      (edge-insert edge)
                                      (edge-unif edge)))
			            newchild newinsert)))

                 ; if the current active edge is not packed and only needs
                 ; the new child in order to become inactive, create an
                 ; inactive edge and place it in the chart.
            (*hold-inactives*
              (hold-inactive newchild (edge-cat edge)
		   (edge-p1 edge)
                   ;(edge-p2 edge)
		   (edge-p1 (aref edge-array newchild)) ; to make p2 reliable
		   (shorten-by-len (edge-needed edge) newinsert)
		   (append (edge-children edge) newinsert)
		   (edge-rule edge)
                   (append (edge-skip edge)
                      (addskip (edge-p2 edge)
                           (edge-p1 (aref edge-array newchild))))
                   ;(append (edge-skip edge) 
                   ;        (addskip 
                   ;           (edge-p2 edge) 
                   ;           (edge-p1 (aref edge-array newchild))))
                   (edge-insert edge)
                   ;(append 
                   ;   (append (edge-insert edge) 
                   ;        (edge-insert (aref edge-array newchild)))
                   ;   (mapcar #'(lambda (newch) 
                   ;       (edge-cat (aref edge-array newch))) newinsert))
                   (edge-unif edge)
                   ;(append (edge-unif edge)
                   ;        (edge-unif (aref edge-array newchild)))
               ))
	      (t (insert-inactive-ext (edge-cat edge) (edge-p1 edge)
		                  (edge-p2 (aref edge-array newchild))
		                  (setf children
			                (append (append (edge-children edge) 
                                                   newinsert)
                                                (list newchild)))
		                  (do-unification-ext children (edge-rule edge))
                                  (append (edge-skip edge)
                                          (append (addskip 
                                                     (edge-p2 edge)
                                                     (edge-p1 
                                                       (aref edge-array 
                                                             newchild)))
                                                  (edge-skip (aref edge-array 
                                                                   newchild))))
                                   (append (append (edge-insert edge)
                                         (mapcar #'(lambda (newch) (edge-cat (aref edge-array newch))) newinsert))
                                           (edge-insert 
                                              (aref edge-array newchild)))
                                   (append (edge-unif edge)
                                           (edge-unif
                                              (aref edge-array newchild))))))))

(defun lc-parse (cat sent)
 (lc-parse2 cat sent))

(defun skipok (skips1 skips2 sent)

  (let ((stat t))
  (dotimes (x (length sent))
    (if (> (skipnum skips1) (skipnum skips2)) (setf stat nil))
    (setf skips1 (skipit skips1))
    (setf skips2 (skipit skips2)))
  stat))

(defun skipnum (skips)
  (do ((num 0)
       (skip (cdr skips) (cdr skip)))
    ((not (equal (car skip) '*skip*)) num)
    (setf num (+ num 1))))

(defun skipit (skips)
  (do ((num 0)
       (skip (cdr skips) (cdr skip)))
    ((not (equal (car skip) '*skip*)) skip)
    (setf num (+ num 1))))


; This function implements the basic LC parsing algorithm with beam skipping.
(defun lc-parse2 (cat sent)
  (setf *input-list* (make-input-list sent))
  (setf *active-hold* nil)
  ; Loop once for each word
  (dotimes (x (length sent))
     (setf *token-position* (+ *token-position* 1))
     (init-rule-bookkeeping)
     ; insert inactive '<start> edges at each non-zero vertex in order to
     ; allow for restarts.
	     (dotimes (y x) ; copy references to active edges from previous 
                            ; vertices until beam is filled up.
	     	(if (or (not (or *skip-beam* *skip-limit*))
                        (and *skip-beam* (null *skip-limit*)
                             (< (length (get-active x)) *skip-beam*))
                        (and *skip-limit* (null *skip-beam*)
                             (< y *skip-limit*))
                        (and (< y *skip-limit*) (< (length (get-active x)) *skip-beam*)))
	     	         (progn 
                                (copy-actives (- (- x y) 1) x)
                                )))

             ; pack active edges ending after the current word.
	     (packactives-pos x) 
	     (print (elt sent x))
     ; Loop once for each lexical entry for the current word
     (reset-inactives)
     (dolist (ent (get-lex-ents (elt sent x)))

             ; insert an inactive edge for the current lexical entry
             (insert-inactive (car ent) x (+ x 1) nil
		  (cadr ent) nil nil nil)
             (process-inactive-hold (+ x 1))
             (ambiguity-packing x (+ x 1))
             (do ()
                ((null *active-hold*) t)
                (process-active-hold (+ x 1))
                (if (null *active-hold*)
                    (process-inactive-hold (+ x 1)))))))



;***********************************************************************
;                    Optimizing Rule Reordering
;***********************************************************************


(defun nt< (a b)
  (if (member a (cadr (assoc b *ord2*)))
      t nil))

(defvar *ord* nil)
(defvar *ord2* nil)

(defun write-ord ()
  (with-open-file (file "ordfile" :direction :output :if-exists :supersede
                     :if-does-not-exist :create)
     (format file "~a~%" *ord2*))
  t)

(defun read-ord ()
  (with-open-file (file "ordfile" :direction :input)
     (setf *ord2* (read file)))
  (dotimes (x (length *ord2*))
     (setf (get (car (elt *ord2* x)) 'ntval) x))
  t)

(defun convert-ord ()
  (let ((ord nil)
	(rule nil)
	(syms nil)
	(ent nil))
    (dotimes (x (length *grammar-table*))
       (setf rule (aref *grammar-table* x))
       (if (not (member (car rule) syms))
	   (push (car rule) syms))
       (dolist (c (poss-pred-cats rule))
            (if (not (member c syms))
	        (push c syms))))
    (dolist (sym syms)
       (setf ent nil)
       (dotimes (x (length syms))
	  (if (equal (aref *ord* (get sym 'num) x) 1)
	      (push (elt syms x) ent)))
       (push (list sym ent) ord))
    (setf *ord2* ord)
    t))

(defun make-ord-table ()
  (let ((syms nil)
	(rule nil)
	(table nil))

    (dotimes (x (length *grammar-table*))
       (setf rule (aref *grammar-table* x))
       (if (not (member (car rule) syms))
	   (push (car rule) syms))
       (dolist (c (poss-pred-cats rule))
            (if (not (member c syms))
	        (push c syms))))
    (setf table (make-array (list (length syms) (length syms))
			    :initial-element 0))
    ;(print syms)
    (dotimes (x (length syms))
       (setf (get (elt syms x) 'num) x))

    (dotimes (x (length *grammar-table*))
       ;(print x)
       (setf rule (aref *grammar-table* x))
       (if (equal (length (elt rule 1)) 1)
           (progn
            ;(print (list (car rule) (car (cadr rule))))
           (setf (aref table (get (car rule) 'num)
                   (get (car (cadr rule)) 'num)) 1))))
    (dotimes (i (length syms))
       (dotimes (j (length syms))
          (if (equal 1 (aref table j i))
	     (dotimes (k (length syms))
		 (if (equal 1 (aref table i k))
		     (setf (aref table j k) 1))))))

    (setf *ord* table)
    t))

;;; Functions for pre-handling contractions prior to parsing

; dcp decontracts and then parses using p

(defun dcp (sentence)
   (p (decontraction sentence)))

; decontraction does the actual decontraction of sentence words

(defun decontraction (sentence)  ; sentence is a list of words.
  (let ((contract-list '((i+ll i will) 
			 (you+ll you will) 
			 (he+ll he will) 
			 (she+ll she will) 
			 (it+ll it will) 
			 (we+ll we will) 
			 (they+ll they will)

			 (i+m i am) 
			 (you+re you are) 
			 (he+s he is) 
			 (she+s she is) 
			 (it+s it is) 
			 (we+re we are) 
			 (they+re they are) 

			 (i+ve i have) 
			 (you+ve you have) 
			 (he+s he has) 
			 (she+s she has) 
			 (it+s it has)
			 (we+ve we have)
			 (they+ve they have)
			 
			 (i+d i would)
			 (you+d you would)
			 (he+d he would)
			 (she+d she would)
			 (it+d it would)
			 (we+d we would)
			 (they+d they would)
			 
			 (that+s that is)
			 (what+s what is)        ; AL
			 (there+s there is )
			 ))
	(new-sentence nil))
    
    (dolist (word sentence new-sentence)
	    (setq new-sentence (concatenate 'list new-sentence
					    (if (assoc word contract-list)
						(cdr (assoc word contract-list))
					      (list word)))))))





