;;;==================================================================;
;;;  -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;==================================================================;
;;;
;;;               Center for Machine Translation
;;;                 Carnegie-Mellon University
;;;                                                                       
;;;------------------------------------------------------------------;
;;;                                                                       
;;; Copyright (c) 1994
;;; Carnegie Mellon University. All Rights Reserved.                      
;;;                                                                       
;;;------------------------------------------------------------------;
;;;
;;;          File: irules.lisp
;;;  File created: Wed Nov 10 1993 by willy
;;;    Maintainer: Nicholas Brownlow <ndb@cs.cmu.edu>
;;; Last Modified: 17-Apr-95 at 21:23
;;;		   14-Nov-96 by garof@cs.cmu.edu
;;;   
;;;------------------------------------------------------------------;
;;; File Description
;;;
;;; This file contains the functions and global variable definitions
;;; used by the interpreter to read and hash interpreter rules.


;;;==================================================================;

;;; Package statements:                                                   

(in-package :user)

;..............................................................................
; 14-Nov-96-garof:  For version control.
;..............................................................................
;(load (compile-file "/afs/cs.cmu.edu/project/cmt/catalyst/amt-ana/code/interpreter/5.0Analyzer/ic_vers_ctrl.lisp"))


;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
; Added 03-Dec-96 by garof.
;------------------------------------------------------------------------------
(defconstant *IRULES-VS* '5.0Analyzer)


;;;==================================================================;

;;; Miscellaneous

(defun validate-irule (keys-values valid-keys rule-type)
  "Checks all keys in KEYS-VALUES against VALID-KEYS for RULE-TYPE; checks that
last key has value.  Issues warnings when errors found; returns T iff all
valid."
  (do ((keys keys-values (cddr keys))
       (valid t))
      ((endp keys) valid)
    (unless (member (first keys) valid-keys :test #'eq)
      (interpreter-warn "~A: bad key ~S in rule ~S"
			rule-type (first keys) keys-values)
      (setf valid nil))
    (unless (rest keys)
      (interpreter-warn "~A: no value for key ~S in rule ~S"
			rule-type (first keys) keys-values)
      (setf valid nil))))


;;;==================================================================;

;;; INTERPRETER-CAT-HIER
;;;
;;; The *INTERPRETER-CAT-HIER* determines rule inheritance.  It is
;;; called the "CAT" hierarchy because it is based on the fillers for
;;; the f-structure feature CAT which gives the part of speech of that
;;; element of the f-structure.  *INTERPETER-CAT-ROOT* determines the
;;; root element of the heirarchy.  The hierarchy is usually built
;;; with a single call to the macro ADD-TO-CAT-HIERARCHY.  The
;;; arguments to that macro lay out the hierarchy in a list structure.
;;; The top-level arguments to that function are made children of
;;; *INTERPRETER-CAT-ROOT*.

;;;;; Readers of *INTERPRETER-CAT-ROOT*: (ADD-TO-CAT-HIERARCHY*
;;;;; GET-HEAD-FROM-FS-AUX GET-PARENT-CAT GETMAPINFO
;;;;; PRINT-REVERSE-CAT-HIERARCHY SEMANTIC-MAP-NO-OR-OR-MULT)

(defvar *interpreter-cat-root* nil)
(defvar *interpreter-cat-default* nil)

;;;;; Readers of *INTERPRETER-CAT-HIER*: (ADD-ONE-CAT ADD-TO-CAT-HIERARCHY*
;;;;; BUILD-REVERSE-CAT-HIERARCHY GET-PARENT-CAT)

(defvar *interpreter-cat-hier* (make-hash-table :size 150 :test #'eq)
  "Interpreter category hierarchy")

(defun clear-imcats ()
  "Clear the interpreter category hierarchy."
  (clrhash *interpreter-cat-hier*))

(defun define-imcat (cat parent)
  "Define new interpreter mapping category CAT(s) with PARENT(s)."
  (setf parent (cond ((null parent)
		      (list *interpreter-cat-root*))
		     ((atom parent)
		      (list parent))
		     (t
		      parent)))
  (if (listp cat)
      (dolist (cat cat)
	(setf (gethash cat *interpreter-cat-hier*) parent))
    (setf (gethash cat *interpreter-cat-hier*) parent)))

(defmacro defimcat (&key cat parent)
  `(define-imcat ',cat ',parent))


(defmacro defimcat-root (cat)
  `(progn
     (setf (gethash ',cat *interpreter-cat-hier*) nil)
     (setf *interpreter-cat-root* ',cat)))

(defmacro defimcat-default (cat)
  `(setf *interpreter-cat-default* ',cat))


;;;==================================================================;

;;; Interpreter category inheritance 

(defun imcat-parents (cat)
  "Retrieves the parent cats for a given CAT in the *INTERPRETER-CAT-HIER*."
  (or (gethash cat *interpreter-cat-hier*)
      (and cat
	   (not (eq cat *interpreter-cat-root*))
	   (interpreter-warn "non-root CAT has no parents: ~s" cat))))


(defun some-imcat-inherit-aux (cat)
  (declare (special function))
  (or (funcall function cat)
      (some #'some-imcat-inherit-aux (imcat-parents cat))))

(defun some-imcat-inherit (cat function)
  "Maps FUNCTION over the categories in the *INTERPRETER-CAT-HIER* in
DFS order, starting with CAT and stopping when FUNCTION returns
non-nil.  FUNCTION takes one argument: the current category.  Returns
the result of FUNCTION."
  (declare (special function))
  (some-imcat-inherit-aux cat))


;;;==================================================================;

;;; FEATURES-TO-SKIP
;;;
;;; Defines the list of f-structure features to be skipped by the
;;; interpreter.

;;;;; Readers of *FEATURES-TO-SKIP*: (INTERPRETER-MAP-RULE)

(defvar *features-to-skip* '())

(defmacro features-to-skip (&rest features)
  `(progn
    (unless *features-to-skip*
      (setq *features-to-skip* (make-hash-table :size (length ',features) :test #'eq)))
    (dolist (feature ',features)
      (setf (gethash feature *features-to-skip*) t))))


;;;------------------------------------------------------------------;

;;; Function DEFINE-HEAD-RULE
;;;
;;; Define the list of head rules for a given "CAT".  Head rules are
;;; not additive in that you cannot define a set of head rules for a
;;; CAT and then add more rules to the head rules for that CAT.  The
;;; new set would replace the old set of rules.
;;;
;;; Head rules are stored in the *INTERPETER-HEAD-RULE-HASH* keyed on
;;; CAT.
;;;
;;; A head rule may have a constraint that is a lisp form or a partial
;;; f-structure that is unified with an input f-structure using
;;; STRICT-UNIFY (if STRICT-UNIFY suceeds for the constraint then the
;;; head rule lisp form is evaluated).
;;;
;;; NOTE: The order of the head rules is preserved so that they are
;;; applied in the order that they are listed in the rule file.

(defstruct head-rule
  sem-head
  sem-head-is-func
  syn-constraint
  syn-constraint-is-func
  comment)

;;;;; Readers of *INTERPRETER-HEAD-RULE-HASH*: (DEFINE-HEAD-RULE INTERPRETER-GET-SEM-HEAD)

(defvar *interpreter-head-rule-hash* (make-hash-table :size 5))

(defun define-head-rule (cat rules)
  (setf (gethash cat *interpreter-head-rule-hash*)
	(mapcar #'(lambda (rule)
		    (validate-irule rule '(:SYN-CONSTRAINT :SEM-HEAD :COMMENT) "Head")
		    (let ((sem-head (second (member :sem-head rule)))
			  (syn-constraint (second (member :syn-constraint rule)))
			  (comment (second (member :comment rule))))
		      (if sem-head
			  (make-head-rule
			   :sem-head sem-head
			   :sem-head-is-func (listp sem-head)
			   :syn-constraint (or syn-constraint t)
			   :syn-constraint-is-func (or (null syn-constraint)
						       (not (any-fs-p
							     syn-constraint)))
			   :comment comment)
			  (interpreter-warn ":SEM-HEAD: not given for cat (~a)~
                                             rule ~s"
					    cat
					    rule))))
		rules)))

;;;------------------------------------------------------------------;

;;; DEFINE-FEATURE-SLOT-RULE
;;;
;;; Define a feature-slot rule.  Since features and slots look about
;;; the same, the rules for them are put into a single hash table.
;;; They are keyed by both the CAT and f-structure feature (or slot)
;;; that they serve.  feature-slot rules are additive, so even when
;;; some feature-slot rules have been defined for a CAT, additional
;;; rules can be defined for other features or overwrite rules for
;;; features already defined; but, defining a new feature-slot rule
;;; for a CAT will not remove all previous feature-slot rules for that
;;; cat. This function is also used to define multiple-feature-slot
;;; rules: these rules simply list multiple features that are mapped
;;; by a single rule. (See the function INTERPRETER-MAP-CONSUMES in
;;; interpreter.lisp.)
;;;
;;; (See ../doc/rule-syntax.txt)

;;;;; Readers of *FEATURE-SLOT-RULE-HASH*: (ADD-FEATURE-SLOT-RULE-AUX GETMAPINFO)

(defvar *feature-slot-rule-hash* (make-hash-table :size 100 :test #'equal))

(defstruct fs-rule
  mapforms
  consumes)

(defun mapform-feature (mapform)
  "Returns the semantic feature from MAPFORM."
  (first mapform))

(defun mapform-form (mapform)
  "Returns the Lisp form from MAPFORM."
  (second mapform))


(defun validate-mapforms (cat feat-or-slot mapforms)
  "Check all the MAPFORMS to be sure each contains either a semantic
feature or a Lisp form.  Delete offending mapforms, issuing warning.
Returns validated mapforms."
  (let ((new-mapforms nil))
    (dolist (mapform mapforms)
      (if (or (mapform-feature mapform)
	      (mapform-form mapform))
	  (push mapform new-mapforms)
	(interpreter-warn "Bad mapform in rule for feature or slot ~S under cat ~S"
			  feat-or-slot cat)))
    (nreverse new-mapforms)))


(defun add-feature-slot-rule-aux (cat feat-or-slot mapforms &optional consumes)
  (let* ((key (cons cat feat-or-slot))
	 (oldrule (gethash key *feature-slot-rule-hash*))
	 (oldmapforms (when oldrule
			(fs-rule-mapforms oldrule)))
	 (oldconsumes (when oldrule
			(fs-rule-consumes oldrule))))
    (setf mapforms (validate-mapforms cat feat-or-slot mapforms))
    (when oldrule
      (unless (equal oldmapforms mapforms)
	(interpreter-warn "RULE RE-DEFINITION: ~s~%  ~
                             Old Rule: ~s~%  ~
                             New Rule: ~s"
			  key
			  oldmapforms
			  mapforms))
      (unless (equal oldconsumes consumes)
	(interpreter-warn "RULE CONSUMES RE-DEFINITION: ~s~%  ~
                             Old Rule: ~s~%  ~
                             New Rule: ~s"
			  key
			  oldmapforms
			  mapforms)))
    (setf (gethash key *feature-slot-rule-hash*)
	  (make-fs-rule :mapforms mapforms
			:consumes consumes))))


(defun add-multiple-feature-slot-rule (cat feat-slot-list mapforms)
  (dolist (feat-or-slot feat-slot-list)
    (add-feature-slot-rule-aux cat
			       feat-or-slot
			       mapforms
			       feat-slot-list)))


(defun add-feature-slot-rule (cat feat-or-slot mapforms)
  (add-feature-slot-rule-aux cat
			     feat-or-slot
			     mapforms))


(defun parse-feat-slot-rule (rule)
  (let ((sem-slot (second (member :sem-slot rule)))
	(slot-value (second (member :slot-value rule))))
    (list sem-slot slot-value)))


(defun define-feature-slot-rule (cat rule)
  (validate-irule rule '(:SYN-SLOT :SEM-SLOT :SLOT-VALUE) "Slot")
  (add-feature-slot-rule cat
			 (second (member :syn-slot rule))
			 (if (list-of-lists-p rule)
			     (mapcar #'parse-feat-slot-rule rule)
			     (list (parse-feat-slot-rule rule)))))


;;;------------------------------------------------------------------;
;;; Function DEFINE-FEATURE-SLOT-RULES
;;; 

(defun define-feature-slot-rules (cat rules)
  (dolist (rule rules)
    (define-feature-slot-rule cat rule)))


;;;------------------------------------------------------------------;

;;; Function DEFINE-MULTIPLE-FEATURE-SLOT-RULES
;;; 

(defun define-multiple-feature-slot-rule (cat rule)
  (validate-irule rule '(:SYN-SLOTS :EVAL-FORMS) "Multiple Slot")
  (let* ((feat-slot-list (second (member :syn-slots rule)))
	 (mapforms (second (member :eval-forms rule))))
    (if (and feat-slot-list
	     mapforms
	     (listp feat-slot-list))
	(add-multiple-feature-slot-rule cat
					feat-slot-list
					(if (list-of-lists-p mapforms)
					    mapforms
					    (list mapforms)))
	(interpreter-warn "MULTIPLE-FEATURE-SLOT-RULE has bad format! ~%  ~
                           cat: ~s~%  syn-slots: ~s~%  eval-forms: ~s"
			  cat
			  feat-slot-list
			  mapforms))))
  
(defun define-multiple-feature-slot-rules (cat rules)
  (if (list-of-lists-p rules)
      (dolist (rule rules)
	(define-multiple-feature-slot-rule cat rule))
      (interpreter-warn "MULTIPLE-FEATURE-SLOT-RULE has bad format! ~%  ~
                         cat: ~s~%  rules: ~s"
			cat
			rules)))
  
;;;------------------------------------------------------------------;

;;;;; Callers of GETMAPINFO: (GETMAPINFO INTERPRETER-MAP-NON-SEMSLOT)

(defun getmapinfo (cat slot)
  "Get the slot mapping rule for CAT and SLOT.  Searches the category
hierarchy until it finds a parent category which includes a rule for SLOT."
  (some-imcat-inherit cat
		      #'(lambda (the-cat)
			  (gethash (cons the-cat slot) *feature-slot-rule-hash*))))


;;;------------------------------------------------------------------;

;;; Function DEFINE-PRE-MAPPING-RULES
;;; 
;;; Defines pre-mapping rules for f-structures with a given CAT.
;;;
;;; Pre-mapping rules are the first rules applied to an f-structure.
;;; After pre-mapping, then the head-rules are applied to an
;;; f-structure.  These pre-mapping rules are not additive.

;;;;; Readers of *PRE-MAP-RULE-HASH*: (ADD-PRE-MAP-RULES APPLY-PRE-MAP-RULES)

(defvar *pre-map-rule-hash* (make-hash-table :size 20 :test #'equal))

(defstruct pre-map-rule
  map-func
  constraint
  constraint-is-func
  comment)

(defun add-pre-map-rules (cat rule-list)
  (let ((old-rule-list (gethash cat *pre-map-rule-hash*)))
    (when old-rule-list
      (interpreter-warn "Redefining PRE-MAP-RULES (~s): ~%~
                         OLD: ~s~%~
                         NEW: ~s"
			cat
			old-rule-list
			rule-list))
    (setf (gethash cat *pre-map-rule-hash*) rule-list)))

(defun define-pre-mapping-rules (cat pre-map-rules)
  (let (pre-map-func
	syn-constraint
	comment
	rule-list) 
    (dolist (pre-map-rule pre-map-rules)
      (validate-irule pre-map-rule '(:SYN-CONSTRAINT :PRE-MAP-FUNC :COMMENT) "Pre-Map")
      (setf pre-map-func (second (member :pre-map-func pre-map-rule))
	    syn-constraint (second (member :syn-constraint pre-map-rule))
	    comment (second (member :comment pre-map-rule)))
      (if pre-map-func
	  (push (make-pre-map-rule :map-func pre-map-func
				   :constraint (or syn-constraint t)
				   :constraint-is-func
				   (or (null syn-constraint)
				       (not (any-fs-p syn-constraint)))
				   :comment comment)
		rule-list)
	(interpreter-warn "PRE-MAP-RULE missing :PRE-MAP-FUNC (~s): ~s"
			  cat
			  pre-map-rule)))
    (add-pre-map-rules cat (nreverse rule-list))))
			    
				 


;;;------------------------------------------------------------------;

;;; Function DEFINE-POST-MAPPING-RULES
;;;
;;; Defines post-mapping rules for f-structuers with a given CAT.
;;; These rules are not additive.

;;;;; Readers of *POST-MAP-RULE-HASH*: (ADD-POST-MAP-RULES APPLY-POST-MAP-RULES)

(defvar *post-map-rule-hash* (make-hash-table :size 20 :test #'equal))

(defstruct post-map-rule
  map-func
  constraint
  comment)

(defun add-post-map-rules (cat rule-list)
  (let ((old-rule-list (gethash cat *post-map-rule-hash*)))
    (when old-rule-list
      (interpreter-warn "Redefining POST-MAP-RULES (~s): ~%~
                         OLD: ~s~%~
                         NEW: ~s"
			cat
			old-rule-list
			rule-list))
    (setf (gethash cat *post-map-rule-hash*) rule-list)))

(defun define-post-mapping-rules (cat post-map-rules)
  (let (post-map-func
	sem-constraint
	comment
	rule-list)
    (dolist (post-map-rule post-map-rules)
      (validate-irule post-map-rule '(:SEM-CONSTRAINT :POST-MAP-FUNC :COMMENT) "Pre-Map")
      (setf post-map-func (second (member :post-map-func post-map-rule))
	    sem-constraint (second (member :sem-constraint post-map-rule))
	    comment (second (member :comment post-map-rule)))
      (when (and sem-constraint (symbolp sem-constraint))
	(setq sem-constraint `(assoc ,sem-constraint (rest $sem))))
      (if post-map-func
	  (push (make-post-map-rule :map-func post-map-func
				    :constraint (or sem-constraint t)
				    :comment comment)
		rule-list)
	  (interpreter-warn "POST-MAP-RULE missing :POST-MAP-FUNC (~s): ~s"
			    cat
			    post-map-rule)))
    (add-post-map-rules cat (nreverse rule-list))))
			    
				 


;;;------------------------------------------------------------------;

;;; Function DEFINE-CAT-RULES
;;;
;;; This is the macro to use when writing interpreter rules.
;;;
;;; NOTE: :FEATURE-SLOT-RULES and :MULTIPLE-FEATURE-SLOT-RULES are
;;; additive; so, you may have multiple INTERPRETER-CAT-RULES listed
;;; for the same :CAT in your rule files. However, only one of those
;;; INTERPRETER-CAT-RULES calls should have :PRE-MAP, :HEAD-RULES, and
;;; :POST-MAP rules defined; otherwise, some of those rules will be
;;; overwritten but subsequent calls to INTERPRETER-CAT-RULES with the
;;; same :CAT.

(defmacro interpreter-cat-rules
    (&key
     cat
     pre-map
     head-rules
     feature-slot-rules
     multiple-feature-slot-rules
     post-map)
  `(let ((good-cat (or ',cat *interpreter-cat-root*)))
    (when ',pre-map
      (define-pre-mapping-rules good-cat ',pre-map))
    (when ',head-rules
      (define-head-rule good-cat ',head-rules))
    (when ',feature-slot-rules
      (define-feature-slot-rules good-cat ',feature-slot-rules))
    (when ',multiple-feature-slot-rules
      (define-multiple-feature-slot-rules
	  good-cat
	  ',multiple-feature-slot-rules))
    (when ',post-map
      (define-post-mapping-rules good-cat ',post-map))))


;;;==================================================================;

;;; Function ADD-INTERPRETER-CODA-FUNCTION
;;;
;;; Interpreter coda functions are functions that are applied to the
;;; resulting IRs from semantic mapping after all semantic mapping has
;;; been completed.  

;;;;; Readers of *INTERPRETER-CODA-HOOKS*: (ADD-INTERPRETER-CODA-FUNCTION APPLY-INTERPRETER-CODA-HOOKS)

(defvar *interpreter-coda-hooks* nil)

(defun clear-interpreter-coda-hooks ()
  (setf *interpreter-coda-hooks* nil))

(defun add-interpreter-coda-hook (func)
  (setf *interpreter-coda-hooks*
	(nconc *interpreter-coda-hooks* (list func))))

(defun apply-interpreter-coda-hooks (ir)
  (dolist (hook *interpreter-coda-hooks* ir)
    (setf ir (funcall hook ir))))

(defun interpreter-coda (ir-list)
  (delete nil (mapcar #'apply-interpreter-coda-hooks ir-list)))

