;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  verb-rules.lisp
;	By:  Willy Walker	<wkw@pitt.edu>
;     Path:  ../cmt/catalyst/amt-ana/english/interpreter-rules/<release-vs>
;  Started:  30 September 1993
; Modified:  03 December 1996	by <garof>
;
; Comments:  Defines functions for handling verb argument mapping and for
;	     building argument mapping structures.
;
; Modified:  21 June 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "Interpreter" maintainer until then.
;
; Modified:  19 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To garof-ify this file, to add version control markers to it, and
;	     to comment "pre-map-be-verb".
;	     WARNING:  There are several functions of this file which need to
;	     be modularized and simplified.  Will take a bit of effort over
;	     time.
;
; Modified:  03 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  Removed references to "ir_vers_ctrl.lisp".
;
; Modified:  <date>
;	By:  <name>		<full e-mail>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;		     Copyright (c) 1993, 1994, 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)


;..............................................................................
;			  Attempts at Version Control
;..............................................................................

;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
;------------------------------------------------------------------------------
(defconstant *VERB-RULES-VS* '5.0Analyzer)



;..............................................................................
;				   Let It BE
;..............................................................................


;------------------------------------------------------------------------------
;			      Predicate Be-Head-P
;
; Returns non-NIL if a given semantic head is derived from the "be" verb.
;
; Callers of Be-Head-P: (Post-Map-Any-Verb Verb-Slot-Map-And-Gap)
;------------------------------------------------------------------------------
(defun Be-Head-P (head)
  "Returns non-NIL if a given semantic head is derived from the \"be\" verb.
Callers of Be-Head-P: (Post-Map-Any-Verb Verb-Slot-Map-And-Gap)"
  (member head
	  '(*A-BE
	    *A-BE-PREDICATE
	    *A-BE-EQUIVALENCE
	    *A-BE-EXISTENCE)))



;..............................................................................
;			  Verb Argument Class Mapping
;..............................................................................

;------------------------------------------------------------------------------
; Either :DMK or :HASH-TABLE, tells where to look first for verb argument
; mapping information.
;
; 19-Nov-96-garof:
; *Verb-Arg-Class-Hash* is the table which is consulted.
;------------------------------------------------------------------------------
(defvar *Argclass-Priority* :DMK
  "Either :DMK or :HASH-TABLE, tells where to look first for verb argument
mapping information.  *Verb-Arg-Class-Hash* is the table which is consulted.")


;------------------------------------------------------------------------------
; 19-Nov-96-garof:
; The table which maintains verb class argument mapping information.
;------------------------------------------------------------------------------
(defvar *Verb-Arg-Class-Hash* (make-hash-table :size 100 :test #'equal)
  "The table which maintains verb class argument mapping information.")


;------------------------------------------------------------------------------
; Callers of Verbs-And-Roles: NIL
;------------------------------------------------------------------------------
(defmacro Verbs-And-Roles (&rest verbs-and-roles-list)
  "Callers of Verbs-And-Roles: NIL"
  `(dolist (verb-and-role ',verbs-and-roles-list)
	   (setf (gethash (first verb-and-role) *verb-arg-class-hash*)
		 (second verb-and-role))))


;------------------------------------------------------------------------------
; Callers of Get-DMK-Verb-Arg-Class: (Get-Verb-Arg-Class)
;------------------------------------------------------------------------------
(defun Get-DMK-Verb-Arg-Class (head root)
  "Callers of Get-DMK-Verb-Arg-Class: (Get-Verb-Arg-Class)"
  (let ((dmk-entries (find-all-dmk-entries root :pos 'V :concept head)))
    (and dmk-entries
	 (dmk-class (first dmk-entries)))))


;------------------------------------------------------------------------------
; Callers of Get-Hash-Table-Verb-Arg-Class: (Get-Verb-Arg-Class)
;------------------------------------------------------------------------------
(defun Get-Hash-Table-Verb-Arg-Class (root)
  "Callers of Get-Hash-Table-Verb-Arg-Class: (Get-Verb-Arg-Class)"
  (gethash root *Verb-Arg-Class-Hash*))


;------------------------------------------------------------------------------
;			      Get-Verb-Arg-Class
;
; Get a verb's argument class from the DMK.
; Callers of Get-Verb-Arg-Class: (Get-Verb-Arg-Mappings)
;------------------------------------------------------------------------------
(defun Get-Verb-Arg-Class (head root)
  "Get a verb's argument class from the DMK. Callers of Get-Verb-Arg-Class:
\(Get-Verb-Arg-Mappings\)"
  
  (case *Argclass-Priority*
    (:DMK	 (or (Get-DMK-Verb-Arg-Class head root)
		     (Get-Hash-Table-Verb-Arg-Class root)))
    (:HASH-TABLE (or (Get-Hash-Table-Verb-Arg-Class root)
		     (Get-DMK-Verb-Arg-Class head root)))
    (t
     (interpreter-warn "Improper value for *ARGCLASS-PRIORITY*: ~s"
		       *Argclass-Priority*)
     nil)))



;..............................................................................
;			 Verb Argument Class MappingS
;
; Get the mappings of F-Structure verb slots (SUBJ, OBJ, XCOMP, etc.) to IR
; verb slots (AGENT, PATIENT, COMPLEMENT, etc.) from the hierarchy built in the
; file "verb-argclasses.lisp".
;..............................................................................

(defstruct argclass
  (name nil :type symbol)		;Argclass name
  (type nil :type symbol)		;:CLASS, :MULTICLASS, :HIERCLASS
  (map nil :type list)			;List of (SOURCE . TARGET) pairs
  (features nil :type list)		;Features to add to FS
  (inherit nil :type list)		;List of class symbols/constraint lists
  )


;------------------------------------------------------------------------------
(defvar *Argclass-Hierarchy* (make-hash-table :size 100 :test #'eq)
  "Verb argument mapping class hierarchy")


;------------------------------------------------------------------------------
; Defines a new verb argument mapping class and adds it to the hierarchy.
;------------------------------------------------------------------------------
(defun Define-Argclass (type name map features inherit)
  "Defines a new verb argument mapping class and adds it to the hierarchy."
  (setf (gethash name *Argclass-Hierarchy*)
	(make-argclass :name name
		       :type type
		       :map map
		       :features features
		       :inherit inherit)))


;------------------------------------------------------------------------------
; Defines a new verb argument mapping class and adds it to the hierarchy.
;------------------------------------------------------------------------------
(defmacro defargclass (type name &key (map nil) (features nil) (inherit nil))
  "Defines a new verb argument mapping class and adds it to the hierarchy."
  `(define-argclass ',type ',name ',map ',features ',inherit)) 


;------------------------------------------------------------------------------
; Clears the verb argument mapping class hierarchy and slotname list.
;------------------------------------------------------------------------------
(defun Clear-Argclasses ()
  "Clears the verb argument mapping class hierarchy and slotname list."
  (clrhash *Argclass-Hierarchy*)) 



;..............................................................................
;			       Class Inheritance
;..............................................................................


;------------------------------------------------------------------------------
; Returns the FS constraint from a CONstrained inHERIT record.
;------------------------------------------------------------------------------
(defmacro Conherit-Constraint (conherit)
  "Returns the FS constraint from a CONstrained inHERIT record."
  `(second ,conherit))


;------------------------------------------------------------------------------
; Returns the parents list from a CONstrained inHERIT record.
;------------------------------------------------------------------------------
(defmacro Conherit-Parents (conherit)
  "Returns the parents list from a CONstrained inHERIT record."
  `(cddr ,conherit))


;------------------------------------------------------------------------------
; Inherits argument mappings for verb argument CLASS appearing in FS.  Returns
; three values: 
; 1. list of mapping pairs,
; 2. list of feature-value pairs, and
; 3. most specific :CLASS symbol.
;
; 19-Nov-96-garof:
; Not much modularity nor readability here.  Break it up!
;------------------------------------------------------------------------------
(defun Verb-Argclass-Inherit (class fs)
  "Inherits argument mappings for verb argument CLASS appearing in FS.
Returns three values:
1. list of mapping pairs,
2. list of feature-value pairs, and
3. most specific :CLASS symbol."

  (let ((argclass (gethash class *argclass-hierarchy*))
	map
	features
	(result-class nil)
	(collect-map nil)
	(collect-features nil))
    (if argclass
	(progn
	  ;; We iterate over the inheritance items
	  (dolist
	   (inherit (Argclass-Inherit argclass))
	   (typecase
	    inherit
	    (symbol    ; If it's a symbol, inherit and continue
	     (multiple-value-setq
	      (map features result-class)
	      (Verb-Argclass-Inherit inherit fs))
	     
	     (setf collect-map (append map collect-map)
		   collect-features (append features collect-features)))
	    ;; If it's a list, it has a test.
	    ;; If the test passes, inherit from the parents and STOP.
	    ;; If the test fails, continue.
	    (list (let ((itest (tree-test (conherit-constraint inherit) fs)))
		    (unless itest 
			    (interpreter-warn "Verb argclass test failed : ~s" inherit))
		    (when itest
			(dolist (parent (conherit-parents inherit))
				(multiple-value-setq
				 (map features result-class)
				 (verb-argclass-inherit parent fs))
				(setf collect-map (append map collect-map)
				      collect-features
				      (append features collect-features)))
			(return))))))

	  (values (append (argclass-map argclass) collect-map)
		  (append (argclass-features argclass) collect-features)
		  (if (eq :CLASS (argclass-type argclass))
		      class
		    result-class)))
      (progn
	;;  5-Feb-97 by EHN -- beef up this warning.
	(warn "No argclass hierarchy entry for ~S, ~s" class (list (assoc 'root fs)(assoc 'sem fs)))
	(values nil nil nil))))) 


;;;------------------------------------------------------------------;

;;; Function GET-VERB-ARG-MAPPINGS
;;; 
;;; Given a verb semantic head and root, find the verb's DMK verb
;;; class and from that VERB-CLASS-INHERIT the mappings from
;;; f-structure syntactic to IR semantic slots.
;;;
;;; Returns three values, an association list of syn-slotname to
;;; sem-slotname mappings, whether the verb has passive alternation
;;; (returns + if so), and the verb argument mapping class to get the
;;; syn-slotname to sem-slotname mappings.

;;;;; Callers of GET-BE-VERB-ARG-MAPPINGS: (POST-MAP-BE-VERB)

;; 11-Feb-97 by EHN -- OBJ->OBJECT, SUBJ->SUBJECT
;; 19-Feb-97 by EHN -- OBJECT->PREDICATE, *A-BE; removed APCOMP (obsolete)

#|
(defun get-be-verb-arg-mappings (head)
  (case head

	(*A-BE 
	 '((SUBJECT . THEME)
	   (OBJECT . PREDICATE)
	   (XCOMP . COMPLEMENT)))

	(*A-BE-PREDICATE
	 '((SUBJECT . THEME)
	   (PREDICATE . PREDICATE)
	   (XCOMP . COMPLEMENT)))

	(*A-BE-EQUIVALENCE
	 '((SUBJECT . THEME)
	   (OBJECT . PREDICATE)))

	(*A-BE-EXISTENCE  
	 '((OBJECT . THEME)))

	(t
	 (interpreter-warn "Unknown BE verb head: ~a" head))))
|#

;; 19-Feb-97 by EHN -- Octav's version.

(defun get-be-verb-arg-mappings (head)
  (case head

        (*A-BE
         '((SUBJECT . THEME)
           (OBJECT . PREDICATE)
           (XCOMP . COMPLEMENT)))

        (*A-BE-PREDICATE
         '((SUBJECT . THEME)
           (PREDICATE . PREDICATE)
           (XCOMP . COMPLEMENT)))

        (*A-BE-EQUIVALENCE
         '((SUBJECT . THEME)
           (OBJECT . PREDICATE)))

        (*A-BE-EXISTENCE
         '((OBJECT . THEME)))

        (t
         (interpreter-warn "Unknown BE verb head: ~a" head))))
    
;;;;; Callers of GET-VERB-ARG-MAPPINGS: (POST-MAP-VERB)

(defun get-verb-arg-mappings (head root fs)
  (multiple-value-bind (map features result-class)
      (verb-argclass-inherit (get-verb-arg-class head root) fs)
    (unless map
      (interpreter-warn "No mappings for ~s ~s" head root))
    ;; I'll nreverse the map pairs list for maximum conformity
    (values (nreverse map) features result-class)))

(defun arg-mapping-targets (arg-mappings sources)
  (remove-duplicates
   (safe-mapcan-1 #'(lambda (arg-mapping)
		      (and (member (car arg-mapping) sources :test #'eq)
			   (list (cdr arg-mapping))))
		  arg-mappings)))

;;;------------------------------------------------------------------;

;;; Gap arguments in subclauses

(defun insert-gap (ir path gap-ir)
  "Inserts GAP-IR as gapped argument in the subclause ir at the end of PATH in
IR.  Does not add PATH.  Obeys the GAP-ROLE slot in the subclause IR.  Does
not replace an existing argument (unless its head is *gapped-argument-head*).
Returns modified IR."
  (let ((first t))
    (failsafe
     (tree-path-set
      ir path
      :fn #'(lambda (ir1 class1 subclass1)
	      (declare (ignore class1 subclass1))
	      (let ((gap-path (rest (assoc 'GAP-ROLE (rest ir1) :test #'eq))))
		(when (and gap-path
			   ;; OK if no argument in GAP-PATH or argument is a gapped dummy
			   (failure-p (tree-path-test
				       ir1 gap-path
				       :fn #'(lambda (ir2 class2 subclass2)
					       (declare (ignore class2 subclass2))
					       (fail-if (eq (first ir2)
							    *gapped-argument-head*)))
				       :walk-exp t)))
		  (unless first
		    ;; Make a copy for multiple insertions after the first
		    (setf gap-ir (copy-tree gap-ir)))
		  (setf ir1 (tree-path-set ir1 gap-path :val gap-ir)
			first nil))
		ir1))
      :add-path nil
      :walk-exp t)
     :fail ir)))


(defun make-gap (ir &optional (remove-roles nil))
  "Makes IR into a gapped argument.  Copies IR, removes roles in REMOVE-ROLES,
adds (GAPPED +).  Returns modified IR."
  (setf ir (copy-tree ir))
  (dolist (role remove-roles)
    (setf ir (tree-path-remove ir `(,role))))
  (tree-path-set ir '(GAPPED) :val '+))


;;;==================================================================;

;;; Function VERB-SLOT-MAP-AND-GAP
;;;
;;; This is the function that changes verb argument syntactic slots to IR
;;; semantic slots.  It also handles putting "gapped" arguments (named by the
;;; global variable *GAPPED-ARGUMENT-IR*) into verb argument slots that are
;;; not filled in the f-structure.  However, the SUBJ slot is not filled for
;;; imperative sentences (although gapping it with the IR representation of
;;; "you" might be good--gapped complement event subjects on imperative
;;; sentences are already filled with the IR for "you".)

(defun hoist-by (ir concept-by)
  "Hoists BY-phrase from IR Q-MODIFIER."
  (let ((ir-by nil))
    (setf ir (failsafe
	      (tree-path-set
	       ir '(Q-MODIFIER)
	       ;; Walk over occupants of Q-MODIFIER slot.
	       :fn #'(lambda (ir1 class1 subclass1)
		       (declare (ignore class1 subclass1))
		       (if (eq concept-by
			       ;; 11-Feb-97 by EHN -- failing on "stripped" PPs for
			       ;; UPMC/CNBC, fixed.
			       ;; (tree-head (tree-filler ir1 'CASE class1 subclass1))
			       (tree-head ir1)
			       )
			   ;; This is a BY-phrase.  Remember its object and
			   ;; delete it from Q-MODIFIER.
			   (progn
			     (setf ir-by (or ir-by ; Keep the first if many
					     (tree-filler ir1 'OBJECT class1 subclass1)))
			     nil)
			 ;; Not a BY-phrase.  Keep it in Q-MODIFIER.
			 ir1))
	       :add-path nil
	       :walk-exp t)
	      :fail ir))
    (values ir ir-by)))


;;;;; Callers of VERB-SLOT-MAP-AND-GAP: (POST-MAP-BE-VERB POST-MAP-VERB)

(defun verb-slot-map-and-gap (fs
			      ir
			      arg-mappings
			      &optional
			      (features nil)
			      (verb-arg-class nil))
  (declare (ignore features))
  (let* ((concept (first ir))
	 (passive (or (tree-test '((PASSIVE +)) fs)
		      (tree-test '((FORM PASSIVE)) fs)))
	 (imperative (tree-test '(NIL (MOOD IMPERATIVE)) ir))
	 syn-key
	 sem-key
	 sem-slot
	 topic-role
	 direct-role-p
	 (missing-required-roles nil)
	 (gapped-ir nil)
	 (path-to-insert-gap nil))

    (if passive
	;; FS is syntactically passive: manipulate the IR
	(progn
	  ;; The SUBJ is what would be the OBJ, topicalized.  We'll be inserting
	  ;; the TOPIC-ROLE feature.
	  
	  ;;  3-Feb-97 by EHN -- OBJ -> OBJECT, SUBJ->SUBJECT
	  (setf topic-role (cdr (assoc 'OBJECT arg-mappings))
		;; Get rid of the gapped OBJ
		ir (tree-path-remove ir '(OBJECT)))
	  ;; Rename SUBJ slot to OBJ for mapping purposes
	  (when (setf sem-slot (tree-slot ir 'SUBJECT))
	    (setf (first sem-slot) 'OBJECT))
	  ;; Hoist the first BY-phrase to SUBJ
	  (let (ir-by)
	    ;; 11-Feb-97 by EHN -- this calling failing on UPMC fs
	    (multiple-value-setq (ir ir-by)
	      (hoist-by ir '*K-BY))
	    (when ir-by
	      (setf ir (tree-path-set ir '(SUBJECT) :val ir-by)))))
      ;; FS is active
      (progn
	(setf topic-role (cdr (assoc 'SUBJECT arg-mappings)))))

    ;; Go through the ARG-MAPPINGS, replacing syntactic verb argument
    ;; slotnames in the IR with semantic slotnames.
    (dolist (arg-mapping arg-mappings)
      (setf syn-key (car arg-mapping)
	    direct-role-p (member syn-key '(SUBJECT OBJECT OBJECT2 INDOBJECT) :test #'eq)
	    sem-key (cdr arg-mapping)
	    sem-slot (tree-slot ir syn-key))
      ;;  3-Feb-97 by EHN -- more tracing.
;;      (format t "~%~% ***** SYN-KEY: ~s SEM-KEY: ~s~%~%" syn-key sem-key)
;;      (format t "~%~% ***** IR: ~s ~%~%" ir)
      (if sem-slot
	  ;; Argument found in IR
	  (progn
	    ;; The gapped role for complements is the last direct verb argument
	    (when direct-role-p
	      (setf gapped-ir (second sem-slot)))
	    (cond ((eq syn-key 'XCOMP)
		   ;; If an XCOMP/COMPLEMENT, need to insert a gapped element in it.
		   (setf path-to-insert-gap `(,sem-key EVENT)))
		  ((and (eq syn-key 'APCOMP)
			(tree-test '(NIL (COMPLEMENT :DEFINED)) (second sem-slot)))
		   ;; If an APCOMP/ATTRIBUTE with a COMPLEMENT, need to insert a
		   ;; gapped element in the COMPLEMENT.
		   (setf path-to-insert-gap `(,sem-key COMPLEMENT EVENT))))
	    ;; Change the slot key to the semantic role
	    (setf (first sem-slot) sem-key))
	;; Argument not found in IR
	(when direct-role-p
	  (if (and imperative (eq syn-key 'SUBJECT))
	      ;; Imperatives don't have subjects, but we need something to gap anyway
	      (setf gapped-ir *you-ir*)
	    ;; This missing argument is required
	    (push sem-key missing-required-roles)))))

    ;; Add some stuff to the IR
    (setf ir (nconc ir
		    ;; For passives, add TOPIC-ROLE
		    (and passive
			 topic-role
			 (list (list 'TOPIC-ROLE topic-role)))
		    ;; Note the gappable (topic) role in GAP-ROLE
		    (and topic-role
			 (list (list 'GAP-ROLE topic-role)))
		    ;; Dummy required arguments
		    (mapcar #'(lambda (sem-key)
				(list sem-key
				      (copy-tree *gapped-argument-ir*)))
			    missing-required-roles)
		    ;; If the verb has an ARGUMENT-CLASS ("be" verbs don't???),
		    ;; put it into the IR.
		    (and verb-arg-class
			 (list (list 'ARGUMENT-CLASS verb-arg-class)))))

    ;; Gap the complement
    (when (and gapped-ir
	       path-to-insert-gap
	       (not (member (tree-head gapped-ir)
			    '(*G-PLEONASTIC-PRONOUN-IT *G-PLEONASTIC-PRONOUN-THERE)
			    :test #'eq)))
      (setf ir (insert-gap ir path-to-insert-gap (make-gap gapped-ir))))

    ;;; Looks like a HACK to me. --willy
    (when (and (not (be-head-p concept))
	       (eq (cdr (assoc 'SUBJECT arg-mappings)) 'THEME)
	       (eq (cdr (assoc 'APCOMP arg-mappings)) 'ATTRIBUTE)
	       (not (null (assoc 'ATTRIBUTE (rest ir)))))
      (setf (rest ir) (cons (copy-tree '(PREDICATE-ROLE ATTRIBUTE))
			    (rest ir))))

    ;;; Last-ditch cleanup of any lingering SUBJ, OBJ, OBJ2
    ;;; (Rather a hack.)
    (let (arg-mapping)
      (dolist (slot-filler (rest ir))
	(when (setf arg-mapping (assoc (first slot-filler)
				       ;; 19-Feb-97 by EHN
				       '((SUBJECT . SUBJECT)
					 (OBJECT . OBJECT)
					 (OBJECT2 . OBJECT2)
					 (INDOBJECT . INDOBJECT)
					 (COMPL . COMPLEMENT)
					 (PREDICATE . PREDICATE))
				       :test #'eq))
	      (cond ((eq (first (second slot-filler)) '*G-GAPPED-ARGUMENT)
		     (cond ((assoc (cdr arg-mapping) (rest ir))
			    (interpreter-warn
			     "Ignoring GAP in ~s" (cdr arg-mapping))
			    (setf (first slot-filler) 'IGNORE))
			   (t (interpreter-warn "Keeping GAP in ~s"
						(cdr arg-mapping))
			      (setf (first slot-filler) 
				    (cdr arg-mapping)))))
		    (t
		      ;; Unmapped syntactic slot
		      (interpreter-warn "Leftover! (~s): Mapping ~s (~s) to ~s"
					(first ir) (car arg-mapping) 
					(first (second slot-filler ))
					(cdr arg-mapping))
		      (setf (first slot-filler) (cdr arg-mapping)))))))
    
    ir))



;;;------------------------------------------------------------------;

;;; POST-MAP-MODAL
;;;
;;; For animate objects (not yet marked in the DM but hard-wired for
;;; *O-OPERATOR and "you"), change (NECESSITY +) to (OBLIGATION
;;; MEDIUM) or change (POSSIBILITY +) to (ABILITY +) when there is no
;;; TENTATIVITY.

;;;;; Callers of ANIMATE-P: NIL

(defun animate-p (agent-filler)
  (and agent-filler
       (let ((head (first agent-filler)))
	 (or (eq head '*O-OPERATOR)
	     (and (eq head '*G-PRONOUN)
		  (eq (second (assoc 'PERSON (rest agent-filler))) 'SECOND))
	     ;; (domo-is-a-p head '&O-ANIMATE-OBJECT)
	     ))))

;;;;; Callers of GET-AGENT: NIL

(defun get-agent (sem)
  (cond ((eq (second (assoc 'MOOD (rest sem)))
	     'IMPERATIVE)
	 *you-ir*)
	(t
	 (second (assoc 'AGENT (rest sem))))))

;;;;; Callers of POST-MAP-MODAL: NIL

(defun post-map-modal (ir)
  (let ((position (position '(necessity +) ir :test #'equal)))
    (cond (position
	   (setf (elt ir position) (copy-tree '(obligation medium))))
	  (t
	   (setq position (position '(possibility +) ir :test #'equal))
	   (when (and position
		      (not (assoc 'tentativity (rest ir))))
	     (setf (elt ir position) (copy-tree '(ability +)))))))
  ir)


;;;------------------------------------------------------------------;

;;; Modal mapping

(defparameter *modal-root-map*
  '(("can" (POSSIBILITY +))
    ("cannot" (NEGATION +) (POSSIBILITY +))
    ("could" (POSSIBILITY +) (TENTATIVITY LOW))
    ("may" (POSSIBILITY +) (TENTATIVITY MEDIUM))
    ("might" (POSSIBILITY +) (TENTATIVITY HIGH))
    ("must" (NECESSITY +))
    ("should" (EXPECTATION +))
    ("shall" (OBLIGATION HIGH))
    ("would" (CONDITIONAL +))    
    ))

;;;;; Callers of MAP-MOOD-FORM-AND-MODAL: NIL

(defun map-mood-form-and-modal (slot-values)
  "Since the IR slots MOOD and FORM and all of the IR slots that relate to
modals, these features are mapped together by a :MULTIPLE-FEATURE-SLOT-RULE to
which the function is attached.  This handles making 'will' indicate (TENSE
FUTURE), inserts the proper features for modals, maps the proper value for
TENSE from FORM, and inserts the correct MOOD."
  (let* ((form (second (assoc 'FORM slot-values)))

	 (modal (second (assoc 'MODAL slot-values)))
	 (modal-root (second (assoc 'ROOT modal)))
	 (modal-will-p (string= modal-root "will"))

	 (tense (or
		 ;;  5-Feb-97 by EHN -- allow f-s tense to promote
		 (second (assoc 'TENSE slot-values))

		 (cond (modal-will-p
			'FUTURE)
		       ((tree-test '(:OR PAST PAST13SG PASTPL PASTPART) form)
			'PAST)
		       (t
			;; 'PRESENT
			;; 31-Mar-97 by EHN - rely on the grammar.
			nil))))

	 (mood-value (or (second (assoc 'MOOD slot-values))
			 'DEC))
	 (mood (case mood-value
		     (IMP  'IMPERATIVE)
		     (DEC  'DECLARATIVE)
		     ;;  3-Feb-97 by EHN -- added for CNBC.
		     (IMPERATIVE 'IMPERATIVE)
		     (DECLARATIVE 'DECLARATIVE)
		     (otherwise
		      (warn "Unknown value for MOOD feature: ~a"
			    mood-value))))

	 (infinitive-p (eq '+ (second (assoc 'INFINITIVE slot-values))))
	 (naked-p (eq '+ (second (assoc 'NAKED slot-values)))))

    (nconc (list (list 'form form)) ;; 24-Feb-97 by EHN
	   (and naked-p
		(copy-tree '((NAKED +))))
	   (and (not (or infinitive-p naked-p))
		;; Only finite clauses get TENSE and MOOD
		(if tense
		    (list (list 'TENSE tense)
			  (list 'MOOD mood))
		  ;; 31-Mar-97 by EHN - don't put in (TENSE NIL)
		  (list (list 'mood mood)))
		  )
	   (and modal-root
		(not modal-will-p)
		(or (copy-tree (rest (assoc modal-root *modal-root-map*
					    :test #'string=)))
		    (interpreter-warn "Modal root not found: ~s" modal-root))))))
    
    
;;;==================================================================;

;;; POST-MAP-VERB

(defun post-map-verb (fs root ir)
  "Get the verb's argument mappings and map them using VERB-SLOT-MAP-AND-GAP."
  (multiple-value-bind (arg-mappings features arg-class)
      (get-verb-arg-mappings (tree-head ir) root fs)
    (verb-slot-map-and-gap fs
			   ir
			   arg-mappings
			   features
			   arg-class)))


;------------------------------------------------------------------------------
; Pre-Map-Be-Verb
;
; 19-Nov-96-garof:
; *A-BE-EXISTENCE <== (obj :defined) && (subj ((root "there"))) in FS
; 	:action - removes FS "subj"
;
; *A-BE-EQUIVALENCE <=== (obj :defined) in FS
;
; *A-BE-PREDICATE && (PREDICATE-ROLE ATTRIBUTE)  <=== (apcomp :defined) ||
;						     (adj-comp :defined) in FS
; *A-BE-PREDICATE && (PREDICATE-ROLE COMPLEMENT) <=== (XCOMP :defined) in FS
; *A-BE-PREDICATE && (PREDICATE-ROLE Q-PREDICATE)<=== (predicate-pp :defined)
;------------------------------------------------------------------------------

;; 11-Feb-97 by EHN -- fixed for UPMC/CNBC, OBJ->OBJECT

;; 19-Feb-97 by EHN -- Latest from Violetta, Bob, Octav
#|
As far as *A-BE and family are concerned, Bob and I would both be
happy if *A-BE, *A-BE-PREDICATE, and *A-BE-EQUIVALENCE could all
be merged into one with the same semantic role PREDICATE.  This
would mean that:

VERB               F-structure role            IR role

*A-BE                OBJECT                      PREDICATE
*A-BE-PREDICATE      PREDICATE                   PREDICATE
*A-BE-EQUIVALENCE    OBJECT                      PREDICATE

and

*A-BE-EXISTENCE      OBJECT                      THEME

(Note: in the last entry, it can't be SUBJECT and must be
 OBJECT since the grammar uses SUBJECT for "there".)
|#

#|
(defun pre-map-be-verb (fs)
  (let ((concept nil)
	(predicate-role nil))
    (cond ((tree-test '((OBJECT :DEFINED)) fs)
	   ;; OBJECT present -- choose EXISTENCE or EQUIVALENCE
	   ;; depending on presence of "there" as SUBJECT.
	   (cond ((tree-test '((SUBJECT ((ROOT "there")))) fs)
		  (setf concept '*A-BE-EXISTENCE
			predicate-role 'PREDICATE))
		 (t (setf concept '*A-BE-EQUIVALENCE
			  predicate-role 'PREDICATE))))
	  ((tree-test '((PREDICATE :DEFINED)) fs)
	   ;; PREDICATE (ADJP) present. Choose PREDICATE.
	   (setf concept '*A-BE-PREDICATE
		 predicate-role 'PREDICATE))
	  ((tree-test '((XCOMP :DEFINED)) fs)
	   ;; MAP XCOMP to COMPLEMENT.
	   (setf concept '*A-BE-PREDICATE
		 predicate-role 'COMPLEMENT))
	  ((tree-test '((OBLIQUE :DEFINED)) fs)
	   ;; Not sure this case ever obtains, but if
	   ;; we get BE with no OBJECT and an OBLIQUE,
	   ;; map OBLIQUE -> Q-MODIFIER in BE-PREDICATE.
	   (setf concept '*A-BE-PREDICATE 
		 predicate-role 'Q-MODIFIER))
	  ((tree-test '((PREDICATE-PP :DEFINED)) fs)
	   ;; Don't think this is used in UPMC/CNBC.
	   (setf concept '*A-BE-PREDICATE
		 predicate-role 'Q-PREDICATE))
	  (t (interpreter-warn "Couldn't select concept and predicate role for BE:~%~s"
			       fs)))
    ;; Copy the concept to the SEM slot in the fs, for
    ;; subsequent mapping operations. If BE-EXISTENCE,
    ;; delete SUBJECT.
    (when concept
      (setf fs (tree-path-set fs '(SEM) :val concept))
      (when (eq concept '*A-BE-EXISTENCE)
	(setf fs (tree-path-remove fs '(SUBJECT)))))
    ;; Store the PREDICATE-ROLE in the fs.
    (when predicate-role
      (setf fs (tree-path-set fs '(PREDICATE-ROLE) 
			      :val predicate-role)))
    fs))
|#

;; 19-Feb-97 by EHN -- Octav's version.

(defun pre-map-be-verb (fs)
  (let ((concept nil)
        (predicate-role nil))
    (cond ((tree-test '((OBJECT :DEFINED)) fs)
           ;; OBJECT present -- choose EXISTENCE or EQUIVALENCE
           ;; depending on presence of "there" as SUBJECT.
           (cond ((tree-test '((SUBJECT ((ROOT "there")))) fs)
                  (setf concept '*A-BE-EXISTENCE
                        predicate-role 'THEME))
                 (t (setf concept '*A-BE
                          predicate-role 'PREDICATE))))
          ((tree-test '((PREDICATE :DEFINED)) fs)
           ;; PREDICATE (ADJP) present. Choose PREDICATE.
           (setf concept '*A-BE
                 predicate-role 'PREDICATE))
          ((tree-test '((COMPL :DEFINED)) fs)
           ;; MAP COMPL to COMPLEMENT.
           (setf concept '*A-BE
                 predicate-role 'COMPLEMENT))
          ((tree-test '((OBLIQUE :DEFINED)) fs)
           ;; we get BE with no OBJECT and an OBLIQUE,
           ;; map OBLIQUE -> Q-MODIFIER in BE.
           (setf concept '*A-BE
                 predicate-role 'Q-MODIFIER))
          (t (interpreter-warn "Couldn't select concept and predicate
role for BE:~%~s"
                               fs)))
    ;; Copy the concept to the SEM slot in the fs, for
    ;; subsequent mapping operations. If BE-EXISTENCE,
    ;; delete SUBJECT.
    (when concept
      (setf fs (tree-path-set fs '(SEM) :val concept))
      (when (eq concept '*A-BE-EXISTENCE)
        (setf fs (tree-path-remove fs '(SUBJECT)))))
    ;; Store the PREDICATE-ROLE in the fs.
    (when predicate-role
      (setf fs (tree-path-set fs '(PREDICATE-ROLE)
                              :val predicate-role)))
    fs))


;;;------------------------------------------------------------------;
;;; POST-MAP-BE-VERB

(defun post-map-be-verb (fs ir)
  (verb-slot-map-and-gap fs
			 ir
			 (get-be-verb-arg-mappings (first ir))
			 nil
			 nil))
