;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  interpreter-rules.lisp
;	By:  Willy Walker	<wkw@pitt.edu>
;     Path:  ../cmt/catalyst/amt-ana/english/interpreter-rules/<release-vs>
;  Started:  13 September 1993
; Modified:  03 December 1996	by <garof>
;
; Comments:  Interpreter rules for making interlingua frames from the Catalyst
;	     english grammar.
;
; Modified:  21 June 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "Interpreter" maintainer until then.
;
; Modified:  18 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To garof-ify this file, and to add version control markers to it.
;
; Modified:  03 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  Removed references to "ir_vers_ctrl.lisp".
;
; Modified:  <date>
;	By:  <name>		<full e-mail>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;		     Copyright (c) 1993, 1994, 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)


;..............................................................................
;			  Attempts at Version Control
;..............................................................................

;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
;------------------------------------------------------------------------------
(defconstant *INTERPRETER-RULES-VS* '5.0Analyzer)


;..............................................................................
;			       Interpreter Rules
;..............................................................................

;------------------------------------------------------------------------------
; Runtime concept construction
;------------------------------------------------------------------------------
(defvar *Special-Head-Table* (make-hash-table :size 50 :test #'equal))


;------------------------------------------------------------------------------
; Constructs concept of the form *PREFIX-ROOT.  Replaces spaces in ROOT with
; hyphens.  Caches result in *SPECIAL-HEAD-TABLE* indexed by ROOT and CAT.
;------------------------------------------------------------------------------
(defun Get-Special-Head (root cat &optional (prefix cat))
  "Constructs concept of the form *PREFIX-ROOT.  Replaces spaces in ROOT with
hyphens.  Caches result in *SPECIAL-HEAD-TABLE* indexed by ROOT and CAT."
  (let ((key (cons root cat)))
    (or (gethash key *Special-Head-Table*)
	(setf (gethash key *Special-Head-Table*)
	      (intern (nstring-upcase
		       (format nil "*~A-~A"
			       prefix
			       (if (stringp root)
				   (substitute #\- #\Space root)
				 root))))))))


;------------------------------------------------------------------------------
; Runtime conversion of concept symbol to feature
;------------------------------------------------------------------------------
(defvar *Concept-Feature-Table* (make-hash-table :size 10 :test #'eq))


;------------------------------------------------------------------------------
; Returns the name of CONCEPT -- CONCEPT sans prefix -- as a string copy.
;------------------------------------------------------------------------------
(defun Concept-Name (concept)
  "Returns the name of CONCEPT -- CONCEPT sans prefix -- as a string copy."
  (let* ((name (symbol-name concept))
	 (delim-pos (position #\- name)))
    (cond ((zerop (length name))	; Null CONCEPT name
	   "NOVALUE")
	  ((char/= (schar name 0) #\*)	; CONCEPT is not a concept
	   (copy-seq name))
	  ((and delim-pos
		(< (1+ delim-pos) (length name))) ; CONCEPT has a name
	   (subseq name (1+ delim-pos)))
	  (t				; CONCEPT does not have a name
	   "NOVALUE"))))

;------------------------------------------------------------------------------
(defun Get-Concept-Feature (concept
			    &key (hyphens-to-slashes nil) (keyword nil))
  (or (gethash concept *Concept-Feature-Table*)
      (setf (gethash concept *Concept-Feature-Table*)
	    (let ((feature-string (Concept-Name concept)))
	      (when hyphens-to-slashes
		    (setf feature-string (nsubstitute #\/ #\- feature-string)))
	      (intern feature-string (if keyword
					 :keyword
				       :user))))))


;------------------------------------------------------------------------------
; Semantic head from ROOT and CAT
; Gets the semantic head by looking up ROOT and CAT in the DMK.  Returns NIL 
; if none found.
; 18-Nov-96-garof:
; Should there be a call to "Eng-Idiom-DMK-Entry" in this function, too?
; Probably not.
;------------------------------------------------------------------------------
(defun Get-Semantic-Head (root cat)
  "Gets the semantic head by looking up ROOT and CAT in the DMK.  Returns NIL
if none found."
  (let ((dmk (find cat (or (Eng-DMK-Entry root)
			   (Eng-Phrasal-DMK-Entry root))
		   :key #'dmk-pos)))
    (and dmk (dmk-concept dmk))))


;------------------------------------------------------------------------------
; Prepositional phrases
; Adds CASE feature, looking up PHRASE string if present or ROOT otherwise.
;------------------------------------------------------------------------------
(defun Add-Case (fs ir)
  "Adds CASE feature, looking up PHRASE string if present or ROOT otherwise."
  (let* ((string (or (Tree-Filler fs 'PHRASE)
		     (Tree-Filler fs 'ROOT)))
	 (concept (and string (DoMo-Prep-Concept string))))
    (when concept
	  (setf ir (Tree-Path-Set ir '(CASE) :val (list concept)))))
  ir)


;------------------------------------------------------------------------------
; Gets the first concept available in IR, walking over expressions and
; *G-COORDINATIONs.
;------------------------------------------------------------------------------
(defun Get-First-Concept (ir)
  "Gets the first concept available in IR, walking over expressions and
*G-COORDINATIONs."
  (Tree-Path-Map ir '()
   :fn #'(lambda (ir1 class1 subclass1)
	   (let ((head1 (tree-head ir1 class1 subclass1)))
	     (if (eq head1 '*G-COORDINATION)
		 ;; Walk over coordinated items
		 (return-from Get-First-Concept
			      (Get-First-Concept
			       (Tree-Filler ir1 'CONJUNCTS class1 subclass1)))
	       ;; Found the concept
	       (return-from Get-First-Concept head1))))
   :walk-exp t))


;------------------------------------------------------------------------------
; IR is a single PP IR, so we can go to work on it without further ado.
; 18-Nov-96-garof:
; Not used much, now that we are in the days of Grammar callout,
; "sem-attach-pp".  Gets called only if semrole missing in FS.
;------------------------------------------------------------------------------
(defun Map-PP-AUX (target-concept ir class subclass)
  "IR is a single PP IR, so we can go to work on it without further ado."
  (if (success-p (Tree-Path-Test ir '(SEMSLOT)))
      ;; SEMSLOT present.  PP fully mapped.  Get rid of SEMSLOT.
      (setf ir (Tree-Path-Remove ir '(SEMSLOT)))
    ;; SEMSLOT not present.  PP has default role mapping.  Do full mapping.
    (let ((prep-concept (Tree-Head (Tree-Filler ir 'CASE class subclass)))
	  (filler-concept (Get-First-Concept
			   (Tree-Filler ir 'OBJECT class subclass)))
	  psem)
      
      (unless (DoMo-Get target-concept)
       (interpreter-warn
	"Target concept ~S not in DoMo, using &ALL" target-concept)
       (setf target-concept '&ALL))
      
      (unless (domo-get filler-concept)
	(if filler-concept
	    (interpreter-warn
	     "Filler concept ~S not in DoMo, using &ALL" filler-concept)
	  (interpreter-warn "No filler concept in PP ~S, using &ALL" ir))
	(setf filler-concept '&ALL))
      
      (if (not prep-concept)
	  (interpreter-warn "No case concept in PP ~S" ir)
	(progn
	  (setf psem
		(psem@symbol
		 (DoMo-Search target-concept prep-concept filler-concept)))
	  (if (not psem)
	      (interpreter-warn "DoMo search failed for ~S ~S ~S"
				target-concept prep-concept filler-concept)
	    (progn
	      (setf (first ir) (psem-concept psem)
		    ir (Tree-Path-Set ir '(ROLE) :val (psem-role psem)))))))))
  ir)


;------------------------------------------------------------------------------
; Adjust the PP IR, putting in the appropriate head and ROLE given TARGET-
; CONCEPT and the PP OBJECT.
; 18-Nov-96-garof:
; Not used much, now that we are in the days of Grammar callout,
; "sem-attach-pp".  Gets called only if semrole missing in FS.
;------------------------------------------------------------------------------
(defun Map-PP (target-concept ir)
  "Adjust the PP IR, putting in the appropriate head and ROLE given
TARGET-CONCEPT and the PP OBJECT."
  ;; Walk over any top-level expression
  (failsafe
   (tree-path-set
    ir '()
    :fn #'(lambda (ir1 class1 subclass1)
	    (let ((head1 (tree-head ir1 class1 subclass1)))
	      (if (eq head1 '*G-COORDINATION)
		  ;; Walk over coordinated PP's
		  (Tree-Path-Set
		   ir1 '(CONJUNCTS)
		   :fn #'(lambda (ir2 class2 subclass2)
			   (declare (ignore class2 subclass2))
			   (Map-PP target-concept ir2)))
		(Map-PP-AUX target-concept ir1 class1 subclass1))))
    :walk-exp t)
   :fail ir))



;..............................................................................
;				:MODIFIER Rules
;
; Since Adverbs, adjectives, and quantifiers shared many mapping attributes
; they were collected under this general CAT class of MODIFIERS so that they
; could share semantic mapping rules.
;..............................................................................


;------------------------------------------------------------------------------
;			     Function MAP-ADJ-ADV
;
; For adverb modifiers of adjectives, perform the proper semantic mapping
; depending on wheter the adverb is "more" or "less" (yielding COMPARISON and
; DEGREE features) or a general adverb to fill the MANNER slot.
;------------------------------------------------------------------------------
(defun Map-Adj-Adv (root sem)
  (cond ((equal root "less")
	 (copy-tree '((DEGREE COMPARATIVE)
		      (COMPARISON LESS))))
	((equal root "more")
	 (copy-tree '((DEGREE COMPARATIVE)
		      (COMPARISON MORE))))
	(t (list 'MANNER sem))))


;------------------------------------------------------------------------------
(defun Map-Adj-Adv-Form (fs)
  (let ((value (second (assoc 'FORM fs))))
    (case value
	  (SUPER (copy-tree '((DEGREE SUPERLATIVE)
			      (COMPARISON MORE))))
	  (COMP (copy-tree '((DEGREE COMPARATIVE)
			     (COMPARISON MORE))))
	  ((POS ROOTFORM) (copy-tree '((DEGREE POSITIVE))))
	  (t
	   (interpreter-warn
	    "MAP-ADJ-ADV-FORM: unknown value for FORM feature: ~a" value)
	   nil))))



;..............................................................................
;			    Interpreter-Coda rules
;..............................................................................
(clear-interpreter-coda-hooks)
(add-interpreter-coda-hook
 #'(lambda (ir) (Tree-Delete-Slots ir *Coda-Delete-Slots*)))
(add-interpreter-coda-hook #'bundle-multiples) 



;---eof interpreter-rules.lisp---
