;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  category-rules.lisp
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;     Path:  ../cmt/catalyst/amt-ana/english/interpreter-rules/<release-vs>
;  Started:  14 November 1994
; Modified:  03 December 1996	by <garof>
;
; Comments:  Defines "interpreter" mapping rules associated with categories.
;
; Modified:  22 June 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "Interpreter" maintainer until then.
;
; Modified:  20 April 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  cf. PR cat/3633 "GENERIC + feature missing from f-structure for
;	     COUNT - nouns."
;	     "GENERIC +" is equivalent to "REFERENCE" being absent from the IR,
;	     so I am removing all references to it from rule "INTERPRETER-CAT-
;	     RULES".  Previously, "GENERIC" was removed from "PARENTHETICAL"
;	     features (i.e. "automatic blade control (ABC)").  Now there is no
;	     "GENERIC", but "(REFERENCE NO-REFERENCE)", which may/may not be
;	     what the TLs want or expect.
;     N.B.:  IR for determiners modified:
;  Removed:  GENERIC	(+ -)
;    Added:  REFERENCE	(DEFINITE, INDEFINITE, NO-REFERENCE)
;
; QUESTION:  Should ":multiple-feature-slot-rules" of ":syn-slots" have
;	     "REFERENCE" in the list, now that "GENERIC" has been removed?
;
; Modified:  26 April 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  cf. PR cat/3336 <teststepref> tag not in IR.  Now it should be.
;	     Modification made to "interpreter-cat-rules", below.
;
; Modified:  06 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  cf. PR cat/3926 with conjoined complement subclauses getting
;	     truncated.  Modified ":cat subclause" rule, below.
;
; Modified:  08, 13 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  cf. PR cat/1827 "GRAMMAR: Phrases preceeded by numbers"
;	     Modified ":cat n" rule, below, adding "Calculate-Semantic-Number"
;	     as a post-mapping semantic constraint.
;
; Modified:  27 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  Added mapping for SGML tag "mktver-tag", per PR 4562.
;	     Added mapping from FS "num-adjunct" to IR "measurement-qualifier"
;	     for phrases such as "or less" or "or more", for ":cat fraction",
;	     per PRs 3689 and 4354.
;
; Modified:  03 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  Removed references to "ir_vers_ctrl.lisp".
;
; Modified:  <date>
;	By:  <name>		<full e-mail>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;			   Copyright (c) 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)


;..............................................................................
;			  Attempts at Version Control
;..............................................................................

;------------------------------------------------------------------------------
; 03-Dec-96-garof:  For use by the maintainer's personal program.
;------------------------------------------------------------------------------
(defconstant *CATEGORY-RULES-VS* '5.0Analyzer)



;..............................................................................
;				Category Rules
;..............................................................................


;------------------------------------------------------------------------------
; Category:  super
;
; Here we define almost all simple slot promotions.  (Some more specific
; promotions are defined for the relevant categories.)
;
; 24-Apr-96-garof:
; Per PR cat/3336, I added:
;			(:syn-slot teststepref-tag
;			 :sem-slot teststepref-tag
;			 :slot-value $value)
;
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super
 :head-rules
 ((:syn-constraint ((SEM :DEFINED))
		   :sem-head (tree-filler $fs 'SEM)
		   :comment "SEM")
;; 31-Mar-97 by EHN - flames out because GET-SEMANTIC-HEAD calls
;; the old DMK routines.
;;  (:syn-constraint (stringp $root)
;;		   :sem-head (get-semantic-head $root $orig-cat)
;;		   :comment "ROOT")
  (:syn-constraint ((ROOT "*gap*"))
		   :sem-head (copy-tree *gapped-argument-ir*)
		   :comment "Gap")

  )
 :feature-slot-rules
 (;; Frobbage
  (:syn-slot as-as
	     :slot-value (copy-tree '((DEGREE COMPARATIVE)
				      (COMPARISON EQUAL))))
  (:syn-slot conj
	     :sem-slot conjunction
	     :slot-value $value)
;;	     (get-concept-feature (first $value) :hyphens-to-slashes t))
  (:syn-slot genitive
	     :slot-value (quote nil))
  
  (:syn-slot pp
	     :sem-slot q-modifier
	     :slot-value (map-pp $head $value))

  ;;  6-Jun-97 by EHN -- add for UPMC to handle 'Dr.'
  (:syn-slot title
	     :sem-slot title
	     :slot-value $value)

  ;; 31-Jan-97 by EHN -- add for PPs that are in MODIFIER slot in f-s
  (:syn-slot modifier
	     :sem-slot q-modifier
	     :slot-value $value)

  ;; 20-Feb-97 by EHN -- fix for different types of adjuncts.
  (:syn-slot adjuncts
	     :slot-value (create-adjunct-slot $fs $value))

  ;; 11-Feb-97 by EHN -- add for PPs that are in OBLIQUE, OBLIQUE2 slot in f-s
  (:syn-slot oblique
	     :sem-slot q-modifier
	     :slot-value $value)

  (:syn-slot oblique2
	     ;; 24-Feb-97 by EHN -- stash here, then merge with
	     ;; correct order in the Q-MODIFIER slot using post-map-func.
	     :sem-slot q-modifier-2
	     :slot-value $value)

  ;;  6-Feb-97 by EHN -- add for PPs that are in SENTENTIAL-ADJUNCT slot in f-s.
  ;; 20-Feb-97 by EHN -- These are always in topic position, so add (TOPIC +)
  (:syn-slot sentential-adjunct
	     :slot-value (list 'q-modifier
			       (tree-path-set $value '(TOPIC) :val '+ :do :PUSH :ADD-PATH t)))

  ;; 19-Feb-97 by EHN -- add for QUOTED feature.
  (:syn-slot quoted
	     :sem-slot quoted
	     :slot-value $value)

  ;; 19-Feb-97 by EHN -- add for ABBREV feature.
  (:syn-slot abbrev
	     :sem-slot abbrev
	     :slot-value $value)

  ;; 19-Feb-97 by EHN -- add for ANAPHOR feature.
  (:syn-slot anaphor
	     :sem-slot anaphor
	     :slot-value $value)

  ;; 19-Feb-97 by EHN -- add for GENDER feature.
  (:syn-slot gender
	     :sem-slot gender
	     :slot-value $value)

  ;; 20-Feb-97 by EHN -- add for DISCOURSE feature.
  (:syn-slot discourse
	     :sem-slot discourse
	     :slot-value $value)

  ;; 20-Feb-97 by EHN -- IDENTIFIER
  (:syn-slot identifier
	     :sem-slot number-unit
	     :slot-value $value)

;; 24-Feb-97 by EHN

  (:syn-slot gapped
	     :sem-slot gapped
	     :slot-value $value)

  ;;  3-Mar-97 by EHN

  (:syn-slot position
	     :sem-slot position
	     :slot-value $value)

  ;; 18-Mar-97 by EHN -- for DATE
  (:syn-slot year  :sem-slot year  :slot-value $value)
  (:syn-slot day   :sem-slot day   :slot-value $value)
  (:syn-slot month :sem-slot month :slot-value $value)

  ;; 20-Mar-97 by EHN

  (:syn-slot missing-verb :sem-slot missing-verb :slot-value $value)  

  ;; 25-Mar-97 by EHN

  (:syn-slot apposition     :sem-slot apposition :slot-value $value)
  (:syn-slot level          :sem-slot level      :slot-value $value)
  (:syn-slot purpose-clause :sem-slot purp-qual  :slot-value $value)
  (:syn-slot rate           :sem-slot rate       :slot-value $value)

  ;; 28-Mar-97 by EHN

  (:syn-slot impersonal     :sem-slot impersonal :slot-value $value)
  (:syn-slot elab           :sem-slot elaboration :slot-value $value)
  (:syn-slot percentage     :sem-slot percentage :slot-value $value)

;;------------------------------------------------------------

  (:syn-slot pre-sent-pp
	     :sem-slot q-modifier
	     :slot-value (map-pp $head $value))

  (:syn-slot predicate-pp
	     :sem-slot q-predicate
	     :slot-value (map-pp '&ALL $value))

  ;; 17-Feb-97 by EHN -- for new PREDICATE slot in UPMC/CNBC
  (:syn-slot predicate
	     :sem-slot predicate
	     :slot-value (map-pp '&ALL $value))
  
  ;; Promotions
  (:syn-slot abb
	     :sem-slot abbreviation
	     :slot-value $value)
  (:syn-slot acr
	     :sem-slot acronym
	     :slot-value $value)
  (:syn-slot adj
	     :sem-slot attribute
	     :slot-value $value)
  (:syn-slot adj-comp
	     :sem-slot apcomp
	     :slot-value $value)
  (:syn-slot adj-mod
	     :sem-slot measurement-modifier
	     :slot-value $value)

  ;; 18-Mar-97 by EHN -- split treatment of ADV
  ;; 18-Mar-97 by EHN -- doesn't work, everything is in ADJUNCTS slot
  ;;(:syn-constraint ((time +))
  ;; 		   :syn-slot adjuncts
  ;; 		   :sem-slot time-modifier
  ;;		   :slot-value $value)

  (:syn-slot adv
	     :sem-slot manner
	     :slot-value $value)

  ;; 19-Feb-97 by EHN
  (:syn-slot adverb
	     :sem-slot manner
	     :slot-value $value)
  (:syn-slot all
	     :sem-slot all
	     :slot-value $value)
  (:syn-slot apcomp
	     :sem-slot apcomp
	     :slot-value $value)
  (:syn-slot attribute
	     :sem-slot attribute
	     :slot-value $value)
  (:syn-slot card
	     :sem-slot quantity
	     :slot-value $value)
  (:syn-slot comparand
	     :sem-slot comparison-theme
	     :slot-value $value)
  (:syn-slot comparison
	     :sem-slot comparison
	     :slot-value $value)
  (:syn-slot comparison-theme
	     :sem-slot comparison-theme
	     :slot-value $value)
  (:syn-slot complementizer
	     :sem-slot complementizer-head
	     :slot-value $value)
  (:syn-slot decimal
	     :sem-slot decimal
	     :slot-value $value)
  (:syn-slot degree
	     :sem-slot degree
	     :slot-value $value)
  (:syn-slot emphatic-possessive
	     :sem-slot emphatic-possessive
	     :slot-value $value)
  (:syn-slot event
	     :sem-slot event
	     :slot-value $value)
  (:syn-slot event-frequency
	     :sem-slot event-frequency
	     :slot-value $value)
  (:syn-slot extent
	     :sem-slot extent
	     :slot-value $value)
  (:syn-slot fraction
	     :sem-slot fraction
	     :slot-value $value)
  (:syn-slot genitive-mod
	     :sem-slot possessor
	     :slot-value $value)

  ;; 31-Jan-97 by EHN -- added for POSSESSOR slot in f-s
  (:syn-slot possessor
	     :sem-slot possessor
	     :slot-value $value)

  (:syn-slot implied-reference
	     :sem-slot implied-reference
	     :slot-value $value)
  (:syn-slot int
	     :sem-slot intensifier
	     :slot-value $value)
  (:syn-slot integer
	     :sem-slot integer
	     :slot-value $value)
  (:syn-slot juxtaposition
	     :sem-slot juxtaposition
	     :slot-value $value)
  (:syn-slot label
	     :sem-slot label
	     :slot-value $value)
  (:syn-slot lone-number
	     :sem-slot standalone-phrase
	     :slot-value $value)
  (:syn-slot meas-mod
	     :sem-slot measurement-attribute
	     :slot-value $value)
  (:syn-slot member
	     :sem-slot conjuncts
	     :slot-value $value)
  (:syn-slot naked
	     :sem-slot naked
	     :slot-value $value)
  (:syn-slot negation
	     :sem-slot negation
	     :slot-value $value)
  (:syn-slot nom-np
	     :sem-slot nominal-noun-phrase
	     :slot-value $value)
  (:syn-slot NUMERIC-NP-MOD    ; added 12-Nov-96 by garof
	     :sem-slot NUMBERED-NP-ATTRIBUTE
	     :slot-value $value)
  (:syn-slot nu-nu-mod
	     :sem-slot numbered-number-identifier
	     :slot-value $value)
  (:syn-slot number-value
	     :sem-slot quantity
	     :slot-value $value)
  (:syn-slot numbered-number
	     :sem-slot numbered-number
	     :slot-value $value)
  (:syn-slot numeric-postmod
	     :sem-slot identifier
	     :slot-value $value)
  (:syn-slot obj
	     :sem-slot obj
	     :slot-value $value)

  ;;  3-Feb-97 by EHN -- added OBJECT
  (:syn-slot object
	     :sem-slot object
	     :slot-value $value)
  (:syn-slot obj2
	     :sem-slot obj2
	     :slot-value $value)
  (:syn-slot ord-premod
	     :sem-slot ordinal-attribute
	     :slot-value $value)
  (:syn-slot parenthetical
	     :sem-slot in-parentheses
	     :slot-value $value)
  (:syn-slot parenthetical-number
	     :sem-slot parenthetical
	     :slot-value $value)
  (:syn-slot perfect
	     :sem-slot perfective
	     :slot-value $value)
  (:syn-slot person
	     :sem-slot person
	     :slot-value $value)
  (:syn-slot position-paren-element
	     :sem-slot parenthetical
	     :slot-value $value)
  (:syn-slot predicate-role
	     :sem-slot predicate-role
	     :slot-value $value)
  (:syn-slot prenom-mod
	     :sem-slot prenominal-modifier
	     :slot-value $value)
  (:syn-slot progressive
	     :sem-slot progressive
	     :slot-value $value)
  (:syn-slot punctuated-number
	     :sem-slot punctuated-number
	     :slot-value $value)
  (:syn-slot punctuation
	     :sem-slot punctuation
	     :slot-value $value)
  (:syn-slot qualification
	     :sem-slot qualification
	     :slot-value $value)
  (:syn-slot quant
	     :sem-slot quantifier
	     :slot-value $value)
  (:syn-slot quantifier
	     :sem-slot quantifier
	     :slot-value $value)
  (:syn-slot quantity
	     :sem-slot quantity
	     :slot-value $value)
  (:syn-slot reciprocal
	     :sem-slot reciprocal
	     :slot-value $value)
  (:syn-slot ratio
	     :sem-slot ratio
	     :slot-value $value)
  (:syn-slot adjoined-modifier
	     :sem-slot adjoined-modifier
	     :slot-value $value)
  (:syn-slot SCOMP
	     :sem-slot COMPLEMENT
	     :slot-value $value)
  (:syn-slot sent-adv
	     :sem-slot manner
	     :slot-value $value)
  ;; 19-Feb-97 by EHN
  (:syn-slot sentential-adverb
	     :sem-slot manner
	     :slot-value $value)
  (:syn-slot standalone-phrase
	     :sem-slot standalone-phrase
	     :slot-value $value)
  (:syn-slot subj
	     :sem-slot subj
	     :slot-value $value)
  ;;  3-Feb-97 by EHN -- added SUBJECT
  (:syn-slot subject
	     :sem-slot subject
	     :slot-value $value)
  (:syn-slot substance
	     :sem-slot substance
	     :slot-value $value)
  (:syn-slot token
	     :sem-slot token
	     :slot-value $value)
  (:syn-slot topic
	     :sem-slot topic
	     :slot-value $value)
  (:syn-slot unit
	     :sem-slot unit
	     :slot-value $value)
  (:syn-slot value
	     :sem-slot value
	     :slot-value $value)
  (:syn-slot XCOMP
	     :sem-slot COMPLEMENT
	     :slot-value $value)
  (:syn-slot COMPL
	     :sem-slot COMPLEMENT
	     :slot-value $value)
  
  ;; SGML tag slot rules
  
  (:syn-slot salesmdl-seq
	     :sem-slot salesmdl-tag
	     :slot-value $value)
  
  (:syn-slot begin-tag
	     :sem-slot begin-tag
	     :slot-value $value)
  (:syn-slot blankline-tag
	     :sem-slot blankline-tag
	     :slot-value $value)
  (:syn-slot booktitle-tag
	     :sem-slot booktitle-tag
	     :slot-value $value)
  (:syn-slot callout-tag
	     :sem-slot callout-tag
	     :slot-value $value)
  (:syn-slot circno-tag
	     :sem-slot circno-tag
	     :slot-value $value)
  (:syn-slot code-tag
	     :sem-slot code-tag
	     :slot-value $value)
  (:syn-slot color-tag
	     :sem-slot color-tag
	     :slot-value $value)
  (:syn-slot copyr-tag
	     :sem-slot copyr-tag
	     :slot-value $value)
  (:syn-slot cpn-tag
	     :sem-slot cpn-tag
	     :slot-value $value)
  (:syn-slot cpn-id-tag
	     :sem-slot cpn-id-tag
	     :slot-value $value)
  (:syn-slot cpnmod-tag
	     :sem-slot cpnmod-tag
	     :slot-value $value)
  (:syn-slot date-tag
	     :sem-slot date-tag
	     :slot-value $value)
  (:syn-slot day-tag
	     :sem-slot day-tag
	     :slot-value $value)
  (:syn-slot disa-ietitle-tag
	     :sem-slot disa-ietitle-tag
	     :slot-value $value)
  (:syn-slot emphasis-tag
	     :sem-slot emphasis-tag
	     :slot-value $value)
  (:syn-slot end-tag
	     :sem-slot end-tag
	     :slot-value $value)
  (:syn-slot english-tag
	     :sem-slot english-tag
	     :slot-value $value)
  (:syn-slot figref-tag
	     :sem-slot figref-tag
	     :slot-value $value)
  (:syn-slot formno-tag
	     :sem-slot formno-tag
	     :slot-value $value)
  (:syn-slot ftnref-tag
	     :sem-slot ftnref-tag
	     :slot-value $value)
  (:syn-slot ftr-title-tag
	     :sem-slot ftr-title-tag
	     :slot-value $value)
  (:syn-slot idiom-tag
	     :sem-slot idiom-tag
	     :slot-value $value)
  (:syn-slot ie-topic-tag
	     :sem-slot ie-topic-tag
	     :slot-value $value)
  (:syn-slot ieref-tag
	     :sem-slot ieref-tag
	     :slot-value $value)
  (:syn-slot iesubtitle-tag
	     :sem-slot iesubtitle-tag
	     :slot-value $value)
  (:syn-slot ietitle-tag
	     :sem-slot ietitle-tag
	     :slot-value $value)
  (:syn-slot inref-tag
	     :sem-slot inref-tag
	     :slot-value $value)
  (:syn-slot int-item-tag
	     :sem-slot int-item-tag
	     :slot-value $value)
  (:syn-slot int-item2-tag
	     :sem-slot int-item2-tag
	     :slot-value $value)
  (:syn-slot jobtype-tag
	     :sem-slot jobtype-tag
	     :slot-value $value)
  (:syn-slot label-tag
	     :sem-slot label-tag
	     :slot-value $value)
  (:syn-slot levelref-tag
	     :sem-slot levelref-tag
	     :slot-value $value)
  (:syn-slot mdldesc-tag
	     :sem-slot mdldesc-tag
	     :slot-value $value)
  (:syn-slot mdlgroup-tag
	     :sem-slot mdlgroup-tag
	     :slot-value $value)
  (:syn-slot mdlgroupdesc-tag
	     :sem-slot mdlgroupdesc-tag
	     :slot-value $value)
  (:syn-slot media-tag
	     :sem-slot media-tag
	     :slot-value $value)
  (:syn-slot metric-tag
	     :sem-slot metric-tag
	     :slot-value $value)
  (:syn-slot modifier-tag
	     :sem-slot modifier-tag
	     :slot-value $value)
  (:syn-slot month-tag
	     :sem-slot month-tag
	     :slot-value $value)
  (:syn-slot mktver-tag				; 27-Nov-96-garof: PR 4562
	     :sem-slot mktver-tag
	     :slot-value $value)
  (:syn-slot nomen-tag
	     :sem-slot nomen-tag
	     :slot-value $value)
  (:syn-slot noncatpub-tag
	     :sem-slot noncatpub-tag
	     :slot-value $value)
  (:syn-slot pageno-tag
	     :sem-slot pageno-tag
	     :slot-value $value)
  (:syn-slot partno-tag
	     :sem-slot partno-tag
	     :slot-value $value)
  (:syn-slot pfx-tag
	     :sem-slot pfx-tag
	     :slot-value $value)
  (:syn-slot phoneline-tag
	     :sem-slot phoneline-tag
	     :slot-value $value)
  (:syn-slot phoneno-tag
	     :sem-slot phoneno-tag
	     :slot-value $value)
  (:syn-slot pingroup-tag
	     :sem-slot pingroup-tag
	     :slot-value $value)
  (:syn-slot pinrange-tag
	     :sem-slot pinrange-tag
	     :slot-value $value)
  (:syn-slot pinsnitem-tag
	     :sem-slot pinsnitem-tag
	     :slot-value $value)
  (:syn-slot pinsnlist-tag
	     :sem-slot pinsnlist-tag
	     :slot-value $value)
  (:syn-slot position-tag
	     :sem-slot position-tag
	     :slot-value $value)
  (:syn-slot prod-name-tag
	     :sem-slot prod-name-tag
	     :slot-value $value)
  (:syn-slot proddesc-tag
	     :sem-slot proddesc-tag
	     :slot-value $value)
  (:syn-slot produsage-tag
	     :sem-slot produsage-tag
	     :slot-value $value)
  (:syn-slot propname-tag
	     :sem-slot propname-tag
	     :slot-value $value)
  (:syn-slot ptype-tag
	     :sem-slot ptype-tag
	     :slot-value $value)
  (:syn-slot pubdate-tag
	     :sem-slot pubdate-tag
	     :slot-value $value)
  (:syn-slot pubref-tag
	     :sem-slot pubref-tag
	     :slot-value $value)
  (:syn-slot pubtitle-tag
	     :sem-slot pubtitle-tag
	     :slot-value $value)
  (:syn-slot pubtype-tag
	     :sem-slot pubtype-tag
	     :slot-value $value)
  (:syn-slot qualifier-tag
	     :sem-slot qualifier-tag
	     :slot-value $value)
  (:syn-slot quote-tag
	     :sem-slot quote-tag
	     :slot-value $value)
  (:syn-slot revno-tag
	     :sem-slot revno-tag
	     :slot-value $value)
  (:syn-slot salesmdl-tag
	     :sem-slot salesmdl-tag
	     :slot-value $value)
  (:syn-slot salesmdldesc-tag
	     :sem-slot salesmdldesc-tag
	     :slot-value $value)
  (:syn-slot series-tag
	     :sem-slot series-tag
	     :slot-value $value)
  (:syn-slot size-tag
	     :sem-slot size-tag
	     :slot-value $value)
  (:syn-slot smcscode-tag
	     :sem-slot smcscode-tag
	     :slot-value $value)
  (:syn-slot smcssect-tag
	     :sem-slot smcssect-tag
	     :slot-value $value)
  (:syn-slot sngroup-tag
	     :sem-slot sngroup-tag
	     :slot-value $value)
  (:syn-slot snrange-tag
	     :sem-slot snrange-tag
	     :slot-value $value)
  (:syn-slot specvalue-tag
	     :sem-slot specvalue-tag
	     :slot-value $value)
  (:syn-slot stepref-tag
	     :sem-slot stepref-tag
	     :slot-value $value)
  (:syn-slot subjectdesc-tag
	     :sem-slot subjectdesc-tag
	     :slot-value $value)
  (:syn-slot subproddesc-tag
	     :sem-slot subproddesc-tag
	     :slot-value $value)
  (:syn-slot tblref-tag
	     :sem-slot tblref-tag
	     :slot-value $value)
  (:syn-slot term-tag
	     :sem-slot term-tag
	     :slot-value $value)
  (:syn-slot teststepref-tag
	     :sem-slot teststepref-tag
	     :slot-value $value)
  (:syn-slot title-tag
	     :sem-slot title-tag
	     :slot-value $value)
  (:syn-slot tmantitle-tag
	     :sem-slot tmantitle-tag
	     :slot-value $value)
  (:syn-slot topic-tag
	     :sem-slot topic-tag
	     :slot-value $value)
  (:syn-slot trademark-tag
	     :sem-slot trademark-tag
	     :slot-value $value)
  (:syn-slot unitsgrp-tag
	     :sem-slot unitsgrp-tag
	     :slot-value $value)
  (:syn-slot valtext-tag
	     :sem-slot valtext-tag
	     :slot-value $value)
  (:syn-slot wire-tag
	     :sem-slot wire-tag
	     :slot-value $value)
  (:syn-slot year-tag
	     :sem-slot year-tag
	     :slot-value $value))

 :post-map
 (
  (:sem-constraint (tree-test '(*G-COORDINATION (CONJUNCTION :UNDEFINED)) $ir)
		   :post-map-func (tree-path-set $ir '(CONJUNCTION) :val 'NULL)
		   :comment "CONJUNCTION NULL")
  (:sem-constraint (tree-test '(*G-COORDINATION (TENSE :DEFINED)) $ir)
		   :post-map-func (unify-conjunct-tenses $ir)
		  :comment "Make sure conjuncts have same tense as head")
  (:post-map-func (combine-duplicate-slots $ir :MULTIPLE)
		  :comment "Dup Slots")
  (:post-map-func (merge-q-modifier-slots $ir)
		  :comment "Merge Q-MODIFIER-2 at end of Q-MODIFIER")
;;  (:post-map-func (gap-argument-role $ir)
;;		  :comment "Gap the role in GAP-ROLE")
  ))



;------------------------------------------------------------------------------
; Category:  super-modifier
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super-modifier
 :pre-map
 ((:syn-constraint (*NOT* (*OR* ((ROOT SET-ADJP))
				((CAT QUANT))
				((DEGREE :DEFINED))
				((AS-AS :DEFINED))
				((FORM :DEFINED))))
		   :pre-map-func (tree-path-set $fs '(DEGREE) :val 'POSITIVE)
		   :comment "Positive")

  (:syn-constraint ((FORM :DEFINED)
		    (DEGREE :UNDEFINED))
		   :pre-map-func (append (map-adj-adv-form $fs) $fs)
		   :comment "ADJ/ADV form"))

 :feature-slot-rules
 ((:syn-slot as-possible
	     :slot-value
	     (copy-tree '((COMPARISON-THEME
			   (*P-POSSIBLE
			    (DEGREE POSITIVE))))))))


;------------------------------------------------------------------------------
; Category:  super-nominal 
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super-nominal
 :pre-map
 ((:syn-constraint (*OR* ((DET ((ROOT "such-a"))))
			 ((DET ((ROOT "such")))))
		   :pre-map-func (pre-map-such $fs)
		   :comment "Such")

  (:syn-constraint ((ROOT SET-NP))
		   :pre-map-func (cons (copy-tree '(SEM *G-COORDINATION))
				       (copy-list $fs))
		   :comment "NP set")

  (:syn-constraint (*OR* ((PARTITIVE-QUANT :DEFINED))
			 ((PARTITIVE-PERCENT :DEFINED))
			 ((PARTITIVE-CARD :DEFINED)))
		   :pre-map-func (pre-map-partitive $fs)
		   :comment "Partitive")

  (:syn-constraint ((CARD ((AS-QUANT-AS +) (QUANT :DEFINED))))
		   :pre-map-func (pre-map-as-quant-as-np $fs 'card)
		   :comment "As QUANT as")

  (:syn-constraint (*OR* ((CARD :DEFINED)	  (NUM-ADJUNCT :DEFINED))
			 ((MEAS-MOD :DEFINED)	  (NUM-ADJUNCT :DEFINED))
			 ((NUMBER-VALUE :DEFINED) (NUM-ADJUNCT :DEFINED)))
		   :pre-map-func
		   (let ((num-adjunct (assoc 'num-adjunct $fs))
			 (amount (or (assoc 'meas-mod $fs)
				     (assoc 'card $fs)
				     (assoc 'number-value $fs))))
		     (nconc
		      (remove-features
		       '(num-adjunct card meas-mod number-value)
		       $fs)

		      (list (list (first amount)
				  (cons num-adjunct (second amount))))))
		   :comment "Num ADJ")

  (:syn-constraint ((DET ((ROOT "your"))))
		   :pre-map-func (nconc (delete-slots $fs 'det)
					(copy-tree
					 '((GENITIVE-MOD
					    ((ROOT "you")
					     (NUMBER (*OR* SG PL))
					     (CAT PRON))))))
		   :comment "Your"))

 :feature-slot-rules
 (
  ;; 24-Feb-97 by EHN
  (:syn-slot ord
	     :sem-slot ordinal
	     :slot-value $value)

  (:syn-slot REL-CL
	     :sem-slot rel-qual
	     :slot-value $value)
  ;; 19-Feb-97 by EHN
  (:syn-slot subordinate-clause
	     :sem-slot sub-qual
	     :slot-value $value)
  ;; Not clear why this is necesssary:
  (:syn-slot PAREN-NP
	     :sem-slot PARENTHETICAL
	     :slot-value $value))
 
 :multiple-feature-slot-rules
 ((:syn-slots
   (DET COUNT GENITIVE IMPLIED-REFERENCE LEXICAL-NUMBER NOMINALIZED NUMBER)
   :eval-forms (Map-Noun-Features $slot-sems $fs)))
 )


;------------------------------------------------------------------------------
; Category:  super-number
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super-number
:pre-map
(
  (:syn-constraint ((ROOT SET-NP))
		   :pre-map-func (cons (copy-tree '(SEM *G-COORDINATION))
				       (copy-list $fs))
		   :comment "NP set")
)
 :head-rules
 ((:sem-head (second (assoc 'sem $fs))
	     :comment "SEM"))

 :feature-slot-rules
 ((:syn-slot leading-sign
	     :sem-slot sign
	     :slot-value (get-numeric-operator-keyword $value))))


;------------------------------------------------------------------------------
; Category:  super-number-simple
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super-number-simple
 :pre-map
 ((:pre-map-func (pre-map-number $root $orig-cat $fs)
		 :comment "Number value")

  (:syn-constraint ((as-quant-as +) (quant :DEFINED))
		   :pre-map-func (fix-as-quant-as $fs)
		   :comment "As QUANT as"))

 :feature-slot-rules
 ((:syn-slot num-adjunct
	     :sem-slot measurement-qualifier
	     :slot-value (Get-Concept-Feature (first $value) :keyword t))
  
  (:syn-slot number-form
	     :sem-slot number-form
	     :slot-value $value)
  
  (:syn-slot number-type
	     :sem-slot number-type
	     :slot-value $value)))


;------------------------------------------------------------------------------
; Category:  super-set
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super-set
 :head-rules
 ((:sem-head '*G-COORDINATION
	     :comment "*G-COORDINATION"))

)


;------------------------------------------------------------------------------
; Category:  super-sgml
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super-sgml
 :head-rules
 ((:sem-head (Get-Sgml-Head $root '*S)
	     :syn-constraint (symbolp $root)
	     :comment "SGML ROOT"))

 :post-map
 ;; This should be done more sensibly.
 ((:post-map-func (tree-path-remove $ir '(NUMBER))
		  :comment "Remove NUMBER")
  ))


;------------------------------------------------------------------------------
; Category:  super-special
;
; The semantic head for this category is chosen as follows:
;  - SEM value if present; or
;  - runtime concept constructed from original category and:
;    - PHRASE value if present (idiom ROOT has {IDIOM} markup, PHRASE doesn't);
;	or 
;    - root.
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat super-special
 :head-rules
 ((:sem-head (or (second (assoc 'SEM $fs))
		 (get-special-head (or (second (assoc 'PHRASE $fs)) $root)
				   $orig-cat))
	     :comment "SEM/Special")))


;------------------------------------------------------------------------------
; Category: adj
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat adj
 :pre-map
 ((:syn-constraint ((root SET-ADJP))
		   :pre-map-func (cons (list 'SEM '*G-COORDINATION)
				       (copy-list $fs))
		   :comment "ADJP set"))
 
 :feature-slot-rules
 (
  (:syn-slot adv
	     :slot-value (map-adj-adv $root $value))

;; 19-Feb-97 by EHN
  (:syn-slot adverb
	     :slot-value (map-adj-adv $root $value))

;  (:syn-slot form
;	     :slot-value (quote nil))
))


;------------------------------------------------------------------------------
; Category: adv
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat adv
 :pre-map
 ((:syn-constraint ((root SET-ADV))
		   :pre-map-func (cons (list 'SEM '*G-COORDINATION)
				       (copy-list $fs))
		   :comment "ADV set"))
 :feature-slot-rules
 (
  (:syn-slot form
	     :slot-value (quote nil))))


;------------------------------------------------------------------------------
; Category: comparative
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat comparative
 :head-rules
 ((:sem-head '*G-GENERIC-COMPARATIVE
	     :comment "*G-GENERIC-COMPARATIVE")))


;; ------------------------------------------------------------
;; 21-Feb-97 by EHN -- NUMBER rules

(interpreter-cat-rules
  :cat number
  :head-rules
  ((:syn-constraint ((ROOT NUMBER-RANGE))
		    :sem-head '*O-NUMBER-RANGE
		    :comment "Number ranges"))

  :feature-slot-rules
  ;; 19-Feb-97 by EHN
  (
   (:syn-slot NUMERATOR
	     :sem-slot NUMERATOR
	     :slot-value $value)

  (:syn-slot DENOMINATOR
	     :sem-slot DENOMINATOR
	     :slot-value $value)

   (:syn-slot BASE-VALUE
	     :sem-slot BASE-VALUE
	     :slot-value $value)

  (:syn-slot SECOND-VALUE
	     :sem-slot SECOND-VALUE
	     :slot-value $value)

  (:syn-slot SEPARATOR
	     :sem-slot SEPARATOR
	     :slot-value $value)
))

;------------------------------------------------------------------------------
; Category: fraction
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat fraction
 :head-rules
 ((:sem-head '*C-FRACTION
	     :comment "*C-FRACTION"))

 :feature-slot-rules
 ((:syn-slot NUMER
	     :sem-slot NUMERATOR
	     :slot-value $value)

  (:syn-slot DENOM
	     :sem-slot DENOMINATOR
	     :slot-value $value)



  (:syn-slot num-adjunct		; 27-Nov-96-garof: PRs 3689, 4354
	     :sem-slot measurement-qualifier
	     :slot-value (Get-Concept-Feature (first $value) :keyword t)))
 
 :post-map
 ((:sem-constraint (tree-test '(NIL (SIGN :DEFINED)) $ir)
		   :post-map-func (setf $ir (tree-path-set
					     $ir '(NUMERATOR SIGN)
					     :val (tree-filler $ir 'SIGN))
					$ir (tree-path-remove $ir '(SIGN)))
		   :comment "Move SIGN")
  ))


;------------------------------------------------------------------------------
; Category: mod
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat mod
 :head-rules
 ((:sem-head '*MOD-HEAD
	     :comment "*MOD-HEAD")))


;------------------------------------------------------------------------------
; Category: n
;
; 08-Nov-96-garof:
; Added and then removed:
;      :feature-slot-rules
;      ((:syn-slot numeric-np-mod
;	 :sem-slot numbered-np-attribute
;	 :slot-value $value))
; Per new IR spec. given by PR 1827.  "Numeric-NP-Mod" is handled by the ':CAT
; "SUPER"' rule.
;
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat n
 :head-rules
 ((:syn-constraint ((ROOT DATE))
		   :sem-head '*O-DATE
		   :comment "Dates"))
 :pre-map
 ((:syn-constraint ((SEM *O-NUMBER) (NUMERIC-POSTMOD :DEFINED))
		   :pre-map-func
		   (let ((nu-nu (second (assoc 'numeric-postmod $fs))))
		     (Pre-Map-Number
		      (second (assoc 'root nu-nu))
		      (second (assoc 'cat nu-nu))
		      (append nu-nu
			      (remove-features '(number
						 cat
						 root
						 count
						 implied-reference
						 sem
						 token
						 phrasal-score
						 numeric-postmod)
					       $fs)
			      (copy-tree '((NUMBER-NP +)
					   (NUMBERED-NUMBER +))))))
		   :comment "Numeric postmod"))
 :post-map
 ((:sem-constraint
   (Tree-Test '(NIL (QUALIFICATION :DEFINED)) $ir)
   :post-map-func
   (Insert-Gap $ir '(QUALIFICATION EVENT)
	       (Make-Gap $ir '(QUALIFICATION)))
   :comment "Qualification")

  ;----------------------------------------------------------------------------
  ; 13-Nov-96-garof:
  ; Adding this callout to adjust the NUMBER value based on the value of the
  ; QUANTITY expression.
  ; I.e. Instead of "54 tooth gear" --> (*O-TOOTH (NUMBER SINGULAR) ...)
  ; the IR should contain (*O-TOOTH (NUMBER PLURAL) ...)
  ; Implementation Note: I assume that :sem-constraint must test values
  ; returned by "failure-p" or "success-p", etc.
  ; (tree-path-test *tui* '(number)) --> SINGULAR ...
  ; (tree-path-test *tui* '(foo)) --> :FAIL
  ;----------------------------------------------------------------------------
  (:sem-constraint
   (Success-P (Tree-Path-Test $ir '(QUANTITY)))
   :post-map-func
   (Calculate-Semantic-Number $ir)
   :comment "Overwrite FS morphological number with semantic number")))


;------------------------------------------------------------------------------
; Category: number-range
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat number-range
 :head-rules
 ((:sem-head '*C-NUMBER-RANGE
	     :comment "*C-NUMBER-RANGE"))
 
 :feature-slot-rules
 ((:syn-slot BASE-VALUE
	     :sem-slot FROM
	     :slot-value $value)
  (:syn-slot SEPARATOR
	     :sem-slot SEPARATOR
	     :slot-value (get-numeric-operator-keyword $value))
  (:syn-slot SECOND-VALUE
	     :sem-slot TO
	     :slot-value $value)))


;------------------------------------------------------------------------------
; Category: number-tolerance
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat number-tolerance
 :head-rules
 ((:sem-head '*C-NUMBER-TOLERANCE
	     :comment "*C-NUMBER-TOLERANCE"))
 
 :feature-slot-rules
 ((:syn-slot BASE-VALUE
	     :sem-slot IDEAL-VALUE
	     :slot-value $value)
  (:syn-slot SEPARATOR
	     :sem-slot FIRST-SEPARATOR
	     :slot-value (get-numeric-operator-keyword $value))
  (:syn-slot SECOND-VALUE
	     :sem-slot FIRST-VALUE
	     :slot-value $value)
  (:syn-slot SECOND-SEPARATOR
	     :sem-slot SECOND-SEPARATOR
	     :slot-value (get-numeric-operator-keyword $value))
  (:syn-slot THIRD-VALUE
	     :sem-slot SECOND-VALUE
	     :slot-value $value)))


;------------------------------------------------------------------------------
; Category: prep
;------------------------------------------------------------------------------
(interpreter-cat-rules
 :cat p ;; 31-Jan-97 by EHN -- used to be PREP
 :head-rules					; 13-Nov-96-garof:
 ((:syn-constraint (eq $root 'SET-PP)		; if FS has multiple PPs
		   :sem-head '*G-COORDINATION	; map SET-PP to *G-COORDINATION
		   :comment "*G-COORDINATION")  ; and recurse on rest

  ;; 31-Jan-97 by EHN -- changed to produce simplified *K- concepts
  ;; for PP heads.
  (:syn-constraint ((root :DEFINED))
		   :sem-head
		   (or (tree-filler $fs 'sem)
		       (int-get-prep-K-concept (Tree-Filler $fs 'root)))
		   :comment "*K from root")
  
  (:sem-head '*Q-UNKNOWN
	     :comment "*Q-UNKNOWN"))
 
 :feature-slot-rules
 (;; 31-Jan-97 by EHN -- comment out, don't need
  #|(:syn-slot semslot
	     :slot-value
	     (list (list 'SEMSLOT $value)
		   (list 'ROLE (DoMo-QRole-Role $value))))|#

  (:syn-slot object ;; 31-Jan-97 by EHN -- used to be 'obj'
	     :sem-slot object
	     :slot-value $value))
 
 :post-map
 ;; 31-Jan-97 by EHN -- do we need these? The second messes things
 ;; up when we don't have a semslot, and the first looks unnecessary.
 nil
#|
 ((:sem-constraint (not (eq $root 'SET-PP))		; 13-Nov-96-garof:
		   :post-map-func (Add-Case $fs $ir)	; the prep itself
		   :comment "Case")			; ex. *K-OF

  (:sem-constraint
   (and (stringp $root)
	(failure-p (Tree-Path-Test $ir '(SEMSLOT))))
   
   :post-map-func (Map-PP '&ALL $ir)
   :comment "No SEMSLOT")
  )
|#

)


;------------------------------------------------------------------------------
; Category: pron
;------------------------------------------------------------------------------
(interpreter-cat-rules :cat pron
		       :pre-map
		       ((:syn-constraint
			 (or (equal $root "you")
			     (equal $root "your"))
			 :pre-map-func
			 (tree-path-set $fs '(PERSON) :val 'SECOND)
			 :comment "You/Your")
			(:syn-constraint
			 (equal $root "{idiom} each other {/idiom}")
			 :pre-map-func
			 (setf $fs (tree-path-remove $fs '(NUMBER))
			       $fs (tree-path-set $fs '(RECIPROCAL) :val '+))
			 :comment "Each other"))
		       :head-rules
		       ((:syn-constraint (equal $root "this")
					 :sem-head '*G-SENTENTIAL-PRONOUN
					 :comment "This")
			(:syn-constraint (equal $root "there")
					 :sem-head '*G-PLEONASTIC-PRONOUN-THERE
					 :comment "There")
			(:syn-constraint (equal $root "it")
					 :sem-head '*G-PLEONASTIC-PRONOUN-IT
					 :comment "It")
			(:sem-head '*G-PRONOUN
				   :comment "*G-PRONOUN"))
		       :feature-slot-rules
		       ((:syn-slot number
				   :sem-slot number
				   :slot-value
				   (let ((number (map-number-info $value)))
				     (case number
					   (BOTH
					    (copy-tree '(:OR SINGULAR PLURAL)))
					   (t
					    number))))))


;------------------------------------------------------------------------------
; Category: quant
;------------------------------------------------------------------------------
(interpreter-cat-rules :cat quant
		       :feature-slot-rules
		       ((:syn-slot
			 as-quant-as
			 :slot-value (copy-tree '((DEGREE COMPARATIVE)
						  (COMPARISON EQUAL))))))


;------------------------------------------------------------------------------
; Category: ratio
;------------------------------------------------------------------------------
(interpreter-cat-rules :cat ratio
		       :head-rules
		       ((:sem-head '*C-NUMBER-RATIO
				   :comment "*C-NUMBER-RATIO"))
		       :feature-slot-rules
		       ((:syn-slot FIRST-VALUE
				   :sem-slot FIRST-VALUE
				   :slot-value $value)
			(:syn-slot SECOND-VALUE
				   :sem-slot SECOND-VALUE
				   :slot-value $value)
			(:syn-slot THIRD-VALUE
				   :sem-slot THIRD-VALUE
				   :slot-value $value)))


;------------------------------------------------------------------------------
; Category: subclause
;
; 06-Nov-96-garof:
; The grammar uses SET-<foo> for conjoined structures such as:  set-np, set-pp,
; set-adjp, set-subclause, set-number, set-complement.
;------------------------------------------------------------------------------
(interpreter-cat-rules :cat subclause
		       :head-rules
		       ((:syn-constraint
			 ((ROOT (*OR* PURPOSE RELATIVE
				      SUBORDINATE CONDITIONAL)))
			 :sem-head *G-QUALIFYING-EVENT
			 :comment "*G-QUALIFYING-EVENT")

			; 06-Nov-96-garof re: PR 3926
			(:syn-constraint ((ROOT SET-COMPLEMENT))
					 :sem-head *G-COORDINATION
					 :comment "*G-COORDINATION")
			
			(:syn-constraint ((ROOT COMPLEMENT))
					 :sem-head *G-COMPLEMENT-EVENT
					 :comment "*G-COMPLEMENT-EVENT"))
		       :feature-slot-rules
		       ((:syn-slot connective
				   :sem-slot extent
				   :slot-value $value)
			(:syn-slot content
				   :sem-slot event
				   :slot-value $value)
			))


;------------------------------------------------------------------------------
; Category: tag-sequence
;------------------------------------------------------------------------------
(interpreter-cat-rules :cat tag-sequence
		       :head-rules
		       ((:sem-head (get-sgml-head $root '*BS)
			 :syn-constraint (symbolp $root)
			 :comment "SGML Sequence ROOT"))
		       )


;------------------------------------------------------------------------------
; Category: unit
;------------------------------------------------------------------------------
(interpreter-cat-rules :cat unit
		       :pre-map
		       ((:syn-constraint
			 ((adj-mod :DEFINED)
			  (number-value :DEFINED))
			 :pre-map-func (re-structure-adj-mod $fs)
			 :comment "ADJ mod")
			(:syn-constraint
			 ((number-value ((as-quant-as +) (quant :DEFINED))))
			 :pre-map-func (pre-map-as-quant-as-np $fs
							       'number-value)
			 :comment "As QUANT as"))
		       )


;------------------------------------------------------------------------------
; Category: v
;------------------------------------------------------------------------------
(interpreter-cat-rules :cat v
		       :pre-map
		       ((:syn-constraint
			 (*NOT* ((PASSIVE +)))
			 :pre-map-func (tree-path-set $fs '(PASSIVE) :val '-)
			 :comment "Passive")
			(:syn-constraint
			 ((ROOT SET-S))
			 :pre-map-func
			 (tree-path-set $fs '(SEM) :val '*G-COORDINATION)
			 :comment "S set")
			;; 19-Feb-97 by EHN -- fingers crossed...
			#|
			(:syn-constraint ((ROOT "be"))
					 :pre-map-func (pre-map-be-verb $fs)
					 :comment "Be")|#
			(:syn-constraint
			 ((SENT-ADV ((SEM *M-NEVER))))
			 :pre-map-func (tree-path-set $fs '(NEGATION) :val '+)
			 :comment "SENT-ADV Never")
			;; 19-Feb-97 by EHN
			(:syn-constraint
			 ((SENTENTIAL-ADVERB ((SEM *M-NEVER))))
			 :pre-map-func (tree-path-set $fs '(NEGATION) :val '+)
			 :comment "SENT-ADV Never")
			(:syn-constraint
			 ((ADV ((SEM *M-NEVER))))
			 :pre-map-func (tree-path-set $fs '(NEGATION) :val '+)
			 :comment "ADV Never")
			;; 19-Feb-97 by EHN
			(:syn-constraint
			 ((ADVERB ((SEM *M-NEVER))))
			 :pre-map-func (tree-path-set $fs '(NEGATION) :val '+)
			 :comment "ADV Never"))
		       :feature-slot-rules
		       ((:syn-slot xcomp
				   :sem-slot xcomp
				   :slot-value $value)
			(:syn-slot compl
				   :sem-slot complement
				   :slot-value $value)
			(:syn-slot scomp
				   :sem-slot scomp
				   :slot-value $value)
			(:syn-slot subord-clause
				   :sem-slot qualification
				   :slot-value $value)
			;; 24-Feb-97 by EHN
#|			(:syn-slot compl-form
				   :sem-slot FORM
				   :slot-value $value)|#
			(:syn-slot form
				   :sem-slot FORM
				   :slot-value $value)

			(:syn-slot gapped
				   :sem-slot gapped
				   :slot-value $value)

			;;--

			(:syn-slot purpose-inf
				   :sem-slot qualification
				   :slot-value $value)
			)
		       :multiple-feature-slot-rules
		       ((:syn-slots
			 ;;  5-Feb-97 by EHN -- added TENSE to the list.
			 (form modal mood infinitive naked tense)
			 :eval-forms (map-mood-form-and-modal $slot-values)))
		       :post-map
		       (
			(:sem-constraint
			 (tree-test '(:NOT (*G-COORDINATION)) $ir)
			 :post-map-func
			 (if (tree-test '(:OR (*A-BE)
					      (*A-BE-PREDICATE)
					      (*A-BE-EQUIVALENCE)
					      (*A-BE-EXISTENCE))
					$ir)
			     (post-map-be-verb $fs $ir)
			   (post-map-verb $fs $root $ir))
			 :comment "Any verb")
			(:sem-constraint (animate-p (get-agent $ir))
					 :post-map-func (post-map-modal $ir)
					 :comment "Modal")))




;; 10-Feb-97 by EHN -- UPMC domain
(interpreter-cat-rules :cat LBL
		       :head-rules
		       ((:sem-head 
			 (or (tree-filler $fs 'SEM)
			     (get-special-head $root 'LBL)
			     :syn-constraint (stringp $root)
			     :comment "LBL ROOT")))
		       :feature-slot-rules
		       ((:syn-slot punctuation
				   :sem-slot punctuation
				   :slot-value $value))
		       )

;; 20-Feb-97 by EHN -- support functions.

(defun create-adjunct-slot (head-fs slot-filler)
  (let ((cat (second (assoc 'cat head-fs))))
    (case cat
	  (adv (cond ((eq '+ (second (assoc 'time head-fs)))
		      (list 'time-modifier slot-filler))
		     (t (list 'manner slot-filler))))
	  ;;  3-Mar-97 by EHN -- look for (TIME +) in the POBJ,
	  ;; put those PPs into TIME-MODIFIER.
	  (p   (let* ((obj (second (assoc 'object head-fs)))
		      (time (second (assoc 'time obj))))
		 (cond ((eq time '+)
			(list 'time-modifier slot-filler))
		       (t (list 'q-modifier slot-filler)))))
	  (n   (cond ((eq (second (assoc 'time head-fs)) '+)
		      (list 'time-modifier slot-filler))
		     (t (list 'o-modifier slot-filler))))
	  (t (interpreter-warn "CREATE-ADJUNCT-SLOT: Unknown cat (~s)" cat)
	     (list 'UNKNOWN-ADJUNCT slot-filler)))))

(defun topicalize (ir)
  (tree-path-set ir '(TOPIC) :val '+ :do :PUSH :ADD-PATH t))

;; 18-Mar-97 by EHN

(defun unify-conjunct-tenses (ir)
  (let ((tense (second (assoc 'tense (rest ir))))
	(conjuncts (rest (assoc :multiple 
				(rest (assoc 'conjuncts (rest ir))))))
	target)
    (dolist (pointer conjuncts ir)
	    (if (assoc 'tense (rest pointer))
		(setf (second (assoc 'tense (rest pointer))) tense)
	      (push (list 'tense tense) (rest pointer))))))

;---eof category-rules.lisp---
