(in-package 'user)

;; ----------------------------------------------------------------------
;; 9. Reduce lexical ambiguity via SMCS codes
;;
;; New variables:
;;
;;   *smcs*  -- Hash table keyed on root string, holds assoc list keyed
;;              on POS; entries contain (<concept> <code>+) assignments
;;
;;   *current-smcs-codes* -- Assume this is set to contain codes
;;                           identifying the current IE's subdomain
;;
;; New macros:
;;
;;   smcs (root pos &rest concepts) -- build an entry in *smcs* (see below)
;;
;; 
;; New functions:
;;
;;  FILTER-SMCS-MEANINGS 
;;   FIND-SMCS-CONCEPTS

;; ----------------------------------------------------------------------
#| DOCUMENTATION / EXAMPLES

;; When you're in the Analyzer image, use the function DISPLAY-SMCS
;; to see the SMCS restrictions, if any, that have been assigned to
;; an AMBIGTRUE term:

> (display-smcs "ballast")
((N
    (*O-BALLAST-1)
    (*O-BALLAST-2 1400 1550 1900 1450 4450 5700 7300)))

;; When no SMCS codes are assigned (the default case when you fire
;; up the Analyzer, or after you call SMCS with no arguments), all
;; senses are available:

> (smcs) ; reset SMCS codes 
NIL

> (test-sentence "ballast")

("TYPE" "LEXICAL" "OFFSET" "1" "LENGTH" "7" "STRING" "ballast" "MSG"
"'ballast' is used ambiguously."  "SEM" "*O-BALLAST-1" "SEM"
"*O-BALLAST-2")

;; Note that some terms (like "ballast") include senses for which
;; no SMCS codes are assigned (e.g., *O-BALLAST-1). For senses like
;; these, if some other sense with an SMCS code matches, the sense
;; with no information will be pruned:

> (smcs 1400) ; use the function SMCS to set some codes
(1400)

> (test-sentence "ballast")
1

> (pns *parse-value*)
(((STANDALONE-PHRASE +) (COUNT -) (COUNT-PLUS-MINUS -) (NUMBER SG)
  (CAT N) (TOKEN 1) (ROOT "ballast") (GENPL -) (SEM *O-BALLAST-2)
  (PHRASAL-SCORE 1)))

;; Note that only *O-BALLAST-2 is selected, since the active SMCS
;; code of 1400 matches *O-BALLAST-2 and *O-BALLAST-1 neither matches
;; 1400 nor contains the keyword :ALL; contrast with this example
;; using "board", which does include :ALL keywords on senses without
;; specific code assignments:

> (display-smcs "board")
((N
    (*O-BOARD-1 :ALL)
    (*O-BOARD-2 :ALL)
    (*O-BOARD-3 1400 1500 1900 4800 5700)
    (*O-BOARD-4 1600 7450)
    (*O-BOARD-5 1600 7450)))

> (test-sentence "board")
("TYPE" "LEXICAL" "OFFSET" "1" "LENGTH" "5" "STRING" "board"
 "MSG" "'board' is used ambiguously." "SEM" "*O-BOARD-1"
 "SEM" "*O-BOARD-2" "SEM" "*O-BOARD-3")

;; Senses 1, 2, and 3 are available -- 1 and 2 because they have
;; the keyword :ALL, 3 because it matches the code 1400. If
;; *O-BOARD-1 and *O-BOARD-2 were empty instead, then only
;; *O-BOARD-3 would be selected in this case.

;; When SMCS codes are assigned, but don't match any assigned
;; codes for the term's senses, then all senses are available,
;; even those that lack the :ALL keyword:

> (smcs 78903)
(78903)

> (test-sentence "ballast")

("TYPE" "LEXICAL" "OFFSET" "1" "LENGTH" "7" "STRING" "ballast" "MSG"
"'ballast' is used ambiguously."  "SEM" "*O-BALLAST-1" "SEM"
"*O-BALLAST-2")
|#

(defvar *smcs* (make-hash-table :size 100 :test #'equal))

(defmacro assign-smcs (root pos &rest concepts)
  `(setf (gethash ,root *smcs*)
	 (nconc (gethash ,root *smcs*)
		'((,pos ,@concepts)))))

(clrhash *smcs*)



(defvar *current-smcs-codes*)
(setq  *current-smcs-codes* nil)

(defmacro smcs (&rest codes)
  `(progn (clear-cache *word-fs-cache*)
	  (setq *current-smcs-codes* ',codes)))

(defun filter-smcs-meanings (root pos concept-list)
  ;;  3-Oct-95 by EHN -- if this fails to find an entry
  ;; for a particular concept, although there are SMCS
  ;; entries overall, reverts to the whole set -- don't
  ;; want problems with sparse domain coverage causing
  ;; hard errors.
  (cond ((listp concept-list)
	 ;;  4-Oct-95 by EHN -- assume there is an *OR*
	 (unless (eq '*OR* (first concept-list))
	   (error "BAD CONCEPT-LIST: ~s" concept-list))
	 (let (smcs-concepts good)
	   (cond ((setq smcs-concepts
			(or (find-smcs-concepts root pos)
			    (rest concept-list)))
		  (setq good (or (subset #'(lambda (concept)
					     (member concept smcs-concepts))
					 smcs-concepts)
				 concept-list))
		  (if (> (length good) 1)
		      (cons '*or* good)
		    (first good)))
		 (t concept-list))))
	(t concept-list)))

(defun find-smcs-concepts (root pos)
  (let ((entry (rest (assoc pos (gethash root *smcs*)))))
    (cond (entry
	   (let (result)
	     (dolist (sub entry result)
	       (when (smcs-intersection (rest sub) *current-smcs-codes*)
		 (push (first sub) result)))))
	  (t nil))))

(defun smcs-intersection (codes current)
  (dolist (code codes nil)
    (when (smcs-unify code current)
      (return t))))

(defun smcs-unify (item list)
  (if (find item list :test #'smcs-match)
      t
    nil))

(defun smcs-match (x y)
  (cond ((or (eq x :all)
	     (eq y :all))
	 t)
	((and (numberp x)
	      (numberp y))
	 (if (= x y)
	     t
	   nil))
	((and (numberp x)
	      (smcs-range-p y))
	 (if (smcs-range-include x y)
	     t
	   nil))
	((and (numberp y)
	      (smcs-range-p x))
	 (if (smcs-range-include y x)
	     t
	   nil))
	((and (smcs-range-p x)
	      (smcs-range-p y))
	 (if (smcs-range-intersect x y)
	     t
	   nil))
	(t (error "SMCS-MATCH: Bad args: ~s ~s" x y))))

(defun smcs-range-p (list)
  (and (listp list)
	(eq :plus (first list))))

(defun smcs-range-intersect (x y)
  (eq (second x)
      (second y)))

(defun smcs-range-include (x y)
  (let ((difference (- x (second y))))
    (and (< difference 1000)
	 (>= difference 0))))

(defun display-smcs (root)
  (pns (gethash root *smcs*)))
