;;;-----------------------------------------------------------------------;
;;;  -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;-----------------------------------------------------------------------;
;;;
;;; Last Modified  25-Jun-96, 21:07
;;;
;;;         File: test-sentence.lisp
;;; File created: 26-Apr-96 by EHN 
;;;      Package: USER
;;;       Author: Eric Nyberg [ehn@cs.cmu.edu]
;;;   
;;; File Description: 
;;;
;;; Merges functionality previously provided in files checker-functions.lisp
;;; and phrase-filter.lisp. That functionality is the top-level call for
;;; TEST-SENTENCE, the function called by the CTE grammar checker, and
;;; all of its required subfunctions. The DISAMBIGUATE function is stored
;;; in a separate file (see below).
;;; 
;;;     See also: disamb-bob.lisp
;;; 
;;;-----------------------------------------------------------------------;
;;;
;;; Change Log:
;;; -----------
;;;
;;; 26-Apr-96 by EHN: Created
;;;
;;;-----------------------------------------------------------------------;


;;;-----------------------------------------------------------------------;
;;; Documentation:                                                        ;
;;;-----------------------------------------------------------------------;
#|

Basic Call Structure:

 test-sentence                     ; top-level call / error handler

   test-sentence-1                 ; main function

     parse-string                  ; call to Tomita parser (syntax)

     filter-ors                    ; prune ORs

     filter-phrase-compositions    ; remove spurious NP parses

     complexity-measure            ; text sentence complexity

     disambiguate                  ; call lex/struct disambiguator

     filter-best-triples-internal  ; remove suboptimal PP attachments

|#
;;;-----------------------------------------------------------------------;
;;; Package Statements:                                                   ;
;;;-----------------------------------------------------------------------;

(in-package 'user)

;;;-----------------------------------------------------------------------;
;;; Configuration:
;;;-----------------------------------------------------------------------;

;; (proclaim '(inline foo bar))

;;;-----------------------------------------------------------------------;
;;; Structures etc:                                                       ;
;;;-----------------------------------------------------------------------;


;;;-----------------------------------------------------------------------;
;;; Global Variables:                                                     ;
;;;-----------------------------------------------------------------------;

;;; Defined in cte-checker/release/reader.lisp.
(setq *analyzer-tag-phrases* nil) ;; disable the obsolete IDIOM tagging.

;;; Defined by the parser.
(proclaim '(special
	    *out*			; Output printing stream
	    *max-ambiguity-display*	; How many f-structures printed 
	    *parser-failed*		; Failed on last call to parse-list
	    *parse-value*
	    *parse-value-orig*
	    ))

(proclaim '(special
	    *analyzer-version*
	    *analyzer-path*
	    *sgml-tags-discarded*
	    *analyzer-tag-phrases*
	    *dmk-directory*
	    *sem-attach-verbose*
	    *sem-attach-trace*
	    *sem-attach-much-output*
	    *use-sem-restrictions*
	    ))

(defvar *test-sentence-verbose* nil
  "Controls whether or not TEST-SENTENCE hides all internal
   output or tracing; acceptable values are T or NIL."
  )

(defparameter *dev-null-stream*
  (make-two-way-stream (make-concatenated-stream) (make-broadcast-stream))
  "Data sink -- works like /dev/null"
  )

(defvar *trace-noun-phrase-filter* nil
  "Enable a tracing message each time an fs is pruned."
  )

;; The reason why ADJ, N are on this list should be obvious. V is
;; there because "ing" forms (e.g., 'cooling systems') can be ambig
;; with NP; ORD is there because of ambigs like "first speed" vs. NP;
;; and PREP is there because of things like "shank in" vs. NP.

;; 03-Dec-96 by EHN -- added 'UNIT' to the list for PR 4519
(defvar *np-components* '(n adj v ord prep adv unit)
  "When pruning compositional readings, only do so if the composed NP
   contains only tokens whose CAT feature are in this list."
  )

;; 03-Dec-96 by EHN -- added 'UNIT' to the list for PR 4519
(defvar *np-comparison-categories* '(n adj v ord prep conj adv unit)
  "Filtering will store information about these POS when comparing readings."
  )

(defvar *noun-phrase-filter-data* nil
  "Used to store the results of FIND-TOKEN-COVERAGE in the NP filter."
  )

(defvar *trace-triple-pruning* nil
  "Controls tracing of triple pruning mechanism."
  )

(defvar *or-filter-type* :prune
  "Controls the type of OR pruning (see filter-ors defun)."
  )

(defvar *prefer-NP-top-level-tags* '(ietitle title term specvalue)
  "If these tags appear as the CTE top-level item, NPs will
   be preferred if there is an ambiguity."
  )

(defun test-sentence-debug (flag)
  (cond (flag
	 (setq *test-sentence-verbose* t)
	 (setq *max-ambiguity-display* 0)
	 (setq *trace-noun-phrase-filter* t)
	 (setq *trace-triple-pruning* t))
	(t
	 (setq *test-sentence-verbose* nil)
	 (setq *max-ambiguity-display* 3)
	 (setq *trace-noun-phrase-filter* nil)
	 (setq *trace-triple-pruning* nil))))

(defvar *check-identity-outside-phrases-pruned* t
  "Controls whether context outside phrase range must match exactly for
   compositional reading to be pruned."
  )

(defvar *parse-string-value* nil
  "Holds the value of *PARSE-VALUE* after calling PARSE-STRING"
  )

(defvar *phrase-equivalence-verbose* nil
  "Makes phrase filtering code extra verbose."
  )

(defvar *all-parse-values* nil
  "Holds multiple parse-values when there is idiom tagging variation
   to be dealt with."
  )

;; 25-Jun-96 by EHN -- add vars for controlling raw complexity

(defvar *raw-cons-complexity-limit* 400000
  "If PARSE-STRING creates a *PARSE-VALUE* with more than this number
   of conses, then TEST-SENTENCE will quit."
  )

(defvar *raw-or-complexity-limit* 32000
  "If PARSE-STRING creates a *PARSE-VALUE* with more than this number
   of internal ORs, then TEST-SENTENCE will quit."
    )

(defvar *killer-sentence*
    "Shields, which protect hot exhaust components from oil or fuel spray in the event of a line, tube or seal failure, must be installed correctly."
  "For testing raw complexity measures."
  )

;; 08-06-96 by EHN -- Trace pruning of predicate PPs vs. passive.

(defvar *predicate-pp-pruned* 0
  "Will be a positive integer indicating the number of fs
   removed by the predicate pp heuristic if something was
   pruned."
  )

;;;-----------------------------------------------------------------------;
;;; Macros:                                                               ;
;;;-----------------------------------------------------------------------;

(defmacro test-sentence-trace (s &rest args)
  `(when *test-sentence-verbose*
     (format t ,s ,@args)))

(defmacro trace-number-of-parses (module-string)
  `(when *test-sentence-verbose*
     (format t "~%[TEST-SENTENCE] ~s fs after ~a"
	     (length *parse-value*) ,module-string)))
  
;;;-----------------------------------------------------------------------;
;;; TEST-SENTENCE

(defun test-sentence (sent)
  (handler-case (test-sentence-1 sent)
                (error (condition)
		       (declare (ignore condition))
                       (values nil :parser-error))))

(defun test-sentence-1 (s)
  (let (
	;; Raw return value from the Tomita parser.
	parser-value

	;; Return value from disambiguator.
	disambig-value
	
	;; Variable *OUT* is a stream used by the Tomita parser
	;; to print trace messages; make it NIL to silence the parser,
	;; T to allow tracing to *standard-output*. The use of
	;; *TEST-SENTENCE-VERBOSE* in this binding context implies
	;; that it must be set to T or NIL.
	(*out* (or *test-sentence-verbose* nil))
	
	;; Rebind *STANDARD-OUTPUT* to /dev/null if we don't want any
	;; output. 
	(*standard-output*
	 (if *test-sentence-verbose*
	     *standard-output*
	   *dev-null-stream*))

	;; Control the number of f-structures that the Tomita parser
	;; will print.
	(*max-ambiguity-display*
	 (if *test-sentence-verbose*
	     *max-ambiguity-display*
	   0))
	
	;; Local variables used to store the number of conses and 
	;; internal ORs for the current parse, when testing raw complexity.
	current-conses current-ors
	)
    
    ;; 25-Jun-96 by EHN
    ;; Before we do anything, reset important globals so we don't carry extra 
    ;; (potentially big) storage around while doing syntactic parse. This
    ;; means old values will get GC'ed if GC gets called during this parse.
    
    (setq *parse-value* nil)
    (setq *parse-value-orig* nil)

    ;; Call the Tomita (syntactic) parser.
    (setq parser-value (parse-string s))
    
    ;; 25-Jun-96 by EHN -- Before doing anything, check for complexity
    ;; out of bounds.
    
    (multiple-value-setq (current-conses current-ors)
      (raw-complexity *parse-value*))
    
    (cond (;; Parser-internal timeout. Punt.
	   
	   (eq :timeout parser-value)
	   (values
	    '("TYPE" "COMPLEXITY" "MSG"
	      "Too complex (time limit exceeded). Break up or use fewer prepositions.")
	    :not-cte))
	  
	  (;; Check for raw complexity over threshold.
	   (raw-complexity-too-high current-conses current-ors)
	   (values  
	    '("TYPE" "COMPLEXITY" "MSG"
	      "Too complex (memory limit exceeded). Break up or use fewer prepositions.")
	    :not-cte))

	  (
	   ;; Empty input (e.g., SMCS tag was sucked out). Quit.
	   (eq :empty parser-value)
	   (values nil :parser-failed))
	       
	  (t

	   (cond

	    ;; Look for specific failure cases and return right away.

	    (
	     ;; If the parser failed, then quit. NOTE: This is
	     ;; where 'idiom-retry' used to be.
	     *parser-failed*
	     (values NIL :not-cte))

	     ;;(
	     ;; If there are obsolete tags that were discarded,
	     ;; then fail this sentence.
	     ;;*sgml-tags-discarded*
	     ;;(values NIL :not-cte))

	    (t

	     ;; Otherwise, proceed with filtering and disambiguation.

	     (trace-number-of-parses "PARSE-STRING")
	     (setq *parse-string-value* (copy-tree *parse-value*))
	     
	     (setq *parse-value* (mapcan #'filter-and-reduce-ors *parse-value*))
	     (trace-number-of-parses "FILTER-AND-REDUCE-ORS")
	     (setq *parse-value-orig* (copy-tree *parse-value*))

	     ;; At this point, *parse-value* contains a list of 'flat'
	     ;; f-structures, i.e. there are no *OR*s any more. This
	     ;; means there is no longer ambiguity unless there is
	     ;; strictly >1 f-structure, so we can skip the following
	     ;; steps if <=1 f-s.
	     
	     ;; Filter spurious NP readings.

	     (cond ((> (length *parse-value*) 1)
		    (setq *parse-value* (filter-phrase-compositions *parse-value*))
		    (trace-number-of-parses "NOUN-PHRASE-FILTER"))
		   (t (trace-number-of-parses "skipping NOUN-PHRASE-FILTER")))

	     ;; Filter with top-level heuristics.

	     (cond ((> (length *parse-value*) 1)
		    (setq *parse-value* (filter-top-level *parse-value*))
		    (trace-number-of-parses "FILTER-TOP-LEVEL"))
		   (t (trace-number-of-parses "skipping FILTER-TOP-LEVEL")))
	     
	     ;; 06-Aug-96 by EHN -- Filter out predicate-pp in favor of passive.
	     
     	     (cond ((> (length *parse-value*) 1)
		    (setq *predicate-pp-pruned* 0)
		    ;; 07-Aug-96 by EHN -- if all readings are pruned, 
		    ;; fall back on original parse value.
		    (let ((pruned (prune-predicate-pp *parse-value*)))
		      (when pruned
			(setq *parse-value* pruned)))
		    (trace-number-of-parses "PRUNE-PREDICATE-PP"))
		   (t (trace-number-of-parses "skipping PRUNE-PREDICATE-PP")))

             ;; Filter with triples.

	     (cond ((> (length *parse-value*) 1)
		    (setq *parse-value* (pick-best-triple-scores *parse-value*))
		    (trace-number-of-parses "TRIPLE-PRUNING"))
		   (t (trace-number-of-parses "skipping TRIPLE-PRUNING")))
	     
	     (cond (
		    ;; Then check complexity threshold; if too
		    ;; complex, quit with appropriate return value.
		    
		    (and *halt-on-complexity*
			 (> (length *parse-value*)
			    ;; 22-May-96 by EHN -- used to be
			    ;; (complexity-measure), now obsolete, can
			    ;; be junked in checker-functions.lisp.
			    *analyzer-fs-complexity-limit*))
		    
		    (values '("TYPE" "COMPLEXITY" "MSG" "Too complex. Break up or use fewer prepositions.")
			    :not-cte))
		   (t

		    ;; Otherwise, call the disambiguator. 

		    (clrhash *triple-table*) ;; 22-May-96 by EHN --
					     ;; Obsolete, remove? Need
					     ;; to update disamb-bob sources...
		    (setf disambig-value (disambiguate))

		    ;; If Nothing found, return number of f-structures and :CTE,
		    ;; otherwise return the disambiguator value and :NOT-CTE
			
		    (if (null disambig-value)
			;;  1-Jul-96 by EHN -- if we fall through
			;; here on a null parse, make sure we don't
			;; return "0 :CTE"
			(if *parse-value*
			    (values (length *parse-value*) :cte)
			  (values NIL :not-cte))
		      (values disambig-value :not-cte))))))))))

;;; DISP-VALUE-PRINT is a function used in the parser, so this file must be
;;; loaded after the syntactic parser is loaded to properly shadow the
;;; original definition.

(defun disp-value-print (fs stream)
  "FS printout function used by the parser."
  (if stream ; bind *out* to NIL to silence
      (pns fs :stream stream)))

;;; PARSE-STRING is the main entry into the syntactic parser;
;;; specifically, this function calls PREPROCESS-WORD-STRING
;;; (reader.lisp) and PARSE-LIST (tf-patches.lisp).

(defun parse-string (sent)
  
  "Call Tomita parser via PARSE-LIST, with appropriate global variable
  management and checking for bogus SGML."

  (let (all-token-lists parse-list-return)

    ;; Reset global variables.
    (setq *sgml-tags-discarded* nil)  
    (setq *parser-input-string* sent)

    ;; Call tokenizer. New version returns a list of lists, each
    ;; sublist contains token list and token struct
    (setq all-token-lists (tokenize sent))
    
    (dolist (tagging all-token-lists nil)
      (setq *parser-token-list* (first tagging))
      (setq *tokval* (second tagging))
      (test-sentence-trace "~%[TEST-SENTENCE] Token List: ~s" *parser-token-list*)
      (clrhash *cte-tag-counter*)
      (setq parse-list-return
	    (parse-list *parser-token-list*))
      ;;  1-Jul-96 by EHN -- Check for :timeout and return
      ;; if we get it, otherwise keep trying until we get a
      ;; parse value and return T
      (if (eq parse-list-return :timeout)
	  (return parse-list-return)
	(when *parse-value*
	      (return t)))
)))

(defun parse-string-new (sent)
  
  "Call Tomita parser via PARSE-LIST, with appropriate global variable
  management and checking for bogus SGML."

  (let (all-token-lists)

    ;; Reset global variables.
    (setq *sgml-tags-discarded* nil)  
    (setq *all-parse-values* nil)      
    (setq *parser-input-string* sent)

    ;; Call tokenizer. New version returns a list of lists, each
    ;; sublist contains token list and token struct
    (setq all-token-lists (tokenize sent))
    
    (dolist (tagging all-token-lists)
      (setq *parser-token-list* (first tagging))
      (setq *tokval* (second tagging))
      (test-sentence-trace "~%[TEST-SENTENCE] Token List: ~s" *parser-token-list*)
      (clrhash *cte-tag-counter*)
      (parse-list *parser-token-list*)
      ;; 11-Jun-96 by EHN
      ;; Herein lies a problem. We used to just return once we found a parse,
      ;; but that leads to undesired behavior when we have a sentence like:
      ;; "the gear turns on the shaft." Since we prefer the idiom for
      ;; 'turn on', we don't get the (correct) reading of 'on' as a prep.
      (when *parse-value*
	(dolist (reading *parse-value*)
	  (push (cons reading tagging)
		*all-parse-values*))))
    (cond (*all-parse-values*
	   (setq *parse-value*
	     (mapcar #'first *all-parse-values*))
	   t)
	  (t nil))
    ))

;;;-----------------------------------------------------------------------;
;;; FILTER-PHRASE-COMPOSITIONS

#| Example: "wait until current flows."

> (mapcar #'pprint (mapcar #'pos-token-coverage *parse-value-orig*))

((1 "wait" V) (4 "flow" V)
              (3 "current" N))
((1 "wait" V) ((*MULTIPLE* 4 3) "current flow" N))
((1 "wait" V) (4 "flow" N)
              (3 "current" ADJ))

Algorithm:

"If reading A contains a phrase which covers positions m..n, and
    reading B contains NP-internal tokens for positions m..n, and
    for all other positions not in m..n, A and B contain identical tokens
 then
   prefer A to B."

|#

(defun filter-phrase-compositions (fs-list)
  
  "The top-level call, called from within TEST-SENTENCE and its sub-functions.
   Implements the algorithm described above for pruning compositional structures."
  
  (let* ((relevant-tokens (mapcar #'pos-token-coverage fs-list))
	 (all-phrases (mapcar #'find-noun-phrases relevant-tokens))
	 current-phrases prune result)

    (setq *noun-phrase-filter-data* relevant-tokens)

    ;; For each f-structure, iterate through the list of phrasal nouns that appear,
    ;; if any. For each such phrasal noun, iterate through all the other f-structures,
    ;; looking for those that are equivalent to the current f-structure except for the
    ;; token range covered by the phrasal noun. Further restrict the equivalence to only
    ;; those structures where the tokens covering the range covered by the phrasal noun
    ;; have CAT features that fall in the list *np-components*. Push the index of any
    ;; equivalent structures into the variable PRUNE for later disposal.
    
    (dotimes (i (length all-phrases))
      (cond ((member i prune)
	     nil)
	    (t
	     (setq current-phrases (nth i all-phrases))
	     (when *trace-noun-phrase-filter*
	       (format t "~%[NOUN-PHR-FILT] FS[~a], root: ~s, has phrases: ~s"
		       i (second (assoc 'root (nth i fs-list))) current-phrases))
	     (dolist (phrase current-phrases)
	       (dotimes (j (length all-phrases))
		 (cond ((eq i j) nil)
		       ((member j prune) nil)
		       (t (when (phrase-otherwise-equivalent
				 phrase 
				 (nth i relevant-tokens)
				 (nth j relevant-tokens))
			    (when *trace-noun-phrase-filter*
			      (format t "~%[NOUN-PHR-FILT] Pruning fs[~s], fs[~s] identical with phrase at ~s"
				      j i phrase))
			    (push j prune)))))))))

    ;; Iterate through the original list of f-structures, discarding those whose
    ;; indices were put into the PRUNE list; return the result.

    (dotimes (i (length fs-list) result)
      (unless (member i prune)
	(push (nth i fs-list) result)))
    ))

(defun pos-token-coverage (fs)
  
  "Extract information about those tokens whose CAT feature
   is in the list *NP-COMPARISON-CATEGORIES. Store token range,
   root, and CAT for each such item."
  
  (cond ((listp fs)
	 (cond ((symbolp (first fs))	; handle *OR*, et al.
		(if (every #'symbolp (rest fs))
		    ;; something like (*or* + -)
		    nil
		  (mapcan #'pos-token-coverage
			  (rest fs))))
	       (t			; assume a 'normal' fs.
		(let ((cat (second (assoc 'cat fs)))
		      (root (second (assoc 'root fs)))
		      (token (second (assoc 'token fs)))
		      result)
		  (when (and (member cat *np-comparison-categories*) token)
		    (setq result (list (list token root cat))))
		  (dolist (slot fs result)
		    (when (listp (second slot))
		      (setq result 
			    (nconc result
				   (pos-token-coverage (second slot))))))))))
	(t nil)))

(defun find-noun-phrases (list)
  
  "Called on the result of POS-TOKEN-COVERAGE to identify the phrasal
   entries, and extract their token ranges."
  
  (mapcan #'(lambda (x)
	      (when
		  (and (listp (first x))
		       (eq '*multiple* (first (first x)))
		       ;; 03-Dec-96 by EHN added UNIT, for PR 4519
		       (member (third x) '(unit n)))
		(list (rest (first x)))))
	  list))

(defun phrase-otherwise-equivalent (range phrasal-reading compo-reading)
  
  (let* (
	 ;; Stuff inside the phrase, phrasal reading

	 (phrase (subset #'(lambda (l)
			     (and (listp (first l))
				  (equal (rest (first l))
					 range)))
			 phrasal-reading))

	 ;; Stuff outside the phrase, phrasal reading
	 
	 (nphrase (subset #'(lambda (l)
			      (not (and (listp (first l))
					(equal (rest (first l))
					       range))))
			  phrasal-reading))

	 ;; Stuff inside phrase, compo reading

	 (compo (subset #'(lambda (l)
			    (or
			     (and (numberp (first l))
				  (member (first l) range)
				  (member (third l) *np-components*))
			     ;; Add a case for phrases that are part of a compo reading.
			     (and (listp (first l))
				  (null (set-difference (rest (first l)) range))
				  (member (third l) *np-components*))))
			compo-reading))

	 ;; Stuff outside phrase, compo reading

	 (ncompo (subset #'(lambda (l)
			     (not (or
				   (and (numberp (first l))
					(member (first l) range)
					(member (third l) *np-components*))
				   ;; Add a case for phrases that are part of a compo reading.
				   (and (listp (first l))
					(null (set-difference (rest (first l)) range))
					(member (third l) *np-components*)))))
			 compo-reading))	 

	 ;; Reduce both compo and phrase to their tokens ranges.
    
	 (compo-range (mapcan #'(lambda (x)
				  (if (listp (first x))
				      (copy-tree (rest (first x)))
				    (list (first x))))
			      compo))

	 (phrase-range (mapcan #'(lambda (x)
				   (if (listp (first x))
				       (copy-tree (rest (first x)))
				     (list (first x))))
			       phrase))    
	 )

    (when *phrase-equivalence-verbose*
      (test-sentence-trace "~%[PHRASES-EQUIV] Phrase: ~s" phrase)
      (test-sentence-trace "~%[PHRASES-EQUIV] NPhrase: ~s" nphrase)
      (test-sentence-trace "~%[PHRASES-EQUIV] Compo: ~s" compo)
      (test-sentence-trace "~%[PHRASES-EQUIV] NCompo: ~s" ncompo)
      (test-sentence-trace "~%[PHRASES-EQUIV] PhraseRange: ~s" phrase-range)            
      (test-sentence-trace "~%[PHRASES-EQUIV] CompoRange: ~s" compo-range))

    (when (or (not *check-identity-outside-phrases-pruned*)
	      ;;  5-Jun-96 by EHN -- subtract compo from phrase, or else we will over prune when
	      ;; V isn't prunable (conservative)
	      ;; (null (set-difference nphrase ncompo :test #'equal))
	      ;;  5-Jun-96 by EHN -- Need to allow differences where NP and compo match elsewhere,
	      ;; or else "mutual gotcha" will disallow pruning inside SET-NP... ("you can't prune me
	      ;; because I have a different reading somewhere else, and I can't prune you because we
	      ;; differ in the original position")
	      (material-outside-phrase-matches phrase compo nphrase ncompo)
	      )
      ;; Everything outside the phrase range is equivalent in both readings, so okay to try pruning.
      (cond ((null (set-difference phrase compo :test #'equal))
	     ;; same, don't prune
	     nil)
	    (t
	     (cond ((null (set-difference phrase-range compo-range))
		    ;; Okay to prune, cover same range
		    (test-sentence-trace "~%[PHRASES-EQUIV] Phrase: ~s, **PRUNING** Compo: ~s" phrase compo)		    

		    t)
		   (t nil)))))))

(defun material-outside-phrase-matches (phrase compo nphrase ncompo)
  (declare (ignore phrase compo))
  (let ((phrase-residue (set-difference nphrase ncompo :test #'equal))
	(compo-residue (set-difference ncompo nphrase :test #'equal))
	phrase-residue-accounted-for
	compo-residue-accounted-for)
    ;; Now that we've gotten rid of what's identical, try to match phrases.
    (when *phrase-equivalence-verbose*
      (test-sentence-trace "~%[PHRASES-EQUIV] PHRASE-RESIDUE: ~s" phrase-residue)
      (test-sentence-trace "~%[PHRASES-EQUIV] COMPO-RESIDUE: ~s" compo-residue)    )
    (setq phrase-residue-accounted-for
	  (every #'(lambda (x)
		     (or (and (listp (first x))
			      (eq '*multiple* (first (first x)))
			      (eq 'n (third x)))
			 (dolist (item compo-residue
				       (progn
					 (when *phrase-equivalence-verbose*
					   (test-sentence-trace "~%[PHRASES-EQUIV] Item ~s in phrase-residue not found in compo-residue!" x))
					 nil))
			   (when (and (listp item)
				      (listp (first item))
				      (eq '*MULTIPLE* (first (first item)))
				      (member (first x) (rest (first item))))
			     (when *phrase-equivalence-verbose*
			       (test-sentence-trace "~%[PHRASES-EQUIV] Item ~s in phrase-residue found in compo-residue phrase: ~s" x item)			     )
			     (return t)))))
		 phrase-residue))
    (setq compo-residue-accounted-for
	  (every #'(lambda (x)
		     (or (and (listp (first x))
			      (eq '*multiple* (first (first x)))
			      (eq 'n (third x)))
			 (dolist (item phrase-residue
				       (progn
					 (when *phrase-equivalence-verbose*
					   (test-sentence-trace "~%[PHRASES-EQUIV] Item ~s in compo-residue not found in phrase-residue!" x))
					 nil))
			   (when (and (listp item)
				      (listp (first item))
				      (eq '*MULTIPLE* (first (first item)))
				      (member (first x) (rest (first item))))
			     (when *phrase-equivalence-verbose*
			       (test-sentence-trace "~%[PHRASES-EQUIV] Item ~s in compo-residue found in phrase-residue phrase: ~s" x item))
			     (return t)))
			 ))
		 compo-residue))
    (if (and phrase-residue-accounted-for
	     compo-residue-accounted-for)
	t
      nil)))

;;;----------------------------------------------------------------------
;;; FILTER-BEST-TRIPLES

(defun filter-best-triples-internal ()
  (let ((best (find-best-triple-scores *parse-value*)))
    (setq *parse-value* best)
    T))

(defun pick-best-triple-scores-old (flat-fs-list)
  (let* ((raw (mapcar #'(lambda (fs)
			  (calculate-triple-score fs))
		      flat-fs-list))
	 (best-score (first (sort raw #'<)))
	 best-indices pruned-indices
	 result)
    (dotimes (i (length raw))
      (cond ((= (nth i raw) best-score)
	     (push i best-indices)
	     (setq result (nconc result (list (nth i flat-fs-list)))))
	    (t
	     (push i pruned-indices))))
    (when (and *trace-triple-pruning* pruned-indices)
      (format t "~%[TRIPLE-FILTER] Best score: ~a" best-score)
      (format t "~%[TRIPLE-FILTER] Keeping fs: ~a" best-indices)
;;      (format t "~%[TRIPLE-FILTER] Pruning fs: ~a" pruned-indices)
      )
    result))

(defun pick-best-triple-scores (flat-fs-list)
  (let* ((raw (mapcar #'(lambda (x)
			  (let (a b)
			    (multiple-value-setq (a b)
			      (calculate-triple-score x))
			    (cons a b)))
		      flat-fs-list))
	 (best-score (apply #'min (mapcar #'first raw)))
	 best-indices pruned-indices entry
	 result)
    (dotimes (i (length raw))
      (setq entry (nth i raw))
      ;;(test-sentence-trace "~%[TRIPLE-FILTER] Entry: ~s" entry)
      (cond ((some #'(lambda (x)
		       (and 
			(< (first x) (first entry))
			(same-preps-p (rest x) (rest entry))))
		   raw)
	     (test-sentence-trace "~%[TRIPLE-FILTER] Pruning: ~s" entry)	     
	     (push i pruned-indices))
	    (t (push i best-indices)
	       (setq result (nconc result (list (nth i flat-fs-list)))))))
    (when (and *trace-triple-pruning* pruned-indices)
      (format t "~%[TRIPLE-FILTER] Best score: ~a" best-score)
      (format t "~%[TRIPLE-FILTER] Keeping fs: ~a" best-indices)
      (format t "~%[TRIPLE-FILTER] Pruning fs: ~a" pruned-indices)
      )
    result))

(defun same-preps-p (list1 list2)
  (and (and (listp list1)
	    (listp list2))
       (= (length list1)
	  (length list2))
       (every #'(lambda (x)
		  (find x list2 :test #'equal))
	      list1)))

(defun find-best-triple-scores (fs-list &key (prune t))
  ;; FS-LIST is  *parse-value* or the REST of an *OR*.
  ;; Ergo, we can prune at the level of this call, if we have reason
  ;; to do so.
  (let (flat-fs-list)
    (dolist (fs fs-list)
      (cond ((not (listp fs))
	     ;; Handle atoms like (*OR* + -)
	     (setq flat-fs-list (nconc flat-fs-list (list fs))))
	    ((eq '*or* (first fs))
	     ;; *or* should be pruned at the subcall level.
	     (if (every #'listp (rest fs))
		 (let ((subset (find-best-triple-scores (rest fs))))
		   (if (> (length subset) 1)
		       (setq subset (cons '*or* subset))
		     (setq subset (first subset)))
		   (setq flat-fs-list (nconc flat-fs-list (list subset))))
	       nil))
	    ((symbolp (first fs))
	     ;; *not*, *multiple*, etc. should be the same; no
	     ;; pruning at subcall.
	     (setq flat-fs-list
		   (nconc flat-fs-list
			  (list
			   (cons (first fs)
				 (find-best-triple-scores (rest fs)
							  :prune nil))))))
	    (t
	     ;; assume we have a "regular" fs to work with
	     (unless (every #'listp fs)
	       (error "FIND-BEST-TRIPLE-SCORES: fs exception: ~s" fs))
	     (setq flat-fs-list
		   (nconc flat-fs-list
			  (list
			   (mapcar #'(lambda (slot)
				       (let ((head (first slot))
					     (filler (rest slot)))
					 (if (every #'listp filler)
					     (cons head
						   (find-best-triple-scores filler :prune nil))
					   slot)))
				   fs)))))))
    (if prune
	(pick-best-triple-scores flat-fs-list)
      flat-fs-list)))
  
(defun calculate-triple-score-old (fs)
  (let (tscore)
    (cond ((listp fs)
	   (cond ((every #'listp fs)
		  (or (setq tscore (second (assoc 'tscore fs)))
		      (setq tscore 0))
		  (apply #'+ (cons tscore
				   (mapcar #'(lambda (slot)
					       (calculate-triple-score
						(second slot)))
					   fs))))
		 ((eq (first fs) '*multiple*)
		  (apply #'+ (mapcar #'calculate-triple-score
				     (rest fs))))
		 ((eq (first fs) '*not*)
		  (apply #'+ (mapcar #'calculate-triple-score
				     (rest fs))))
		 ((eq (first fs) '*or*)
		  ;; Can we be more clever than this?
		  ;; (apply #'+ (mapcar #'calculate-triple-score (rest fs)))
		  ;; 27-Feb-96 by EHN -- yes, please.
		  (apply #'min (mapcar #'calculate-triple-score (rest fs))))
		 (t (error "Unknown fs type: ~s" fs))))
	  (t 0))))

(defun calculate-triple-score (fs)
  (let ((total-tscore 0)
	total-pps
	score pps prep token)
    (cond ((listp fs)
	   (cond ((every #'listp fs)
		  (cond ((setq total-tscore
			      (second (assoc 'tscore fs)))
			 (setq prep (second (assoc 'root fs)))
			 (setq token (second (assoc 'token fs)))
			 (setq total-pps (cons (list prep token)
					       total-pps)))
			(t (setq total-tscore 0)))
		  (dolist (slot fs)
		    (multiple-value-setq (score pps)
		      (calculate-triple-score (second slot)))
		    (incf total-tscore score)
		    (setq total-pps (nconc total-pps pps)))
		  (values total-tscore total-pps))
		 ((eq (first fs) '*not*)
		  (values total-tscore total-pps))
		 ((eq (first fs) '*multiple*)
		  (dolist (subfs (rest fs))
		    (multiple-value-setq (score pps)
		      (calculate-triple-score subfs))
		    (incf total-tscore score)
		    (setq total-pps (nconc total-pps pps)))
		  (values total-tscore total-pps))		 
		 ((eq (first fs) '*or*)
		  (if (every #'symbolp (rest fs))
		      (values 0 nil)
		    (error "CALCULATE-TRIPLE-SCORE: OR handling broken.")))
		 (t (error "Unknown fs type: ~s" fs))))
	  (t (values 0 nil)))))

;;;-----------------------------------------------------------------------;
;;; FILTER-ORS

;; 20-May-96 by EHN -- Rewritten to use the new
;; FILTER-PHRASE-COMPOSITIONS and triple scoring routines

(defun filter-and-reduce-ors (fs)
  ;; 29-May-96 by EHN -- used to try filtering phrases internal to *OR*s, but
  ;; not possible to do easily; instead, flatten everything here and let the
  ;; global filtering handle it.
  (let ((result (reduce-ors fs))
	strained (pruned 0))
    (dolist (x result)
      (cond ((find x strained :test #'equal)
	     (incf pruned)
	     nil)
	    (t (setq strained (nconc strained (list x))))))
    (when (> pruned 0)
      (test-sentence-trace "~%[FILT-REDU-ORS] Removed ~s identical fs" pruned))
    strained))

(defun reduce-ors-1 (simple-fs)
  (cond ((listp simple-fs)
	 (let (current next)
	   (dolist (slot simple-fs current)
	     (setq next nil)
	     (let ((name (first slot))
		   (value (second slot)))
	       (cond ((listp value)
		      (if current
			  (dolist (flat-value (reduce-ors value) (setq current next))
			    (dolist (current-fs current)
			      (push (append  current-fs (list (list name flat-value)))
				    next)))
			(dolist (flat-value (reduce-ors value))
			  (push (list (list name flat-value))
				current))))
		     (t (if current
			    (dolist (current-fs current (setq current next))
			      (push (append current-fs (list (list name value)))
				    next))
			  (push (list (list name value))
				current))))))))
	(t (list simple-fs))))

(defun reduce-ors (fs)
  (cond ((not (listp fs))
	 (list fs))
	((every #'listp fs)
	 (reduce-ors-1 fs))
	((and (listp fs)
	      (eq '*or* (first fs))
	      (every #'listp (rest fs)))
	 (mapcan #'reduce-ors-1
		 (rest fs)))
	((and (listp fs)
	      (eq '*multiple* (first fs)))
	 (reduce-ors-multiple (rest fs))
	 )
	((and (listp fs)
	      (eq '*not* (first fs)))
	 ;; 22-May-96 by EHN -- Assumption -- no negation of full f-s, only features.
	 (list fs)
	 )
	(t (list fs))))

(defun reduce-ors-multiple (fs-list)
  (let ((expand-list (mapcar #'reduce-ors fs-list))
	result next) ;  member-counts)
    ; (setq member-counts (mapcar #'length expand-list))
    ; (test-sentence-trace "~%[REDUCE-OR-MUL] ~s -> ~s" member-counts (apply #'* member-counts))
    (setq result (mapcar #'list (pop expand-list)))
    (dolist (expanded-fs expand-list (mapcar #'(lambda (x)
						 (cons '*multiple* x))
					     result))
      (setq next nil)
      (dolist (struct expanded-fs)
	(dolist (current result)
	  (push (append current (list struct))
		next)))
      (setq result next))))


(defun filter-ors (fs)
  (cond ((eq '*OR* (first fs))
	 (filter-ors-1 (rest fs)))
	((eq '*MULTIPLE* (first fs))
	 (cons '*MULTIPLE* (mapcar #'filter-ors (rest fs))))
	(t
	 (mapcar #'(lambda (slot)
		     (if (and (listp slot)
			      ;; d-fs-p is in disamb-bob.lisp
			      (d-fs-p (second slot)))
			 (list (first slot) (filter-ors (second slot)))
			 slot))
		 fs))))


(defun filter-ors-1 (disjuncts)
  (let* ((result disjuncts)		; (mapcan #'filter-and-reduce-ors disjuncts))
	 (going-in (length result))
	 coming-out)
    (when *test-sentence-verbose*
      (test-sentence-trace "~%-->[FILTER_ORS] Called on ~a disjuncts." going-in))
    (setq result (filter-phrase-compositions result))
    (setq coming-out (length result))
    (if (= going-in coming-out)
	(test-sentence-trace "~%<--[FILTER-ORS] No change.")
      (test-sentence-trace "~%<--[FILTER-ORS] ~a original, ~a final f-s" going-in coming-out))
    (cond ((> (length result) 1)
	   (cons '*OR* result))
	  ((= (length result) 1)
	   (first result))
	  (t (error "FILTER-ORS-1 returned NIL!")))))




;;----------------------------------------------------------------------

(defun filter-top-level (fs-list)
  (cond ((and
	  ;; When: all of the f-structures are TITLE tags, and
	  (every #'(lambda (fs)
		     (member (second (assoc 'root fs))
			     *prefer-NP-top-level-tags*))
		 fs-list)
	  ;; ...there is at least one NP reading at top level,
	  (some #'(lambda (fs)
		     (eq 'N (second (assoc 'cat (second (assoc 'value
							      fs))))))
		fs-list))

	 ;; Then: return only the readings that include NP.
	 (test-sentence-trace "~%[FILTER-TOP-LE] Pruning non-NPs in top-level tags: ~s"
			      *prefer-NP-top-level-tags*)
	 (subset #'(lambda (fs)
		     (eq 'N (second (assoc 'cat (second (assoc 'value
							      fs))))))
		fs-list))
	
	;; Otherwise, do nothing.
	(t fs-list)))

;;----------------------------------------------------------------------
;; 06-Aug-96 by EHN -- add a function to check for these ambiguities:
;;
;; ((CAT V)(PASSIVE +)(PP)) vs ((CAT V)(ROOT "be")(PREDICATE-PP)) or
;; ((CAT V)(ADJ-COMP (PP))) vs ((CAT V)(ROOT "be")(PREDICATE-PP))
;; Where the contents of the PPs match on head token.
;; 
;; We want to prune the latter case in favor of the former, so first
;; find any fs that match the latter and see if you can find a passive
;; version of it elsewhere.
;;
;; Challenge: can occur anywhere a clause can occur, not just at the top 
;; level of the fs...

(defun prune-predicate-pp (fs-list)
  (let (hit pruner pruned-list result)
    (dolist (fs fs-list (values result pruned-list))
      ;; (format t "~%Checking fs: ~s" fs)
      (setq hit (find-predicate-pp-clause fs))
      ;; (format t "~%PRUNE-PREDICATE-PP: Hit = ~s" hit)
      (when hit
	(setq pruner (or (find-predicate-pp-pruner hit fs-list)
			 (find-adjcomp-pp-pruner hit fs-list))))
      ;; (format t "~%PRUNE-PREDICATE-PP: Pruner = ~s" pruner)      
      (cond ((and hit pruner)
	     ;; (format t "~%Pruned!")
	     (incf *predicate-pp-pruned*)
	     (setq pruned-list (nconc pruned-list (list fs))))
	    (t ;; (format t "~%Saved.")
	     (setq result (nconc result (list fs))))))))

(defun find-predicate-pp-clause (fs)
  (cond ((and (listp fs)
	      (every #'listp fs))
	 (or (find-predicate-pp-clause-1 fs)
	     (some #'(lambda (slot)
		       (find-predicate-pp-clause (second slot)))
		   fs)))
	((and (listp fs)
	      (eq '*multiple* (first fs))
	      (every #'listp (rest fs)))
	 (some #'(lambda (fs)
		   (find-predicate-pp-clause fs))
	       (rest fs)))
	(t nil)))
  
(defun find-predicate-pp-clause-1 (fs)
  (and 
   (equal "be" (second (assoc 'ROOT fs)))
   (eq 'V (second (assoc 'CAT fs)))
   (assoc 'PREDICATE-PP fs)
   fs))

(defun find-passive-pp-clause (pp-token fs)
  (cond ((and (listp fs)
	      (every #'listp fs))
	 (or (find-passive-pp-clause-1 pp-token fs)
	     (some #'(lambda (slot)
		       (find-passive-pp-clause pp-token (second slot)))
		   fs)))
	((and (listp fs)
	      (eq '*multiple* (first fs))
	      (every #'listp (rest fs)))
	 (some #'(lambda (fs)
		   (find-passive-pp-clause pp-token fs))
	       (rest fs)))
	(t nil)))

(defun find-adjcomp-pp-clause (pp-token fs)
  (when (null pp-token)
      (error "Bad PP-TOKEN value ~s" pp-token))
  (cond ((and (listp fs)
	      (every #'listp fs))
	 (or (find-adjcomp-pp-clause-1 pp-token fs)
	     (some #'(lambda (slot)
		       (find-adjcomp-pp-clause pp-token (second slot)))
		   fs)))
	((and (listp fs)
	      (eq '*multiple* (first fs))
	      (every #'listp (rest fs)))
	 (some #'(lambda (fs)
		   (find-adjcomp-pp-clause pp-token fs))
	       (rest fs)))
	(t nil)))

(defun find-passive-pp-clause-1 (pp-token fs)
  (cond ((and (listp fs)
	      (every #'listp fs))
	 (let (pp token)
	   (and 
	    (eq 'V (second (assoc 'CAT fs)))
	    (find '(PASSIVE +) fs :test #'equal)
	    (setq pp (second (assoc 'PP fs)))
	    (cond ((eq '*multiple* (first pp))
		   (some #'(lambda (x)
			     (let ((token (second (assoc 'token x))))
			       (find token pp-token :test #'equal)))
			 (rest pp)))
		  (t (setq token (second (assoc 'token pp)))
		     (find token pp-token :test #'equal)))
	    fs)))
	((and (listp fs)
	      (eq '*multiple* (first fs)))
	 (some #'(lambda (x)
		   (find-passive-pp-clause-1 pp-token x))
	       (rest fs)))
	(t nil)))

(defun find-adjcomp-pp-clause-1 (pp-token fs)
  (cond ((and (listp fs)
	      (every #'listp fs))
	 (let (adj-comp pp token)
	   (and 
	    (eq 'V (second (assoc 'CAT fs)))
	    (equal "be" (second (assoc 'ROOT fs)))
	    (setq adj-comp (second (assoc 'ADJ-COMP fs)))
	    (setq pp (second (assoc 'PP adj-comp)))
	    (cond ((eq '*multiple* (first pp))
		   (some #'(lambda (x)
			     (let ((token (second (assoc 'token x))))
			       (find token pp-token :test #'equal)))
			 (rest pp)))
		  (t (setq token (second (assoc 'TOKEN pp)))
		     (find token pp-token :test #'equal)))
	    fs)))
	((and (listp fs)
	      (eq '*multiple* (first fs)))
	 (some #'(lambda (x)
		   (find-adjcomp-pp-clause-1 pp-token x))
	       (rest fs)))
	(t nil)))

(defun find-predicate-pp-pruner (bad-fs fs-list)
  (let ((pp-token (find-pp-token bad-fs)))
    (dolist (fs fs-list nil)
      (when (find-passive-pp-clause pp-token fs)
	  (return fs)))))

(defun find-pp-token (bad-fs)
  (let* ((pp (second (assoc 'predicate-pp bad-fs)))
	 (root (second (assoc 'root pp)))
	 token)
    (cond ((eq 'set-pp root)
	   (mapcar #'(lambda (x)
		       (or
			(second (assoc 'token x))
			(error "Bad PP-TOKEN value ~s: ~s" token bad-fs)))
		   (rest (second (assoc 'member pp)))))
	  (t (setq token (second (assoc 'token pp)))
	     (if (null token)
		 (error "Bad PP-TOKEN value ~s: ~s" token bad-fs)
	       (list token))))))

(defun find-adjcomp-pp-pruner (bad-fs fs-list)
  (let ((pp-token (find-pp-token bad-fs)))
    (dolist (fs fs-list nil)
	(when (find-adjcomp-pp-clause pp-token fs)
	  (return fs)))))

(defun test-prune-predicate-pp (s &optional (verbose nil))
  (test-sentence s)
  (when (or verbose (= *predicate-pp-pruned* 0))
    (format t "~%~a ~s" *predicate-pp-pruned* s)
    ))
  
;; ----------------------------------------------------------------------
;; 24-May-96 by EHN -- Add a function that counts up the number of
;; preps, conj, etc. in an f-structure as a way to check hard-coded
;; complexity thresholds based on presence of "problem" terms

(defvar *complex-pos-list* '(PREP CONJ))

(defvar *ignore-root-list* '(SET-PP))

(defun count-complex-pos (fs)
  "Assume we're called after reduce-ors, so we don't need to worry about
   *OR*, only *MULTIPLE*; ignore *NOT*"
  (cond ((listp fs)
	 (cond ((eq '*MULTIPLE* (first fs))
		(apply #'+ (mapcar #'count-complex-pos (rest fs))))
	       ((every #'listp fs)
		(let* ((cat (second (assoc 'cat fs)))
		       (root (second (assoc 'root fs)))
		       (hit (if (and (member cat *complex-pos-list*)
				     (not (member root *ignore-root-list*)))
				1
			      0)))
		  (when (> hit 0)
		    (test-sentence-trace "~%[COMPLEX-POS] +1 ~s ~s" cat root))
		  (apply #'+ (cons hit
				   (mapcar #'(lambda (slot)
					       (count-complex-pos
						(second slot)))
					   fs)))))
	       (t 0)))
	(t 0)))

;; ----------------------------------------------------------------------
;; 28-May-96 by EHN -- Handle multiple tokenizations for IDIOM tags.

(defun insert-phrase-tags-all (token-list)
  (insert-phrase-tags-all-1 nil token-list))

(defun insert-phrase-tags-all-1 (head tail)
  ;;(format t "~%HEAD: ~s TAIL: ~s" head tail)
  (cond ((null tail)
	 (list head))
	(t (append
	    ;; Find all cases where we can insert an idiom at the front of this tail.
	    (mapcan #'(lambda (hit)
			(insert-phrase-tags-all-1
			 (append head (cons '{idiom} (append (subseq tail 0 hit) '({/idiom}))))
			 (nthcdr hit tail)))
		    (find-longest-phrase-rec 1 0 tail))
	    ;; Also include the 'no tag' case
	    (insert-phrase-tags-all-1
	     (append head (list (first tail)))
	     (rest tail))))))

;; (error "This file is in active development!")

;;----------------------------------------------------------------------
;; 29-May-96 by EHN -- Move TOKENIZE here from reader.lisp.

(defun tokenize (s)
  (let (token-struct token-list result new-token-list new-token-struct)
    (load-tokenizer-input s)
    (setq token-struct (find-tokens s))
    (setq token-list
	  (capitalize
	   (remove-if #'null
		      (mapcar #'(lambda (x)
				  (second (assoc 'partok x)))
			      token-struct))))
    
    (dolist (idiom (insert-phrase-tags-all token-list) result)
      (multiple-value-setq (new-token-list new-token-struct)
	(adjust-for-idiom-tags idiom token-struct))
      ;;(format t "~%NEW-TOKEN-LIST: ~s" new-token-list)
      (setq result 
	    (append result 
		   (list (list (append new-token-list (list '$)) 
			       new-token-struct)))))
    ))

(defun adjust-for-idiom-tags (list init-tokval)
  ;;(format t "~%LIST: ~s INIT-TOKVAL: ~s" list init-tokval)
  (let* ((i 0)
	 (tokval (copy-list init-tokval))
	 result last-open-tag)
    (dolist (item list (values list (reverse result)))
      (cond 
       ((eq item '{IDIOM})
	(setq last-open-tag i)
	(setq result (cons `((PARTOK ,item))
			   result)))
       ((eq  item '{/IDIOM})
	(setq result (cons `((PARTOK ,item)
			     (OPENER ,last-open-tag))
			   result)))	   
       (t
	(setq result (cons (first tokval) result))
	(pop tokval)))
      (incf i))))

;;----------------------------------------------------------------------
;; 25-Jun-96 by EHN -- Stuff for checking complexity after syntactic
;; parsing.

(defun count-conses (x)
  (cond ((null x) 0)
	((symbolp x) 0)
	((listp (first x))
	 (+ 1 (count-conses (first x))
	    (count-conses (rest x))))
	((symbolp (first x))
	 (+ 1 (count-conses (rest x))))
	(t 0)))

(defun or-complexity (fs)
  (cond ((listp fs) 
	 (cond ((eq (first fs) '*or*)
		(apply #'+ (mapcar #'or-complexity (rest fs))))
	       ((eq (first fs) '*multiple*)
		(apply #'* (mapcar #'or-complexity (rest fs))))
	       ((eq (first fs) '*not*)
		1)
	       (t
		(let ((total 1))
		  (dolist (slot fs total)
			  (when (listp (second slot))
				(setq total (* total (or-complexity (second slot))))))))))
	(t 1)))

(defun raw-complexity (parse-value)
  (values (count-conses parse-value)
	  (apply #'+ (mapcar #'or-complexity parse-value))))


(defun raw-complexity-too-high (conses ors)
  (or (> conses *raw-cons-complexity-limit*)
      (> ors *raw-or-complexity-limit*)))

#|

From Ken's EMail of Jun 25, 1996:

The 4.9 tagged sentences (untagged)
                    AVG           STDDEV           MAX
CONS              3,785          20,033         441,322
OR                  235           1,926          59,008


The 4.9/4.10 corpus (untagged)
                    AVG           STDDEV           MAX
CONS              1,882          12,267         441,322
OR                  106           1,089          59,008

EHN 6/25:

400,000 = about 32 STDDEV conses using the second number
 32,000 = about 32 STDDEV ORs using the second number

|#

(defun test-complexity (&optional (s *killer-sentence*)) 
  (let (value status conses ors) 
    (let ((*out* nil)
	  (*standard-output* *dev-null-stream*) 
	  (*max-ambiguity-display* 0)) 
      (multiple-value-setq (value status) 
	(test-sentence-1 s))) 
    (format t "~%Value: ~s Status: ~s" value status) 
    (when value (multiple-value-setq (conses ors)
		  (raw-complexity *parse-value*)) 
	  (format t "~%Cons: ~s OR: ~s" conses ors)) 
    (values value status)))

(defun explode-complexity (kernel increment)
  (let (; e.g., (kernel "remove engines")
	;       (increment "in trucks that are driven")
	(n 0)
	current value status)
    (loop
      (incf n)
      (setq current kernel)
      (dotimes (i n)
	(setq current (format nil "~a ~a" current increment)))
      (format t "~%[~s] ~a." n current)
      (multiple-value-setq (value status)
	(test-complexity (format nil "~a." current)))
      (when (or (null value)
		(and (listp value)
		     (equal "COMPLEXITY" (second value))))
	(if (and (listp value)
		     (equal "COMPLEXITY" (second value)))
	    (format t "~%Killer Cons: ~s Killer OR: ~s"
		    (count-conses *parse-value*)
		    (apply #'+ (mapcar #'or-complexity *parse-value*)))
	  (warn "Died on non-COMPLEXITY!"))
	(return)))))
    
    
;; 27-Jun-96 by EHN -- Make this a gentle no-op so the grammar doesn't do
;; extra work; in the next release, make sure that the callouts to PHRASAL-SCORE
;; get eliminated from the grammar.

;; 23-Jul-96 by EHN -- commented out, assume callouts are clean for 4.11
;; (defun phrasal-score (fs) fs)