(in-package 'user)

;;----------------------------------------------------------------------
;; XCHART 1.0
;;
;; File: FUNCTIONS.LISP
;;
;; Last Update: 14-Oct-96 by EHN
;;
;; Copyright (c) 1996, Carnegie Mellon University, All Rights Reserved.

;;----------------------------------------------------------------------
;; COMPGRA

(defun compgra (x)
  (let ((fullname (concatenate 'string x ".gra"))
	(outfile (concatenate 'string x ".lisp"))
	(*print-pretty* nil)
	(count 0)
	fn fnbody rule level strategy)
    (with-open-file (out outfile :direction :output :if-exists :supersede)
		    (format out "(DEFVAR *GRAMMAR*)~%(SETQ *GRAMMAR* NIL)~%")
		    (with-open-file (in fullname :direction :input)
				    (loop
				     (setq rule (read in nil in))
				     (when (eq rule in)
					   (return))
				     (incf count)
				     (setq fn (format nil "GRAMMARF-~a" count))
				     (setq fnbody
					   (compile-augmentation (length (fourth rule))
								 (fifth rule)))
				     (setf (first fnbody) fn)
				     (setf fnbody (cons 'defun fnbody))
				     (format out "~a~%" fnbody)
				     (cond ((listp (first rule))
					    (setq level (first (first rule)))
					    (setq strategy (second (first rule))))
					   (t (setq level (first rule))))
				     (format out "(SETQ *GRAMMAR* (NCONC *GRAMMAR* (LIST (MAKE-RULE :STRATEGY ~s :LEVEL ~s :LHS '~s :RHS '~s :UNIFY '~a))))~%"
					     strategy level (second rule) (fourth rule) fn)
				     )))))

;;----------------------------------------------------------------------
;; LOOKUP-WORD
;;
;; Lexical lookup function
;;
;; Given a word (a Lisp symbol from the parser's input), it returns a
;; list of f-structures denoting the lexical readings for the word. If
;; the word doesn't have an entry, tries to build a default lexical
;; node based on the LISP object type (this is how integers are handled).

(defun lookup-word (word) 
  (or (rest (assoc word *lexicon*))
      (lookup-default word)))

(defun lookup-default (word)
  (let ((type (type-of word)))
    (chart-trace "~% ~20a Reading ~a not in lexicon: ~a"
		 'WILDCARD type word)
    `(((cat ,type)(root ,word)))))

;;----------------------------------------------------------------------
;; CTRACE
;;
;; Set trace vars.

(defun ctrace (val)
  (case val
    (0 (setf *print-parse-tree* nil
	     *print-parse-fs* nil
	     *chart-trace* nil))
    (1 (setf *print-parse-tree* 1
	     *print-parse-fs* nil
	     *chart-trace* nil))
    (2 (setf *print-parse-tree* nil
	     *print-parse-fs* 1
	     *chart-trace* nil))
    (3 (setf *print-parse-tree* nil
	     *print-parse-fs* 1
	     *chart-trace* 1))
    (t (warn "CTRACE levels are 0-3.")))
  val)

;;----------------------------------------------------------------------
;; RULES-STARTED-BY
;;
;; Function which returns a list of the grammar rules which are
;; started by the given constituent.

(defun rules-started-by (constituent highest-level)
  (remove-if-not #'(lambda (rule)
		     (and
		      ;; Level is low enough
		      (<= (rule-level rule) highest-level)
		      ;; Constituent matches first element of RHS,
		      ;; or latter is wildcard
		      (or (eq *wildcard* (first (rule-rhs rule)))
			  (eq (first (rule-rhs rule)) constituent))
		      ;; There are other elements in the RHS to be found
		      (rest (rule-rhs rule))))
		 *grammar*))

;;----------------------------------------------------------------------
;; RULES-COMPLETED-BY
;;
;; Function which returns a list of the grammar rules which are
;; completed by the given constituent (only relevant for rules with a
;; single category in the RHS).

(defun rules-completed-by (constituent highest-level)
  (remove-if-not #'(lambda (rule)
		     (and
		      ;; Level is low enough
		      (<= (rule-level rule) highest-level)
		      ;; Constituent matches first element of RHS, or
		      ;; latter is wildcard
		      (or (eq *wildcard* (first (rule-rhs rule)))
			  (eq (first (rule-rhs rule)) constituent))
		      ;; There are no other elements in the RHS to find
		      (null (rest (rule-rhs rule)))))
		 *grammar*))

;;----------------------------------------------------------------------
;; ARCS-CONTINUED-BY
;;
;; Function which returns a list of the active arcs ending at END
;; which are continued by the given constituent.

(defun arcs-continued-by (end constituent arcs)
  (remove-if-not #'(lambda (arc)
		     (and 
		      ;; Current end point of the arc matches the end
		      ;; point of the constituent.
		      (eq (arc-end arc) end)
		      ;; Constituent matches the next RHS symbol on
		      ;; the arc, or the latter is a wildcard.
		      (or (eq *wildcard*
			      (nth (arc-index arc) (arc-rhs arc)))
			  (eq (nth (arc-index arc) (arc-rhs arc))
			      constituent))
		      ;; There are more symbols on the arc RHS (so
		      ;; we're not completing the arc yet)
		      (nthcdr (1+ (arc-index arc)) (arc-rhs arc))))
		 arcs))

;;----------------------------------------------------------------------
;; ARCS-COMPLETED-BY
;;
;; Function which returns a list of the active arcs ending at END
;; which are continued by the given constituent.

(defun arcs-completed-by (end constituent arcs)
  (remove-if-not #'(lambda (arc)
		     (and 
		      ;; Current end point of the arc matches the end
		      ;; point of the constituent.
		      (eq (arc-end arc) end)
		      ;; Constitutent matches the next RHS symbol on
		      ;; the arc, or latter is a wildcard.
		      (or (eq *wildcard* (nth (arc-index arc) (arc-rhs arc)))
			  (eq (nth (arc-index arc) (arc-rhs arc)) 
			      constituent))
		      ;; There are no more symbols on the arc RHS, so
		      ;; we're completing the arc.
		      (null (nthcdr (1+ (arc-index arc)) (arc-rhs arc)))))
		 arcs))

;;----------------------------------------------------------------------
;; INSERT-INDEX
;;
;; Function to insert '*' into RHS to indicate index position for
;; rules or arcs.
;;
;; For printing purposes only -- equivalent to Allen's circle mark.

(defun insert-index (n list)
  (append (subseq list 0 n) (list '*) (subseq list n)))

;;----------------------------------------------------------------------
;; PARSE
;;
;; The top-level parser call.
;;
;; INPUT: a list of input symbols, which must be in *LEXICON*
;; GRAMMAR: a list of rule structures, such as *GRAMMAR*
;; SEARCHMODE: One of 'dfs (depth-first-search), 'bfs (breadth-first-search)
;; QUITMODE: if non-NIL, causes parser to quit as soon as it finds the
;;   first chart entry that covers the whole input.

;;  1-Apr-97 by EHN -- rewrote arguments to keyword rather than
;; optional, gave them shorter names.

(defun parse (input &key
		    (search *default-searchmode*)
		    (quit *default-quitmode*)
		    (level 999))
  (let* ((position 1)	
	 (final-position (1+ (length input)))
	 (operations 0)
	 (node-counter 0)
	 (parse-count 0)
	 nodeid nodefs chart agenda new-arc arcs word entry constituent start end
	 arc-continues arc-ends rule-starts rule-ends
	 item)
    (setf *parse-value* nil *chart* nil *global-counter* 0)
    (loop
     (when (null agenda)
	   (cond ((null input)
		  (chart-trace "~% ~20a No input, quitting [~a operations]" item operations)
		  (setq *chart* chart)
		  (setq *arcs* arcs)
		  (setq *parse-value* (parse-trees final-position chart))
		  (cond (*print-parse-tree*
			 (cond (*parse-value*
				(when *chart-trace* (terpri))
				(dolist (parse *parse-value* (terpri))
					(format t "~%Parse[~a]:" (incf parse-count))
					(pprint parse)
					(terpri)))
			       (t (format t "~%[Chart contains no full parses, looking for minimal cover.]~%")
				  (dolist (entry (minimal-cover (top-level-entries chart)))
				    (format t "~%FS[~a,~a,~a]:~%"
					    (entry-category entry)
					    (entry-start entry) (entry-end entry)
					    )
				    (pns (entry-fs entry))))))
			(*print-parse-fs*
			 (parse-fs final-position chart)))
		  (return (length *parse-value*)))
		 (t
		  (chart-trace "~% ~20a Reading from input ~a" 'EMPTY input)
		  (incf position)
		  (dolist (reading (lookup-word (setq word (pop input))))
			  (setq entry (make-entry 
				       :start    (1- position) 
				       :end      position
				       :category (second (assoc 'cat reading))
				       :id   (incf node-counter)
				       :lexeme   word
				       :fs reading
				       ))
			  (chart-trace "~% ~20a Entering ~a (~a from ~a to ~a) [agenda ~a]" 
				       'EMPTY (entry-category entry) (entry-lexeme entry)
				       (entry-start entry) (entry-end entry) 
				       (if (eq search 'dfs) "top" "bottom")
				       )
			  (incf operations)
			  (if (eq search 'dfs)
			      (push entry agenda)
			    (setq agenda (nconc agenda (list entry))))))))
     (incf *global-counter*)
     (when *trace-arcs*
	   (dump-arcs arcs))
     (setq item (pop agenda))
     (setf start (entry-start item)
	   end (entry-end item)
	   constituent (entry-category item)
	   nodeid (entry-id item)
	   nodefs (entry-fs item))
     ;; As soon as we've popped off the next item in the agenda, check
     ;; to see if it's a final node; quit if we're supposed to, before
     ;; performing any additional operations, but make sure we put the
     ;; item onto the chart before we quit.
     (when (and (= start 1)
		(= end final-position))
	   (cond (quit
		  (chart-trace "~% ~20a ** Covers the input, quitting [~a operations]" 
			       item operations)
		  (push item chart)
		  (setq *chart* chart)
		  (setq *parse-value* (parse-trees final-position chart))
		  (cond (*print-parse-tree*
			(cond (*parse-value*
			       (when *chart-trace* (terpri))
			       (dolist (parse *parse-value* (terpri))
				       (format t "~%Parse[~a]:" (incf parse-count))
				       (pprint parse)
				       (terpri)))
			      (t (format t "~%[Chart contains no full parses.]~%"))))
			(*print-parse-fs*
			 (parse-fs final-position chart)))
		  (return (length *parse-value*)))
		 (t
		  (chart-trace "~% ~20a ** Covers the input, continuing...[~a operations]" 
			       item operations)
		  )))
     (setq rule-starts (rules-started-by constituent level))
     (chart-trace "~% ~20a Rules started-by: ~a" item (length rule-starts))
     (dolist (rule rule-starts)
	     (setq new-arc (make-arc 
			  :start start
			  :end end
			  :index 1 
			  :lhs (rule-lhs rule) 
			  :rhs (rule-rhs rule) 
			  :children (list nodeid)
			  :unify (rule-unify rule)
			  :strategy (rule-strategy rule)
			  ))
	     (chart-trace "~% ~20a Starts rule ~a -> ~a from ~a to ~a" 
			  item (rule-lhs rule) (insert-index 1 (rule-rhs rule)) start end)
	     (incf operations)
	     (if (eq search 'dfs)
		 (push new-arc arcs)
	       (setq arcs (nconc arcs (list new-arc)))))
     (setq rule-ends (rules-completed-by constituent level))
     (chart-trace "~% ~20a Rules completed-by: ~a" item (length rule-ends))
     (dolist (rule rule-ends)
             (let* (; (child-fs nil)
		    (last-fs nodefs)
		    (fs (funcall (rule-unify rule) last-fs)))
	       (setq entry (make-entry
			    :start start
			    :end end
			    :category (rule-lhs rule) 
			    :id (incf node-counter) 
			    :children (list nodeid)
			    :fs fs))
	       (cond (fs
		      (chart-trace "~% ~20a Completes rule ~a -> ~a from ~a to ~a" 
				   item (rule-lhs rule) (insert-index 1 (rule-rhs rule)) start end)
		      (chart-trace "~% ~20a Entering ~a from ~a to ~a [agenda ~a]" 
				   item (rule-lhs rule) start end
				   (if (eq search 'dfs) "top" "bottom")
				   )
		      (incf operations)
		      (if (or (eq (rule-strategy rule) :dfs)
			      (and (not (rule-strategy rule)) (eq search 'dfs)))
			  (push entry agenda)
			(setq agenda (nconc agenda (list entry)))))
		     (t 
		      (chart-trace "~% ~20a Completes rule ~a -> ~a from ~a to ~a **UNIFY FAILED**" 
				   item (rule-lhs rule) (insert-index 1 (rule-rhs rule)) start end)))
	     ))
     (incf operations)
     ;; Always push processed entries onto the front of the chart.
     (push item chart)
     (setq arc-continues (arcs-continued-by start constituent arcs))
     (chart-trace "~% ~20a Arcs continued-by: ~a" item (length arc-continues))
     (dolist (arc arc-continues)
	     (setq new-arc (make-arc
			    :start (arc-start arc) 
			    :end (entry-end item) ;; 11-Oct-96 by EHN (1+ (arc-end arc)) 
			    :lhs (arc-lhs arc) 
			    :rhs (arc-rhs arc)
			    :index (1+ (arc-index arc))
			    :children (append (arc-children arc)
					      (list nodeid))
			    :unify (arc-unify arc)
			    :strategy (arc-strategy arc)
			    ))
	     (chart-trace "~% ~20a Continues arc ~a -> ~a from ~a to ~a"
			  item (arc-lhs arc) 
			  (insert-index (1+ (arc-index arc)) (arc-rhs arc)) 
			  (arc-start arc) (arc-end arc))
	     (incf operations)
	     (if (or (eq (arc-strategy arc) :dfs)
		     (and (not (arc-strategy arc)) (eq search 'dfs)))
		 (push new-arc arcs)
	       (setq arcs (nconc arcs (list new-arc))))
	     )
     (setq arc-ends (arcs-completed-by start constituent arcs))
     (chart-trace "~% ~20a Arcs completed-by: ~a" item (length arc-ends))
     (dolist (arc arc-ends)
             (let* ((child-fs 
		     (mapcar #'(lambda (x)
				 (entry-fs (find x chart :key #'entry-id)))
			     (arc-children arc)))
		    (last-fs nodefs)
		    (fs (apply (arc-unify arc)
			       (append child-fs (list last-fs)))))
	     (cond ((not (or (eq '*fail* fs) (null fs)))
		    (setq entry (make-entry 
				 :start (arc-start arc) 
				 :end end
				 :category 
				 (let ((lhs (arc-lhs arc)) wpos)
				   (when (eq *wildcard* lhs)
					 (setq wpos (position *wildcard* (arc-rhs arc)))
					 (setq lhs (entry-category (find (nth wpos (arc-children arc))
									  chart :key #'entry-id))))
				   lhs)
				 :id (incf node-counter)
				 :children (append (arc-children arc) 
						   (list nodeid))
				 :fs fs))
		    (chart-trace "~% ~20a Completes arc ~a -> ~a from ~a to ~a"
				 item (arc-lhs arc) 
				 (insert-index (arc-index arc) (arc-rhs arc))
				 (arc-start arc) (arc-end arc))
		    (chart-trace "~% ~20a Entering ~a from ~a to ~a [agenda ~a]" 
				 item (arc-lhs arc) (arc-start arc) end
				 (if (eq search 'dfs) "top" "bottom"))
		    (incf operations)
		    (if (or (eq (arc-strategy arc) :dfs)
			    (and (not (arc-strategy arc))(eq search 'dfs)))
			(push entry agenda)
		      (setq agenda (nconc agenda (list entry)))))
		   (t (chart-trace "~% ~20a Completes arc ~a -> ~a from ~a to ~a **~a FAILED**"
				   item (arc-lhs arc)
				 (insert-index (arc-index arc) (arc-rhs arc))
				 (arc-start arc) (arc-end arc) (arc-unify arc)
				   ))))))))

;;----------------------------------------------------------------------
;; PARSE-TREES
;;
;; Function to build trees for the nodes that cover the input.

(defun parse-trees (final-position chart)
  (let ((parses (remove-if-not #'(lambda (x)
				   (and (= 1 (entry-start x))
					(= final-position (entry-end x))))
			       chart)))
    (if parses
	(mapcar #'(lambda (x)
		    (parse-tree x chart))
		parses)
      nil)))

;;----------------------------------------------------------------------
;; PARSE-FS

(defun parse-fs (final-position chart)
  (let ((parses (remove-if-not #'(lambda (x)
				   (and (= 1 (entry-start x))
					(= final-position (entry-end x))))
			       chart))
	(count 0))
    (setq *parse-fs* nil)
    (dolist (fs parses (terpri))
      (format t "~%FS[~a,~a]:~%" (incf count) (entry-category fs))
      (pns (entry-fs fs))
      (setq *parse-fs* (nconc *parse-fs* (list (entry-fs fs))))
      (terpri))))

;;----------------------------------------------------------------------
;; PARSE-TREE
;;
;; Function to create the tree structure for a single node; recursive.

(defun parse-tree (node chart)
  (let ((children (entry-children node))
	(lexeme (entry-lexeme node)))
    (cond (lexeme (list (entry-category node) lexeme))
	  (t (setq children (mapcar #'(lambda (x)
					(find x chart :key #'entry-id))
				    children))
	     (cons (entry-category node) 
		   (mapcar #'(lambda (x)
			       (parse-tree x chart))
			   children))))))

;;----------------------------------------------------------------------
;; TOP-LEVEL-ENTRIES
;;
;; Subset of the chart s.t. the node id doesn't appear in any child lists.

(defun top-level-entries (chart)
  (remove-if #'(lambda (entry)
		 (find (entry-id entry) chart
		       :key #'entry-children
		       :test #'member))
	     chart))

;;----------------------------------------------------------------------
;; MINIMAL-COVER
;;
;; Subset of top nodes, throw out those who span a subrange covered by
;; another.

(defun minimal-cover (entries)
  (remove-if #'(lambda (entry)
		 (some #'(lambda (x)
			   (and 
			    (>= (entry-end x)
				(entry-end entry))
			    (<= (entry-start x)
				(entry-start entry))
			    (eq (entry-category x)
				(entry-category entry))
			    (not (equal x entry))))
		       entries))
	     entries))


;;----------------------------------------------------------------------

(defun dump-chart ()
  (dolist (entry *chart*)
	  (format t "~%Entry: ~s~%FS: ~s~%" entry (entry-fs entry))))

;;----------------------------------------------------------------------

(defun dump-arcs (arcs)
  (chart-trace "~%----------[Round ~a]------------------------------------------------------------" *global-counter*)
  (dolist (arc arcs)
	  (chart-trace "~% ~20a ~a -> ~a from ~a to ~a"
		       'ACTIVE-ARC (arc-lhs arc) 
		       (insert-index (arc-index arc) (arc-rhs arc)) 
		       (arc-start arc) (arc-end arc))))