;;; Grammar callouts
;;;-----------------


(in-package 'user)


;;; returns the f-structure for the number num

;;; Example: (check-if-number '|1|):
;;;          ((NUMBER SG) 
;;;           (INTEGER "1") 
;;;           (SEM *C-DECIMAL-NUMBER) 
;;;           (CAT NUMBER))

;;;          (check-if-number '|1/2|):
;;;          ((CAT NUMBER) 
;;;           (SEM *C-FRACTION) 
;;;           (FRACTION +) 
;;;           (NUMBER (*OR* SG PL)) 
;;;           (NUMERATOR ((INTEGER "1") 
;;;                       (SEM *C-DECIMAL-NUMBER))) 
;;;           (DENOMINATOR ((INTEGER "2") 
;;;                         (SEM *C-DECIMAL-NUMBER))))

;;;          (check-if-number '|1.2|):
;;;          ((CAT NUMBER)
;;;           (NUMBER PL)
;;;           (SEM *C-DECIMAL-NUMBER)
;;;           (INTEGER "1")
;;;           (DECIMAL "2"))

(defun check-if-number (num)
  (let ((val (read-from-string (symbol-name num))))
    (cond ((integerp val)
	   (let ((out '((sem *c-decimal-number) (cat number))))
	     (if (= 1 val)
		 (progn 
		   (push '(integer "1") out)
		   (push '(number sg) out))
	       (progn
		 (push `(integer ,(format nil "~a" val)) out)
		 (push '(number pl) out)))))
	  ((ratiop val)
	   `((cat number) 
	     (sem *C-FRACTION) 
	     (fraction +)
	     (number (*OR* sg pl))
	     (numerator ((integer ,(format nil "~a" (numerator val)))
			 (sem *c-decimal-number)))
	     (denominator ((integer ,(format nil "~a" (denominator val)))
			   (sem *c-decimal-number)))))
	  ((floatp val)
	   (let* ((out '((sem *c-decimal-number) (cat number) 
			 (number pl)))
		  (str (format nil "~a" val))
		  (dot (position #\. str)))
	     (push `(integer ,(substring str 0 (- dot 1))) out)
	     (push `(decimal ,(substring str (+ 1 dot))) out))))))

(defun no-initial-adjuncts (list)
  (cond ((null list) t)
        ((eq '*multiple* (first list))
         (notany #'(lambda (x) (eq 'initial (second (assoc 'position x))))
                 (rest list)))
	((eq '*or* (first list))
	 t)
        (t (not (eq 'initial (second (assoc 'position list)))))))

(defun no-final-adjuncts (list)
  (cond ((eq '*multiple* (first list))
	 (if (notany #'(lambda (x) (eq 'final (second (assoc 'position x))))
		     (rest list))
	     list
	   '()))
	((eq '*or* (first list))
	 (let* ((out (remove nil (mapcar #'no-final-adjuncts (rest list))))
		(l (length out)))
	   (cond ((eq l 0) nil)
		 ((eq l 1) (first out))
		 (t `(*or* ,@out)))))
        (t (if (not (eq 'final (second (assoc 'position list))))
	       list
	     '()))))

;(defun no-same-adjuncts (root list)
;  (cond ((eq '*multiple* (first list))
;	 (if (notany #'(lambda (x) (equal root (second (assoc 'root x))))
;		     (rest list))
;	     list
;	   '()))
;	((eq '*or* (first list))
;	 (let* ((out (remove nil (mapcar #'no-same-adjuncts (list root) (rest list))))
;		(l (length out)))
;	   (cond ((eq l 0) nil)
;		 ((eq l 1) (first out))
;		 (t `(*or* ,@out)))))
;        (t (if (not (equal root (second (assoc 'root list))))
;	       list
;	     '()))))

(defun no-same-adjuncts (root list)
  (cond ((eq '*multiple* (first list))
	 (if (notany #'(lambda (x) (equal root (second (assoc 'root x))))
		     (rest list))
	     list
	   '()))
	((eq '*or* (first list))
	 (let* ((out (remove nil (mapcar #'(lambda (x) (no-same-adjuncts root x)) (rest list))))
		(l (length out)))
	   (cond ((eq l 0) nil)
		 ((eq l 1) (first out))
		 (t `(*or* ,@out)))))
        (t (if (not (equal root (second (assoc 'root list))))
	       list
	     '()))))

(defun no-time-adjuncts (list)
  (cond ((null list) t)
        ((eq '*multiple* (first list))
         (notany #'(lambda (x) (eq '+ (second (assoc 'time x))))
                 (rest list)))
	((eq '*or* (first list))
	 t)
        (t (not (eq '+ (second (assoc 'time list)))))))



;;;
;;;
;;; (valency subject) is not processed explicitly; the function returns nil
;;; which is expanded by the grammar rule;
;;;
;;;

(defun process-single-valency (valency)
  (case valency
	(subject '((valency subject)))
	(subject+object '((valency subject+object)
			  (object ((cat (*or* n number))))))
	(subject+indobject+object '((valency subject+indobject+object)
				    (object ((cat (*or* n number))))
				    (indobject ((cat n)))))
	(subject+object+object2 '((valency subject+object+object2)
				  (object ((cat (*or* n number))))
				  (object2 ((cat (*or* n number))))))
	(subject+object+oblique '((valency subject+object+oblique)
				  (object ((cat (*or* n number))))
				  (oblique ((cat p)))))
	(subject+object+predicate '((valency subject+object+predicate)
				    (object ((cat (*or* n number))))
				    (predicate ((cat (*or* adj adv))))))
	(subject+object+compl '((valency subject+object+compl)
				(compl ((cat v)))
				(object ((cat (*or* n number))))))
	(subject+object+oblique+oblique2 '((valency subject+object+oblique+oblique2)
					   (object ((cat (*or* n number))))
					   (oblique ((cat p)))
					   (oblique2 ((cat p)))))
	(subject+oblique '((valency subject+oblique)
			   (oblique ((cat p)))))
	(subject+oblique+compl '((valency subject+oblique+compl)
				 (oblique ((cat p)))
				 (compl ((cat v)))))
	(subject+predicate '((valency subject+predicate)
			     (predicate ((cat (*or* adv adj))))))
	(subject+oblique+predicate '((valency subject+oblique+predicate)
				     (oblique ((cat p)))
				     (predicate ((cat *or* adv adj)))))
	(subject+compl '((valency subject+compl)
			 (compl ((cat v)))))))

(defun process-valency (valency)
  (if (listp valency)
      `(*OR* ,@(mapcar #'process-single-valency (rest valency)))
    (process-single-valency valency)))
	

;;;
;;;  PARSE-ENG-MORPH analyzes morphology of a given word,
;;;  without consulting the dictionary.
;;;
(defun parse-eng-morph (word &aux result-list rev-word)
  
  (setq rev-word (reverse (coerce word 'list)))
  
  ;;
  ;;   Verb +S   and Noun +S
  ;;
  (when (and (< 3 (length rev-word))
	     (eq (first rev-word) #\s)
  	     (not (member (second rev-word)
			  '(#\s #\u #\z))))
	(push (list (coerce (reverse (cdr rev-word)) 'string)
		    '+S)
	      result-list))
  
  (when (and (< 3 (length rev-word))
  	     (eq (first rev-word) #\s)
  	     (eq (second rev-word) #\e)
	     (member (third rev-word) '(#\s #\z #\h #\o)))
	(push (list (coerce (reverse (cddr rev-word)) 'string)
		    '+S)
	      result-list))
  
  (when (and (< 4 (length rev-word))
  	     (eq (first rev-word) #\s)
  	     (eq (second rev-word) #\e)
  	     (eq (third rev-word) #\v))
	(push (list (coerce (reverse (cons #\f (cdddr rev-word))) 'string)
		    '+S)
	      result-list))
  
  (when (and (< 4 (length rev-word))
  	     (eq (first rev-word) #\s)
  	     (eq (second rev-word) #\e)
  	     (eq (third rev-word) #\i)
	     (not (member (fourth rev-word) '(#\a #\i #\u #\e #\o))))
	(push (list (coerce (reverse (cons #\y (cdddr rev-word))) 'string)
		    '+S)
	      result-list))
  
  ;;
  ;;   Adjective/Adverb +ER
  ;; 
  
  
  (when (and (< 3 (length rev-word))
	     (eq (first rev-word) #\r)
	     (eq (second rev-word) #\e))
	(push (list (coerce (reverse (cddr rev-word)) 'string)
		    '+ER)
	      result-list)
	(if (eq (third rev-word) (fourth rev-word))
	    (push (list (coerce (reverse (cdddr rev-word)) 'string)
			'+ER)
		  result-list)
	  (push (list (coerce (reverse (cdr rev-word)) 'string)
		      '+ER)
		result-list))
	(if (eq (third rev-word) #\i)
	    (push (list (coerce (reverse (cons #\y (cdddr rev-word))) 'string)
			'+ER)
		  result-list)))


  ;;
  ;; Adjective/Adverb +EST
  ;;
  
  (when (and (< 4 (length rev-word))
	     (eq (first rev-word) #\t)
	     (eq (second rev-word) #\s)
	     (eq (third rev-word) #\e))
	(push (list (coerce (reverse (cdddr rev-word)) 'string)
		    '+EST)
	      result-list)
	(if (eq (fifth rev-word) (fourth rev-word))
	    (push (list (coerce (reverse (cddddr rev-word)) 'string)
			'+EST)
		  result-list)
	  (push (list (coerce (reverse (cddr rev-word)) 'string)
		      '+EST)
		result-list))
	(if (eq (fourth rev-word) #\i)
	    (push (list (coerce (reverse (cons #\y (cddddr rev-word))) 'string)
			'+EST)
		  result-list)))

  ;;
  ;;   Verb +ED
  ;;
  (when (and (< 3 (length rev-word))
  	     (eq (first rev-word) #\d)
  	     (eq (second rev-word) #\e))
	(push (list (coerce (reverse (cddr rev-word)) 'string)
		    '+ED)
	      result-list)
	(if (eq (third rev-word)(fourth rev-word))
	    (push (list (coerce (reverse (cdddr rev-word)) 'string)
			'+ED)
		  result-list)
	  (push (list (coerce (reverse (cdr rev-word)) 'string)
		      '+ED)
		result-list))
	(if (and (eq (third rev-word) #\k) (eq (fourth rev-word) #\c))
	    (push (list (coerce (reverse (cdddr rev-word)) 'string) '+ED)
		  result-list))
	(if (eq (third rev-word) #\i)
	    (push (list (coerce (reverse (cons #\y (cdddr rev-word))) 'string)
			'+ED)
		  result-list)))
  
  ;;
  ;;   Verb +ING
  ;;
  (when (and (< 4 (length rev-word))
  	     (eq (first rev-word) #\g)
  	     (eq (second rev-word) #\n)
	     (eq (third rev-word) #\i))
	(push (list (coerce (reverse (cdddr rev-word)) 'string)
		    '+ING)
	      result-list)
	(if (eq (fourth rev-word)(fifth rev-word))
	    (push (list (coerce (reverse (cddddr rev-word)) 'string)
			'+ING)
		  result-list)
	  (push (list (coerce (reverse (cons #\e (cdddr rev-word))) 'string)
		      '+ING)
		result-list))
	(if (and (eq (fourth rev-word) #\k) (eq (fifth rev-word) #\c))
	    (push (list (coerce (reverse (cddddr rev-word)) 'string) '+ING)
		  result-list))
	(if (eq (fourth rev-word) #\y)
	    (push (list (coerce (reverse `(#\e #\i .,(cddddr rev-word))) 'string)
			'+ING)
		  result-list)))
  
  (cons (list word '+NIL) result-list))	;;; return from MORPH-ENG
  

;;;==================================================================;

;;; This function parses the PI tag and creates an f-structure by pulling
;;; out the relevant information in the tag

(defun parse-cte-tag (object)
  (let* ((string (string object))
	 (len (length string))
	 result ortho sem head modi val dump head-pos stem result eval)
    (when (and string
	       (stringp string)
	       ;; new, old, string
	       ;; 10-Jun-96 by EHN -- fix bug when *percent* is a head.
	       (setf string (replace-string "%%PERCENT%%" "*PERCENT*" string))
	       (setf string (replace-string "_" "__" string))	       
	       (setf string (replace-string " " "_" string))
	       (multiple-value-setq (val result)
		 (get-next-token string #\{))
	       (multiple-value-setq (val result)
		 (get-next-token result))
	       (string-equal val "?CTE"))
      (cond ((string-equal (multiple-value-setq (val result) (get-next-token result))
			   "means")
	     (dotimes (i 2)
	       (cond ((string-equal (multiple-value-setq (val result)
				      (get-next-token result #\=))
				    "text")
		      (multiple-value-setq (dump result) (get-next-token result))
		      (multiple-value-setq (ortho result) (get-next-token result #\*))
		      (multiple-value-setq (dump result) (get-next-token result)))
		     ((string-equal val "val")
		      (multiple-value-setq (dump result) (get-next-token result))
		      (multiple-value-setq (sem result) (get-next-token result))
		      (multiple-value-setq (dump result) (get-next-token result))))))

	    ((string-equal val "eval")
	     (multiple-value-setq (dump result) (get-next-token result #\{))
	     (multiple-value-setq (eval result) (get-next-token result #\}))
	     )	    
			    
	    ((string-equal val "attach")
	     (dotimes (i 3)
	       (cond ((string-equal (multiple-value-setq (val result)
				      (get-next-token result #\=))
				    "head")
		      (multiple-value-setq (dump result) (get-next-token result)) 
		      (multiple-value-setq (head result) (get-next-token result #\*))
		      (multiple-value-setq (dump result) (get-next-token result)))
		     ((string-equal val "head-pos")
		      (multiple-value-setq (dump result) (get-next-token result))
		      (multiple-value-setq (head-pos result) (get-next-token result #\*))
		      (multiple-value-setq (dump result) (get-next-token result)))
		     ((string-equal val "modi")
		      (multiple-value-setq (dump result) (get-next-token result))
		      (multiple-value-setq (modi result) (get-next-token result #\*))
		      (multiple-value-setq (dump result) (get-next-token result)))
		     (t nil))))))

    ;; format the appropriate f-structures
    (cond (sem
	   (if (stringp sem)
	       `((cat cte-means)
		 (ortho ,(string-downcase ortho))
		 (sem ,(read-from-string sem)))
	     `((cat cte-means)
	       (ortho ,(string-downcase ortho))
	       (sem ,sem))))
	  (eval
	   `((cat cte-eval)
		 (eval ,(string-downcase eval))))
	  (head
	   ;;; the structural PI tag is position dependent but since
	   ;;; the location of attachment sites is unknown apriori and could
	   ;;; be in any order, the position information in the PI tag
	   ;;; is computed so that no new structural PI tags are included
	   ;;; in the position information.  This is handled by the 
	   ;;; disambiguator by presenting structural ambiguities in a
	   ;;; right to left order.  Then when these tags are parsed here
	   ;;; the location of each tag is recorded while going left to
	   ;;; right during the parse and the position information
	   ;;; is adjusted to include the preceding PI tags via the function
	   ;;; tag-count.  Positions of structural PI tags is recorded
	   ;;; in the hash table *cte-tag-counter*

	   ;; 31-Aug-94 by EHN - fix LABEL problem
	   ;;  7-Sep-94 by EHN - extend to deal with "-TAG" mark.
	   ;; 7/20/95-igo: need to change references to *disambiguator-non-string-roots*
	   ;; since this var will be no longer supported in the disambiguator.
	   (when (stringp head)
					;(format t "~%~%HEAD: ~s" head)
	     ;; 10-Jun-96 by EHN -- fix bug when *percent* is a head.
	     (setf head (replace-string "*PERCENT*" "%%PERCENT%%" head))
	     (if (and (> (length head) 4)
		      (equal "-TAG" (substring head -4))
		      (setq stem (substring head 0 -5))
		      (find stem *disambiguator-non-string-roots*
			    :key #'symbol-name :test #'equal))
		 (setq head (read-from-string stem))
	       (setq head (string-downcase head))))
		   
	   (setf result
		 (cond ((and (or (stringp head)
				 (find head *disambiguator-non-string-roots*))
			     head-pos)
			`((cat cte-attach)
			  (head ,head)
			  (head-pos ,(tag-count (read-from-string head-pos)))
			  (modi ,(string-downcase modi))))
		       ((or (stringp head)
			    (find head *disambiguator-non-string-roots*))
			`((cat cte-attach)
			  (head ,head)
			  (modi ,(string-downcase modi))))
		       (head-pos
			`((cat cte-attach)
			  (head ,head)
			  (head-pos ,(read-from-string head-pos))
			  (modi ,(string-downcase modi))))
		       (t
			`((cat cte-attach)
			  (head ,head)
			  (modi ,(string-downcase modi))))))
	   (setf (gethash *token-position* *cte-tag-counter*) 1)
	   result))))



(defun get-next-token (string &optional (delimiter #\Space))
  (let (token result token-end (char #\>)
	      max-len values)
    (setf result (string-left-trim " " string))
    (setf max-len (length result))
    (setf token-end
	  (do ((i 0 (incf i)))
	      ((or (char= delimiter char) (= i max-len)) (1- i))
	    (setf char (char result i))))
    (setf token (string-right-trim " "
				   (subseq result 0 token-end)))
    (setf result (string-left-trim `(,delimiter)
				   (subseq result token-end)))
    (when (string-equal token "*not*")
      (setf token
	    (do ()
		((string-equal token "*apostrophe*") `(*not* ,(append '(*or*) values)))
	      (multiple-value-setq (token result) (get-next-token result))
	      (when (not (string-equal token "*apostrophe*"))
		(push (read-from-string token) values)))))
    (when (not (= (1+ token-end) max-len))
      (values token result))))

(defun set-form (form list)
  (let (out)
    (dolist (el (rest list))
	    (let ((fs-form (second (assoc 'FORM el))))
	      (cond ((eq fs-form form)
		     (push el out))
		    ((listp fs-form) 
		     (cond ((member form (rest fs-form))
			    (setq el1 (remove (assoc 'FORM el)
					     el :test #'equal))
			    (push `(FORM ,form) el1)
			    (push el1 out))
			   (t (push 'ERROR out))))
		    (t (push 'ERROR out)))))
    (if (member 'ERROR out)
	nil
      `(:multiple ,@(reverse out)))))
