#!/atm/release/fore/bin/sntclsh

##################################################################################
# This TCL script inserts rc.pvc atmarp entries into the CDB
#
# The following atmarp commands are supported:
# -s host dev vpi vci aal [encap] [peak](set permanent ARP entry for outgoing PVC)
# -l dev vpi vci aal [decap]            (attach IP to an incoming PVC)
# -c host dev vpi vci revalidate [peak] (establish Classical IP PVC)
# -m dev vpi vci aal [encap] [peak]     (set IP multicast default outgoing PVC)
# -n NSAPaddr dev                       (configure our NSAP address)
# -p NSAPaddr dev                       (put (set) ARP Server address)
##################################################################################


##################################################################################
# Open rc.pvc
# Read all atmarp entries into a list
# Close rc.pvc
##################################################################################
set filename /atm/etc/rc.pvc
set result [catch {set rc  [open $filename r]}]
if {$result == 1} {
    puts "rc2cdb: WARNING: Could not open \"$filename\""
    exit
}
set i 0
while {[gets $rc line] >= 0} {
    if {[llength $line] > 0 && [lindex $line 0] == "atmarp"} {
	set lines($i) $line
	incr i
    }
}
close $rc
if {$i == 0} {
# If list is empty, nothing left to do, exit
    exit
}


##################################################################################
# Open SNMP session on switch
# Wait for asxd to start accepting SNMP requests
##################################################################################
while {1} {
    set result [catch {snmp open s localhost}]
    if {$result == 0} {
	break
    }
}
while {1} {
    set result [catch {s switch get 0 -uptime}]
    if {$result == 0} {
	break
    }
}


##################################################################################
# Determine device indices
##################################################################################
set devices(aa0)  0
set devices(asx0) 0
set devices(le0)  0
set devices(lo0)  0
set devices(qaa0) 0
set devices(qaa1) 0
set devices(qaa2) 0
set devices(qaa3) 0
set devices(sbf0) 0
set result [catch {set device [s if list -ifDescr]}]
if {$result == 1} {
    puts "rc2cdb: ERROR: Could not list devices"
    exit
}
for {set j 0} {$j < [llength $device]} {incr j} {
    set dev [lindex $device $j]
    set devices([lindex $dev 1]) [lindex $dev 0]
}


##################################################################################
# Insert rc.pvc atmarp entries into cdb
##################################################################################
for {set j 0} {$j < $i} {incr j} {
    set line $lines($j)
    if {[llength $line] < 3} {
	continue
    }
    set flag [lindex $line 1]
    if {$flag == "-s"} {
	if {[llength $line] < 7} {
	    continue
	}
	set ip  [lindex $line 2]
	set dev [lindex $line 3]
	set vpi [lindex $line 4]
	set vci [lindex $line 5]
	set aal [lindex $line 6]
	for {set k 0} {$k < $i} {incr k} {
	    set l $lines($k)
	    if {[llength $l] >= 6 && [lindex $l 1] == "-l"} {
		set vp [lindex $l 3]
		set vc [lindex $l 4]
		if {$vp == $vpi && $vc == $vci} {
		    set lines($k) "dummy"
		    break
		}
	    }
	}
	if {$k == $i} {
	    puts "rc2cdb: WARNING: $line"
	    puts "\tNo matching \"atmarp -l\" found"
	    puts "\tCreating bidirectional FORE IP VC ($vpi,$vci) anyway"
	}
	if {$devices($dev) == 0} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tNo $dev device present on switch"
	    puts "\tNot creating FORE IP VC ($vpi,$vci)"
	    continue
	}
	if {$aal == 3 || $aal == 4} {
	    set aal 34
	}
	set result [catch {s atmarp foreip $ip $devices($dev) $vpi $vci aal$aal}]
	if {$result == 1} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tCould not create FORE IP VC ($vpi,$vci)"
	}
    } elseif {$flag == "-c"} {
	if {[llength $line] < 6} {
	    continue
	}
	set ip  [lindex $line 2]
	set dev [lindex $line 3]
	set vpi [lindex $line 4]
	set vci [lindex $line 5]
	if {$dev == "asx0"} {
	    set dev qaa0
	}
	if {$devices($dev) == 0} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tNo $dev device present on switch"
	    puts "\tNot creating classical IP VC ($vpi,$vci)"
	    continue
	}
	set result [catch {s atmarp classicalip $ip $devices($dev) $vpi $vci}]
	if {$result == 1} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tCould not create classical IP VC ($vpi,$vci)"
	}
    } elseif {$flag == "-m"} {
	if {[llength $line] < 5} {
	    continue
	}
	set dev [lindex $line 2]
	set vpi [lindex $line 3]
	set vci [lindex $line 4]
	if {$dev == "asx0"} {
	    set dev qaa0
	}
	if {$devices($dev) == 0} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tNo $dev device present on switch"
	    puts "\tNot creating classical IP multicast VC ($vpi,$vci)"
	    continue
	}
	set result [catch {s atmarp classicalip 224.0.0.0 $devices($dev) $vpi $vci}]
	if {$result == 1} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tCould not create classical IP multicast VC ($vpi,$vci)"
	}
    } elseif {$flag == "-p"} {
	if {[llength $line] < 4} {
	    continue
	}
	set nsap [lindex $line 2]
	set dev  [lindex $line 3]
	if {$devices($dev) == 0} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tNo $dev device present on switch"
	    puts "\tNot setting ARP server NSAP address"
	    continue
	}
	set result [catch {s arpserver set $devices($dev) 1 -classicalIpManualConfigArpServerAddr $nsap}]
	if {$result == 1} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tCould not set ARP server NSAP address"
	}
    } elseif {$flag == "-n"} {
	if {[llength $line] < 4} {
	    continue
	}
	set nsap [lindex $line 2]
	set dev  [lindex $line 3]
	if {$devices($dev) == 0} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tNo $dev device present on switch"
	    puts "\tNot setting my NSAP address"
	    continue
	}
	set result [catch {s mynsap set $devices($dev) $nsap}]
	if {$result == 1} {
	    puts "rc2cdb: ERROR: $line"
	    puts "\tCould not set my NSAP address"
	}
    }
}


##################################################################################
# Warn user of "atmarp -l" entries that were not matched
##################################################################################
for {set j 0} {$j < $i} {incr j} {
    set line $lines($j)
    if {[llength $line] > 1} {
	set flag [lindex $line 1]
	if {$flag == "-l"} {
	    if {[llength $line] < 5} {
		continue
	    }
	    set dev [lindex $line 2]
	    set vci [lindex $line 3]
	    set vpi [lindex $line 4]
	    puts "rc2cdb: Warning: $line"
	    puts "\tNo matching \"atmarp -s\" found"
	    puts "\tNot creating FORE IP VC ($vci,$vpi)"
	}
    }
}
