#include <types.h>
#include <sys/time.h>
#include <sys/systm.h>
#include <dev/pci/jetland/include/ijet_coding.h>
#include <dev/pci/jetland/include/ijet_jet.h>
#include <dev/pci/jetland/include/ijet_ll.h>
#include <dev/pci/jetland/include/ijet_sonet.h>
#include <dev/pci/jetland/include/ijet_sonet_internals.h>

/*
 * sonet.c: SONET Initialization and Interrupt
 *          handling stuff
 *
 * slightly hacked by kosak, mostly left alone
 */

private void jet_write_SONET (int unit, uint32 reg, uint32 value);
private uint32 jet_read_SONET (int unit, uint32 reg);

/* Initialization module */

public int jet_SonetInit (int unit)
begin (jet_SonetInit);
{
    /* Give a RESET to SONET chip */
    jet_write_SONET(unit, MASTER_RESET_REG, RESET);
    /* Take SONET out of RESET */
    jet_write_SONET(unit, MASTER_RESET_REG, 0);

    /* Enable Cell Delineation Interrupt / Disable Diagnostics Loopback */
    jet_write_SONET(unit, MASTER_CONTROL_REG, FIX_PTR|LCD_ENAB);

    /* Enable Transmit Reference Out of Lock Interrupt */
    jet_write_SONET(unit, CLOCK_SYN_REG, TROOL_ENAB);

    /* Enable Receive Data Out of Lock Interrupt */
    jet_write_SONET(unit, CLOCK_REC_REG, RDOOL_ENAB);

    /* Enable Out of Frame, Loss of Frame, and Loss of Signal Interrupts */
    /* Descrambling of the STS-3c stream is enabled */
    jet_write_SONET(unit, RSOP_CONTROL_REG, OOF_ENAB|LOF_ENAB|LOS_ENAB);

    /* Enable scrambling of the STS-3c stram */
    jet_write_SONET(unit, TSOP_CONTROL_REG, 0);

    /* Enable Far End Receive Failure and Line AIS Interrupts */
    jet_write_SONET(unit, RLOP_INTERRUPT_REG, RDI_ENAB|LAIS_ENAB);

    /* Enable Path Remote Defect, Path AIS, and Loss of Pointer Interrupts */
    jet_write_SONET(unit, RPOP_INT_ENAB_REG, PRDI_ENAB|PAIS_ENAB|LOP_ENAB);

    /* Enable Descrambling of Cell Payload and Coset Polynomial */
    /* Addition to the Receive HEC */
    jet_write_SONET(unit, RACP_CONTROL_REG, RHCSADD);

    /* Enable Out of Cell Delineation Interrupt */
    jet_write_SONET(unit, RACP_INTERRUPT_REG, OOCD_ENAB);

    /* Enable Scrambling of Cell Payload and Coset Polynomial */
    /* Addition to the Transmit HEC */
    jet_write_SONET(unit, TACP_CONTROL_REG, THCSADD);

    ret SUCCESS;
}
end (jet_SonetInit);


/* Turn on/off SONET LoopBack */
public int jet_SetSonetLoopback (int unit, bool loopback)
begin (jet_SetSonetLoopback);
{
    if (loopback)
        jet_write_SONET (unit, MASTER_CONTROL_REG,
                         jet_read_SONET(unit, MASTER_CONTROL_REG) | DL_ENAB);
    else
        jet_write_SONET (unit, MASTER_CONTROL_REG,
                         jet_read_SONET(unit, MASTER_CONTROL_REG) & ~DL_ENAB);

    ret SUCCESS;
}
end (jet_SetSonetLoopback);


/* num cells since last time */

public void jet_SonetNumCells (int unit, uint32 *numRxCells,
			       uint32 *numTxCells)
begin (jet_SonetNumCells);
{
    if (numRxCells)
        *numRxCells= jet_read_SONET(unit, RACP_RX_CELL) |
                     jet_read_SONET(unit, RACP_RX_CELL+1)<<8 |
                     (jet_read_SONET(unit, RACP_RX_CELL+2)&0x07)<<16;

    if (numTxCells)
        *numTxCells= jet_read_SONET(unit, TACP_TX_CELL) |
                     jet_read_SONET(unit, TACP_TX_CELL+1)<<8 |
                     (jet_read_SONET(unit, TACP_TX_CELL+2)&0x07)<<16;
}
end (jet_SonetNumCells);


private void jet_write_SONET (int unit, uint32 reg, uint32 value)
begin (jet_write_SONET);
{
    ((volatile char *) jet_sonetBase[unit]) [reg*4]= value;
}
end (jet_write_SONET);


private uint32 jet_read_SONET (int unit, uint32 reg)
begin (jet_read_SONET);
{
    ret ((volatile char *) jet_sonetBase[unit]) [reg*4];
}
end (jet_read_SONET);



#if 0
/* SONET interrupt handler */
void sonet_intr()
{
    int  master_status, module_status, module_val;

    master_status = jet_read_SONET(unit, MASTER_INT_REG);

    if (master_status & RSOP_INT) {
        module_status = jet_read_SONET(unit, RSOP_STATUS_REG);
        module_val = module_status;
        if (module_status & OOF_INT) {
            sonet.oof = (module_val&OOF_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_OOF);
        }
        if (module_status & LOF_INT) {
            sonet.lof = (module_val&LOF_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_LOF);
        }
        if (module_status & LOS_INT) {
            sonet.los = (module_val&LOS_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_LOS);
        }
    }

    if (master_status & RLOP_INT) {
        module_status = jet_read_SONET(unit, RLOP_INTERRUPT_REG);
        module_val = jet_read_SONET(unit, RLOP_STATUS_REG);
        if (module_status & RDI_INT) {
            sonet.rdi = (module_val&RDI_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_RDI);
        }
        if (module_status & LAIS_INT) {
            sonet.lais = (module_val&LAIS_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_LAIS);
        }
    }
    if (master_status & RPOP_INT) {
        module_status = jet_read_SONET(unit, RPOP_INT_STAT_REG);
        module_val = jet_read_SONET(unit, RPOP_STATUS_REG);
        if (module_status & PRDI_INT) {
            sonet.prdi = (module_val&PRDI_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_PRDI);
        }
        if (module_status & PAIS_INT) {
            sonet.pais = (module_val&PAIS_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_PAIS);
        }
        if (module_status & LOP_INT) {
            sonet.lop = (module_val&LOP_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_LOP);
        }
    }

    if (master_status & RACP_INT) {
        module_status = jet_read_SONET(unit, RACP_INTERRUPT_REG);
        module_val = jet_read_SONET(unit, RACP_CONTROL_REG);
        if (module_status & OOCD_INT) {
            sonet.oocd = (module_val&OOCD_VAL)>0;
            if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_OOCD);
        }
    }

    if (master_status & RDOOL_INT) {
        module_val = jet_read_SONET(unit, CLOCK_REC_REG);
        sonet.rdool = (module_val&RDOOL_VAL)>0;
        if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_RDOOL);
    }

    if (master_status & LCD_INT) {
        module_val = jet_read_SONET(unit, MASTER_CONTROL_REG);
        sonet.lcd = (module_val&LCD_VAL)>0;
        if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_LCD);
    }

    if (master_status & TROOL_INT) {
        module_val = jet_read_SONET(unit, CLOCK_SYN_REG);
        sonet.trool = (module_val&TROOL_VAL)>0;
        if (hDlgBox[SONET_BOX]) SendMessage(hDlgBox[SONET_BOX], WM_COMMAND, IDC_REDRAWLIGHT, IDC_TROOL);
    }
}

/*
 * Poll the SONET counters
 */
poll_sonet_counters()
{
    /* Write to SONET reg 00 to latch all the counter values */
    write_SONET(unit, MASTER_RESET_REG, 0);

    /* Read the counters and update data struct */
    sonet.sectionbip8 += jet_read_SONET(unit, RSOP_SECTION_BIP8) |
                         jet_read_SONET(unit, RSOP_SECTION_BIP8+1)<<8;
    sonet.linebip8 += jet_read_SONET(unit, RLOP_LINE_BIP8) |
                      jet_read_SONET(unit, RLOP_LINE_BIP8+1)<<8 |
                      (jet_read_SONET(unit, RLOP_LINE_BIP8+2)&0x0F)<<16;
    sonet.linefebe += jet_read_SONET(unit, RLOP_LINE_FEBE) |
                      jet_read_SONET(unit, RLOP_LINE_FEBE+1)<<8 |
                      (jet_read_SONET(unit, RLOP_LINE_FEBE+2)&0x0F)<<16;
    sonet.pathbip8 += jet_read_SONET(unit, RPOP_PATH_BIP8) |
                      jet_read_SONET(unit, RPOP_PATH_BIP8+1)<<8;
    sonet.pathfebe += jet_read_SONET(unit, RPOP_PATH_FEBE) |
                      jet_read_SONET(unit, RPOP_PATH_FEBE+1)<<8;
    sonet.corrhec += jet_read_SONET(unit, RACP_CHCS_ERROR);
    sonet.uncorrhec += jet_read_SONET(unit, RACP_UHCS_ERROR);
    sonet.rxcells += jet_read_SONET(unit, RACP_RX_CELL) |
                     jet_read_SONET(unit, RACP_RX_CELL+1)<<8 |
                     (jet_read_SONET(unit, RACP_RX_CELL+2)&0x07)<<16;
    sonet.txcells += jet_read_SONET(unit, TACP_TX_CELL) |
                     jet_read_SONET(unit, TACP_TX_CELL+1)<<8 |
                     (jet_read_SONET(unit, TACP_TX_CELL+2)&0x07)<<16;
    /* Poll the clock monitor reg */
    sonet.tclk = (jet_read_SONET(unit, MASTER_CLOCK_REG)&TCLKA)>0;
    sonet.rclk = (jet_read_SONET(unit, MASTER_CLOCK_REG)&RCLKA)>0;
    sonet.trclk = (jet_read_SONET(unit, MASTER_CLOCK_REG)&TRCLKA)>0;
    sonet.rrclk = (jet_read_SONET(unit, MASTER_CLOCK_REG)&RRCLKA)>0;
}
#endif
