/*
 * The interrupt dispatch module.
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/systm.h>
#include <dev/pci/jetland/include/ijet_coding.h>
#include <dev/pci/jetland/include/ijet_jet.h>
#include <dev/pci/jetland/include/ijet_sonet.h>
#include <dev/pci/jetland/include/ijet_ll.h>
#include <dev/pci/jetland/include/ijet_intr.h>
#include <dev/pci/jetland/include/ijet_intr_internals.h>

/*
 * For each possible interrupt, we maintain in a table what action to take
 * (what routine to call) when that interrupt occurs.
 * 
 * The interrupts are indexed by the order that they occur in the interrupt
 * status registers.
 */

private intr_handler_t interruptAction[NIJET][INTR_NUM_INTERRUPTS];

private void Dispatch (int unit, char *reg_name, uint32 status,
                       int offset, int size);

private int numActiveAdaptors;


/*
 * intr_Init -- initialize the interrupt module and enable interrupts
 *
 */

public intr_error_t intr_Init (int unit)
begin (intr_Init);
{
  /*
   * set interrupt action tables to default values
   */

  {
    int i;

    for (i= 0; i < INTR_NUM_INTERRUPTS; i++)
      interruptAction[unit][i]= NULL;
  }

  /* removed by todd
     if (!numActiveAdaptors)
     glue_InstallInterruptHandler (InterruptHandler);
     */
  numActiveAdaptors++;

  ret SUCCESS;
}
end (intr_Init);



/*
 * intr_Shutdown -- shut down the interrupt module and disable interrupts
 *
 */

public intr_error_t intr_Shutdown (int unit)
begin (intr_Shutdown);
{
  /* removed by todd... FIX this later
     if (numActiveAdaptors == 1)
     glue_UninstallInterruptHandler ();
     */
  numActiveAdaptors--;

  ret SUCCESS;
}
end (intr_Shutdown);



/*
 * intr_InstallHandler -- install a handler for an interrupt
 *
 * Parameters:
 *    interrupt -- the name of the interrupt (from intr.h)
 *    thread -- the thread to wake up when the interrupt occurs
 */

public void intr_InstallHandler (int unit, intr_interrupt_t interrupt,
                                 intr_handler_t handler)
begin (intr_InstallHandler);
{
    int index;
    volatile uint32 *maskregp;

    assert (((int) interrupt >= 0) && (interrupt < INTR_NUM_INTERRUPTS));
/* FIX: put this back */
/*    assert (interruptAction[unit][interrupt] == NULL); */
    assert (handler != NULL);

    debug ("Adding handler for unit %d, interrupt %d\n", unit, interrupt);

    interruptAction[unit][interrupt]= handler;

    /*
     * enable the interrupt
     */

    index= jet_InterruptToMaskStatusRegp (unit, interrupt, &maskregp, NULL);

    *maskregp |= (1 << index);
}
end (intr_InstallHandler);



/*
 * intr_RemoveHandler -- remove an interrupt handler
 *
 * Parameters:
 *    interrupt -- the name of the interrupt
 */

public void intr_RemoveHandler (int unit, intr_interrupt_t interrupt)
begin (intr_RemoveHandler);
{
    int index;
    volatile uint32 *maskregp;

    assert (((int) interrupt >= 0) && (interrupt < INTR_NUM_INTERRUPTS));

    debug ("Removing handler for unit %d, interrupt %d\n", unit, interrupt);

    interruptAction[unit][interrupt]= NULL;

    index= jet_InterruptToMaskStatusRegp (unit, interrupt, &maskregp, NULL);

    *maskregp &= ~(1 << index);
}
end (intr_RemoveHandler);



/*
 * InterruptHandler
 *
 * Called when any of the Jet Engine interfaces interrupts the host.
 * It figures out the source(s) of the interrupt,
 * and wakes up the appropriate handler(s).
 */

public void ijet_InterruptHandler (int unit)
begin (ijet_InterruptHandler);
{
    static int insideHandler = 0;
    uint32 isr_msc;
    uint32 isr_mbx;

    if (++insideHandler > 1)
      panic ("bogus! reentry into ijet_InterruptHandler\n");

    if (debug_directTextScreenMapDone) {
      static uint16 megaCow;
      debug_directTextScreenp[80] = 0xdead;
      debug_directTextScreenp[81]++;
    }

    if (!jet_adaptorFound[unit]) {
      printf("ijet_intr: spurious interrupt on unit %d\n", unit);
      return;
    }
    
    isr_msc = jet_asicBase[unit]->isr_msc;
    isr_mbx = jet_asicBase[unit]->isr_mbx;
    
    debug ("InterruptRoutine: checking msc bits for unit %d\n", unit);
    Dispatch (unit, "isr_msc", isr_msc, INTR_MSC_OFFSET,
              INTR_NUM_MSC_INTERRUPTS);
        
    debug ("InterruptRoutine: checking mbox bits for unit %d\n", unit);
    Dispatch (unit, "isr_mbx", isr_mbx, INTR_MBX_OFFSET,
              INTR_NUM_MBX_INTERRUPTS);
    
    --insideHandler;
}
end (ijet_InterruptHandler);



/*
 * Dispatch
 *
 * Utility routine called by InterruptRoutine
 *
 * Parameters:
 *  unit:      the unit number of the Jet Interface
 *  reg_name:  the name (in ASCII) of the register being accessed
 *             (used for debugging)
 *  status:    the value of the register
 *  offset:    offset into the relevent section of the action table
 *  size:      size of the relevant section of the action table
 */

private void Dispatch (int unit, char *reg_name, uint32 status, int offset,
                       int size)
begin (Dispatch);
{
  int i = 0;

  while (status && size) {
    if (status & 1) {
      intr_handler_t action= interruptAction[unit][offset + i];
      
      if (action == NULL)
        debug ("No handler for interrupt %d of %s\n", i, reg_name);
      else {
        debug ("Calling handler for interrupt %d of %s\n", i, reg_name);
        (*action) (unit);
        debug ("Returning from handler\n");
      }
    }
    
    status >>= 1;
    i++;
    size--;
  }
}
end (Dispatch);

