#include <sys/types.h>

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/proc.h>

#include <vm/vm.h>
#include <vm/vm_param.h>
#include <vm/vm_extern.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/netisr.h>
#include <net/route.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>

#include <machine/cpu.h>
#include <machine/pio.h>

#include <dev/pci/jetland/include/ijet_coding.h>
#include <dev/pci/jetland/include/ijet_ll.h>
#include <dev/pci/jetland/include/ijet_reg.h>
#include <dev/pci/jetland/src/vpc/vpc_user.h>
#include <dev/pci/jetland/include/timestamp.h>

#undef min
#define min(a,b) ((a)<(b)?(a):(b))


/*------------------------------------------------------------------------*/

extern vm_map_t kernel_map;
extern int hz;

/*------------------------------------------------------------------------*/

#undef debug
#undef DEBUG_PREFIX
#define debug debugr
#define DEBUG_PREFIX "txrate:"

/*------------------------------------------------------------------------
 * my beautiful macros
 *------------------------------------------------------------------------*/

#define VRINDEX(unit,vrp) ((vrp)-ll_vrBase[(unit)])
#define VRP(unit,index) (ll_vrBase[(unit)]+(index))
#define VRPTOVRNODEP(unit,vpcUnitp,vrp) \
        ((vpcUnitp)->mappingTable[VRINDEX(unit,vrp)-(vpcUnitp)->firstVrIndex])

#define SCREEN_SLAM(ypos,xpos,value)                            \
        if (!debug_directTextScreenMapDone) ;                   \
        else debug_directTextScreenp[(ypos)*80+(xpos)]=(value)

#define warnreturn(errorcode, fmt, args...) do {  \
	warn (fmt , ##args);                      \
	return (errorcode);                       \
        } while (0)

#define MALLOC_ARRAY(destptr, count)                                        \
  if (((destptr) = malloc ((count)*sizeof(*(destptr)), M_DEVBUF, M_NOWAIT)) \
      != NULL) ;                                                            \
  else warnreturn (ENOMEM, "Can't malloc %d bytes!\n",                      \
		 (count)*sizeof(*(destptr)))

#define FREE_ARRAY(p) free(p, M_DEVBUF)

#define ENQUEUE(queue, nodep) do {           \
	if ((queue).headp == NULL)           \
	     (queue).headp = (nodep);        \
        else                                 \
	     (queue).tailp->nextp = (nodep); \
        (queue).tailp = (nodep);             \
	(nodep)->nextp = NULL;               \
    } while(0)

#define DEQUEUE(queue) ({                             \
	typeof((queue).headp) retval = (queue).headp; \
	assert (retval != NULL);                      \
	(queue).headp = (queue).headp->nextp;         \
	retval; })

/*------------------------------------------------------------------------*
 * some bits in ATM cell header
 *------------------------------------------------------------------------*/

#define EFCI_OFFSET 1 /* in word 1 */
#define EFCI_BIT_OC3    0x00040000
#define EFCI_BIT_OC12   0x00000400

/*------------------------------------------------------------------------*
 * some other parameters
 *------------------------------------------------------------------------*/

#define VCID_BASE          0x100
#define CELL_SIZE          56
#define TOTAL_DATA_CELL_BUFFER  16384
#define TOTAL_RM_CELL_BUFFER 2340

#define FIRST_RATE_VCI    0x400
#define LAST_RATE_VCI     0x403
#define NUM_RATE_VCS      (LAST_RATE_VCI-FIRST_RATE_VCI+1)

#define RXDATA_VCI        1
#define RXRM_VCIa         0xaa
#define RXRM_VCIb         0xac

#define TX_VPIVCI         0x100

typedef uint16 rateFmt_t;

typedef struct {
    uint32 header[2];
    uint16 pid         : 8;
    uint16 dontCare    : 3;
    uint16 ra          : 1;
    uint16 ni          : 1;
    uint16 ci          : 1;
    uint16 bn          : 1;
    uint16 dir         : 1;
    uint16 er;
    uint16 ccr;
    uint16 mcr;
    uint ql;
    uint sn;
    uint8 reserved1[30];
    uint16 crc         : 10;
    uint16 reserved2   : 6;
} abrRmCell_t;

typedef enum {
    status_tag_lastCellHeader,
    status_tag_rateDesired=8,
    status_tag_fairRate,
    status_tag_datacellcount,
    status_tag_datacelldrop,
    status_tag_rmcellcount,
    status_tag_rmcelldrop,
    status_tag_txidle,
    status_tag_rxBothWatcher,
    status_tag_startTx,
    status_tag_waitTx,
    status_tag_walker,
    status_tag_MACR,
    status_tag_bar
} status_tag_t;

typedef enum {whoami_tag_rxdata, whoami_tag_rxrm} whoami_tag_t;
typedef enum {setup_tag_tx, setup_tag_rx} setup_tag_t;

typedef enum {
    ABR_EFCI,
    ABR_EPRCA,
    ABR_ERICA
} abrAlgo_t;

/* NetBSD 1.2 Hack */
uint32 *vpcRateBuf;

/*uint32 vpcRateBuf[524288];*/       /* 2MB cell buffer */

/*------------------------------------------------------------------------*
 * my beautiful data structures
 *------------------------------------------------------------------------*/

/* queues of (pointers to VRs).  Used to maintain pools of free VRs for
 * each unit */

typedef struct vrnode {
    struct vrnode *nextp;
    vr_t *vrp;
    bool recycleMe;
} vrnode_t;

typedef struct {
    vrnode_t *headp, *tailp;
} vrqueue_t;

/* queues of buffer descriptors.  Used to remember what part of the buffer
 * is where */

typedef enum {bdnode_tag_rxdata, bdnode_tag_rxrm, bdnode_tag_idle}
bdnode_tag_t;

typedef struct bdnode {
    struct bdnode *nextp;
    bdnode_tag_t tag;
    vrnode_t *vrnodep;
    void *virtualAddress;
    uint32 physicalAddress;
    int length;
} bdnode_t;

typedef struct {
    bdnode_t *headp, *tailp;
} bdqueue_t;

/*------------------------------------------------------------------------
 * all of our other info goes here
 *------------------------------------------------------------------------*/

/* global info */

typedef struct {
    bool goHappened;
    bool vpcHomogenous;

    int  numUnits;
    int  numRxdataCells;
    int  numRxrmCells;
    int  cellsPerVr;
    int  outputPortFullThreshold;
    int  lowCongThreshold;
    int  highCongThreshold;
    abrAlgo_t  abrAlgo;
    int  DQT;
    int  rmCellMergeUnit;
    int  kernelLoopCount;

    int vrTableSize;
} vpcConf_t;

/* per unit info */

typedef struct {
    int numCells;
    int numDroppedCells;
    bool overrunHappened;
    int cellsInTx; /* number of cells in txqueue and wait queue */
    char *statusMsg;
    uint32 hexData;
    bool hexDataValid;
} watcherStats_t;

typedef struct {
    int otherUnit;
    mailbox_t mailbox;
    freelist_t rxdataFreelist, rxrmFreelist;
    vcid_t rxdataVcid, rxrmVcida, rxrmVcidb, txVcid;
    vr_t *mailboxHeadp;

    vr_t **vrTable;
    int firstVrIndex, lastVrIndex;
    vrnode_t **mappingTable;

    vrqueue_t vrPool;
    vrnode_t *vrPoolMemoryp;

    bdqueue_t rxdataQueue;
    bdnode_t *rxdataQueueMemoryp;

    bdqueue_t rxrmQueue;
    bdnode_t *rxrmQueueMemoryp;

    bdqueue_t txQueue, waitQueue;

    watcherStats_t rxdataStats, rxrmStats;

    char *startTxMsg, *waitTxMsg, *walkerMsg;
    bool startTxHexDataValid, waitTxHexDataValid;
    uint32 startTxHexData, waitTxHexData, walkerHexData;

    bool congestionHappened;

    int maxBothInTx; /* high water mark for (data+rm) cells */

    /* keep track of last cell headers */
    uint32 lastCellHeader[8][2];

    uint32 desiredRate[NUM_RATE_VCS];
    uint32 fairShare;
  
    /* EPRCA crapola */

    uint32 MACR, DQT;
} vpcUnitInfo_t;

private vpcConf_t vpcConf;
private vpcUnitInfo_t *vpcUnits;


/*------------------------------------------------------------------------*/

public int vpcopen (dev_t dev, int flags, int mode, struct proc *p);
public int vpcclose (dev_t dev, int flags, int mode, struct proc *p);
public int
vpcioctl (dev_t dev, int cmd, caddr_t addr, int flag, struct proc *p);
private int VpcGo (int numUnits,
                   int numRxdataCells,
                   int numRxrmCells,
                   int cellsPerVr, /*applies only to rxdata */
                   int outputPortFullThreshold,
		   int lowCongThreshold,
		   int highCongThreshold,
                   int abrAlgo,
                   int DQT,
                   int rmCellMergeUnit,
                   int kernelLoopCount);
private int Setup (vpcUnitInfo_t *vpcUnitp, int otherUnit);
private int BuildFreelist (vpcUnitInfo_t *vpcUnitp, int freelist, 
			   bdqueue_t *qp, bdnode_t **qdatapp,
			   int numVrs, int cellsPerVr, bdnode_tag_t bdnodeTag,
			   void *bufferAddress);
private void VpcStop (void);
private void UnSetup (vpcUnitInfo_t *vpcUnitp);
private int VpcPoll (void);
private void UpdateScreen (vpcUnitInfo_t *vpcUnitp);
private void RxBothWatcherMan(vpcUnitInfo_t *vpcUnitp, bdqueue_t *qp, 
			      whoami_tag_t whoami);
private void StartTxMan (vpcUnitInfo_t *vpcUnitp);
private void WaitTxMan (vpcUnitInfo_t *vpcUnitp);
private void MailboxWalker (vpcUnitInfo_t *vpcUnitp);
private void SetupVr (bdnode_t *bdnodep, setup_tag_t tag);
private void UpdateBar (int newValue, int *oldValuep, int maxValue, int ypos);
private void ProcessRMCell (vpcUnitInfo_t *invpcUnitp, 
			    vpcUnitInfo_t *outvpcUnitp, abrRmCell_t *cellp);
private uint32 RateFmtToInt (rateFmt_t rate);
private rateFmt_t IntToRateFmt (uint32 value);
private void DumpScreen (int unit, int ypos, char *msg);
private void DumpStr (char *msg);
private void DumpHex (uint32 data);

/*------------------------------------------------------------------------
 * the device interface
 *------------------------------------------------------------------------*/

public int vpcopen (dev_t dev, int flags, int mode, struct proc *p)
begin();
{
    if (vpcUnits) /* don't allow multiple opens */
	return EACCES;

    MALLOC_ARRAY (vpcUnits, NIJET);
    bzero (&vpcConf, sizeof(vpcConf));
    bzero (vpcUnits, NIJET*sizeof(*vpcUnits));

    vpcConf.goHappened = FALSE;
    return 0;
}
end();


public int vpcclose (dev_t dev, int flags, int mode, struct proc *p)
begin();
{
    if (vpcConf.goHappened)
	VpcStop ();

    free (vpcUnits, M_DEVBUF);
    vpcUnits = NULL;

    return 0;
}
end();


public int
vpcioctl (dev_t dev, int cmd, caddr_t addr, int flag, struct proc *p)
begin();
{
    int32 *data = (int32*)addr;

    switch (cmd) {
    case VPC_IOCTL_GO_RATE: {
	return VpcGo (data[0], data[1], data[2], data[3], data[4], data[5],
		      data[6], data[7], data[8], data[9], data[10]);
    } break;

    case VPC_IOCTL_POLL: {
	return VpcPoll ();
    } break;

    default: {
	warnreturn (EINVAL, "ioctl 0x%x not implemented\n", cmd);
    } break;
    }
}
end();


private int VpcGo (int numUnits,
                   int numRxdataCells,
                   int numRxrmCells,
                   int cellsPerVr, /*applies only to rxdata */
                   int outputPortFullThreshold,
		   int lowCongThreshold,
		   int highCongThreshold,
                   int abrAlgo,
                   int DQT,
                   int rmCellMergeUnit,
                   int kernelLoopCount)
begin();
{
    int error;

    if (numUnits<0 || numUnits>2)
	warnreturn (EINVAL, "%d is an absurd value for numUnits\n", numUnits);

    vpcConf.numUnits = numUnits;
    vpcConf.numRxdataCells = numRxdataCells;
    if (vpcConf.numRxdataCells > TOTAL_DATA_CELL_BUFFER) {
	warn("Cannot support more than %d cells of data buffer space\n",
	     TOTAL_DATA_CELL_BUFFER);
	return EINVAL;
    }
    vpcConf.numRxrmCells = numRxrmCells;
    if (vpcConf.numRxrmCells > TOTAL_RM_CELL_BUFFER) {
	warn("Cannot support more than %d cells of credit buffer space\n",
	     TOTAL_RM_CELL_BUFFER);
	return EINVAL;
    }
    vpcConf.cellsPerVr = cellsPerVr;
    vpcConf.outputPortFullThreshold = outputPortFullThreshold;
    vpcConf.lowCongThreshold = lowCongThreshold;
    vpcConf.highCongThreshold = highCongThreshold;
    vpcConf.abrAlgo = abrAlgo;
    vpcConf.DQT = DQT;
    vpcConf.rmCellMergeUnit = rmCellMergeUnit;
    vpcConf.kernelLoopCount = kernelLoopCount;

    /* NetBSD 1.2 hack */
    vpcRateBuf = (uint32 *)debug_bigBufp;

    if (numUnits == 1)
	error = Setup (&vpcUnits[0], 0);
    else {
	if (!(error = Setup (&vpcUnits[0], 1)))
	    error = Setup (&vpcUnits[1], 0);
    }

    if (jet_oc12[0] != jet_oc12[1])
	vpcConf.vpcHomogenous = FALSE;
    else 
	vpcConf.vpcHomogenous = TRUE;

    if (!error)
	vpcConf.goHappened = TRUE;

    return error;
}
end();


private int Setup (vpcUnitInfo_t *vpcUnitp, int otherUnit)
begin();
{
    int unit = vpcUnitp-vpcUnits;

    vpcUnitp->otherUnit = otherUnit;

    /* allocate the VRs we need and establish the vrnode<-->vrp mapping */
    {
	/* This includes:
	 * incoming data cells
	 * incoming rm cells
	 * outgoing (data or rm) cells
	 * slop
	 */
	/* the 2* here is a little bonus to keep it from crashing */
	int vrsNeeded = 2*vpcConf.numRxdataCells/vpcConf.cellsPerVr
	    + vpcConf.numRxrmCells
	    + vpcConf.outputPortFullThreshold
	    + 16;
	int count, i;

	warn ("unit %d: Asking for %d vrs\n", unit, vrsNeeded);
	vpcConf.vrTableSize = vrsNeeded;

	MALLOC_ARRAY (vpcUnitp->vrTable, vrsNeeded);
	if ((count = ll_AllocateVrs (unit, vrsNeeded, vpcUnitp->vrTable))
	    != vrsNeeded)
	    warnreturn (ENETDOWN, "Unit %d couldn't get the %d VRs that"
			" it wanted\n", unit, vrsNeeded);
	{
	    int mappingSize;

	    /* set up the mapping from vr pointers to vrnode pointers */
	    vpcUnitp->firstVrIndex = VRINDEX(unit, vpcUnitp->vrTable[0]);
	    vpcUnitp->lastVrIndex  = VRINDEX(unit, 
					     vpcUnitp->vrTable[vrsNeeded-1]);

	    mappingSize = vpcUnitp->lastVrIndex-vpcUnitp->firstVrIndex+1;
	    MALLOC_ARRAY (vpcUnitp->mappingTable, mappingSize);
	}

	vpcUnitp->vrPool.headp = NULL;
	MALLOC_ARRAY (vpcUnitp->vrPoolMemoryp, vrsNeeded);
	for (i=0; i<vrsNeeded; i++) {
	    vrnode_t *vrnodep = &vpcUnitp->vrPoolMemoryp[i];
	    int index;

	    vrnodep->vrp = vpcUnitp->vrTable[i];
	    index = VRINDEX (unit, vrnodep->vrp) - vpcUnitp->firstVrIndex;
	    vpcUnitp->mappingTable[index] = vrnodep;
	    ENQUEUE (vpcUnitp->vrPool, vrnodep);
	}
    }

    /* we get "optimistic notifications" of receive/transmit done because
     * we peer directly into the VRs.  However, we still process the mailbox
     * in the usual way so that we don't return VRs to the free pool until
     * the ASIC is done with them. */
    {
	static vr_t hostVr = {{0}};
	vrnode_t *vrnodep = DEQUEUE (vpcUnitp->vrPool);
	vrnodep->recycleMe = TRUE;
	vpcUnitp->mailboxHeadp = vrnodep->vrp;

	if ((vpcUnitp->mailbox = ll_AllocMailbox (unit, NULL, NULL,
						  vpcUnitp->mailboxHeadp)) < 0)
	    warnreturn (ENETDOWN, "Unit %d can't open mailbox (error %d)\n",
			unit, vpcUnitp->mailbox);
	*vpcUnitp->mailboxHeadp = hostVr;
    }


    /* the data freelist */
    if ((vpcUnitp->rxdataFreelist = ll_AllocFreelist (unit, NULL, NULL)) < 0)
	warnreturn (ENETDOWN, "Unit %d can't allocate data freelist "
		    "(error %d)\n", unit, vpcUnitp->rxdataFreelist);

    /* the RM cell freelist */
    if ((vpcUnitp->rxrmFreelist = ll_AllocFreelist (unit, NULL, NULL)) < 0)
	warnreturn (ENETDOWN, "Unit %d can't allocate RM freelist "
		    "(error %d)\n", unit, vpcUnitp->rxrmFreelist);

    /* I should check error return but ha ha, I won't
     * There is a 2M physically/virtually contiguous region.  It is divvied up
     * like so:
     * 1M for adapter 0, 1M for adapter 1
     * for adapter N:
     *  64K*2 for rm cells
     *  64K*14 for data cells
     */

    {
	void *nextAddress = (void*) vpcRateBuf + (65536*16*unit);
	
	bzero (nextAddress, 65536*16);
	
	(void)BuildFreelist (vpcUnitp, vpcUnitp->rxrmFreelist,
			     &vpcUnitp->rxrmQueue, &vpcUnitp->rxrmQueueMemoryp,
			     vpcConf.numRxrmCells, 1,
			     bdnode_tag_rxrm, nextAddress);
	nextAddress += 65536*2;
	
	(void)BuildFreelist (vpcUnitp, vpcUnitp->rxdataFreelist,
			     &vpcUnitp->rxdataQueue, 
			     &vpcUnitp->rxdataQueueMemoryp,
			     vpcConf.numRxdataCells/vpcConf.cellsPerVr,
			     vpcConf.cellsPerVr,
			     bdnode_tag_rxdata, nextAddress);
    }
    
    /* the other queues are empty */
    vpcUnitp->txQueue.headp   = NULL;
    vpcUnitp->waitQueue.headp = NULL;
    
    /* this is the promiscuous version */
    {
	int vpivci;
	
	/* open data VC */
	warn ("Using promiscuous mode\n");
	vpivci = RXDATA_VCI;
	if ((vpcUnitp->rxdataVcid = ll_OpenRx (unit, CELL_PROCESSING_RAW,
					       vpcUnitp->mailbox,
					       vpcUnitp->rxdataFreelist, 
					       vpcUnitp->rxdataFreelist,
					       &vpivci)) < 0)
	    warnreturn (ENETDOWN, "unit %d can't open rx %d error %d\n",
			unit, vpivci, vpcUnitp->rxdataVcid);
	
	/* open RM VC */
	vpivci = RXRM_VCIa;
	if ((vpcUnitp->rxrmVcida = ll_OpenRx (unit, CELL_PROCESSING_RAW,
					      vpcUnitp->mailbox,
					      vpcUnitp->rxrmFreelist, 
					      vpcUnitp->rxrmFreelist,
					      &vpivci)) < 0)
	    warnreturn (ENETDOWN, "unit %d can't open rx %d error %d\n",
			unit, vpivci, vpcUnitp->rxrmVcida);
	
	vpivci = RXRM_VCIb;
	if ((vpcUnitp->rxrmVcidb = ll_OpenRx (unit, CELL_PROCESSING_RAW,
					      vpcUnitp->mailbox,
					      vpcUnitp->rxrmFreelist, 
					      vpcUnitp->rxrmFreelist,
					      &vpivci)) < 0)
	    warnreturn (ENETDOWN, "unit %d can't open rx %d error %d\n",
			unit, vpivci, vpcUnitp->rxrmVcidb);

	/* go into promiscuous mode */
	{
	    jet_reg_cfg_reg_t reg = jet_asicBase[unit]->cfg_reg;
	    reg.rxCollectOutOfRangeVcs = 1;
	    jet_asicBase[unit]->cfg_reg= reg;
	}
	jet_asicBase[unit]->vc_err = vpcUnitp->rxdataVcid;
    }

    /* open the transmit vc */
    if ((vpcUnitp->txVcid = ll_OpenTx (unit, CELL_PROCESSING_RAW,
				       vpcUnitp->mailbox, SCHEDULE_STATIC,
				       TX_VPIVCI)) < 0)
	warnreturn (ENETDOWN, "unit %d can't open tx: error %d\n",
		    unit, vpcUnitp->txVcid);

    return 0;
}
end();

/* build the data freelist and the corresponding buffer descriptor
 * structure on the host. */

private int
BuildFreelist (vpcUnitInfo_t *vpcUnitp, int freelist, bdqueue_t *qp, 
	       bdnode_t **qdatapp, int numVrs, int cellsPerVr, 
	       bdnode_tag_t bdnodeTag, void *bufferAddress)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    int i;
    vr_t *firstVrp=NULL, *lastVrp=NULL;

    warn ("Using virtual address 0x%x, physical address 0x%x\n",
	  (uint32)bufferAddress, (uint32)vtophys(bufferAddress));

    qp->headp = NULL;
    MALLOC_ARRAY (*qdatapp, numVrs);
    for (i=0; i<numVrs; i++) {
	/* get next bdnode */
	bdnode_t *bdnodep = &(*qdatapp)[i];

	bdnodep->tag = bdnodeTag;

	/* compute its physical address */
	bdnodep->virtualAddress  = bufferAddress;
	bdnodep->physicalAddress = vtophys (bufferAddress);
	bdnodep->length          = cellsPerVr * CELL_SIZE;
	bufferAddress += bdnodep->length;

	/* bind it to a vrnode */
	bdnodep->vrnodep = DEQUEUE (vpcUnitp->vrPool);

	/* initialize its VR */
	SetupVr (bdnodep, setup_tag_rx);

	/* add to receive pool */
	ENQUEUE (*qp, bdnodep);

	/* add corresponding VR to asic freelist */
	if (0)
	    warn ("unit %d: appending VR 0x%x to freelist %d\n",
		  unit, VRINDEX(unit, bdnodep->vrnodep->vrp), freelist);

	if (!firstVrp)
	    firstVrp = lastVrp = bdnodep->vrnodep->vrp;
	else {
	    lastVrp->vr.nextIndex = VRINDEX (unit, bdnodep->vrnodep->vrp);
	    lastVrp = bdnodep->vrnodep->vrp;
	}
    }
    if (firstVrp)
	ll_AppendVrlistToFreelist (unit, freelist, firstVrp, lastVrp);

    return 0;
}
end();


private void VpcStop (void)
begin();
{
    UnSetup (&vpcUnits[0]);
    if (vpcConf.numUnits == 2)
	UnSetup (&vpcUnits[1]);
}
end();



private void UnSetup (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit = vpcUnitp-vpcUnits;

    warn ("Hi!  Shutting down unit %d\n", unit);
    ll_CloseTx (unit, vpcUnitp->txVcid);
    ll_CloseRx (unit, vpcUnitp->rxdataVcid);
    ll_CloseRx (unit, vpcUnitp->rxrmVcida);
    ll_CloseRx (unit, vpcUnitp->rxrmVcidb);
    ll_FreeMailbox (unit, vpcUnitp->mailbox);
    ll_FreeVrs (unit, vpcConf.vrTableSize, vpcUnitp->vrTable);
    ll_FreeFreelist (unit, vpcUnitp->rxdataFreelist);
    ll_FreeFreelist (unit, vpcUnitp->rxrmFreelist);

    FREE_ARRAY (vpcUnitp->rxrmQueueMemoryp);
    FREE_ARRAY (vpcUnitp->rxdataQueueMemoryp);
    FREE_ARRAY (vpcUnitp->vrPoolMemoryp);
    FREE_ARRAY (vpcUnitp->mappingTable);
    FREE_ARRAY (vpcUnitp->vrTable);

    vpcConf.goHappened = FALSE;
}
end();


private int VpcPoll (void)
begin();
{
    int i, unit;

    for (i=0; i<vpcConf.kernelLoopCount; i++) {
	for (unit=0; unit<vpcConf.numUnits; unit++) {
	    RxBothWatcherMan (&vpcUnits[unit], &vpcUnits[unit].rxdataQueue,
			      whoami_tag_rxdata);
	    RxBothWatcherMan (&vpcUnits[unit], &vpcUnits[unit].rxrmQueue,
			      whoami_tag_rxrm);
	    StartTxMan    (&vpcUnits[unit]);
	    WaitTxMan     (&vpcUnits[unit]);
	    MailboxWalker (&vpcUnits[unit]);
	}
    }

    for (unit=0; unit<vpcConf.numUnits; unit++)
	UpdateScreen (&vpcUnits[unit]);

    return 0;
}
end();


/*
 * This routine watches for incoming cells, either data or rm.
 * If data cells, they are passed through virtually untouched (except for
 * setting of the EFCI bit).  rm cells are processed though.
 */

private void RxBothWatcherMan (vpcUnitInfo_t *vpcUnitp, bdqueue_t *qp, 
			       whoami_tag_t whoami)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    watcherStats_t *watcherStatsp;
    int cellsPerVr;

    if (whoami == whoami_tag_rxdata) {
	watcherStatsp = &vpcUnitp->rxdataStats;
	cellsPerVr    = vpcConf.cellsPerVr;
    } 
    else {
	watcherStatsp = &vpcUnitp->rxrmStats;
	cellsPerVr    = 1;
    }

    for (;;) {
	bdnode_t *bdnodep = qp->headp;

	/* rx queue totally empty? (should only happen in pathological cases)*/
	if (bdnodep == NULL) {
	    watcherStatsp->statusMsg       = "!Empty";
	    watcherStatsp->hexDataValid    = FALSE;
	    watcherStatsp->overrunHappened = TRUE;
	    break;
	}

	/* can't proceed until asic fills incoming vr */
	if (bdnodep->vrnodep->vrp->message.remain != 0) {
	    watcherStatsp->statusMsg    = "wait";
	    watcherStatsp->hexData      = VRINDEX(unit,bdnodep->vrnodep->vrp);
	    watcherStatsp->hexDataValid = TRUE;
	    break;
	}
	DEQUEUE (*qp);

	/* slurp up the cell headers */
	if (0)
	    {
		int i, j, wordnum;
		for (wordnum=0; wordnum<2; wordnum++)
		    for (i=0; i<cellsPerVr; i++) {
			for (j=0; j<7; j++)
			    vpcUnitp->lastCellHeader[j][wordnum] =
				vpcUnitp->lastCellHeader[j+1][wordnum];
			vpcUnitp->lastCellHeader[7][wordnum] =
			    ((uint32*)(bdnodep->virtualAddress))[14*i+wordnum];
		    }
	    }

	watcherStatsp->numCells += cellsPerVr;

	/* unbind VR from buffer descriptor.  MailboxWalker will eventually
	 * return it to the pool */
	bdnodep->vrnodep->recycleMe = TRUE;
	bdnodep->vrnodep = NULL;

	/* if output port is "full", drop the cells, recycle the buffer desc */

	{
	    int outunit = vpcUnitp->otherUnit;
	    vpcUnitInfo_t *outvpcUnitp = &vpcUnits[outunit];

	    /* WARNING: we never drop RM cells, per Prashant's request */
	    if ((whoami == whoami_tag_rxdata) &&
		((outvpcUnitp->rxdataStats.cellsInTx + 
		  outvpcUnitp->rxrmStats.cellsInTx)
		 > vpcConf.outputPortFullThreshold)) {
		watcherStatsp->numDroppedCells += cellsPerVr;
		bdnodep->vrnodep = DEQUEUE (vpcUnitp->vrPool);
		SetupVr (bdnodep, setup_tag_rx);

		/* link to tail of freelist (man rx linking--yay!) */
		assert (qp->headp != NULL);
		qp->tailp->vrnodep->vrp->vr.nextIndex =
		    VRINDEX (unit, bdnodep->vrnodep->vrp);
		ENQUEUE (*qp, bdnodep);
	    } 
	    else {
		/* if it's an RM cell, process it */

		if (whoami == whoami_tag_rxrm) {
		    abrRmCell_t *cellp = 
			(abrRmCell_t *)bdnodep->virtualAddress;
		    /* horrible hack merges rm cells onto the same vc in one
		     * direction only--it's a superhack because we require that
		     * RXRM_VCIa & RXRM_VCIb differ only in the low nibble! 
		     */
#define HACK_NIBBLE_OC3 0x00f00000
#define HACK_NIBBLE_OC12 0x0000f000
#define HACK_BITPOS_OC3 20
#define HACK_BITPOS_OC12 12
		    if (unit == vpcConf.rmCellMergeUnit) {
			uint32 wasHeader = cellp->header[1];
			if (jet_oc12[unit]) 
			    cellp->header[1] = 
				(cellp->header[1] & ~HACK_NIBBLE_OC12) |
				((RXRM_VCIa & 0xf)<<HACK_BITPOS_OC12);
			else
			    cellp->header[1] = 
				(cellp->header[1] & ~HACK_NIBBLE_OC3) |
				((RXRM_VCIa & 0xf) << HACK_BITPOS_OC3);
			
			if (0)
			    warn ("Unit %d merged: from 0x%x to 0x%x...\n", 
				  unit, wasHeader, cellp->header[1]);
		    }
#undef HACK_NIBBLE
		}

		if (vpcConf.abrAlgo && (whoami == whoami_tag_rxrm))
		    ProcessRMCell (vpcUnitp, outvpcUnitp,
				   (abrRmCell_t*)bdnodep->virtualAddress);
		/* 
		 * append to txQueue of output unit 
		 */

		/* set the congestion bit appropriately */

		if (!vpcConf.abrAlgo) {
		    int Q = outvpcUnitp->rxdataStats.cellsInTx + 
			outvpcUnitp->rxrmStats.cellsInTx;
		    
		    if ((Q >= vpcConf.highCongThreshold) ||
			(outvpcUnitp->congestionHappened && 
			 (Q > vpcConf.lowCongThreshold))) {
			if (jet_oc12[unit]) 
			    ((uint32*)bdnodep->virtualAddress)[EFCI_OFFSET]
				|= EFCI_BIT_OC12;
			else
			    ((uint32*)bdnodep->virtualAddress)[EFCI_OFFSET]
				|= EFCI_BIT_OC3;
			outvpcUnitp->congestionHappened = TRUE;
		    }
		    else {
			outvpcUnitp->congestionHappened = FALSE;
		    }
		}

		/* 
		 * Adjust raw cell format when working between OC-3 and OC-12 
		 * units.
		 */
		if (!vpcConf.vpcHomogenous) {
		    uint32 *headerp;
		    int i;

		    for (i = 0; i < cellsPerVr; i++) {
			headerp = (uint32 *) 
			    &(((uint32 *)bdnodep->virtualAddress)[14*i]);
			if (jet_oc12[unit]) {
			    /* Input is OC12 and output is OC3 */
			    headerp[1] = 
				(headerp[1] << 8) | (headerp[0] >> 24);
			    headerp[0] = (headerp[0] << 8);
			}
			else {
			    /* Input is OC3 and output is OC12 */
			    headerp[0] = 
				(headerp[0] >> 8) | (headerp[1] << 24);
			    headerp[1] = (headerp[1] >> 8);
			}
		    }
		}

		/* now finally append the buffer descriptor of the data */

		/* get a VR from out's pool, bind to our buffer descriptor */
		bdnodep->vrnodep = DEQUEUE (outvpcUnitp->vrPool);

		/* set up the VR. */
		SetupVr (bdnodep, setup_tag_tx);

		/* if the tx queue is nonempty, link this VR to the tail of the
		   previous */

		if (outvpcUnitp->txQueue.headp != NULL)
		    outvpcUnitp->txQueue.tailp->vrnodep->vrp->vr.nextIndex =
			VRINDEX (outunit, bdnodep->vrnodep->vrp);

		/* Put this buffer descriptor on the transmit queue */
		ENQUEUE (outvpcUnitp->txQueue, bdnodep);

		if (whoami == whoami_tag_rxdata)
		    outvpcUnitp->rxdataStats.cellsInTx += cellsPerVr;
		else
		    outvpcUnitp->rxrmStats.cellsInTx ++;
	    }
	}
    }
}
end();

/*
 * If the transmit vcid is free, transmit the whole chain waiting
 * in the txQueue (the VRs have already been linked together)
 */

private void StartTxMan (vpcUnitInfo_t *vpcUnitp)
begin ();
{
    int unit = vpcUnitp-vpcUnits;

    {
	bdnode_t *headp = vpcUnitp->txQueue.headp;

	/* nothing to transmit? */
	if (headp == NULL) {
	    vpcUnitp->startTxMsg     = "tx empty";
	    vpcUnitp->startTxHexDataValid = FALSE;
	    return;
	}

	/* is my vcid free? */
	if (ll_txTableBase[unit][vpcUnitp->txVcid].vrIndex != 0) {
	    vpcUnitp->startTxMsg = "manlink wait VC:";
	    vpcUnitp->startTxHexData = vpcUnitp->txVcid;
	    vpcUnitp->startTxHexDataValid = TRUE;
	    return;
	}

	/* transmit the whole pile */
	ll_txTableBase[unit][vpcUnitp->txVcid].vrIndex =
	    VRINDEX (unit, headp->vrnodep->vrp);
    }

    /* Append the new buffer descriptors onto the waitQueue */
    if (vpcUnitp->waitQueue.headp == NULL)
	vpcUnitp->waitQueue = vpcUnitp->txQueue;
    else {
	vpcUnitp->waitQueue.tailp->nextp = vpcUnitp->txQueue.headp;
	vpcUnitp->waitQueue.tailp = vpcUnitp->txQueue.tailp;
    }
    vpcUnitp->txQueue.headp = NULL;
}
end();


private void WaitTxMan (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit=vpcUnitp-vpcUnits;
    int inunit = vpcUnitp->otherUnit;
    vpcUnitInfo_t *invpcUnitp = &vpcUnits[inunit];

    for (;;) {
	bdnode_t *bdnodep = vpcUnitp->waitQueue.headp;

	if (bdnodep == NULL) { /* nothing to wait for? */
	    vpcUnitp->waitTxMsg     = "wait tx empty";
	    vpcUnitp->waitTxHexDataValid = FALSE;
	    break;
	}

	/* VR not transmitted yet? */
	if (bdnodep->vrnodep->vrp->message.remain != 0) {
	    vpcUnitp->waitTxMsg     = "wait complete VR:";
	    vpcUnitp->waitTxHexDataValid = TRUE;
	    vpcUnitp->waitTxHexData = VRINDEX(unit, bdnodep->vrnodep->vrp);
	    break;
	}
	DEQUEUE (vpcUnitp->waitQueue);

	/* unbind vrnode */
	bdnodep->vrnodep->recycleMe = TRUE;
	bdnodep->vrnodep = NULL;

	/* return buffer descriptor to the appropriate queue */
	if (bdnodep->tag == bdnode_tag_rxdata) {
	    vpcUnitp->rxdataStats.cellsInTx -= vpcConf.cellsPerVr;
	    /* get a fresh VR from in's pool */
	    bdnodep->vrnodep = DEQUEUE (invpcUnitp->vrPool);

	    SetupVr (bdnodep, setup_tag_rx);

	    /* add this buffer descriptor to in's rxQueue, and append the new
	     * VR to in's freelist: MAN RX LINKING---yay! */

	    assert (invpcUnitp->rxdataQueue.headp != NULL);
	    invpcUnitp->rxdataQueue.tailp->vrnodep->vrp->vr.nextIndex =
		VRINDEX (inunit, bdnodep->vrnodep->vrp);
	    ENQUEUE (invpcUnitp->rxdataQueue, bdnodep);
	} else if (bdnodep->tag == bdnode_tag_rxrm) {
	    vpcUnitp->rxrmStats.cellsInTx--;
	    /* get a fresh VR from in's pool */
	    bdnodep->vrnodep = DEQUEUE (invpcUnitp->vrPool);
	    SetupVr (bdnodep, setup_tag_rx);

	    /* add this buffer descriptor to in's rxQueue, and append the new
	     * VR to in's freelist: MAX RX LINKING----yay! */
	    assert (invpcUnitp->rxrmQueue.headp != NULL);
	    invpcUnitp->rxrmQueue.tailp->vrnodep->vrp->vr.nextIndex =
		VRINDEX (inunit, bdnodep->vrnodep->vrp);
	    ENQUEUE (invpcUnitp->rxrmQueue, bdnodep);
	} 
    }
}
end();


private void MailboxWalker (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    int nextIndex;

    if (vpcUnitp->mailboxHeadp->message.nextIndex == 0) {
	vpcUnitp->walkerMsg = "wait head:";
	vpcUnitp->walkerHexData = VRINDEX (unit, vpcUnitp->mailboxHeadp);
    } else {
	while ((nextIndex=vpcUnitp->mailboxHeadp->message.nextIndex) != 0) {
	    vrnode_t *vrnodep = VRPTOVRNODEP (unit, vpcUnitp, 
					      vpcUnitp->mailboxHeadp);
	    if (!vrnodep->recycleMe) {
		vpcUnitp->walkerMsg = "recycle:";
		vpcUnitp->walkerHexData = VRINDEX (unit, 
						   vpcUnitp->mailboxHeadp);
		return;
	    }
	    ENQUEUE (vpcUnitp->vrPool, vrnodep);
	    if ((nextIndex<vpcUnitp->firstVrIndex)||
		(nextIndex > vpcUnitp->lastVrIndex)) {
		warn ("**** MEGA WARNING: VR 0x%x has nextIndex 0x%x::"
		      "how absurd\n", VRINDEX(unit, vpcUnitp->mailboxHeadp), 
		      nextIndex);
		panic ("vpc");
	    } else
		vpcUnitp->mailboxHeadp = VRP (unit, nextIndex);
	}
    }
}
end();


private void SetupVr (bdnode_t *bdnodep, setup_tag_t tag)
begin();
{
    static vr_t hostVr = {{0}};

    /* guess what: the "which" parameter doesn't matter */
    (void)tag;

    hostVr.vr.sarLocation           = SAR_HOST;
    hostVr.vr.address               = bdnodep->physicalAddress;
    hostVr.vr.remain                = bdnodep->length;
    /* hostVr.vr.nextIndex defaults to 0 */

    bdnodep->vrnodep->recycleMe = FALSE;
    *(bdnodep->vrnodep->vrp) = hostVr; /* slam into SRAM */
}
end();

/* Rate code helper routines */

private void ProcessRMCell (vpcUnitInfo_t *invpcUnitp, 
			    vpcUnitInfo_t *outvpcUnitp, abrRmCell_t *cellp)
begin();
{
    vpcUnitInfo_t *vpcUnitp;
    uint32 vci = cellp->sn;
    uint32 Q, ACR, MACR, DQT;
    bool congested;

    if (0)
	warn ("vci 0x%x, dir %d, ccr 0x%x\n", vci, cellp->dir, cellp->ccr);

    if ((vci < FIRST_RATE_VCI) || (vci > LAST_RATE_VCI))
	return;

    /* Q is the queue length */

    if (cellp->dir == 0)
	vpcUnitp = outvpcUnitp;
    else
	vpcUnitp = invpcUnitp;

    switch (vpcConf.abrAlgo) {

    case ABR_EFCI: {
    } break;

    case ABR_EPRCA: {
	Q = vpcUnitp->rxdataStats.cellsInTx + vpcUnitp->rxrmStats.cellsInTx;
	congested = Q > vpcConf.highCongThreshold;  /* 
						     * In this case both high
						     * and low thresholds 
						     * should be same.
						     */
	
	ACR  = RateFmtToInt (cellp->ccr);
	MACR = vpcUnitp->MACR;
	DQT  = vpcUnitp->DQT;
	
	if (cellp->dir == 0) { /* forward */
	    if ((congested && (ACR < MACR)) || 
		(!congested && (ACR > MACR*7/8)))
		vpcUnitp->MACR = (MACR*15/16) + (ACR/16);  /* 1/16 is the AV */
	} else { /* backward */
	    if (congested) {
		uint32 ER = RateFmtToInt (cellp->er);
		
		if (Q > DQT)
		    cellp->er = IntToRateFmt (min (MACR/4, ER));
		else {
		    if (ACR>MACR*7/8) { /* 7/8 is the DPF */
			cellp->er = IntToRateFmt (min (MACR*15/16, ER));
			/*15/16 is the erf*/
		    }
		}
	    }
	}
    } break;

    case ABR_ERICA: {
    } break;

    default: {
	warn ("Unknown ABR Algorithm %d\n", vpcConf.abrAlgo);
    } 
    }
}
end();

#if 0
private uint32 RecomputeFairShare (int numItems, uint32 *desiredRate)
begin();
{
    uint32 fairShare = megalinkrate / numItems;

    /* pass 1 */
    for (i=0; i<numItems; i++) {
	if (desiredRate[i] < fairShare) {
	    isPig[i] = FALSE;
	    fairRate[i] = desiredRate[i];
	    numPigs--;
	    linkRateDeduction += desiredRate[i];
	} else
	    isPig[i] = TRUE;
    }

    fairShare = (megalinkrate - linkRateDeduction) / numPigs;
    for (i=0; i<numItems; i++)
	if (isPig[i] && (desiredRate[i] < fairShare)) {

	    isPig
		if (!isPiglet[i])
		    }

    int permute[numItems];
    int i,j;
    uint32 totalPie;

    {
	uint32 total = 0;
	for (i=0; i<numItems; i++)
	    total += desiredRate[i];
	if (total < megalinkrate)
	    return megalinkrate/numItems;
    }

    /* sort according to increasing rate */
    for (i=0; i<numItems; i++)
	permute[i] = i;

    /* use bubble sort because numItems is small */
    for (i=0; i<numItems-1; i++) {
	for (j=i+1; j<numItems; j++) {
	    if (desiredRate[permute[i]] > desiredRate[permute[j]]) {
		int temp = permute[i];
		permute[i] = permute[j];
		permute[j] = temp;
	    }
	}
    }

    /* do the fair share algorithm */
  
    totalPie = megalinkrate;
    for (i=0; i<numItems; i++) {
	uint32 fairShare = totalPie/(numItems-i);

	if (desiredRate[permute[i]] < fairShare)
	    totalPie -= desiredRate[permute[i]];
	else
	    return fairShare;
    }
    notreached;
}
end();
#endif


private uint32 RateFmtToInt (rateFmt_t rate)
begin();
{
    int nz = (rate >> 14) & 0x1;
    int e  = (rate >> 9)  & 0x1f;
    int m  = (rate     )  & 0x1ff;

    if (nz == 0)
	return 0;
    else if (e>=9)
	return (m+512)<<(e-9);
    else
	return (m+512)>>(9-e);
}
end();


private rateFmt_t IntToRateFmt (uint32 value)
begin();
{
    int e, m;
    for (e=31;e>=0;e--)
	if (value & (1<<e))
	    break;
    if (e == -1)
	return 0;
    if (e >= 9)
	m = (value>>(e-9)) & 0x1ff;
    else
	m = (value<<(9-e)) & 0x1ff;

    return (1<<14) | (e<<9) | m;
}
end();

typedef enum {
    color_black,
    color_blue,
    color_green,
    color_aqua,
    color_red,
    color_purple,
    color_orange,
    color_grey,
    attrib_intense=0x800,
    attrib_blink=0x8000
} color_t;

#define FG(x) ((x)<<8)
#define BG(x) ((x)<<12)
#define COLOR_NORMAL   (attrib_intense | FG(color_orange) | BG(color_blue))
#define COLOR_EMPHASIS (attrib_intense | FG(color_red)    | BG(color_blue))

#define BAR_COLOR_VALUE     (attrib_intense | FG(color_green))
#define BAR_COLOR_WATERMARK FG(color_green)
#define BAR_NEVER_VALUE     1
#define BAR_HASH_VALUE      2

private int screen_ypos, screen_xpos, screen_color;

private void DumpScreen (int unit, int ypos, char *msg)
begin();
{
    screen_ypos = ypos;
    screen_xpos = 40*unit;
    DumpStr (msg);
}
end();


private void DumpStr (char *msg)
begin();
{
    screen_color = COLOR_NORMAL;
    while (*msg) {
	if (*msg == '!')
	    screen_color = COLOR_EMPHASIS;
	else {
	    SCREEN_SLAM (screen_ypos, screen_xpos, screen_color+*msg);
	    screen_xpos++;
	}
	msg++;
    }
}
end();


private void DumpHex (uint32 data)
begin();
{
    static char hextable[] = "0123456789abcdef";
    int i;

    for (i=7; i>=0; i--) {
	SCREEN_SLAM (screen_ypos, screen_xpos+i, 
		     screen_color+hextable[data&0xf]);
	data = data>>4;
    }
    SCREEN_SLAM (screen_ypos, screen_xpos+8, screen_color+' ');
    screen_xpos += 9;
}
end();

private void UpdateBar (int newValue, int *highWaterMarkp, int maxValue,
                        int ypos)
begin();
{
    int i;
    int valueIndex=newValue;
    int highWaterMarkIndex = *highWaterMarkp;

    if (newValue > *highWaterMarkp)
	*highWaterMarkp = newValue;

    if (maxValue > 80) {
	valueIndex         *= 80;
	highWaterMarkIndex *= 80;
	valueIndex         /= maxValue;
	highWaterMarkIndex /= maxValue;
    }

    for (i=0; i<valueIndex; i++)
	SCREEN_SLAM (ypos, i, BAR_COLOR_VALUE     | BAR_HASH_VALUE);
    for (; i<highWaterMarkIndex; i++)
	SCREEN_SLAM (ypos, i, BAR_COLOR_WATERMARK | BAR_HASH_VALUE);
    for (; i<80; i++)
	SCREEN_SLAM (ypos, i, BAR_COLOR_WATERMARK | BAR_NEVER_VALUE);
}
end();


private void UpdateScreen (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit=vpcUnitp-vpcUnits;
    int i;

    if (0)
	for (i=0; i<256; i++)
	    SCREEN_SLAM (0, i, 0x1200+i);

    DumpScreen (unit, status_tag_rxBothWatcher, "dt:");
    DumpStr (vpcUnitp->rxdataStats.statusMsg);
    if (vpcUnitp->rxdataStats.hexDataValid);
    DumpHex (vpcUnitp->rxdataStats.hexData);

    DumpStr ("  rm:");
    DumpStr (vpcUnitp->rxrmStats.statusMsg);
    if (vpcUnitp->rxrmStats.hexDataValid);
    DumpHex (vpcUnitp->rxrmStats.hexData);

    DumpScreen (unit, status_tag_datacellcount, "datacount:");
    DumpHex (vpcUnitp->rxdataStats.numCells);
    DumpStr ("in tx:");
    DumpHex (vpcUnitp->rxdataStats.cellsInTx);

    DumpScreen (unit, status_tag_datacelldrop,  "data drop:");
    DumpHex (vpcUnitp->rxdataStats.numDroppedCells);
    DumpStr("ov:");
    if (vpcUnitp->rxdataStats.overrunHappened)
	DumpStr ("!YES");
    else
	DumpStr ("no");
    DumpStr (" EFCI:");
    if (vpcUnitp->congestionHappened)
	DumpStr ("!YES");
    else
	DumpStr ("no");

    DumpScreen (unit, status_tag_rmcellcount,   "   rm cnt:");
    DumpHex (vpcUnitp->rxrmStats.numCells);
    DumpStr ("in tx:");
    DumpHex (vpcUnitp->rxrmStats.cellsInTx);

    DumpScreen (unit, status_tag_rmcelldrop, "  rm drop:");
    DumpHex (vpcUnitp->rxrmStats.numDroppedCells);
    DumpStr ("ov:");
    if (vpcUnitp->rxrmStats.overrunHappened)
	DumpStr ("!YES");
    else
	DumpStr ("no");

    if (vpcConf.abrAlgo) {
	DumpScreen (unit, status_tag_rateDesired, "ccr:");
	for (i=0; i<4; i++)
	    DumpHex (vpcUnitp->desiredRate[i]);
	DumpScreen (unit, status_tag_fairRate, "fair share:");
	DumpHex (vpcUnitp->fairShare);
    }

    DumpScreen (unit, status_tag_startTx, vpcUnitp->startTxMsg);
    if (vpcUnitp->startTxHexDataValid)
	DumpHex (vpcUnitp->startTxHexData);

    DumpScreen (unit, status_tag_waitTx, vpcUnitp->waitTxMsg);
    if (vpcUnitp->waitTxHexDataValid)
	DumpHex (vpcUnitp->waitTxHexData);

    DumpScreen (unit, status_tag_walker, vpcUnitp->walkerMsg);
    DumpHex (vpcUnitp->walkerHexData);

    DumpScreen (unit, status_tag_MACR, "MACR:");
    DumpHex (vpcUnitp->MACR);

    UpdateBar (vpcUnitp->rxdataStats.cellsInTx + vpcUnitp->rxrmStats.cellsInTx,
	       &vpcUnitp->maxBothInTx,
	       vpcConf.outputPortFullThreshold,
	       status_tag_bar+unit);

    if (0) {
	int i;
	for (i=0; i<8; i++) {
	    DumpScreen (unit, status_tag_lastCellHeader+i, "h:");
	    DumpHex (vpcUnitp->lastCellHeader[i][0]);
	    DumpHex (vpcUnitp->lastCellHeader[i][1]);
	}
    }
}
end();
