int vpcKeepGoing = 1;

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/proc.h>

#include <vm/vm.h>
#include <vm/vm_param.h>
#include <vm/vm_extern.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/netisr.h>
#include <net/route.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>

#include <machine/cpu.h>
#include <machine/pio.h>

#include <dev/pci/jetland/include/ijet_coding.h>
#include <dev/pci/jetland/include/ijet_ll.h>
#include <dev/pci/jetland/include/ijet_reg.h>
#include <dev/pci/jetland/src/vpc/vpc_user.h>

#undef min
#define min(a,b) ((a)<(b)?(a):(b))

#undef max
#define max(a,b) ((a)>(b)?(a):(b))

/*------------------------------------------------------------------------*/

extern vm_map_t kernel_map;
extern int hz;

/*------------------------------------------------------------------------*/

#undef debug
#undef DEBUG_PREFIX
#define debug debugr
#define DEBUG_PREFIX "vpcCredit:"

/*------------------------------------------------------------------------
 * my beautiful macros
 *------------------------------------------------------------------------*/

#define VRINDEX(unit,vrp) ((vrp)-ll_vrBase[(unit)])
#define VRP(unit,index) (ll_vrBase[(unit)]+(index))
#define VRPTOVRNODEP(unit,vpcUnitp,vrp) \
        ((vpcUnitp)->mappingTable[VRINDEX(unit,vrp)-(vpcUnitp)->firstVrIndex])

#define SCREEN_SLAM(ypos,xpos,value)                           \
        if (!debug_directTextScreenMapDone) ;                  \
        else debug_directTextScreenp[(ypos)*80+(xpos)]=(value)

#define warnreturn(errorcode, fmt, args...) do {  \
	warn (fmt , ##args);                      \
	return (errorcode);                       \
        } while (0)

#define MALLOC_ARRAY(destptr, count)                                      \
if (((destptr) = malloc ((count)*sizeof(*(destptr)), M_DEVBUF, M_NOWAIT)) \
    != NULL) ;                                                            \
else warnreturn (ENOMEM, "Can't malloc %d bytes!\n",                      \
		 (count)*sizeof(*(destptr)))

#define FREE_ARRAY(p) free(p, M_DEVBUF)

#define ENQUEUE(queue, nodep) do {           \
	if ((queue).headp == NULL)           \
	     (queue).headp = (nodep);        \
        else                                 \
	     (queue).tailp->nextp = (nodep); \
        (queue).tailp = (nodep);             \
	(nodep)->nextp = NULL;               \
    } while(0)

#define DEQUEUE(queue) ({                             \
	typeof((queue).headp) retval = (queue).headp; \
	assert (retval != NULL);                      \
	(queue).headp = (queue).headp->nextp;         \
	retval; })

#define CELL_SIZE          56
#define MAX_FREELISTS      4
#define TOTAL_DATA_CELL_BUFFER  16384
#define TOTAL_CREDIT_CELL_BUFFER 1024

#define CREDIT_CELL_HEADER_OC3(creditVci) ((((creditVci) & 0xf) << 20) | \
                                ((((creditVci) >> 4) & 0xff) << 8) |     \
			        (((creditVci) >> 12) & 0xf))

#define CREDIT_CELL_HEADER_OC12_WORD0(creditVci) \
        (((creditVci) & 0xf000) << 12)

#define CREDIT_CELL_HEADER_OC12_WORD1(creditVci) \
        ((((creditVci) & 0xf) << 12) | (((creditVci) & 0xff0) >> 4))

#define INJECT_VPIVCI     0x110

#define DATA_VCI_BASE     98
#define CREDIT_VCI_BASE  298

#define VC_TO_INDEX(vc) (((vc) - DATA_VCI_BASE)/2)
#define INDEX_TO_VC(index) (2*(index) + DATA_VCI_BASE)

#define CIRC_SUBTRACT(x, y) ((0xffffffff + (x) - (y)) % 0xffffffff)

#define INITIAL_BUFFER_ALLOC 1000			    
#define INITIAL_RTT 1000
#define INITIAL_ALLOC_INTERVAL 2 * INITIAL_RTT
#define INITIAL_MEAS_INTERVAL 2 * INITIAL_ALLOC_INTERVAL
#define RTT_UPDATE_INTERVAL 100000
/*
 * Credit cell raw format...all 56 bytes
 */

typedef struct {
    uint32 header[2];
    uint   msgType:16;
    uint   highvci:8;
    uint   unused1:4;
    uint   lowvci:4;
    uint   value:16;
    uint   valueredux:16;
    uint32 deadbeef;
    uint32 timeStamp;
    uint32 bufferLimit;
    uint32 fwdCells;
    uint32 unused4[6];
} vpcCreditCell_t;

uint32 vpcCreditBuf[524288];        /* 2MB cell buffer */

typedef enum {
    CREDIT_UPDATE,
    RTT_ESTIMATE_REQUEST,
    RTT_ESTIMATE_RESPONSE,
    BUFFER_STATE_UPDATE,
    BUFFER_LIMIT_INDICATE
} vpcCreditMsg_t;

typedef enum {
    status_tag_vci=1,
    status_tag_rxdata,
    status_tag_datadrop,
    status_tag_creditcount,
    status_tag_credit,
    status_tag_n123,
    status_tag_bw,
    status_tag_msgs,
    status_tag_rxmsg,
    status_tag_bar,
    status_tag_mboxwalker = 23,
    status_tag_debug,
} status_tag_t;

/*------------------------------------------------------------------------*
 * my beautiful data structures
 *------------------------------------------------------------------------*/

/* queues of (pointers to VRs).  Used to maintain pools of free VRs for
 * each unit */

typedef struct vrnode {
    struct vrnode *nextp;
    vr_t *vrp;
    bool recycleMe;
} vrnode_t;

typedef struct {
    vrnode_t *headp, *tailp;
} vrqueue_t;

/* queues of buffer descriptors.  Used to remember what part of the buffer
 * is where */

typedef enum {bdnode_tag_rxdata, bdnode_tag_rxcredit, bdnode_tag_txcredit}
bdnode_tag_t;

typedef struct bdnode {
    struct bdnode *nextp;
    bdnode_tag_t tag;
    vrnode_t *vrnodep;
    void *virtualAddress;
    uint32 physicalAddress;
    int length;
} bdnode_t;

typedef struct {
    bdnode_t *headp, *tailp;
} bdqueue_t;

/*------------------------------------------------------------------------
 * all of our other info goes here
 *------------------------------------------------------------------------*/

/* global info */

typedef struct {
    bool goHappened;
    bool vpcHomogenous;

    int       numUnits;                          
    int       numOfVcs;
    int       numRxdataCells;                    
    int       numRxcreditCells;
    int       numTxcreditCells;
    int       creditCellSplitUnit;
    vpivci_t  creditVci;
    int       cellsPerVr;
    int       totalTxQueue;
    int       totalMemory;
    int       kernelLoopCount;

    int       limit;

    /* adaptive credit parameters */
    bool      adaptiveCredit;
    int       allocInterval;
    int       measInterval;
    int       rttInterval;
    uint32    allocTimer;
    uint32    measTimer;
    uint32    rttTimer;
    uint32    RTTmax;
    uint32    RTTmin;
    
    int       traceVcs[2];

    int       vrTableSize;

    bool      injectHappened;
    int       injectVcid;
} vpcConf_t;

/* per unit info */

typedef struct {
    uint32 numRxDataCells;
    uint32 numTxDataCells;
    uint32 numRxCreditCells;
    uint32 numTxCreditCells;
    uint32 numDroppedCells;
    bool overrunHappened;
    uint32 cellsInTx; /* number of cells in txqueue and wait queue */
    uint32 N2Counter;
    bool fcvc;
    uint32 rxBufferAlloc;
    uint32 rxFwdCells;
    uint32 N3, N2;
    int RTTAvg;
    uint32 prevTxCellCount;
    uint32 bw;
} vcStats_t;

typedef struct {
    int otherUnit;

    uint32 linkRate;    /* speed to which the output link is throttled (kbps)*/
    uint32 lineRate;    /* speed of the output link (kbps) */

    uint32 cellsInTx;

    mailbox_t mailbox;
    freelist_t rxdataFreelist[MAX_FREELISTS], rxcreditFreelist;
    vcid_t *dataVcid, *rxCreditVcid, txCreditVcid;
    vr_t *mailboxHeadp;
  
    vr_t **vrTable;
    int firstVrIndex, lastVrIndex;
    vrnode_t **mappingTable;

    vrqueue_t vrPool;
    vrnode_t *vrPoolMemoryp;

    /* Queues for receiving data cells */
    bdqueue_t *rxdataQueue;
    bdnode_t **rxdataQueueMemoryp;
  
    /* Queue for receiving credit cells */
    bdqueue_t rxcreditQueue;
    bdnode_t *rxcreditQueueMemoryp;

    /* Per VC transmit and wait queues for data cells */
    bdqueue_t *txQueue, *waitQueue;
  
    /* Queues for transmitting credit cells */
    bdqueue_t txcreditFreelist, txcreditQueue, waitcreditQueue;
    bdnode_t *txcreditFreelistMemoryp;

    vcStats_t *vcStats, rxcreditStats;

    char **startTxMsg, **waitTxMsg, *walkerMsg;
    bool *startTxHexDataValid, *waitTxHexDataValid;
    uint32 *startTxHexData, *waitTxHexData, walkerHexData;

    char **rxMsg;
    uint32 *rxHexData;
    bool *rxHexDataValid;

    /* keep track of last cell headers */
    uint32 lastCellHeader[8][2];

} vpcUnitInfo_t;

typedef enum {whoami_tag_rxdata, whoami_tag_rxcredit} whoami_tag_t;
typedef enum {setup_tag_tx, setup_tag_rx} setup_tag_t;

private vpcConf_t vpcConf;
private vpcUnitInfo_t *vpcUnits;
private bool rateSchRunning = FALSE;

/*------------------------------------------------------------------------*/

public int vpcopen (dev_t dev, int flags, int mode, struct proc *p);
public int vpcclose (dev_t dev, int flags, int mode, struct proc *p);
public int vpcioctl (dev_t dev, int cmd, caddr_t addr, int flag, 
                     struct proc *p);
private int VpcGo (int numUnits,
                   int numOfVcs,
                   int numRxdataCells,
                   int numRxcreditCells,
		   int numTxcreditCells,
		   vpivci_t creditVci,
		   int creditCellSplitUnit,
                   int cellsPerVr, /*applies only to rxdata */
                   int totalTxQueue,
                   int kernelLoopCount,
		   int limit,
		   bool adaptiveCredit);
private int Setup (vpcUnitInfo_t *vpcUnitp, int otherUnit);
private int BuildFreelist (vpcUnitInfo_t *vpcUnitp, int freelist, 
                           bdqueue_t *qp, bdnode_t **qdatapp,
                           int numVrs, int cellsPerVr, bdnode_tag_t bdnodeTag,
                           void *bufferAddress);
private int VpcSetLinkRate (int unit, uint32 linkRate, uint32 lineRate);
private int VpcAddVc (vpivci_t vpivci, uint16 N2, uint16 N3, bool fcvc);
private int VpcAddTraceVc (vpivci_t vpivci);
private void VpcStop (void);
private void UnSetup (vpcUnitInfo_t *vpcUnitp);
private int VpcPoll (void);
private void RxDataWatcherMan(vpcUnitInfo_t *vpcUnitp, bdqueue_t *qp, 
			      vpivci_t vci);
private void RxCreditWatcherMan(vpcUnitInfo_t *vpcUnitp, bdqueue_t *qp);
private void StartTxMan (vpcUnitInfo_t *vpcUnitp);
private void WaitTxMan (vpcUnitInfo_t *vpcUnitp);
private void StartCreditTxMan (vpcUnitInfo_t *vpcUnitp);
private void WaitCreditTxMan (vpcUnitInfo_t *vpcUnitp);
private void MailboxWalker (vpcUnitInfo_t *vpcUnitp);
private void SetupVr (bdnode_t *bdnodep, setup_tag_t tag);
private void ProcessCreditCell (vpcUnitInfo_t *invpcUnitp, 
                                vpcUnitInfo_t *outvpcUnitp, 
                                vpcCreditCell_t *cellp);
private void SendCreditCell (vpcUnitInfo_t *vpcUnitp, 
			     vpcUnitInfo_t *invpcUnitp, vpcCreditMsg_t msgType,
			     vcid_t vcid, uint32 timeStamp, 
			     uint32 bufferLimit);
private uint32 RateToInterval (vpcUnitInfo_t *vpcUnitp, double rate);
private void AllocUpdate (vpcUnitInfo_t *vpcUnitp);
public void MeasureUpdate (vpcUnitInfo_t *vpcUnitp);
private void RttUpdate (vpcUnitInfo_t *vpcUnitp);
private uint32 inline now(void);
private void UpdateScreen (vpcUnitInfo_t *vpcUnitp);
private void UpdateBar (int unit, int newValue, int maxValue, int ypos);
private void DumpScreen (int unit, int ypos, char *msg);
private void DumpStr (char *msg);
private void DumpHex (uint32 data);

/*------------------------------------------------------------------------
 * the device interface
 *------------------------------------------------------------------------*/

public int vpcopen (dev_t dev, int flags, int mode, struct proc *p)
begin();
{
    return 0;
}
end();


public int vpcclose (dev_t dev, int flags, int mode, struct proc *p)
begin();
{
    if (vpcConf.goHappened) {
	VpcStop ();
	
	free (vpcUnits, M_DEVBUF);
	vpcUnits = NULL;
    }

    return 0;
}
end();


public int
vpcioctl (dev_t dev, int cmd, caddr_t addr, int flag, struct proc *p)
begin();
{
    int32 *data = (int32*)addr;

    switch (cmd) {
    case VPC_IOCTL_GO_CREDIT: {
	return VpcGo (data[0], data[1], data[2], data[3], data[4], data[5],
		      data[6], data[7], data[8], data[9], data[10], data[11]);
    } break;

    case VPC_IOCTL_POLL: {
	vpcConf.goHappened = TRUE;
	return VpcPoll ();
    } break;

    case VPC_IOCTL_SET_LINK_RATE: {
	warn ("IOCTL_SET_LINK_RATE: %d %d\n", data[0], data[1], data[2]);
	return VpcSetLinkRate(data[0], data[1], data[2]);
    } break;

    case VPC_IOCTL_ADD_VC: {
	warn ("IOCTL_ADD_VC: %d %d %d %d %d\n", data[0], data[1], data[2],
	      data[3]);
	return VpcAddVc(data[0], data[1], data[2], data[3]);
    } break;
    
    case VPC_IOCTL_TRACE_VC: {
	warn ("IOCTL_TRACE_VC: %d\n", data[0]);
	return VpcAddTraceVc(data[0]);
    } break;

    default: {
	warnreturn (EINVAL, "ioctl 0x%x not implemented\n", cmd);
    } break;
    }
}
end();


private inline uint32 now(void)
begin();
{
    struct timeval result;
    uint32 retVal;

    microtime(&result);
    retVal = (uint32) result.tv_sec;
    retVal = 1000000 * retVal + result.tv_usec;

    return (retVal);
}
end();

private int VpcGo (int numUnits,
                   int numOfVcs,
                   int numRxdataCells,
                   int numRxcreditCells,
		   int numTxcreditCells,
		   vpivci_t creditVci,
		   int creditCellSplitUnit,
                   int cellsPerVr, /*applies only to rxdata */
                   int totalTxQueue,
                   int kernelLoopCount,
		   int limit,
		   bool adaptiveCredit)
begin();
{
    int error;

    if (vpcUnits)
	return EACCES;

    MALLOC_ARRAY (vpcUnits, NIJET);
    bzero (&vpcConf, sizeof(vpcConf));
    bzero (vpcUnits, NIJET*sizeof(*vpcUnits));

    vpcConf.goHappened = FALSE;

    if (numUnits<0 || numUnits>2)
	warnreturn (EINVAL, "%d is an absurd value for numUnits\n", numUnits);

    vpcConf.numUnits = numUnits;
    vpcConf.numOfVcs = numOfVcs;
    if (vpcConf.numOfVcs > MAX_FREELISTS) {
	warn("Cannot support more than %d VCs\n", MAX_FREELISTS);
	return EINVAL;
    }

    vpcConf.numRxdataCells = numRxdataCells;
    if (vpcConf.numRxdataCells * (vpcConf.numOfVcs-1) > 
	TOTAL_DATA_CELL_BUFFER) {
	warn("Cannot support more than %d cells of data buffer space\n",
	     TOTAL_DATA_CELL_BUFFER);
	return EINVAL;
    }

    vpcConf.numRxcreditCells = numRxcreditCells;
    vpcConf.numTxcreditCells = numTxcreditCells;
    if ((vpcConf.numRxcreditCells > TOTAL_CREDIT_CELL_BUFFER) ||
	(vpcConf.numTxcreditCells > TOTAL_CREDIT_CELL_BUFFER)) {
	warn("Cannot support more than %d cells of credit buffer space\n",
	     TOTAL_CREDIT_CELL_BUFFER);
	return EINVAL;
    }

    vpcConf.creditVci = creditVci;
    vpcConf.creditCellSplitUnit = creditCellSplitUnit;
    vpcConf.cellsPerVr = cellsPerVr;
    vpcConf.totalTxQueue = totalTxQueue;
    vpcConf.totalMemory = totalTxQueue;
    vpcConf.kernelLoopCount = kernelLoopCount;
    vpcConf.limit = limit;
    vpcConf.adaptiveCredit = adaptiveCredit;
    vpcConf.allocInterval = INITIAL_ALLOC_INTERVAL;
    vpcConf.measInterval = INITIAL_MEAS_INTERVAL;
    vpcConf.rttInterval = RTT_UPDATE_INTERVAL;
    if (!vpcConf.adaptiveCredit)
	vpcConf.allocInterval = 1000000; /* Hack */
    vpcConf.allocTimer = now();
    vpcConf.measTimer = now();
    vpcConf.rttTimer = now();
    vpcConf.RTTmax = 0;
    vpcConf.RTTmin = 0xffffffff;
    
    if (numUnits == 1)
	error = Setup (&vpcUnits[0], 0);
    else {
	if (!(error = Setup (&vpcUnits[0], 1)))
	    error = Setup (&vpcUnits[1], 0);
    }

    if (jet_oc12[0] != jet_oc12[1])
	vpcConf.vpcHomogenous = FALSE;
    else
	vpcConf.vpcHomogenous = TRUE;
    /* if (!error)
       vpcConf.goHappened = TRUE;*/
    
    return error;
}
end();


private int Setup (vpcUnitInfo_t *vpcUnitp, int otherUnit)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    vpivci_t vpivci;

    vpcUnitp->otherUnit = otherUnit;
    vpcUnitp->cellsInTx = 0;

    /* allocate the VRs we need and establish the vrnode<-->vrp mapping */
    {
	/* This includes:
	 * incoming data cells
	 * incoming credit cells
	 * outgoing (data or credit) cells
	 * slop
	 */

	/* the 2* here is a little bonus to keep it from crashing */
	int vrsNeeded = 
	    2*vpcConf.numRxdataCells*(vpcConf.numOfVcs-1)/vpcConf.cellsPerVr
	    + vpcConf.numRxcreditCells + vpcConf.numTxcreditCells
	    + vpcConf.totalTxQueue/vpcConf.cellsPerVr
	    + 16;
	int count, i;

	warn ("unit %d: Asking for %d vrs\n", unit, vrsNeeded);
	vpcConf.vrTableSize = vrsNeeded;

	MALLOC_ARRAY (vpcUnitp->vrTable, vrsNeeded);
	if ((count = ll_AllocateVrs (unit, vrsNeeded, vpcUnitp->vrTable))
	    != vrsNeeded)
	    warnreturn (ENETDOWN, 
			"Unit %d couldn't get the %d VRs that it wanted\n",
			unit, vrsNeeded);
	{
	    int mappingSize;

	    /* set up the mapping from vr pointers to vrnode pointers */
	    vpcUnitp->firstVrIndex = VRINDEX(unit, vpcUnitp->vrTable[0]);
	    vpcUnitp->lastVrIndex  = 
		VRINDEX(unit, vpcUnitp->vrTable[vrsNeeded-1]);

	    mappingSize = vpcUnitp->lastVrIndex-vpcUnitp->firstVrIndex+1;
	    MALLOC_ARRAY (vpcUnitp->mappingTable, mappingSize);
	}

	vpcUnitp->vrPool.headp = NULL;
	MALLOC_ARRAY (vpcUnitp->vrPoolMemoryp, vrsNeeded);
	for (i=0; i<vrsNeeded; i++) {
	    vrnode_t *vrnodep = &vpcUnitp->vrPoolMemoryp[i];
	    int index;

	    vrnodep->vrp = vpcUnitp->vrTable[i];
	    index = VRINDEX (unit, vrnodep->vrp) - vpcUnitp->firstVrIndex;
	    vpcUnitp->mappingTable[index] = vrnodep;
	    ENQUEUE (vpcUnitp->vrPool, vrnodep);
	}
    }

    /* we get "optimistic notifications" of receive/transmit done because
     * we peer directly into the VRs.  However, we still process the mailbox
     * in the usual way so that we don't return VRs to the free pool until
     * the ASIC is done with them. */
    {
	static vr_t hostVr = {{0}};
	vrnode_t *vrnodep = DEQUEUE (vpcUnitp->vrPool);
	vrnodep->recycleMe = TRUE;
	vpcUnitp->mailboxHeadp = vrnodep->vrp;

	if ((vpcUnitp->mailbox = ll_AllocMailbox (unit, NULL, NULL,
						  vpcUnitp->mailboxHeadp)) < 0)
	    warnreturn (ENETDOWN, "Unit %d can't open mailbox (error %d)\n",
			unit, vpcUnitp->mailbox);
	*vpcUnitp->mailboxHeadp = hostVr;
    }


    /* the data freelists */
    {
	int i;
	for (i = 1; i < vpcConf.numOfVcs; i++) 
	    vpcUnitp->rxdataFreelist[i] = 0;

	for (i = 1; i < vpcConf.numOfVcs; i++) {	    
	    if ((vpcUnitp->rxdataFreelist[i] = 
		 ll_AllocFreelist (unit, NULL, NULL)) < 0)
		warnreturn (ENETDOWN, 
			    "Unit %d can't allocate data freelist (error %d)\n",
			    unit, vpcUnitp->rxdataFreelist);
	}
    }

    /* the credit cell freelist */
    if ((vpcUnitp->rxcreditFreelist = 
	 ll_AllocFreelist (unit, NULL, NULL)) < 0)
	warnreturn (ENETDOWN, 
		    "Unit %d can't allocate RM freelist (error %d)\n",
		    unit, vpcUnitp->rxcreditFreelist);
    
    /* I should check error return but ha ha, I won't
     * There is a 2M physically/virtually contiguous region.  It is divvied up
     * like so:
     * 1M for adapter 0, 1M for adapter 1
     * for adapter N:
     *  64K for tx credit cells
     *  64K for rx credit cells
     *  64K*14 for data cells
     */

    {
	/* void *nextAddress = (void*)debug_bigBufp + (65536*8*unit); */
	
	void *nextAddress = (void *) vpcCreditBuf + 65536*16*unit;

	bzero (nextAddress, 65536*16);

	(void)BuildFreelist (vpcUnitp, 0, &vpcUnitp->txcreditFreelist,
			     &vpcUnitp->txcreditFreelistMemoryp,
			     vpcConf.numTxcreditCells, 1,
			     bdnode_tag_txcredit, nextAddress);

	nextAddress += 65536;
	(void)BuildFreelist (vpcUnitp, vpcUnitp->rxcreditFreelist,
			     &vpcUnitp->rxcreditQueue, 
			     &vpcUnitp->rxcreditQueueMemoryp,
			     vpcConf.numRxcreditCells, 1,
			     bdnode_tag_rxcredit, nextAddress);
	
	nextAddress += 65536;
	
	MALLOC_ARRAY (vpcUnitp->rxdataQueue, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->rxdataQueueMemoryp, vpcConf.numOfVcs);
	{
	    int i;
	    for (i = 1; i < vpcConf.numOfVcs; i++) {
		(void)BuildFreelist 
		    (vpcUnitp, vpcUnitp->rxdataFreelist[i],
		     &vpcUnitp->rxdataQueue[i], 
		     &vpcUnitp->rxdataQueueMemoryp[i],
		     vpcConf.numRxdataCells/vpcConf.cellsPerVr,
		     vpcConf.cellsPerVr,
		     bdnode_tag_rxdata, nextAddress);
		nextAddress += vpcConf.numRxdataCells * CELL_SIZE;
	    }
	}
    }

    /* allocate the per-VC structures */
    {
	int i;

	MALLOC_ARRAY (vpcUnitp->txQueue, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->waitQueue, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->vcStats, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->startTxMsg, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->rxMsg, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->waitTxMsg, vpcConf.numOfVcs);    
	MALLOC_ARRAY (vpcUnitp->dataVcid, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->rxCreditVcid, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->startTxHexDataValid, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->startTxHexData, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->waitTxHexDataValid, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->waitTxHexData, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->rxHexData, vpcConf.numOfVcs);
	MALLOC_ARRAY (vpcUnitp->rxHexDataValid, vpcConf.numOfVcs);

	/* the tx and wait queues are empty */
	for (i = 0; i < vpcConf.numOfVcs; i++) {
	    vpcUnitp->txQueue[i].headp   = NULL;
	    vpcUnitp->waitQueue[i].headp = NULL;
	    vpcUnitp->startTxMsg[i] = "Init";	    
	    vpcUnitp->waitTxMsg[i] = "Init";
	    vpcUnitp->dataVcid[i] = 0;
	    vpcUnitp->rxCreditVcid[i] = 0;
	    bzero ((char *) &vpcUnitp->vcStats[i], sizeof(vcStats_t));
	}
    }
  
    /* the credit tx and wait queues are empty */
    vpcUnitp->txcreditQueue.headp = NULL;
    vpcUnitp->waitcreditQueue.headp = NULL;

    vpivci = vpcConf.creditVci;
    if (unit != vpcConf.creditCellSplitUnit) {
	/* open credit VC for reception */
	if ((vpcUnitp->rxCreditVcid[1] = ll_OpenRx (unit, 
						 CELL_PROCESSING_RAW,
						 vpcUnitp->mailbox,
						 vpcUnitp->rxcreditFreelist, 
						 vpcUnitp->rxcreditFreelist,
						 &vpivci)) < 0)
	    warnreturn (ENETDOWN, "unit %d can't open rx %d error %d\n",
			unit, vpivci, vpcUnitp->rxCreditVcid);
	warn("Opened Rx. Credit VCI %d as VCID %d\n", vpivci, 
	     vpcUnitp->rxCreditVcid[1]);
	
    }	
    /* open the credit VC for transmit */
    if ((vpcUnitp->txCreditVcid = 
	 ll_OpenCreditTx (unit, CELL_PROCESSING_RAW, vpcUnitp->mailbox, 
			  SCHEDULE_STATIC, vpcConf.creditVci)) < 0)
	warnreturn (ENETDOWN, "unit %d can't open tx: error %d\n",
		    unit, vpcUnitp->txCreditVcid);
    
    warn("Opened Tx. Credit VCI %d as VCID %d\n", vpcConf.creditVci, 
	 vpcUnitp->txCreditVcid); 
    return 0;
}
end();

/* build the data freelist and the corresponding buffer descriptor
 * structure on the host. */

private int BuildFreelist (vpcUnitInfo_t *vpcUnitp, int freelist, 
                           bdqueue_t *qp, bdnode_t **qdatapp, int numVrs, 
                           int cellsPerVr, bdnode_tag_t bdnodeTag, 
                           void *bufferAddress)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    int i;
    vr_t *firstVrp=NULL, *lastVrp=NULL;

    /* warn ("Using virtual address 0x%x, physical address 0x%x\n",
       (uint32)bufferAddress, (uint32)vtophys(bufferAddress)); */

    qp->headp = NULL;
    MALLOC_ARRAY (*qdatapp, numVrs);
    for (i=0; i<numVrs; i++) {
	/* get next bdnode */
	bdnode_t *bdnodep = &(*qdatapp)[i];

	bdnodep->tag = bdnodeTag;

	/* compute its physical address */
	bdnodep->virtualAddress  = bufferAddress;
	bdnodep->physicalAddress = vtophys (bufferAddress);
	bdnodep->length          = cellsPerVr * CELL_SIZE;
	bufferAddress += bdnodep->length;

	if (bdnodeTag != bdnode_tag_txcredit) {
	    /* bind it to a vrnode */
	    bdnodep->vrnodep = DEQUEUE (vpcUnitp->vrPool);
	    if (bdnodep->vrnodep == NULL) {
		warn("Ran out of VRs!! freelist %d, numVrs %d\n", freelist, numVrs);
		return ENOMEM;
	    }
	    /* initialize its VR */
	    SetupVr (bdnodep, setup_tag_rx);
	}

	/* add to receive pool */
	ENQUEUE (*qp, bdnodep);

	/* add corresponding VR to asic freelist */
	if (bdnodeTag != bdnode_tag_txcredit) {
	    if (0)
		warn ("unit %d: appending VR 0x%x to freelist %d\n",
		      unit, VRINDEX(unit, bdnodep->vrnodep->vrp), freelist);
      
	    if (!firstVrp)
		firstVrp = lastVrp = bdnodep->vrnodep->vrp;
	    else {
		lastVrp->vr.nextIndex = VRINDEX (unit, bdnodep->vrnodep->vrp);
		lastVrp = bdnodep->vrnodep->vrp;
	    }
	}
    }
    if ((bdnodeTag != bdnode_tag_txcredit) && (firstVrp)) {
	warn ("Appending to freelist %d startIndex 0x%x endIndex 0x%x\n",
	      freelist, VRINDEX(unit, firstVrp), VRINDEX(unit, lastVrp));
	ll_AppendVrlistToFreelist (unit, freelist, firstVrp, lastVrp);
    }
  
    return 0;
}
end();

private int VpcSetLinkRate (int unit, uint32 linkRate, uint32 lineRate)
begin();
{
    vpcUnitInfo_t *vpcUnitp = &vpcUnits[unit];
    int error;
    vcid_t preferredVc = 0xfffe;

    vpcUnitp->linkRate = linkRate;
    vpcUnitp->lineRate = lineRate;
    warn ("Setting link rate to %d on unit %d\n", vpcUnitp->linkRate, unit);
    /*
     * Here we need to set up the static scheduler with the correct 
     * number of idle slots.
     */
    error = ll_SetLinkRate (unit, linkRate, lineRate, preferredVc);
    return error;
}
end();

private int VpcAddVc (vpivci_t vpivci, uint16 N2, uint16 N3, bool fcvc)
begin();
{
    vpcUnitInfo_t *firstUnitp, *secondUnitp;
    vcid_t vcid;
    int index;
    vpivci_t vpvc;

    /* start the rate scheduler */
    if (!rateSchRunning) {
	uint32 *p;
	uint32 parms[3];
	int i, unit;
	
	for (unit = 0; unit < vpcConf.numUnits; unit++) {
	    if (jet_oc12[unit]) 
		p = (uint32 *) (jet_sramBase[unit] + 0x1a0000);
	    else
		p = (uint32 *) jet_sramBase[unit];
	    
	    for (i = 0x1000; i < 0x3fff; i++) p[i] = 0x0;
	    p[8] = 0x0; p[9] = 32;
	    
	    parms[0] = JET_DYN_VR_LINK_OPCODE << 27;
	    parms[1] = 0;
	    parms[2] = 0;
	    IssueSchedulerCommand_func (unit, parms, 3); 
	  } 
	rateSchRunning = TRUE;
    }

    firstUnitp = &vpcUnits[0];
    secondUnitp = &vpcUnits[1];

    /* open a bi-directional VC */
    vpvc = vpivci;
    index = VC_TO_INDEX(vpvc);
    if ((vcid = ll_OpenRx (0, CELL_PROCESSING_RAW,
			   firstUnitp->mailbox, 
			   firstUnitp->rxdataFreelist[index], 
			   firstUnitp->rxdataFreelist[index], &vpvc)) < 0)
	warnreturn (ENETDOWN, "can't open %d on unit 0: error %d\n", 
		    vpvc, vcid);

    warn ("Opening Rx. VCI %d as VCID %d on unit 0\n", vpivci, vcid);

    firstUnitp->dataVcid[index] = vcid;
    firstUnitp->vcStats[index].N3 = N3;
    firstUnitp->vcStats[index].N2 = N2;
    firstUnitp->vcStats[index].fcvc = fcvc;
    firstUnitp->vcStats[index].RTTAvg = INITIAL_RTT;
    firstUnitp->vcStats[index].rxBufferAlloc = INITIAL_BUFFER_ALLOC;

    if ((vcid = ll_OpenTx (1, CELL_PROCESSING_RAW,
			   secondUnitp->mailbox, SCHEDULE_STATIC,
			   vpivci)) < 0)
	warnreturn (ENETDOWN, "unit 1 can't open tx: error %d\n", vcid);  
    warn ("Opening Tx. VCI %d as VCID %d on unit 1\n", vpivci, vcid);  

    index = VC_TO_INDEX(vcid);
    secondUnitp->dataVcid[index] = vcid;

    /* Give the VC it's rate and inform the rate scheduler */
    {
	uint32 *p;
	uint32 interval;

	if (jet_oc12[1]) 
	    p = (uint32 *) (jet_sramBase[1] + 0x1a0000);
	else
	    p = (uint32 *) jet_sramBase[1];
	
	interval = RateToInterval(secondUnitp, secondUnitp->lineRate);

	p[0x2c00 + vcid] = interval;
	p[8] = vcid;
    }

    vpvc = vpivci;
    index = VC_TO_INDEX(vpvc);
    if ((vcid = ll_OpenRx (1, CELL_PROCESSING_RAW,
			   secondUnitp->mailbox, 
			   secondUnitp->rxdataFreelist[index], 
			   secondUnitp->rxdataFreelist[index], &vpvc)) < 0)
	warnreturn (ENETDOWN, "can't open %d on unit 1: error %d\n", 
		    vpvc, vcid);
    warn ("Opening Rx. VCI %d as VCID %d on unit 1\n", vpivci, vcid);

    secondUnitp->dataVcid[index] = vcid;
    secondUnitp->vcStats[index].N3 = N3;
    secondUnitp->vcStats[index].N2 = N2;
    secondUnitp->vcStats[index].fcvc = fcvc;
    secondUnitp->vcStats[index].RTTAvg = INITIAL_RTT;
    secondUnitp->vcStats[index].rxBufferAlloc = INITIAL_BUFFER_ALLOC;

    if ((vcid = ll_OpenTx (0, CELL_PROCESSING_RAW,
			   firstUnitp->mailbox, SCHEDULE_STATIC,
			   vpivci)) < 0)
	warnreturn (ENETDOWN, "unit 0 can't open tx: error %d\n", vcid);  
    warn ("Opening Tx. VCI %d as VCID %d on unit 0\n", vpivci, vcid);
  
    index = VC_TO_INDEX(vcid);
    firstUnitp->dataVcid[index] = vcid;

    /* Give the VC it's rate and inform the rate scheduler */
    {
	uint32 *p;
	uint32 interval;

	if (jet_oc12[0]) 
	    p = (uint32 *) (jet_sramBase[0] + 0x1a0000);
	else
	    p = (uint32 *) jet_sramBase[0];

	/* interval = RateToInterval(firstUnitp, firstUnitp->linkRate); */
	interval = RateToInterval(firstUnitp, firstUnitp->lineRate);

	p[0x2c00 + vcid] = interval;
	p[8] = vcid;
    }

    /* 
     * Open additional credit VCs on the unit which splits credit cells
     * into different VCs
     */
    if (vpcConf.creditCellSplitUnit != -1) {
	int splitUnit = vpcConf.creditCellSplitUnit;
	vpcUnitInfo_t *vpcUnitp = &vpcUnits[splitUnit];
	vpivci_t creditVci = vpivci - DATA_VCI_BASE + CREDIT_VCI_BASE;

	if ((vpcUnitp->rxCreditVcid[index] = 
	     ll_OpenRx (splitUnit, CELL_PROCESSING_RAW,
			vpcUnitp->mailbox, vpcUnitp->rxcreditFreelist, 
			vpcUnitp->rxcreditFreelist, &creditVci)) < 0)
	    warnreturn (ENETDOWN, "unit %d can't open rx %d error %d\n",
			splitUnit, vpivci, vpcUnitp->rxCreditVcid);	   
	warn("Opened Rx. Credit VCI %d as VCID %d\n", creditVci, 
	     vpcUnitp->rxCreditVcid[index]);
    }
    
    return 0;
}
end();

private int VpcAddTraceVc (vpivci_t vpvc) 
begin();
{
    static int traceIndex = 0;
    vpcUnitInfo_t *vpcUnitp = &vpcUnits[0];
    int i, index = 0;
    
    index = VC_TO_INDEX(vpvc);
    if ((index == 0) || (index >= vpcConf.numOfVcs))
	return (EINVAL, "VCI %d not found\n", vpvc);

    vpcConf.traceVcs[traceIndex] = index;
    warn ("Adding VCI %d as trace VC %d\n", vpvc, traceIndex);
    if (traceIndex == 0) traceIndex = 1;
    else traceIndex = 0;
    return 0;
}
end();

private void VpcStop (void)
begin();
{
    if (vpcConf.injectHappened)
	ll_CloseTx (0, vpcConf.injectVcid);
    UnSetup (&vpcUnits[0]);
    if (vpcConf.numUnits == 2)
	UnSetup (&vpcUnits[1]);
}
end();

private void UnSetup (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    int i;

    warn ("Hi!  Shutting down unit %d\n", unit);
    for (i = 1; i < vpcConf.numOfVcs; i++) {
	if (vpcUnitp->dataVcid[i]) {
	    ll_CloseTx (unit, vpcUnitp->dataVcid[i]);
	    ll_CloseRx (unit, vpcUnitp->dataVcid[i]);
	}
	if (vpcUnitp->rxCreditVcid[i])
	    ll_CloseRx (unit, vpcUnitp->rxCreditVcid[i]);
    }
    ll_CloseTx (unit, vpcUnitp->txCreditVcid);
    ll_FreeMailbox (unit, vpcUnitp->mailbox);
    ll_FreeVrs (unit, vpcConf.vrTableSize, vpcUnitp->vrTable);
    ll_FreeFreelist (unit, vpcUnitp->rxcreditFreelist);

    FREE_ARRAY (vpcUnitp->rxcreditQueueMemoryp);
    for (i = 1; i < vpcConf.numOfVcs; i++) {
	if (vpcUnitp->rxdataFreelist[i])
	    ll_FreeFreelist (unit, vpcUnitp->rxdataFreelist[i]);
	FREE_ARRAY (vpcUnitp->rxdataQueueMemoryp[i]);
    }
    FREE_ARRAY (vpcUnitp->rxdataQueueMemoryp);
    FREE_ARRAY (vpcUnitp->rxdataQueue);
    FREE_ARRAY (vpcUnitp->vrPoolMemoryp);
    FREE_ARRAY (vpcUnitp->mappingTable);
    FREE_ARRAY (vpcUnitp->vrTable);
    FREE_ARRAY (vpcUnitp->txQueue);
    FREE_ARRAY (vpcUnitp->waitQueue);
    FREE_ARRAY (vpcUnitp->vcStats);
    FREE_ARRAY (vpcUnitp->startTxMsg);
    FREE_ARRAY (vpcUnitp->rxMsg);
    FREE_ARRAY (vpcUnitp->waitTxMsg);    
    FREE_ARRAY (vpcUnitp->dataVcid);
    FREE_ARRAY (vpcUnitp->rxCreditVcid);
    FREE_ARRAY (vpcUnitp->startTxHexDataValid);
    FREE_ARRAY (vpcUnitp->startTxHexData);
    FREE_ARRAY (vpcUnitp->waitTxHexDataValid);
    FREE_ARRAY (vpcUnitp->waitTxHexData);
    FREE_ARRAY (vpcUnitp->rxHexData);
    FREE_ARRAY (vpcUnitp->rxHexDataValid);

    rateSchRunning = FALSE;
    vpcConf.goHappened = FALSE;
}
end();


private int VpcPoll (void)
begin();
{
    int i, vc, unit;
    int s;

    for (i=0; i<vpcConf.kernelLoopCount; i++) {
	/* i = 0;
	   s = splimp();
	   while (vpcKeepGoing) {
	   i++; */
	for (unit=0; unit<vpcConf.numUnits; unit++) {
	    for (vc = 1; vc < vpcConf.numOfVcs; vc++)
		RxDataWatcherMan (&vpcUnits[unit], 
				  &vpcUnits[unit].rxdataQueue[vc], 
				  INDEX_TO_VC(vc));
	    RxCreditWatcherMan(&vpcUnits[unit], &vpcUnits[unit].rxcreditQueue);
	    StartTxMan    (&vpcUnits[unit]);
	    WaitTxMan     (&vpcUnits[unit]);
	    StartCreditTxMan (&vpcUnits[unit]);
	    WaitCreditTxMan (&vpcUnits[unit]);
	    MailboxWalker (&vpcUnits[unit]);
	}
	if (vpcConf.adaptiveCredit) {
	    int diff;
	    
	    diff = (int) (now() - vpcConf.rttTimer);	    
	    if (diff > vpcConf.rttInterval) {
		RttUpdate(&vpcUnits[0]);
		RttUpdate(&vpcUnits[1]);
		vpcConf.rttTimer = now() + vpcConf.rttInterval;
	    }
	    
	    diff = (int) (now() - vpcConf.allocTimer);
	    if (diff > vpcConf.allocInterval) {
		AllocUpdate(&vpcUnits[0]);
		AllocUpdate(&vpcUnits[1]);
		vpcConf.allocTimer = now() + vpcConf.allocInterval;
	    }

	    diff = (int) (now() - vpcConf.measTimer);
	    if (diff > vpcConf.measInterval) {
		MeasureUpdate(&vpcUnits[0]);
		MeasureUpdate(&vpcUnits[1]);
		vpcConf.measTimer = now() + vpcConf.measInterval;
	    }	    
	}
	else {
	    int diff = (int) (now() - vpcConf.allocTimer);
	    if (diff > vpcConf.allocInterval) {
		AllocUpdate(&vpcUnits[0]);
		AllocUpdate(&vpcUnits[1]);
		vpcConf.allocTimer = now() + vpcConf.allocInterval;
	    }	    
	}

	/* if (i == 1) {
	   for (unit=0; unit<vpcConf.numUnits; unit++)
	   UpdateScreen (&vpcUnits[unit]);
	   i = 0;
	   }*/
    }

    /* splx(s);*/
    for (unit=0; unit<vpcConf.numUnits; unit++)
	UpdateScreen (&vpcUnits[unit]);

    return 0;
}
end();

/*
 * This routine watches for incoming data cells which are passed through 
 * untouched.
 */

private void RxDataWatcherMan (vpcUnitInfo_t *vpcUnitp, bdqueue_t *qp,
			       vpivci_t vci)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    int cellsPerVr = vpcConf.cellsPerVr;
    int index;
    uint32 header;

    index = VC_TO_INDEX(vci);
    for (;;) {
	bdnode_t *bdnodep = qp->headp;

	/* rx queue totally empty? (should only happen in pathological 
	 * cases) 
	 */
	if (bdnodep == NULL) {
	    vpcUnitp->rxMsg[index]       = "!Empty";
	    vpcUnitp->rxHexDataValid[index]    = FALSE; 
	    break;
	}
	
	/* can't proceed until asic fills incoming vr */
	if (bdnodep->vrnodep->vrp->message.remain != 0) {
	    /* (bdnodep->vrnodep->vrp->message.vcid == 0)) { */
	    vpcUnitp->rxMsg[index] = "wait:";
	    vpcUnitp->rxHexData[index] = VRINDEX(unit,bdnodep->vrnodep->vrp);
	    vpcUnitp->rxHexDataValid[index] = TRUE; 
	    break;
	}
 	DEQUEUE (*qp);
	
	/* if (jet_oc12[unit]) {
	   header = ((uint32 *) bdnodep->virtualAddress)[1];
	   vci = ((header & 0xff) << 4) | ((header & 0xf000) >> 12);
	   header = ((uint32 *) bdnodep->virtualAddress)[0];
	   vci |= (vci | ((header & 0xf000000) >> 12));
	   }
	   else {
	   header = ((uint32 *) bdnodep->virtualAddress)[1];
	   vci = (((header & 0xf) << 12) | ((header & 0xff00) >> 4) | 
	   ((header & 0xf00000) >> 20));
	   } */
	vpcUnitp->rxMsg[index] = "rx vcid:";
	vpcUnitp->rxHexData[index] = vci;
	vpcUnitp->rxHexDataValid[index] = TRUE;

	if (!((index >= 0) && (index < vpcConf.numOfVcs))) {
	    warn("Absurd index %d vcid %d header 0x%x VR 0x%x virt 0x%x\n",  
		 index, vci, header,
		 VRINDEX(unit, bdnodep->vrnodep->vrp), 
		 vtophys(bdnodep->vrnodep->vrp));
	    panic("vpc");
	}
	vpcUnitp->vcStats[index].numRxDataCells += cellsPerVr;

	/* unbind VR from buffer descriptor.  MailboxWalker will eventually
	 * return it to the pool */
	bdnodep->vrnodep->recycleMe = TRUE;
	bdnodep->vrnodep = NULL;

	/* if output port is "full", drop the cells, recycle the buffer desc */

	{
	    int outunit = vpcUnitp->otherUnit;
	    vpcUnitInfo_t *outvpcUnitp = &vpcUnits[outunit];

	    if ((vpcConf.adaptiveCredit && 
		 (outvpcUnitp->cellsInTx > vpcConf.totalMemory)) ||
		(!vpcConf.adaptiveCredit && 
		 (outvpcUnitp->vcStats[index].cellsInTx
		  > outvpcUnitp->vcStats[index].N3 + 
		  outvpcUnitp->vcStats[index].N2))) {
		vpcUnitp->vcStats[index].numDroppedCells += cellsPerVr;
#if 0
		/* 
		 * Treat dropped cells as forwarded cells. Hence update
		 * N2 counter appropriately.
		 */
		outvpcUnitp->vcStats[index].N2Counter += cellsPerVr;
		/* Transmit credit cell if N2 data cells have been sent */
		if (outvpcUnitp->vcStats[index].N2Counter >= 
		    outvpcUnitp->vcStats[index].N2) {
		    outvpcUnitp->vcStats[index].N2Counter -= 
			vpcUnitp->vcStats[index].N2;
		    SendCreditCell(outvpcUnitp, vpcUnitp, 
				   BUFFER_STATE_UPDATE, INDEX_TO_VC(i), 0, 0);
		}
#endif
		vpcUnitp->vcStats[index].overrunHappened = TRUE;
		bdnodep->vrnodep = DEQUEUE (vpcUnitp->vrPool);
		SetupVr (bdnodep, setup_tag_rx);
			       
		/* link to tail of freelist (man rx linking--yay!) */
		assert (qp->headp != NULL);
		qp->tailp->vrnodep->vrp->vr.nextIndex =
		    VRINDEX (unit, bdnodep->vrnodep->vrp);
		ENQUEUE (*qp, bdnodep);
	    } 
	    else {
		/* 
		 * Adjust raw cell format when working between OC-3 and OC-12 
		 * units.
		 */
		if (!vpcConf.vpcHomogenous) {
		    uint32 *headerp;
		    int i;

		    for (i = 0; i < cellsPerVr; i++) {
			headerp = (uint32 *) 
			    &(((uint32 *)bdnodep->virtualAddress)[14*i]);
			if (jet_oc12[unit]) {
			    /* Input is OC12 and output is OC3 */
			    headerp[1] = 
				(headerp[1] << 8) | (headerp[0] >> 24);
			    headerp[0] = (headerp[0] << 8);
			}
			else {
			    /* Input is OC3 and output is OC12 */
			    headerp[0] = 
				(headerp[0] >> 8) | (headerp[1] << 24);
			    headerp[1] = (headerp[1] >> 8);
			}
		    }
		}

		/* get a VR from out's pool, bind to our buffer descriptor */
		bdnodep->vrnodep = DEQUEUE (outvpcUnitp->vrPool);
        
		/* set up the VR. */
		SetupVr (bdnodep, setup_tag_tx);
        
		/* if the tx queue is nonempty, link this VR to the tail of the
		   previous */
        
		if (outvpcUnitp->txQueue[index].headp != NULL)
		    outvpcUnitp->txQueue[index].tailp->
			vrnodep->vrp->vr.nextIndex =
			VRINDEX (outunit, bdnodep->vrnodep->vrp);
        
		/* Put this buffer descriptor on the transmit queue */
		ENQUEUE (outvpcUnitp->txQueue[index], bdnodep);
        
		outvpcUnitp->vcStats[index].cellsInTx += cellsPerVr;
		outvpcUnitp->cellsInTx += cellsPerVr;
	    }
	}
    }
}
end();

/*
 * This routine watches for incoming credit cells cells, 
 * and performs appropriate credit processing. 
 */

private void RxCreditWatcherMan (vpcUnitInfo_t *vpcUnitp, bdqueue_t *qp)
begin();
{
    int unit = vpcUnitp-vpcUnits;

    for (;;) {
	bdnode_t *bdnodep = qp->headp;

	/* rx queue totally empty? (should only happen in 
	 * pathological cases) 
	 */
	if (bdnodep == NULL) {
	    /* vcStatsp->statusMsg       = "!Empty";
	    vcStatsp->hexDataValid    = FALSE; */
	    break;
	}

	/* can't proceed until asic fills incoming vr */
	if (bdnodep->vrnodep->vrp->message.remain != 0) {
	    /* vcStatsp->statusMsg    = "wait";
	       vcStatsp->hexData      = VRINDEX(unit,bdnodep->vrnodep->vrp);
	    vcStatsp->hexDataValid = TRUE; */
	    break;
	}
	DEQUEUE (*qp);
    
	/* vcStatsp->numCells += cellsPerVr; */

	/* unbind VR from buffer descriptor.  MailboxWalker will eventually
	 * return it to the pool */
	bdnodep->vrnodep->recycleMe = TRUE;
	bdnodep->vrnodep = NULL;

	/* Process the credit cell and recycle the buffer desc */

	{
	    int outunit = vpcUnitp->otherUnit;
	    vpcUnitInfo_t *outvpcUnitp = &vpcUnits[outunit];

	    ProcessCreditCell (vpcUnitp, outvpcUnitp,
			       (vpcCreditCell_t *)bdnodep->virtualAddress);
	    bdnodep->vrnodep = DEQUEUE (vpcUnitp->vrPool);
	    SetupVr (bdnodep, setup_tag_rx);
	    assert (qp->headp != NULL);

	    /* Manual Rx. linking to the tail of the credit freelist */
	    qp->tailp->vrnodep->vrp->vr.nextIndex = 
		VRINDEX (unit, bdnodep->vrnodep->vrp);
	    ENQUEUE (*qp, bdnodep);
	    return;
      
	}
    }
}
end();

/*
 * Cycle through the transmit queues.
 * If the transmit vcid is free, transmit the whole chain waiting
 * in the txQueue (the VRs have already been linked together)
 */

private void StartTxMan (vpcUnitInfo_t *vpcUnitp)
begin ();
{
    int unit = vpcUnitp-vpcUnits;
    int i;
    uint32 *p;
    
    if (jet_oc12[unit]) 
	p = (uint32 *) (jet_sramBase[unit] + 0x1a0000);
    else
	p = (uint32 *) jet_sramBase[unit];

    for (i = 1; i < vpcConf.numOfVcs; i++) 
	for (;;) {
	    
	    bdnode_t *headp = vpcUnitp->txQueue[i].headp;
	    bdnode_t *tailp = vpcUnitp->txQueue[i].tailp;

	    /* nothing to transmit? */
	    if (headp == NULL) {
		vpcUnitp->startTxMsg[i] = "Tx empty";
		vpcUnitp->startTxHexDataValid[i] = FALSE;
		break;
	    }
	    
	    /* is my vcid free? */
	    if ((ll_txTableBase[unit][INDEX_TO_VC(i)].vrIndex != 0) ||
		(p[8] != 0x0)) {
		/*if (ll_txTableBase[unit][INDEX_TO_VC(i)].vrIndex != 0) {*/
		vpcUnitp->startTxMsg[i] = "Tx link wait:";
		vpcUnitp->startTxHexData[i] = INDEX_TO_VC(i);
		vpcUnitp->startTxHexDataValid[i] = TRUE;
		break;
	    }
	    
	    /* transmit the number of cells I have credit for */
	    {
		int credit = vpcUnitp->vcStats[i].rxBufferAlloc - 
		    CIRC_SUBTRACT(vpcUnitp->vcStats[i].numTxDataCells, 
				  vpcUnitp->vcStats[i].rxFwdCells);

		int cellsInTx = vpcUnitp->vcStats[i].cellsInTx;
		int bdsWithCredit = credit/vpcConf.cellsPerVr;
		int count;
		bool fcvc = vpcUnitp->vcStats[i].fcvc;
		bdnode_t *breakp;
		
		if (!fcvc || (fcvc && (credit >= cellsInTx))) {

		    /* transmit the whole pile */
		    ll_txTableBase[unit][INDEX_TO_VC(i)].vrIndex = 
			VRINDEX (unit, headp->vrnodep->vrp);
		    /* inform the rate scheduler */
		    if (p[8])
			warn ("p[8] is 0x%x\n", p[8]);
		    p[8] = INDEX_TO_VC(i);

		    /* ll_AppendVrThenScheduleDynamic(unit, 0, INDEX_TO_VC(i),
						   8, headp->vrnodep->vrp,
						   tailp->vrnodep->vrp);*/
		    
		    /*
		     * FIX THIS!!!!!!!
		     */
		    /* if (fcvc)
		       vpcUnitp->vcStats[i].credit -= cellsInTx; */
		    
		    /* Append the new buffer descriptors onto the waitQueue */
		    if (vpcUnitp->waitQueue[i].headp == NULL)
			vpcUnitp->waitQueue[i] = vpcUnitp->txQueue[i];
		    else {
			vpcUnitp->waitQueue[i].tailp->nextp = 
			    vpcUnitp->txQueue[i].headp;
			vpcUnitp->waitQueue[i].tailp = 
			    vpcUnitp->txQueue[i].tailp;
		    }
		    vpcUnitp->txQueue[i].headp = NULL;
		}
		else {
		    /* Adjust the credit value */
		    /* vpcUnitp->vcStats[i].credit -= 
		       bdsWithCredit * vpcConf.cellsPerVr; */
		    
		    /* Find the break point in the Tx queue */
		    breakp = headp;
		    for (count = 0; count < bdsWithCredit - 1; count++) 
			breakp = breakp->nextp;
		    
		    /* Terminate transmit chain at the breakpoint 
		       and schedule it*/
		    breakp->vrnodep->vrp->vr.nextIndex = 0;

		    ll_txTableBase[unit][INDEX_TO_VC(i)].vrIndex = 
			VRINDEX (unit, headp->vrnodep->vrp);
		    /* inform the rate scheduler */
		    if (p[8])
			warn("p[8] is 0x%x\n", p[8]);
		    p[8] = INDEX_TO_VC(i);
		    
		    /* ll_AppendVrThenScheduleDynamic(unit, 0, INDEX_TO_VC(i),
		       8, headp->vrnodep->vrp,
		       tailp->vrnodep->vrp);*/

		    /* Append the scheduled buffers to the wait queue */
		    if (vpcUnitp->waitQueue[i].headp == NULL)
			vpcUnitp->waitQueue[i] = vpcUnitp->txQueue[i];
		    else {
			vpcUnitp->waitQueue[i].tailp->nextp = 
			    vpcUnitp->txQueue[i].headp;
			vpcUnitp->waitQueue[i].tailp = breakp;
		    }
		    
		    /* Adjust the tx queue */
		    vpcUnitp->txQueue[i].headp = breakp->nextp;        
		    breakp->nextp = NULL;
		}
	    }
	}
}
end();


private void WaitTxMan (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int i;
    int unit=vpcUnitp-vpcUnits;
    int inunit = vpcUnitp->otherUnit;
    vpcUnitInfo_t *invpcUnitp = &vpcUnits[inunit];

    for (i = 1; i < vpcConf.numOfVcs; i++)
	for (;;) {
	    bdnode_t *bdnodep = vpcUnitp->waitQueue[i].headp;
      
	    if (bdnodep == NULL) { /* nothing to wait for? */
		vpcUnitp->waitTxMsg[i] = " Wait empty";
		vpcUnitp->waitTxHexDataValid[i] = FALSE;
		break;
	    }
      
	    /* VR not transmitted yet? */
	    if (bdnodep->vrnodep->vrp->message.remain != 0) {
		vpcUnitp->waitTxMsg[i] = " Wait VR:";
		vpcUnitp->waitTxHexDataValid[i] = TRUE;
		vpcUnitp->waitTxHexData[i] = 
		    VRINDEX(unit, bdnodep->vrnodep->vrp); 
		break;
	    }
	    DEQUEUE (vpcUnitp->waitQueue[i]);
      
	    /* unbind vrnode */
	    bdnodep->vrnodep->recycleMe = TRUE;
	    bdnodep->vrnodep = NULL;
      
	    /* return buffer descriptor to the appropriate queue */
	    if (bdnodep->tag == bdnode_tag_rxdata) {
		vpcUnitp->vcStats[i].cellsInTx -= vpcConf.cellsPerVr;
		vpcUnitp->cellsInTx -= vpcConf.cellsPerVr;
		vpcUnitp->vcStats[i].numTxDataCells += vpcConf.cellsPerVr;
		vpcUnitp->vcStats[i].N2Counter += vpcConf.cellsPerVr;

		/* Transmit credit cell if N2 data cells have been sent */
		if (vpcUnitp->vcStats[i].N2Counter >= 
		    vpcUnitp->vcStats[i].N2) {
		    vpcUnitp->vcStats[i].N2Counter -= vpcUnitp->vcStats[i].N2;
		    SendCreditCell(vpcUnitp, invpcUnitp, 
				   BUFFER_STATE_UPDATE, INDEX_TO_VC(i), 0, 0);
		}

		/* get a fresh VR from in's pool */
		bdnodep->vrnodep = DEQUEUE (invpcUnitp->vrPool);
        
		SetupVr (bdnodep, setup_tag_rx);
        
		/* add this buffer descriptor to in's rxQueue, and append
		 * the new VR to in's freelist: MAN RX LINKING---yay! */
        
		assert (invpcUnitp->rxdataQueue[i].headp != NULL);
		invpcUnitp->rxdataQueue[i].tailp->vrnodep->vrp->vr.nextIndex =
		    VRINDEX (inunit, bdnodep->vrnodep->vrp);
		ENQUEUE (invpcUnitp->rxdataQueue[i], bdnodep);
	    } else if (bdnodep->tag == bdnode_tag_rxcredit) {
		warn("Credit buffer being recycled in waitTxMain!!!!\n");
		/* get a fresh VR from in's pool */
		bdnodep->vrnodep = DEQUEUE (vpcUnitp->vrPool);
		SetupVr (bdnodep, setup_tag_rx);
        
		/* add this buffer descriptor to in's rxQueue, and append 
		 * the new VR to in's freelist: MAX RX LINKING----yay! */
		assert (vpcUnitp->rxcreditQueue.headp != NULL);
		vpcUnitp->rxcreditQueue.tailp->vrnodep->vrp->vr.nextIndex =
		    VRINDEX (inunit, bdnodep->vrnodep->vrp);
		ENQUEUE (vpcUnitp->rxcreditQueue, bdnodep);
	    } 
	}
}
end();

/*
 * If the credit transmit vcid is free, transmit the whole chain waiting
 * in the txcreditQueue (the VRs have already been linked together)
 */

private void StartCreditTxMan (vpcUnitInfo_t *vpcUnitp)
begin ();
{
    int unit = vpcUnitp - vpcUnits;

    {
	bdnode_t *headp = vpcUnitp->txcreditQueue.headp;

	/* nothing to transmit? */
	if (headp == NULL) {
	    return;
	}

	/* is my vcid free? */
	if (ll_txTableBase[unit][vpcUnitp->txCreditVcid].vrIndex != 0) {
	    return;
	}

	/* transmit the whole pile */
	ll_txTableBase[unit][vpcUnitp->txCreditVcid].vrIndex =
	    VRINDEX (unit, headp->vrnodep->vrp);
    }

    /* Append the new buffer descriptors onto the waitcreditQueue */
    if (vpcUnitp->waitcreditQueue.headp == NULL)
	vpcUnitp->waitcreditQueue = vpcUnitp->txcreditQueue;
    else {
	vpcUnitp->waitcreditQueue.tailp->nextp = vpcUnitp->txcreditQueue.headp;
	vpcUnitp->waitcreditQueue.tailp = vpcUnitp->txcreditQueue.tailp;
    }
    vpcUnitp->txcreditQueue.headp = NULL;
}
end();

private void WaitCreditTxMan (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit=vpcUnitp-vpcUnits;

    for (;;) {
	bdnode_t *bdnodep = vpcUnitp->waitcreditQueue.headp;

	if (bdnodep == NULL) { /* nothing to wait for? */
	    break;
	}

	/* VR not transmitted yet? */
	if (bdnodep->vrnodep->vrp->message.remain != 0) {
	    break;
	}
	DEQUEUE (vpcUnitp->waitcreditQueue);

	/* unbind vrnode */
	bdnodep->vrnodep->recycleMe = TRUE;
	bdnodep->vrnodep = NULL;

	/* return buffer descriptor to the tx credit freelist */

	ENQUEUE (vpcUnitp->txcreditFreelist, bdnodep);
    }
}
end();

private void MailboxWalker (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit = vpcUnitp-vpcUnits;
    int nextIndex;

    if (vpcUnitp->mailboxHeadp->message.nextIndex == 0) {
	vpcUnitp->walkerMsg = "wait head:";
	vpcUnitp->walkerHexData = VRINDEX (unit, vpcUnitp->mailboxHeadp);
    } else {
	while ((nextIndex=vpcUnitp->mailboxHeadp->message.nextIndex) != 0) {
	    vrnode_t *vrnodep = VRPTOVRNODEP (unit, vpcUnitp, 
					      vpcUnitp->mailboxHeadp);
	    if (!vrnodep->recycleMe) {
		vpcUnitp->walkerMsg = "recycle:";
		vpcUnitp->walkerHexData = 
		    VRINDEX (unit, vpcUnitp->mailboxHeadp);
		return;
	    }
	    ENQUEUE (vpcUnitp->vrPool, vrnodep);
	    if ((nextIndex<vpcUnitp->firstVrIndex)||
		(nextIndex > vpcUnitp->lastVrIndex)) {
		warn ("**** MEGA WARNING: VR 0x%x has nextIndex 0x%x\n",
		      VRINDEX(unit, vpcUnitp->mailboxHeadp), nextIndex);
		panic ("vpc");
	    } else
		vpcUnitp->mailboxHeadp = VRP (unit, nextIndex);
	}
    }
}
end();


private void SetupVr (bdnode_t *bdnodep, setup_tag_t tag)
begin();
{
    static vr_t hostVr = {{0}};

    /* guess what: the "which" parameter doesn't matter */
    (void)tag;

    hostVr.vr.sarLocation           = SAR_HOST;
    hostVr.vr.address               = bdnodep->physicalAddress;
    hostVr.vr.remain                = bdnodep->length;
    /* hostVr.vr.nextIndex defaults to 0 */

    bdnodep->vrnodep->recycleMe = FALSE;
    *(bdnodep->vrnodep->vrp) = hostVr; /* slam into SRAM */
}
end();

private void ProcessCreditCell (vpcUnitInfo_t *invpcUnitp, 
                                vpcUnitInfo_t *outvpcUnitp, 
                                vpcCreditCell_t *cellp)
begin();
{
    int index = 0;
    vpivci_t vpvc;
    
    vpvc = (cellp->highvci<<4)+cellp->lowvci;
    index = VC_TO_INDEX(vpvc);
    if ((index == 0) || (index >= vpcConf.numOfVcs))
	warn ("Credit cell received on unknown VCI %d\n", vpvc);
    /* invpcUnitp->vcStats[index].numRxCreditCells++;*/
    switch (cellp->msgType) {

    case BUFFER_STATE_UPDATE: {
	invpcUnitp->vcStats[index].numRxCreditCells++;
    } break;

    case BUFFER_LIMIT_INDICATE: {
    } break;

    case RTT_ESTIMATE_RESPONSE: {
	uint32 RTTSample; 

	RTTSample = CIRC_SUBTRACT(now(), cellp->timeStamp);
	outvpcUnitp->vcStats[index].RTTAvg += (int)
	    (((int) RTTSample - outvpcUnitp->vcStats[index].RTTAvg) >> 3);

	/* warn("RTTAvg is now %d\n", outvpcUnitp->vcStats[index].RTTAvg);*/

	if (outvpcUnitp->vcStats[index].RTTAvg > vpcConf.RTTmax) {	    
	    vpcConf.RTTmax = outvpcUnitp->vcStats[index].RTTAvg;
	    /* warn ("RTTmax is now %u\n", vpcConf.RTTmax);*/
	}
	if (outvpcUnitp->vcStats[index].RTTAvg < vpcConf.RTTmin) {	    
	    vpcConf.RTTmin = outvpcUnitp->vcStats[index].RTTAvg;
	    /* warn ("RTTmin is now %u\n", vpcConf.RTTmin);*/
	}
	
	/* Adjust the total memory available */
#if 0	
	{
	    vpcConf.totalMemory = (int) 
		((double) (4.0 * vpcConf.RTTmax * 
			   outvpcUnitp->linkRate / (424.0 * 1000.0))) +
		2 * vpcConf.numOfVcs;
	    
	    if (vpcConf.totalMemory > vpcConf.totalTxQueue) {
		warn ("Total memory required greater than queue size\n");
		vpcConf.totalMemory = vpcConf.totalTxQueue;
	    }
	}
#endif
	
	vpcConf.allocInterval = (vpcConf.RTTmax << 1);
	vpcConf.measInterval = (vpcConf.RTTmax << 2);
    } break;
    case RTT_ESTIMATE_REQUEST: {
	SendCreditCell (outvpcUnitp, invpcUnitp, RTT_ESTIMATE_RESPONSE, 
			vpvc, cellp->timeStamp, 0);
    } break;
    default: {
	warn ("Unknown credit message type %d\n", cellp->msgType);
    }
    }
}
end();

private void SendCreditCell (vpcUnitInfo_t *vpcUnitp, 
			     vpcUnitInfo_t *invpcUnitp, vpcCreditMsg_t msgType,
			     vcid_t vcid, uint32 timeStamp, uint32 bufferLimit)
begin();
{
    int inUnit = vpcUnitp->otherUnit;
    vpcCreditCell_t *cellp;
    int index = VC_TO_INDEX(vcid);
    bdnode_t *bdnodep;
    vpivci_t creditVci;

    /* increment the tx credit cell count */
    /* invpcUnitp->vcStats[index].numTxCreditCells++;*/

    /* get a bdnode from the tx freelist */
    bdnodep = invpcUnitp->txcreditFreelist.headp;
    DEQUEUE(invpcUnitp->txcreditFreelist);

    /* set up the bdnode as a credit cell */
    cellp = (vpcCreditCell_t *) bdnodep->virtualAddress;

    if (inUnit == vpcConf.creditCellSplitUnit) 
	creditVci = vcid - DATA_VCI_BASE + CREDIT_VCI_BASE;
    else
	creditVci = vpcConf.creditVci;

    if (jet_oc12[inUnit]) {
	cellp->header[0] = CREDIT_CELL_HEADER_OC12_WORD0(creditVci);
	cellp->header[1] = CREDIT_CELL_HEADER_OC12_WORD1(creditVci);
    }
    else {
	cellp->header[0] = 0;
	cellp->header[1] = CREDIT_CELL_HEADER_OC3(creditVci);
    }
    cellp->msgType = msgType;
    cellp->highvci = (vcid & 0xff0) >> 4;
    cellp->lowvci = (vcid & 0xf);

    switch (msgType) {

    case BUFFER_STATE_UPDATE: {
	invpcUnitp->vcStats[index].numTxCreditCells++;
	cellp->fwdCells = vpcUnitp->vcStats[index].numTxDataCells;
    } break;
    
    case BUFFER_LIMIT_INDICATE: {
	cellp->bufferLimit = bufferLimit;
    } break;

    case RTT_ESTIMATE_RESPONSE: {
	cellp->timeStamp = timeStamp;
    } break;

    case RTT_ESTIMATE_REQUEST: {
	cellp->timeStamp = now();
    } break;

    default: {
	warn ("Unknown credit message type %d\n", msgType);
    } break;
    }
	
    /* get a VR from in's pool and bind it to our buffer descriptor */
    bdnodep->vrnodep = DEQUEUE (invpcUnitp->vrPool);

    /* set up the VR */
    SetupVr (bdnodep, setup_tag_tx);

    /* if the credit tx queue is non-empty link this VR to the tail of the 
       previous */
    if (invpcUnitp->txcreditQueue.headp != NULL)
	invpcUnitp->txcreditQueue.tailp->vrnodep->vrp->vr.nextIndex = 
	    VRINDEX (inUnit, bdnodep->vrnodep->vrp);
  
    /* Put this buffer descriptor on the transmit queue */
    ENQUEUE (invpcUnitp->txcreditQueue, bdnodep);
}
end();

private uint32 RateToInterval (vpcUnitInfo_t *vpcUnitp, double rate)
begin();
{
    uint32 x;
  
    if (rate == 0) x = 0;
    else
	x = (uint32) ((vpcUnitp->lineRate * 16.0)/rate);
    return ((x & 0xffff) | (vpcConf.limit << 16));    
}
end();

public void MeasureUpdate (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int i;
    uint32 cellsForwarded;

    for (i = 1; i < vpcConf.numOfVcs; i++) {
	cellsForwarded = CIRC_SUBTRACT(vpcUnitp->vcStats[i].numTxDataCells, 
			  vpcUnitp->vcStats[i].prevTxCellCount);
	vpcUnitp->vcStats[i].bw = 
	    (cellsForwarded * 1000000) / vpcConf.measInterval;
	vpcUnitp->vcStats[i].prevTxCellCount = 
	    vpcUnitp->vcStats[i].numTxDataCells;
    }
}
end();

private void RttUpdate (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int i;
    vcid_t vcid;
    vpcUnitInfo_t *otherUnitp = &vpcUnits[vpcUnitp->otherUnit];

    for (i = 1; i < vpcConf.numOfVcs; i++) {
	vcid = INDEX_TO_VC(i);
	SendCreditCell (vpcUnitp, otherUnitp, RTT_ESTIMATE_REQUEST, vcid, 
			0, 0);
    }
}
end();

private void AllocUpdate (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int i;
    int totalQueue = 0;
    uint32 allocation;
    double totalRTTBw = 0.0;
    double rttBwFraction;
    int inunit = vpcUnitp->otherUnit;
    vpcUnitInfo_t *invpcUnitp = &vpcUnits[inunit];

    if (!vpcConf.adaptiveCredit) {
	for (i = 1; i < vpcConf.numOfVcs; i++) 
	    /* Send the allocation to the transmitter */
	    SendCreditCell(vpcUnitp, invpcUnitp, BUFFER_LIMIT_INDICATE, 
			   INDEX_TO_VC(i), 0, vpcUnitp->vcStats[i].N2 + 
			   vpcUnitp->vcStats[i].N3);
	return;
    }

    for (i = 1; i < vpcConf.numOfVcs; i++) {
	totalQueue += vpcUnitp->vcStats[i].cellsInTx;
	totalRTTBw += (double)
	    (vpcUnitp->vcStats[i].bw * vpcUnitp->vcStats[i].RTTAvg);
    }

    for (i = 1; i < vpcConf.numOfVcs; i++) {
	if (totalRTTBw != 0.0) {
	    rttBwFraction = (double)
		(vpcUnitp->vcStats[i].bw * vpcUnitp->vcStats[i].RTTAvg /
		 totalRTTBw);
	    allocation = (uint32) 
		((vpcConf.totalMemory / 2.0 - totalQueue - vpcConf.numOfVcs) * 
		 rttBwFraction);
	}
	else
	    allocation = 0;
	
	allocation = max(allocation, 300);  /* minimum 32 cells of credit */
	/* vpcUnitp->vcStats[i].N2 = allocation >> 2;*/
	vpcUnitp->vcStats[i].N2 = 32;
	vpcUnitp->vcStats[i].N3 = (allocation - vpcUnitp->vcStats[i].N2);
	
	/* Send the new allocation to the transmitter */
	SendCreditCell(vpcUnitp, invpcUnitp, BUFFER_LIMIT_INDICATE, 
		       INDEX_TO_VC(i), 0, allocation);

	DumpScreen (i - 1, status_tag_debug, "Alloc:");
	DumpHex (allocation);
    }
}
end();
    
/*
 * Now some debugging routines 
 */

typedef enum {
    color_black,
    color_blue,
    color_green,
    color_aqua,
    color_red,
    color_purple,
    color_orange,
    color_grey,
    attrib_intense=0x800,
    attrib_blink=0x8000
} color_t;

#define FG(x) ((x)<<8)
#define BG(x) ((x)<<12)
#define COLOR_NORMAL   (attrib_intense | FG(color_orange) | BG(color_blue))
#define COLOR_EMPHASIS (attrib_intense | FG(color_red)    | BG(color_blue))
#define BAR_COLOR_VALUE(unit) (attrib_intense | FG(color_green + unit*4))
#define BAR_COLOR_WATERMARK(unit) FG(color_green + unit*4)
#define BAR_NEVER_VALUE     1
#define BAR_HASH_VALUE      2

private int screen_ypos, screen_xpos, screen_color;

private void DumpScreen (int unit, int ypos, char *msg)
begin();
{
    screen_ypos = ypos;
    screen_xpos = 40*unit;
    DumpStr (msg);
}
end();

private void DumpStr (char *msg)
begin();
{
    screen_color = COLOR_NORMAL;
    while (*msg) {
	if (*msg == '!')
	    screen_color = COLOR_EMPHASIS;
	else {
	    SCREEN_SLAM (screen_ypos, screen_xpos, screen_color+*msg);
	    screen_xpos++;
	}
	msg++;
    }
}
end();


private void DumpHex (uint32 data)
begin();
{
    static char hextable[] = "0123456789abcdef";
    int i;

    for (i=7; i>=0; i--) {
	SCREEN_SLAM (screen_ypos, screen_xpos+i, 
		     screen_color+hextable[data&0xf]);
	data = data>>4;
    }
    SCREEN_SLAM (screen_ypos, screen_xpos+8, screen_color+' ');
    screen_xpos += 9;
}
end();

private void UpdateBar (int unit, int newValue, int maxValue, int ypos)
begin();
{
    int i;
    int valueIndex=newValue;
    
    /* warn ("Updating bar for unit %d newValue %d maxValue %d ypos %d\n",
       unit, newValue, maxValue, ypos); */
    if (maxValue == 0) return;

    valueIndex         *= 40;
    valueIndex         /= maxValue;

    for (i=0; i<valueIndex; i++)
	SCREEN_SLAM (ypos, unit*40 + i, BAR_COLOR_VALUE(unit)|BAR_HASH_VALUE);
    for (; i<40; i++)
	SCREEN_SLAM (ypos, unit*40 + i, 
		     BAR_COLOR_WATERMARK(unit)|BAR_NEVER_VALUE);
}
end();

private void UpdateScreen (vpcUnitInfo_t *vpcUnitp)
begin();
{
    int unit=vpcUnitp-vpcUnits;
    int i, traceVc;

    for (traceVc = 0; traceVc < 2; traceVc++) {
	i = vpcConf.traceVcs[traceVc];
	/* warn ("Tracing index %d on unit %d\n", i, unit); */
	DumpScreen (unit, status_tag_vci + 10*traceVc, "VCI:");
	DumpHex (INDEX_TO_VC(i));
	DumpStr (" FCVC:");
	if (vpcUnitp->vcStats[i].fcvc)
	    DumpStr ("!YES");
	else
	    DumpStr ("NO");

	DumpScreen (unit, status_tag_rxdata + 10*traceVc, "rxdata:");
	DumpHex (vpcUnitp->vcStats[i].numRxDataCells);
	DumpStr (" in tx:");
	DumpHex (vpcUnitp->vcStats[i].cellsInTx);
	
	DumpScreen (unit, status_tag_datadrop + 10*traceVc,  "data drop:");
	DumpHex (vpcUnitp->vcStats[i].numDroppedCells);
	DumpStr(" ov:");
	if (vpcUnitp->vcStats[i].overrunHappened)
	    DumpStr ("!YES");
	else
	    DumpStr ("NO");

	DumpScreen (unit, status_tag_creditcount + 10*traceVc, "rxcredit:");
	DumpHex (vpcUnitp->vcStats[i].numRxCreditCells);
	DumpStr (" txcredit:");
	DumpHex (vpcUnitp->vcStats[i].numTxCreditCells);

	DumpScreen (unit, status_tag_credit + 10*traceVc, "credit:");
	DumpHex (vpcUnitp->vcStats[i].rxBufferAlloc - 
		 CIRC_SUBTRACT(vpcUnitp->vcStats[i].numTxDataCells, 
			       vpcUnitp->vcStats[i].rxFwdCells));
	DumpStr ("  N2Count:");
	DumpHex (vpcUnitp->vcStats[i].N2Counter);

	DumpScreen (unit, status_tag_n123 + 10*traceVc, "N2:");
	DumpHex (vpcUnitp->vcStats[i].N2);
	DumpStr (" N3:");
	DumpHex (vpcUnitp->vcStats[i].N3);

	DumpScreen (unit, status_tag_bw + 10*traceVc, "BW:");
	DumpHex (vpcUnitp->vcStats[i].bw);
	DumpStr (" RTT:");
	DumpHex (vpcUnitp->vcStats[i].RTTAvg);

  	DumpScreen (unit, status_tag_msgs + 10*traceVc, 
		    vpcUnitp->startTxMsg[i]);
	if (vpcUnitp->startTxHexDataValid[i])
	    DumpHex (vpcUnitp->startTxHexData[i]);
	else 
	    DumpStr ("        ");
	DumpStr (vpcUnitp->waitTxMsg[i]); 
	if (vpcUnitp->waitTxHexDataValid[i])
	    DumpHex (vpcUnitp->waitTxHexData[i]);    

	DumpScreen (unit, status_tag_rxmsg + 10*traceVc, "Rx:");
	DumpStr (vpcUnitp->rxMsg[i]);
	if (vpcUnitp->rxHexDataValid[i]) 
	    DumpHex(vpcUnitp->rxHexData[i]);
	else
	    DumpStr ("         ");

	UpdateBar (unit, vpcUnitp->vcStats[i].cellsInTx,
		   vpcConf.adaptiveCredit ? vpcConf.totalMemory :
		   vpcUnitp->vcStats[i].N2 + vpcUnitp->vcStats[i].N3,
		   status_tag_bar + 10*traceVc);

    }


    DumpScreen (unit, status_tag_mboxwalker, "Walker:");
    DumpStr (vpcUnitp->walkerMsg);
    DumpHex (vpcUnitp->walkerHexData);
}
end();
