/*
 * Routines for writing to ASIC registers
 */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/systm.h>
#include <sys/param.h>

#include <sys/socket.h>
#include <sys/device.h>
#include <net/if.h>
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>

#include <dev/pci/jetland/include/ijet_coding.h>
#include <dev/pci/jetland/include/ijet_jet.h>
#include <dev/pci/jetland/include/ijet_reg.h>
#include <dev/pci/jetland/include/ijet_ll.h>

#define IssueSchedulerCommand(UNIT,COMMANDP) \
 IssueSchedulerCommand_func (UNIT, (uint32*)(COMMANDP),\
 sizeof(*(COMMANDP))/sizeof(uint32))

public void bh(void) {}

void IssueSchedulerCommand_func (int unit, uint32 *ucmd, int numwords);

public void IssueCommand1 (int unit, uint32 command1);
public void IssueCommand2 (int unit, uint32 command1, uint32 command2);
public void IssueCommand3 (int unit, uint32 command1, uint32 command2,
                           uint32 command3);


public void
reg_AppendVrlistToFreelist (int unit, freelist_t freelist, vr_t *currentTailp,
                            vr_t *headp)
begin (reg_AppendVrlistToFreelist);
{
    static jet_free_buf_link_cmd_t cmd;  /* all fields initially 0 */

    if (currentTailp) {
        cmd.tailIndex = currentTailp-ll_vrBase[unit];
        cmd.vrStatus = currentTailp->status.status;
    } else {
        cmd.tailIndex = 0;
        cmd.vrStatus  = 0;
    }

    ifdebug(0)
        debug ("STATUS is 0x%x\n", cmd.vrStatus);

    cmd.headIndex = headp-ll_vrBase[unit];
    cmd.opcode    = JET_FREE_BUF_LINK_OPCODE;
    cmd.freelist  = freelist;
    
    IssueSchedulerCommand (unit, &cmd);

    ifdebug (0) {
        uint32 vrStatus;
        if (currentTailp)
            vrStatus = currentTailp->status.status;
        else
            vrStatus = 0;
        
        debug ("STATUS NOW 0x%x\n", vrStatus);
    }
}
end (reg_AppendVrlistToFreelist);



public void
reg_AppendVrThenScheduleStatic (int unit, vcid_t vcid, vr_t *headp,
                                vr_t *currentTailp)
begin (reg_AppendVrThenScheduleStatic);
{
    static jet_sta_vr_link_cmd_t cmd;  /* all fields initially 0 */

/* the tail pointer needs to be sent as an SRAM address; the
 * head pointer needs to be sent as a VR Index
 */

    if (currentTailp) {
        cmd.tailPtr  = ((uint32*)currentTailp-(uint32*)jet_sramBase[unit]);
        cmd.vrStatus = currentTailp->status.status;
    } else {
        cmd.tailPtr  = 0;
        cmd.vrStatus = 0;
    }
    
    cmd.headIndex = headp-ll_vrBase[unit];
    cmd.opcode    = JET_STA_VR_LINK_OPCODE;
    cmd.vcid      = vcid;

    IssueSchedulerCommand (unit, &cmd);
}
end (reg_AppendVrThenScheduleStatic);



public void
reg_AppendVrThenScheduleDynamic (int unit, int listNum, vcid_t vcid, 
                                 int burstSize, vr_t *headp,
                                 vr_t *currentTailp)
begin();
{
    static jet_dyn_vr_link_cmd_t cmd;  /* all fields initially 0 */

/* the tail pointer needs to be sent as an SRAM address; the
 * head pointer needs to be sent as a VR Index
 */

    if (currentTailp) {
        cmd.tailPtr  = ((uint32*)currentTailp-(uint32*)jet_sramBase[unit]);
        cmd.vrStatus = currentTailp->status.status;
    } else {
        cmd.tailPtr  = 0;
        cmd.vrStatus = 0;
    }
    
    cmd.headIndex = headp-ll_vrBase[unit];
    cmd.listNum   = listNum;
    cmd.vcid      = vcid;
    cmd.burstSize = burstSize;
    cmd.opcode    = JET_DYN_VR_LINK_OPCODE;

    IssueSchedulerCommand (unit, &cmd);
}
end ();



public void reg_AddVcToDynamicList (int unit, int listNum, vcid_t vcid,
                                    int burstSize)
begin (reg_AddVcToDynamicList);
{
    static jet_dyn_vc_add_cmd_t cmd;  /* fields initially 0 */

    cmd.vcid = vcid;
    cmd.burstSize = burstSize;
    cmd.listNum = listNum;
    cmd.opcode = JET_DYN_VC_ADD_OPCODE;

    IssueSchedulerCommand (unit, &cmd);
}
end (reg_AddVcToDynamicList);



public uint32 reg_ReadStaticRegister (int unit,
                                      jet_static_regs_t whichReg)
begin (reg_ReadStaticRegister);
{
    assert (whichReg < JET_STATIC_NUM_REGISTERS);
    /* FIX:: assert (JET_UNIT_VALID (unit));*/

    switch (whichReg)
    {
        case JET_STATIC_TABLE_BASEADDR:
        case JET_STATIC_TABLE_LIMIT:
        case JET_STATIC_OFFSET: {
            static jet_gp_cmd_rd_cmd_t cmd;  /* fields initially 0 */
            cmd.reg      = whichReg;
            cmd.opcode   = JET_GP_CMD_RD_OPCODE;
            IssueSchedulerCommand (unit, &cmd);
            ret jet_asicBase[unit]->sch_addr;
        } break;

        default: { 
            printf ("ERROR: Reading from Static Register %d "
                    "has not been approved\n", whichReg);
            panic ("");
        } break;
    }
}
end (reg_ReadStaticRegister);


public void
reg_WriteStaticRegister (int unit, jet_static_regs_t whichReg, uint32 value)
begin (reg_WriteStaticRegister);
{
    static jet_gp_cmd_wr_cmd_t cmd;  /* fields initially 0 */

    assert (whichReg < JET_STATIC_NUM_REGISTERS);
    /* FIX:: assert (JET_UNIT_VALID (unit));*/
   
    cmd.opcode = JET_GP_CMD_WR_OPCODE;
    cmd.reg    = whichReg;
    cmd.data   = value;

    switch (whichReg)
    {
        case JET_STATIC_TABLE_BASEADDR:
        case JET_STATIC_TABLE_LIMIT: {
            IssueSchedulerCommand (unit, &cmd);
        } break;

        case JET_STATIC_DRAM_REG0:
        case JET_STATIC_DRAM_REG1:
        case JET_STATIC_DRAM_ADDR: {
            uint32 *p=(uint32*)&cmd;
            while (jet_asicBase[unit]->sch_status.host_pending)
                ;
            jet_asicBase[unit]->sch_addr = p[0];
            jet_asicBase[unit]->sch_addr = p[1];
        } break;

        default: {
            printf ("ERROR: Writing to Static Register %d "
                    "has not been approved\n", whichReg);
            panic ("");
        } break;
    }
}
end (reg_WriteStaticRegister);



public uint32
reg_ReadDynamicRegister (int unit, int listNum, jet_dynamic_regs_t whichReg)
begin (reg_ReadDynamicRegister);
{
    static jet_dyn_reg_rd_cmd_t cmd;  /* fields initially 0 */
    cmd.opcode  = JET_DYN_REG_RD_OPCODE;
    cmd.reg     = whichReg;
    cmd.listNum = listNum;
    IssueSchedulerCommand (unit, &cmd);
    ret jet_asicBase[unit]->sch_addr;
}
end (reg_ReadDynamicRegister);


public void reg_WriteDynamicRegister (int unit, int listNum,
                                      jet_dynamic_regs_t whichReg,
                                      uint32 value)
begin (reg_WriteDynamicRegister);
{
    static jet_dyn_reg_wr_cmd_t cmd;  /* fields initially 0 */
    cmd.opcode  = JET_DYN_REG_WR_OPCODE;
    cmd.reg     = whichReg;
    cmd.listNum = listNum;
    cmd.value   = value;
    IssueSchedulerCommand (unit, &cmd);
}
end (reg_WriteDynamicRegister);



public int reg_ReadStaticBitVector (int unit, int vcid)
begin (reg_ReadStaticBitVector);
{
    ret 0;
}
end (reg_ReadStaticBitVector);


public void reg_WriteStaticBitVector (int unit, int vcid, int bit)
begin (reg_WriteStaticBitVector);
{
    static jet_bit_vec_wr_cmd_t cmd;  /* fields initially 0 */
    if (vcid < JET_NUM_STATIC_VCS) {
        cmd.opcode   = JET_BIT_VEC_WR_OPCODE;
        cmd.vcid     = vcid;
        cmd.bitvalue = bit;
        IssueSchedulerCommand (unit, &cmd);
    }
}
end (reg_WriteStaticBitVector);



void IssueSchedulerCommand_func (int unit, uint32 *ucmd, int numwords)
{
    if (numwords == 1)
        IssueCommand1 (unit, ucmd[0]);
    else {
        jet_intel_hack_cmd_t *secondWord = (jet_intel_hack_cmd_t*)&ucmd[1];
        secondWord->opcode = JET_INTEL_HACK_OPCODE;

        if (numwords == 2)
            IssueCommand2 (unit, ucmd[0], ucmd[1]);
        else if (numwords == 3)
            IssueCommand3 (unit, ucmd[0], ucmd[1], ucmd[2]);
        else
            notreached;
    }
}


#define SCH_TIMEOUT 100

void IssueCommand1 (int unit, uint32 command1)
{
  int timeout;

  for (timeout=0;timeout<SCH_TIMEOUT;timeout++) {
    jet_reg_sch_status_t status;
    
    jet_asicBase[unit]->sch_addr = command1;
    status = jet_asicBase[unit]->sch_status;
    if ((status.busy_status == JET_REG_SCH_STATUS_BUSY_STATUS_ACCESS) &&
        (status.data_src    == JET_REG_SCH_STATUS_DATA_SRC_HOST))
      ;
    else
      break;
  }
  
  if (timeout == SCH_TIMEOUT)
    warn ("IssueCommand1 (0x%x) timed out!\n", command1);
}


void IssueCommand2 (int unit, uint32 command1, uint32 command2)
{
  jet_reg_sch_status_t status;
  int timeout0, timeout1;

  for (;;) {
    jet_asicBase[unit]->sch_addr = command1;
    jet_asicBase[unit]->sch_addr = command2;

    for (timeout0=0; timeout0<SCH_TIMEOUT; timeout0++) {
      status = jet_asicBase[unit]->sch_status;

      if (status.idle)
        return;

      if ((status.busy_status == JET_REG_SCH_STATUS_BUSY_STATUS_ACCESS)&&
          (status.data_src    == JET_REG_SCH_STATUS_DATA_SRC_HOST)) {
        static jet_gp_cmd_rd_cmd_t cmd;  /* fields initially 0 */
        volatile uint32 x;  /* volatile=>don't optimize me away */
        cmd.opcode = JET_GP_CMD_RD_OPCODE;

        jet_asicBase[unit]->sch_addr = *(uint32*)&cmd;
        for (timeout1=0; timeout1<SCH_TIMEOUT; timeout1++) {
          x = jet_asicBase[unit]->sch_addr;
          status = jet_asicBase[unit]->sch_status;
          if (status.idle)
            break;
        }
        if (timeout1 == SCH_TIMEOUT) {
          warn ("Hack blast never succeeded on IssueCommand2(0x%x,0x%x)\n",
                command1, command2);
          return;
        }

        if (0) /* HACK */
          warn ("Hack flag set on IssueCommand2(0x%x,0x%x), retrying\n",
                command1, command2);
        break;
      }
    }
    if (timeout0 == SCH_TIMEOUT) {
      warn ("Scheduler never went idle on IssueCommand2(0x%x,0x%x), "
            "last status was 0x%x\n", command1, command2, *(uint32*)&status);
      return;
    }
  }
}


void IssueCommand3 (int unit, uint32 command1, uint32 command2,
                    uint32 command3)
{
  int timeout;
  for (;;) {
    jet_reg_sch_status_t status;
    volatile uint32 x;

    *jet_hackFlag[unit] = 0;

    jet_asicBase[unit]->sch_addr = command1;
    jet_asicBase[unit]->sch_addr = command2;
    jet_asicBase[unit]->sch_addr = command3;

    for (timeout=0; timeout<SCH_TIMEOUT; timeout++) {
      status = jet_asicBase[unit]->sch_status;
      if ((status.busy_status == JET_REG_SCH_STATUS_BUSY_STATUS_EXE) &&
          (status.data_src    == JET_REG_SCH_STATUS_DATA_SRC_HOST))
        ;
      else break;
    }
    if (timeout == SCH_TIMEOUT) {
      warn ("IssueCommand3(0x%x,0x%x,0x%x) never left host exe\n",
            command1, command2, command3);
      return;
    }

    for (timeout=0; timeout<SCH_TIMEOUT; timeout++) {
      x = jet_asicBase[unit]->sch_addr;
      status = jet_asicBase[unit]->sch_status;
      if (status.idle)
        break;
    }
    if (timeout == SCH_TIMEOUT) {
      warn ("IssueCommand3(0x%x,0x%x,0x%x) never went idle\n",
            command1, command2, command3);
      return;
    }
        
    if (*jet_hackFlag[unit] == 0)
      break;

    warn ("hack flag set--retrying\n");
  }
}

int blowjob = 1;

uint32 IssueCommand2_writeback (int unit, uint32 command1, uint32 command2)
{
  jet_reg_sch_status_t status;
  int timeout0, timeout1;

  warn ("cow::::0x%x 0x%x 0x%x\n", unit, command1, command2);

  if (blowjob)
    return 0xdeadb00b;

  for (;;) {
    jet_asicBase[unit]->sch_addr = command1;
    jet_asicBase[unit]->sch_addr = command2;
    /* get the result back */
    
    status = jet_asicBase[unit]->sch_status;
    warn ("Here my status is 0x%x\n", *(uint32*)&status);

    for (timeout0=0; timeout0<SCH_TIMEOUT; timeout0++) {
      volatile uint32 x = jet_asicBase[unit]->sch_addr;
      status = jet_asicBase[unit]->sch_status;
      warn ("Here(%d) my status is 0x%x\n", timeout0, *(uint32*)&status);
      if (status.idle)
        return x;
    }
    if (timeout0 == SCH_TIMEOUT)
      warn ("Didn't get my word back on (0x%x,0x%x)\n",
            command1, command2);

    /* the "lost word" case */
    if ((status.busy_status == JET_REG_SCH_STATUS_BUSY_STATUS_ACCESS)&&
        (status.data_src    == JET_REG_SCH_STATUS_DATA_SRC_HOST)) {
      static jet_gp_cmd_rd_cmd_t cmd;  /* fields initially 0 */
      cmd.opcode = JET_GP_CMD_RD_OPCODE;
      
      jet_asicBase[unit]->sch_addr = *(uint32*)&cmd;
      for (timeout1=0; timeout1<SCH_TIMEOUT; timeout1++) {
        volatile uint32 x = jet_asicBase[unit]->sch_addr;
        status = jet_asicBase[unit]->sch_status;
        if (status.idle)
          break;
      }
      if (timeout1 == SCH_TIMEOUT) {
        warn ("Hack blast never succeeded on IssueCommand2(0x%x,0x%x)\n",
              command1, command2);
        return 0xdeadbeef;
      }

      warn ("Hack flag set on IssueCommand2(0x%x,0x%x), retrying\n",
            command1, command2);
      break;
    } else
      warn ("I have no idea what's going on.  Time to panic?\n");
  }
}
