#ifndef _IJET_JET_H_
#define _IJET_JET_H_

#include <ijet.h>
#include <dev/pci/jetland/include/ijet_coding.h>

typedef int mailbox_t;
typedef int freelist_t;

typedef int32 vpivci_t;

typedef int dynamicList_t;

typedef enum { CELL_PROCESSING_AAL5, CELL_PROCESSING_DISCARD_HEADER,
    CELL_PROCESSING_RAW } cellProcessing_t;

typedef enum { SCHEDULE_STATIC, SCHEDULE_DYNAMIC } scheduling_t;

typedef enum { SAR_LOCAL, SAR_HOST } sar_t;

typedef uint16 staticTableEntry_t;
typedef uint32 dynamicListEntry_t;

typedef enum {
    JET_DYNAMIC_LIST_END_MARKER=0xfffd,
    JET_STATIC_SCHEDULE_DYNAMIC,
    JET_STATIC_SCHEDULE_IDLE
} vcid_t;

#define AAL5_PT_END_OF_PDU_MASK             1
#define AAL5_PT_CONGESTION_ENCOUNTERED_MASK 2

/*------------------------------------------------------------------------*/

#define JET_MAX_REAL_MAILBOXES  8
#define JET_MAX_FREELISTS       16

#define JET_MAILBOXOK(MBOX) ((MBOX >= 0) && (MBOX < JET_MAX_REAL_MAILBOXES))
#define JET_FREELISTOK(FL)  ((FL   >= 0) && (FL   < JET_MAX_FREELISTS))

/*------------------------------------------------------------------------*/

#define JET_INITIAL_CRC     (0xffffffff)

#define STATIC_FIRST_VCID   0
#define JET_NUM_STATIC_VCS  256
#define DYNAMIC_FIRST_VCID  256

#define JET_NULL_CELL_VPIVCI    0

/*------------------------------------------------------------------------*
 * Cell
 *------------------------------------------------------------------------*/

/*
 * Note: sizeof(cell_t) is probably not 53, due to the way the compiler pads
 * structs.  There is probably no easy way to get around this.
 */

typedef struct
{
    uint     gfc: 4;
    uint  vpivci:24;
    uint     pti: 3;
    uint     clp: 1;    /* or does clp come before pt? */

    uint8 hec;

    uint8 payload [48];
} cell_t;

/*--------------------------------------------------------------------------*
 * VR format
 *--------------------------------------------------------------------------*/

typedef enum {VR_MESSAGE_TX, VR_MESSAGE_RX} vr_message_rxtx_t;

typedef union {
    struct {
        uint nextIndex            :18;
        uint depadding            : 1;
        uint sarLocation          : 1; /* 0=local, 1=host */
        uint clp                  : 1;
        uint pti                  : 1;
        uint gfc                  : 4;
        uint endOfPdu             : 1;
        uint freeListMarker       : 1;
        uint bufferLengthIndicator: 4;

        uint pduLength            :16;
        uint cpcsUuCpi            :16;
        uint length               :16;
        uint remain               :16;

        uint address              :32;
    } vr;

/* the highlighted fields are the ones that are different for the message
 * type
 */

    struct {
        uint nextIndex            :18;
        uint depadding            : 1;
        uint rxTx                 : 1; /**/ /* 0=tx, 1=rx */
        uint clp                  : 1;
        uint pti                  : 1;
        uint gfc                  : 4;
        uint endOfPdu             : 1;
        uint crcError             : 1; /**/
        uint bufferLengthIndicator: 4;

        uint cpcsUuCpi            :16;     /**/
        uint vcid                 :16;     /**/
        uint pduLength            :16;     /**/
        uint remain               :16;

        uint address              :32;
    } message;

    struct {
        uint nextIndex:18;
        uint status   :14;
    } status;
} vr_t;

/*--------------------------------------------------------------------------*
 * Rx Table, Tx Table
 *--------------------------------------------------------------------------*/

typedef struct
{
    uint vrIndex        :18;
    uint cellProcessing : 2;
    uint mailbox        : 3;
    uint firstBuffer    : 1;
    uint freelistA      : 4;
    uint freelistB      : 4;

    uint32 crc;
} rxTableEntry_t;

typedef struct
{
    uint creditValue    :16;
    uint dynamicListMask:16;
    uint vrIndex        :18;
    uint cellProcessing : 2;
    uint mailbox        : 3;

    uint32 crc;
    uint32 vpivci       : 24;
} txTableEntry_t;

/*--------------------------------------------------------------------------*
 * Notification buffer entry
 *--------------------------------------------------------------------------*/

typedef struct
{
    uint vcid     :16;
    uint freelist :16;
} notification_t;

/*--------------------------------------------------------------------------*
 * Registers broken down into their fields
 *--------------------------------------------------------------------------*/

typedef struct {
    uint linkLevelFlowControl     :1;
    uint rxVcExtraction           :3;
    uint rxCollectOutOfRangeVcs   :1;
    uint notificationBufferEnable :1;
} jet_reg_cfg_reg_t;

typedef struct {
    uint enable :1;
} jet_reg_cmd_reg_t;

typedef struct {
    uint rxCount :4;
    uint txCount :4;
} jet_reg_txrx_arb_t;

typedef enum {
    JET_REG_SCH_STATUS_DATA_SRC_960=1,
    JET_REG_SCH_STATUS_DATA_SRC_HOST,
    JET_REG_SCH_STATUS_DATA_SRC_SAR
} jet_reg_sch_status_data_src_t;

typedef enum {
    JET_REG_SCH_STATUS_BUSY_STATUS_ACCESS=1,
    JET_REG_SCH_STATUS_BUSY_STATUS_EXE,
    JET_REG_SCH_STATUS_BUSY_STATUS_WRITE_BACK
} jet_reg_sch_status_busy_status_t;

typedef struct {
    uint i960_pending:1;
    uint host_pending:1;
    uint data_src    :2;
    uint busy_status :2;
    uint idle        :1;
    uint unused1    :24;
    uint timeout     :1;
} jet_reg_sch_status_t;

/*--------------------------------------------------------------------------* 
 * All Jet Engine Registers
 *--------------------------------------------------------------------------*/

typedef struct {
    uint32 tb_rega [3];
    uint32 vr_rega [4];

    uint32 unused0 [9];

    uint32 tb_regb [3];
    uint32 vr_regb [4];

    uint32 unused1 [9];

    uint32 isr_mbx;
    uint32 isr_msc;
    uint32 imr_mbx;
    uint32 imr_msc;

    jet_reg_cfg_reg_t   cfg_reg;        /* 0x90 */
    jet_reg_cmd_reg_t   cmd_reg;
    jet_reg_txrx_arb_t  txrx_arb;

    uint32 h960ack;
    uint32 h960int;

    uint32 rx_vtba;
    uint32 tx_vtba;
    uint32 rx_vtba_msk;

    uint32 vc_oam;
    uint32 vc_err;

    uint32 flshpgadd;

    uint32 unused2;

    uint32 new_ptr;
    uint32 tail_ptr;

    uint32 unused3 [14];

    uint32 mbox [JET_MAX_REAL_MAILBOXES];

    uint32 nbuff;

    uint32 unused4 [7];

    uint32 freelist [JET_MAX_FREELISTS];

    uint32 t1c_fifo;
    uint32 r1c_fifo;

    uint32 hdr_reg1;
    uint32 hdr_reg2;

    uint32 unused5 [28];

    uint32 tx_idle_cell_gen_enable;

    uint32 rx_empty;
    uint32 rx_full;
    uint32 rx_half;
    uint32 tx_empty;
    uint32 tx_full;
    uint32 tx_half;

    uint32 unused6 [25];

    uint32 io0;
    uint32 io1;
    uint32 arbiter;

    uint32 unused7 [29];

    uint32 sch_addr;
    jet_reg_sch_status_t sch_status;
} jet_regs_t;

/*--------------------------------------------------------------------------*
 * Static scheduler registers (these can only be accessed indirectly)
 *--------------------------------------------------------------------------*/

typedef enum {
    JET_STATIC_TABLE_BASEADDR=0x10,
    JET_STATIC_TABLE_LIMIT,
    JET_STATIC_OFFSET,

    JET_STATIC_DRAM_REG0=0x1d,
    JET_STATIC_DRAM_REG1,
    JET_STATIC_DRAM_ADDR,
    JET_STATIC_NUM_REGISTERS
} jet_static_regs_t;

typedef struct {
    uint stat_table_size    :16;
    uint unused             : 4;
    uint bit_vector_en      : 1;
    uint stat_sch_en        : 1;
    uint cmd_proc_timeout_en: 1;
} jet_stat_table_limit_t;

/*------------------------------------------------------------------------*
 * Scheduler commands
 *------------------------------------------------------------------------*/

typedef enum {
        JET_GP_CMD_WR_OPCODE=1,
        JET_GP_CMD_RD_OPCODE=4,
    JET_FREE_BUF_LINK_OPCODE=5,
       JET_BIT_VEC_WR_OPCODE=8,
       JET_BIT_VEC_RD_OPCODE=0x0c,
       JET_DYN_REG_RD_OPCODE=0x10,
       JET_DYN_REG_WR_OPCODE=0x11,
      JET_DYN_VR_LINK_OPCODE=0x12,
       JET_DYN_VC_ADD_OPCODE=0x14,
       JET_INTEL_HACK_OPCODE=0x15,
      JET_STA_VR_LINK_OPCODE=0x1a
} jet_scheduler_opcode_t;

/* INTEL HACK */

typedef struct {
    uint unused:27;
    uint opcode: 5;
} jet_intel_hack_cmd_t;

/* FREE BUF LINK */
  
typedef struct {
    uint tailIndex:18;
    uint freelist : 4;
    uint unused   : 5;
    uint opcode   : 5;

    uint headIndex:18;
    uint vrStatus :14;
} jet_free_buf_link_cmd_t;

/* STA VR LINK */

typedef struct {
    uint vcid    :16;
    uint unused0 :11;
    uint opcode  : 5;

    uint tailPtr :20;  /* SRAM address of tail ptr (addr>>2) */
    uint unused1 :12;

    uint headIndex:18;  /* VR Index of head ptr (addr>>4) */
    uint vrStatus :14;
} jet_sta_vr_link_cmd_t;

/* DYN VR LINK */

typedef struct {
    uint vcid     :16;
    uint burstSize: 6;
    uint listNum  : 4;
    uint unused0  : 1;
    uint opcode   : 5;

    uint tailPtr  :20;  /* SRAM address of tail ptr (addr>>2) */
    uint unused1  :12;

    uint headIndex:18;  /* VR Index of head ptr (addr>>4) */
    uint vrStatus :14;
} jet_dyn_vr_link_cmd_t;

/* DYN VC ADD */

typedef struct {
    uint vcid     :16;
    uint burstSize: 6;
    uint listNum  : 4;
    uint unused   : 1;
    uint opcode   : 5;
} jet_dyn_vc_add_cmd_t;

/* GP CMD RD */

typedef struct {
    uint reg   : 5;
    uint unused:22;
    uint opcode: 5;
} jet_gp_cmd_rd_cmd_t;

/* GP CMD WR */

typedef struct {
    uint reg     : 5;
    uint unused  :22;
    uint opcode  : 5;

    uint data    :32;
} jet_gp_cmd_wr_cmd_t;

/* DYN REG RD */

typedef struct {
    uint reg    : 4;
    uint unused0:18;
    uint listNum: 4;
    uint unused1: 1;
    uint opcode : 5;
} jet_dyn_reg_rd_cmd_t;

/* DYN REG WR */

typedef struct {
    uint reg     : 4;
    uint unused0 :18;
    uint listNum : 4;
    uint unused1 : 1;
    uint opcode  : 5;

    uint value   :32;
} jet_dyn_reg_wr_cmd_t;

/* BIT VEC WR */

typedef struct {
    uint vcid    : 8;
    uint unused  : 18;
    uint bitvalue: 1;
    uint opcode  : 5;
} jet_bit_vec_wr_cmd_t;

/*--------------------------------------------------------------------------*
 * Dynamic registers (these can only be accessed indirectly)
 *--------------------------------------------------------------------------*/

/* per-list registers */

typedef enum {
    JET_DYNAMIC_LIST_HEAD_PTR=2,
    JET_DYNAMIC_LIST_TAIL_PTR,
    JET_DYNAMIC_LIST_MCNT_RELOAD,

    JET_DYNAMIC_LIST_BIT_REG=6,

    JET_DYNAMIC_LIST_LM_VC_BURST_ADDR=10,
    JET_DYNAMIC_LIST_LM_HEAD_TAIL_ADDR,
    JET_DYNAMIC_LIST_MCNT_ADDR
} jet_dynamic_regs_t;

/* global registers */

typedef enum {
    JET_DYNAMIC_GLOBAL_TABLE_BASE,
    JET_DYNAMIC_TB_REG0,
    JET_DYNAMIC_TB_REG1,
    JET_DYNAMIC_TB_MCNT,
    JET_DYNAMIC_GLOBAL_PRIORITIES,
    JET_DYNAMIC_GLOBAL_VC_BASE,
    JET_DYNAMIC_GLOBAL_SIZEARB,
    JET_DYNAMIC_TB_CP_DIN,
    JET_DYNAMIC_RB_CP_ADDR,
    JET_DYNAMIC_TB_LBI_DIN,
    JET_DYNAMIC_TB_SAR_INT,
    JET_DYNAMIC_TB_LIMIT,
    JET_DYNAMIC_TB_REG2,
    JET_DYNAMIC_TB_REG3
} jet_dynamicGlobalRegs_t;
 
#define JET_DYNAMIC_ARBITRATION_DISCIPLINE 0  /* HACK */

/*--------------------------------------------------------------------------*
 * This gives symbolic names to all the interrupts generated by the ASIC
 *
 * This table needs to match the order of the bits in the hardware.
 *--------------------------------------------------------------------------*/

typedef enum {
/* interrupt bits from isr_mbx */

    INTR_BUFSW_MBOX0=0,
    INTR_EOPDU_MBOX0,
    INTR_BUFSW_MBOX1,
    INTR_EOPDU_MBOX1,
    INTR_BUFSW_MBOX2,
    INTR_EOPDU_MBOX2,
    INTR_BUFSW_MBOX3,
    INTR_EOPDU_MBOX3,
    INTR_BUFSW_MBOX4,
    INTR_EOPDU_MBOX4,
    INTR_BUFSW_MBOX5,
    INTR_EOPDU_MBOX5,
    INTR_BUFSW_MBOX6,
    INTR_EOPDU_MBOX6,
    INTR_BUFSW_MBOX7,
    INTR_EOPDU_MBOX7,

    INTR_BUFSW_MBOX8,
    INTR_EOPDU_MBOX8,

/* interrupt bits from isr_msc */

    INTR_FREELIST0,
    INTR_FREELIST1,
    INTR_FREELIST2,
    INTR_FREELIST3,
    INTR_FREELIST4,
    INTR_FREELIST5,
    INTR_FREELIST6,
    INTR_FREELIST7,
    INTR_FREELIST8,
    INTR_FREELIST9,
    INTR_FREELIST10,
    INTR_FREELIST11,
    INTR_FREELIST12,
    INTR_FREELIST13,
    INTR_FREELIST14,
    INTR_FREELIST15,

    INTR_TX_INCOMPLETE_CELL,
    INTR_RX_PARTIAL_CELL,
    INTR_RX_OUT_OF_RANGE_VC,
    INTR_END_OF_FREELIST,
    INTR_960_HOST,
    INTR_SONET_HOST,

    INTR_NUM_INTERRUPTS

} intr_interrupt_t;

#define INTR_COMPUTE_FREELIST_INDEX(N)  (INTR_FREELIST0+(N))
#define INTR_COMPUTE_BUFSW_INDEX(N)     (INTR_BUFSW_MBOX0+2*(N))
#define INTR_COMPUTE_EOPDU_INDEX(N)     (INTR_EOPDU_MBOX0+2*(N))

/*
 * intr_interrupt_t is divided into two regions: those interrupts
 * reported by isr_mbx, and those reported by isr_msc.  These constants
 * define the starting number and length of those regions
 */

#define INTR_MSC_OFFSET         INTR_FREELIST0
#define INTR_NUM_MSC_INTERRUPTS (INTR_SONET_HOST-INTR_MSC_OFFSET+1)

#define INTR_MBX_OFFSET         INTR_BUFSW_MBOX0
#define INTR_NUM_MBX_INTERRUPTS (INTR_EOPDU_MBOX8-INTR_BUFSW_MBOX0+1)

/*--------------------------------------------------------------------------*
 * error codes for jet routines.  Error if <0, success if 0.
 *--------------------------------------------------------------------------*/

#define SUCCESS (0)

#define LL_ERROR_BASE   (-1999)
#define INTR_ERROR_BASE (-2999)
#define DIAG_ERROR_BASE (-3999)

/*--------------------------------------------------------------------------*
 * other stuff
 *--------------------------------------------------------------------------*/

#define JET_UNIT_VALID(unit) ((unit)>=0 && (unit)<NIJET)

#define BITMASK_BIT(mask,bitnum) ((mask) & (1 << (bitnum)))
#define BITMASK_SET0(object,bitnum) ((object) &= ~(1 << (bitnum)))
#define BITMASK_SET1(object,bitnum) ((object) |=  (1 << (bitnum)))

#endif /* _IJET_JET_H_ */
