#include <sys/types.h>
#include <sys/param.h>
#include <fcntl.h>
#include <sys/proc.h>
#include <sys/systm.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <sys/malloc.h>
#include <vm/vm.h>
#include <sys/mbuf.h>
#include <string.h>

#include <sys/errno.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/syslog.h>
#include <sys/device.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/if_llc.h>
#include <net/netisr.h>
#include <net/route.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>

#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>

#include <dev/pci/jetland/include/ijet_coding.h>
#include <dev/pci/jetland/include/ijet_ll.h>
#include <dev/pci/jetland/include/jetmsg_internals.h>
#include <dev/pci/jetland/include/jetmsg_user.h>


extern vm_map_t kernel_map;
extern int hz;

private bool upAndRunning;
private int numRxBuffers, numTxBuffers, numSnapBuffers;
private int numjds, numVrs, numBuffers;
private mailbox_t mailbox;
private freelist_t freelist;
private vr_t *mailboxHead;
private vm_offset_t virtBase;      /* the first buffer in our contig block */
private vr_t *vrBase;              /* the first VR in our contiguous block */
private bufferdesc_t *bufferdesc;  /* the buffer descriptor array */
private jetdesc_t    *jetdesc;     /* the file desriptor array */
private int txFreelistHead;
private int txFreelistTail;        /* head and tail indices of tx freelist */
private int snapRxFreelistHead;    /* head and tail indices of snap buffer...*/
private int snapRxFreelistTail;    /* ...freelist */
private int mailboxEvent;          /* unique pointer for sleep/wakeup */
private int snapEvent;
private struct selinfo selinfo;
private uint32 tempBuffer [PAGESIZE/sizeof(uint32)];
private int SKIP_JD;		   /* SKIP file descriptor */
 
extern inline void DumpVr (char *message, vr_t *vrp)
{
  uint32 *p=(uint32*)vrp;
  debug ("%s--0x%x:%x %x %x %x\n", message,VRINDEX(vrp),p[0],p[1],p[2],p[3]);
}

public int jetmsgopen (dev_t dev, int flags, int mode, struct proc *p)
begin();
{
  int jd = minor (dev);
  debug ("dev=0x%x flags=%d mode=%d proc=0x%x, jd=%d\n", dev, flags, mode,
         (uint32)p, jd);

  if (jd == 0)  /* OK to open device 0 without initializing first */
    return 0;
  if ((jd >= numjds) || !upAndRunning)
    return ENXIO;
  if (jetdesc[jd].state & JD_STATE_OPEN)
    return EBUSY;
  if (jd >= numjds)
    return ENXIO;

  jetdesc[jd].state = JD_STATE_OPEN;
  if (flags & FREAD)
    jetdesc[jd].state |= JD_STATE_CAN_BIND_READ;
  if (flags & FWRITE)
    jetdesc[jd].state |= JD_STATE_CAN_BIND_WRITE;

  return 0;
}
end();


public int jetmsgclose (dev_t dev, int flags, int mode, struct proc *p)
begin();
{
  int jd = minor (dev);
  debug ("dev=0x%x flags=%d mode=%d proc=0x%x, jd=%d\n", dev, flags, mode,
         (uint32)p, jd);

  if (jd == 0)
    return 0;

  if (jetdesc[jd].state & JD_STATE_READ_VC)
    ll_CloseRx (UNIT, jetdesc[jd].rx.msg.vcid);
  if (jetdesc[jd].state & JD_STATE_WRITE_VC)
    ll_CloseTx (UNIT, jetdesc[jd].tx.msg.vcid);
  jetdesc[jd].state = 0;

  return 0;
}
end();


public int jetmsgread (dev_t dev, struct uio *uio, int ioflag)
begin();
{
  int jd = minor (dev);
  int length = CountUio (uio);
  int xferamt;
  void *source;
  bufferdesc_t *bd;
  int bdIndex;
  bool snapLand, skipLand;
  struct llc llc;

  debug ("dev=0x%x, uio=0x%x, ioflag=%d, jd=%d\n", dev, (uint32)uio, ioflag,
         jd);

  if (jd == 0) /* read and write don't work for minor device 0 */
    return EACCES;
  if (!(jetdesc[jd].state & JD_STATE_CAN_READ))
    return EACCES;

  snapLand = jetdesc[jd].state & JD_STATE_READ_SNAP_DEMUX;
  skipLand = jetdesc[jd].state & JD_STATE_READ_SKIP;
    
  for (;;) {
    int error;

    if (!snapLand && !skipLand)
      WalkMailbox();
    if (jetdesc[jd].rx.msg.head != -1)
      break;
    debug ("going to sleep (snapLand %d), because there is nothing to "
           "read\n", snapLand);
    if (!snapLand) {
      if ((error = tsleep(&mailboxEvent, PWAIT | PCATCH, "wormboy", 0))
          != 0) {
        debug ("tsleep returned %d\n", error);
        return error;
      }
    } else {
      if ((error = tsleep(&snapEvent, PWAIT | PCATCH, "wormboy", 0))
          != 0) {
        debug ("tsleep returned %d\n", error);
        return error;
      }
    }
    debug ("waking up! (maybe there's something)\n");
  }
  bdIndex = jetdesc[jd].rx.msg.head;
  bd = &bufferdesc[bdIndex];
  debug ("bdIndex is %d.  %d bytes in buffer, user can take %d\n",
         bdIndex, bd->length, length);
 
  if (!snapLand && !skipLand && (bd->offset == 0)) { 
    debug ("jmread: Read offset is zero and length is %d\n",
           bd->length); 
    /*
     * Strip LLC/SNAP header if one exists
     */
       
    if (jetdesc[jd].rx.msg.flag != 0) {
      debug ("Stripping LLC/SNAP header\n");
      debug ("jmread: Updating offset to %d and length to %d\n",
             bd->offset + sizeof(llc), bd->length - sizeof(llc));
      bd->offset += sizeof(llc);
      bd->length -= sizeof(llc);
      if (bd->length < 0) {
        warn ("jetmsgread: packet length < 0\n");
        return EIO;
      }
    }
  }

  source = BDINDEX_TO_DATABUF(bdIndex) + bd->offset;
  xferamt = MIN(length,bd->length);
  debug ("uiomove(0x%x,%d,0x%x)\n", (uint32)source, xferamt, (uint32)uio);
  uiomove (source, xferamt, uio);
        
  bd->length -= xferamt;
  if (bd->length != 0)
    bd->offset += xferamt;
  else {
    if ((jetdesc[jd].rx.msg.head = bd->next) == -1)
      jetdesc[jd].rx.msg.tail = -1;
            
    debug ("buffer descriptor %d:", bdIndex);
    if (bd->mailboxDone) {
      debug ("returning to RxFreelist\n");
      AppendAppropriateFreelist_one(bdIndex);
    }
    else {
      debug ("consumer is done with it, but mbox still wants it\n");
      bd->consumerDone = TRUE;
    }
  }
   
  return 0;
}
end();


public int jetmsgwrite (dev_t dev, struct uio *uio, int ioflag)
begin();
{
  int jd = minor (dev);
  int length = CountUio (uio);
  int bdIndex;
  uint32 dest;

  debug ("dev=0x%x, uio=0x%x, ioflag=%d, jd=%d\n", dev, (uint32)uio, ioflag,
         jd);

  if (jd == 0)  /* read and write don't work for minor device 0 */
    return EACCES;
  if (!(jetdesc[jd].state & JD_STATE_CAN_WRITE))
    return EACCES;
  if (jetdesc[jd].tx.msg.flag != 0) {
    if (length > (PAGESIZE - 8)) 
      return EFBIG;
  }
  else 
    if (length > PAGESIZE)
      return EFBIG;

  if (jetdesc[jd].state & JD_STATE_WRITE_VC) {
    /* wait for a free VR and for the txlist to be empty */
    for (;;) {
      vcid_t vcid = jetdesc[jd].tx.msg.vcid;
      bool txBusy = (ll_txTableBase[UNIT][vcid].vrIndex != 0);
      debug ("vcid=%d, txBusy=%d, txFreelistHead=%d\n",
             vcid, txBusy, txFreelistHead);
            
      WalkMailbox();
            
      if ((txFreelistHead==-1) || txBusy) {
        int error;
        debug ("going to sleep because I can't get what I want\n");
        if ((error = tsleep(&mailboxEvent, PWAIT | PCATCH,
                            "wormboy", 0)) != 0) {
          debug ("tsleep returned %d\n", error);
          return error;
        }
        debug ("Waking up, with some sense of hope\n");
      }
      else
        break;
    }

    bdIndex=txFreelistHead;
    debug ("using buffer descriptor %d\n", bdIndex);
    if ((txFreelistHead = bufferdesc[bdIndex].next) == -1)
      txFreelistTail = -1;
    dest = virtBase + bdIndex*PAGESIZE;
 
    if (jetdesc[jd].tx.msg.flag != 0) {
 
      /*
       * Add LLC/SNAP header to packet
       */
           
      struct llc llc;

      debug ("jetmsgwrite: Adding LLC/SNAP header\n");
      llc.llc_dsap = LLC_SNAP_LSAP;
      llc.llc_ssap = LLC_SNAP_LSAP;
      llc.llc_un.type_snap.control = LLC_UI;
      llc.llc_un.type_snap.org_code[0] = 0;
      llc.llc_un.type_snap.org_code[1] = 0;
      llc.llc_un.type_snap.org_code[2] = 0;
      llc.llc_un.type_snap.ether_type = htons(jetdesc[jd].tx.msg.header);

      debug ("copying LLC/SNAP header to tx buffer\n");
      bcopy(&llc, (void*) dest, sizeof(llc));

      debug ("copying %d bytes from user to kernel virt 0x%x\n",
             length, (uint32)dest + sizeof(llc));
      uiomove ((char *) dest + sizeof(llc), length, uio);
      AppendTx_one(jetdesc[jd].tx.msg.vcid, bdIndex, length + sizeof(llc));
    }
    else {
      debug ("copying %d bytes from user to kernel virt 0x%x\n",
             length, (uint32)dest);
      uiomove ((char *) dest, length, uio);
      AppendTx_one(jetdesc[jd].tx.msg.vcid, bdIndex, length);
    }
  } else { /* do snap multiplexing */
    struct mbuf *mbuf = NULL;
    struct llc llc;

    /* copy uio buffer into a contiguous area.  Why?  Because we suck. */
    uiomove ((char *) tempBuffer, length, uio);

    llc.llc_dsap = LLC_SNAP_LSAP;
    llc.llc_ssap = LLC_SNAP_LSAP;
    llc.llc_un.type_snap.control = LLC_UI;
    llc.llc_un.type_snap.org_code[0] = 0;
    llc.llc_un.type_snap.org_code[1] = 0;
    llc.llc_un.type_snap.org_code[2] = 0;
    llc.llc_un.type_snap.ether_type = htons(jetdesc[jd].tx.snap.header);

    warn ("Looking for mbuf to send snap packet\n");

    if (MbufAppend (&mbuf, &llc, sizeof(llc)) < 0)
      return ENOMEM;  /* then the llc/snap header */
    if (MbufAppend (&mbuf, tempBuffer, length) < 0)
      return ENOMEM;  /* then the data */

    warn ("Sending mbuf to IP land on VPIVCI %d!\n",
          jetdesc[jd].tx.snap.vpivci);
    jetmsg_ip_tx_interface (UNIT, mbuf, jetdesc[jd].tx.snap.vpivci);
  }

  return 0;
}
end();
   

public int jetmsgselect (dev_t dev, int rw, struct proc *p)
begin();
{
  int jd = minor (dev);
  debug ("dev=0x%x, rw=%d, proc=0x%x, jd is %d\n", dev, rw, (uint32)p, jd);

  WalkMailbox();

  if (rw == FREAD) {
    if (!(jetdesc[jd].state & JD_STATE_CAN_READ)) {
      warn ("Did you forget to bind a rx connection?\n");
      return 0;
    }
 
    if (jetdesc[jd].rx.msg.head != -1) {
      debug ("ready to read jd %d\n", jd);
      return 1;
    }
  } else {
    assert (rw == FWRITE);
    if (txFreelistHead != -1) {
      debug ("ready to write\n");
      return 1;
    }
  }
  debug ("Calling selrecord...\n");
  selrecord (p, &selinfo);
  return 0;
}
end();


public int
jetmsgioctl (dev_t dev, int cmd, caddr_t addr, int flag, struct proc *p)
begin();
{
  int jd = minor (dev);
  int32 *data = (int32*)addr;

  debug ("dev=%d, cmd=0x%x, addr=0x%x, flag=%d, proc=0x%x, jd=%d\n",
         dev, cmd, (uint32)addr, flag, (uint32)p, jd);

  switch (cmd) {
  case JETMSG_IOCTL_JET_INIT: {
    int error;
    int unit        = data[0];
    uint32 *address = (uint32*)data[1];
    uint32 length   = data[2];
    uint32 *ucode;

    warn ("unit=%d, ucode user address 0x%x, length %d\n",
          unit, (uint32)address, length);

    if ((ucode = malloc(length, M_DEVBUF, M_WAITOK)) == NULL)
      return ENOMEM;
    copyin(address, ucode, length);
  
    if (1) {
      warn ("Asking for a lot of resources!!!\n");
#ifdef IJET_VPC_CREDIT
      error = ll_Init (unit, 1024, 1024, 10464, 1024, 2, 64, 0,
                       (caddr_t)ucode, length);
#else
      error = ll_Init (unit, 1024, 1024, 13000, 1024, 2, 64, 0,
                       (caddr_t)ucode, length);
#endif
    } else
      error = ll_Init (unit, 1024, 1024, 1024, 1024, 2, 64, 0,
                       (caddr_t)ucode, length);

            
    free (ucode, M_DEVBUF);
    return error;
  } break;

  case JETMSG_IOCTL_INIT: {
    return msg_Init (data[0], data[1], data[2], data[3]);
  } break;

  case JETMSG_IOCTL_RELOAD_UCODE: {
    int error;
    int unit = data[0];
    uint32 *address = (uint32*)data[1];
    uint32 length   = data[2];
    uint32 *ucode;

    debug ("Reloading ucode from address 0x%x, length %d\n", (uint32)address,
           length);

    if ((ucode = malloc(length, M_DEVBUF, M_WAITOK)) == NULL)
      return ENOMEM;
    copyin(address, ucode, length);

    error = jet_DownloadUcode (unit, ucode, length);
            
    free (ucode, M_DEVBUF);
    return error;
  }
  
  case JETMSG_IOCTL_SET_LOOPBACK: {
    int unit  = data[0];
    int value = data[1];
    return jet_SetSonetLoopback (data[0], data[1]);
  } break;

  case JETMSG_IOCTL_OPEN_RX: {
    return msg_OpenRx (jd, data[0], data[1], data[2]);
  } break;

  case JETMSG_IOCTL_OPEN_TX: {
    return msg_OpenTx (jd, data[0], data[1], data[2]);
  } break;

  case JETMSG_IOCTL_BIND_RX_IPVC: {
    return msg_BindRxIPVC (jd, data[0], data[1]);
  } break;
        
  case JETMSG_IOCTL_BIND_TX_IPVC: {
    return msg_BindTxIPVC (jd, data[0], data[1]);
  } break;

  case JETMSG_IOCTL_BIND_SKIP: {
    return msg_OpenSkip(jd);
  } break;

  case FIONREAD: {
    return msg_Fionread (jd, &data[0]);
  } break;

  default: {
    warn ("ioctl 0x%x not implemented\n", cmd);
    return EINVAL;
  } break;
  }
}
end();




/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

public int msg_Init (int p_numRxBuffers, int p_numTxBuffers,
                     int p_numSnapBuffers, int p_numjds)
begin();
{
  int i;

  numRxBuffers   = p_numRxBuffers;
  numTxBuffers   = p_numTxBuffers;
  numSnapBuffers = p_numSnapBuffers;
  numjds         = p_numjds;

  numVrs     = numRxBuffers + numTxBuffers;
  numBuffers = numRxBuffers + numTxBuffers + numSnapBuffers;

  warn ("numRxBuffers=%d, numTxBuffers=%d, numSnapBuffers=%d, "
        "numjds=%d\n", numRxBuffers, numTxBuffers, numSnapBuffers, numjds);

  /* get the VRs.  For painfree programming I require a contiguous block */
    
  {
    vr_t *vrTable[numVrs];
    int count, i;

    if ((count = ll_AllocateVrs (UNIT, numVrs, vrTable))!=numVrs) {
      warn("Can't allocate %d VR's (error %d)\n", numVrs, count);
      return count;
    }
    for (i=1;i<numVrs;i++) {
      if (VRINDEX(vrTable[i]) != VRINDEX(vrTable[i-1])+1) {
        warn("VRs not sequential!\n");
        return -1;
      }
    }
    vrBase=vrTable[0];
    debug("vrBase is index 0x%x\n", VRINDEX(vrBase));
  }

  {
    /* allocate tx/rx buffers.  I don't know if kmem_alloc guarantees that
     * memory will be aligned on a page boundary, so I do it myself. */
        
    vm_offset_t base;
    uint32 size = (numBuffers+1)*PAGESIZE;
        
    if ((base = kmem_alloc(kernel_map, size)) <= 0) {
      warn("Couldn't allocate %d bytes for Big Buffer!\n", size);
      return -1;
    }
    virtBase = (base+PAGESIZE-1)&~(PAGESIZE-1);
    debug ("virtBase is 0x%x\n", virtBase);
  }

  /* allocate the buffer descriptors */
        
  if ((bufferdesc = malloc(numBuffers*sizeof(*bufferdesc),
                           M_DEVBUF, M_NOWAIT)) == NULL) {
    warn("can't get memory for buffer descriptors!\n");
    return -1;
  }
  for (i=0; i<numBuffers; i++) {
    bufferdesc[i].physAddress = vtophys(i*PAGESIZE+virtBase);
    debug("Buffer %d lives at virtual 0x%x physical 0x%x\n",
          i, i*PAGESIZE+virtBase, bufferdesc[i].physAddress);
  }

  /* allocate the file descriptor table */

  if ((jetdesc = malloc(numjds*sizeof(*jetdesc),
                        M_DEVBUF, M_NOWAIT)) == NULL) {
    warn("can't get memory for file descriptor table!\n");
    return -1;
  }
  bzero (jetdesc, numjds*sizeof(*jetdesc));

  /* steal buffer descriptor 0, use its VR to initialize mailbox */

  mailboxHead = &vrBase[0];
  {
    static vr_t hostVr = {{0}};
    *mailboxHead = hostVr;  /* zero the VR */
  }
  if ((mailbox = ll_AllocMailbox (UNIT, NULL, MailboxInterruptHandler,
                                  mailboxHead)) < 0) {
    warn ("Can't open mailbox (error %d)\n", mailbox);
    return mailbox;

    bufferdesc[0].mailboxDone  = FALSE;
    bufferdesc[0].consumerDone = TRUE;
  }
  debug ("mailbox is %d\n", mailbox);
    
  /* get a freelist */
    
  if ((freelist = ll_AllocFreelist (UNIT, NULL, NULL)) < 0) {
    warn ("Can't allocate freelist (error %d)\n", freelist);
    return freelist;
  }
  debug ("freelist is %d\n", freelist);

  /* build rx,tx,snap freelist */

  txFreelistHead     = -1;
  snapRxFreelistHead = -1;
  AppendAppropriateFreelist_init();
  for(i=1;i<numBuffers;i++)  /* 0 lives on the mailbox */
    AppendAppropriateFreelist(i);
  AppendAppropriateFreelist_done();

  warn ("first rx VR is 0x%x\n", VRINDEX(vrBase));
  warn ("first tx VR is 0x%x\n", VRINDEX(vrBase+numRxBuffers));
  warn ("Let's go!\n");
  upAndRunning = TRUE;
    
  return 0;
}
end();


private int msg_OpenRx (int jd, vpivci_t vpivci, uint16 llc_flag, uint16 snap)
begin();
{
  vcid_t vcid;
  debug ("jd %d, vpivci %d\n", jd, vpivci);

  if (!(jetdesc[jd].state & JD_STATE_CAN_BIND_READ))
    return EACCES;

  if ((vcid = ll_OpenRx (UNIT, CELL_PROCESSING_AAL5, mailbox,
                         freelist, freelist, &vpivci)) < 0) {
    warn ("can't open rx: error %d\n", vcid);
    return EPERM;
  }
  debug ("opened receive vcid %d\n", vcid);

  jetdesc[jd].rx.msg.vcid = vcid;
  jetdesc[jd].rx.msg.head = -1;
  jetdesc[jd].rx.msg.tail = -1;
  jetdesc[jd].rx.msg.flag = llc_flag;
  jetdesc[jd].rx.msg.header = snap;
  jetdesc[jd].state &= ~JD_STATE_CAN_BIND_READ;
  jetdesc[jd].state |=  JD_STATE_CAN_READ | JD_STATE_READ_VC;
  return 0;
}
end();


private int msg_BindRxIPVC (int jd, vpivci_t vpivci, uint16 snap)
begin();
{
  debug ("jd %d, vpivci %d\n", jd, vpivci);

  if (!(jetdesc[jd].state & JD_STATE_CAN_BIND_READ))
    return EACCES;

  jetdesc[jd].rx.snap.vpivci = vpivci;
  jetdesc[jd].rx.snap.header = snap;
  jetdesc[jd].rx.snap.head = -1;
  jetdesc[jd].rx.snap.tail = -1;
  jetdesc[jd].state &= ~JD_STATE_CAN_BIND_READ;
  jetdesc[jd].state |=  JD_STATE_CAN_READ | JD_STATE_READ_SNAP_DEMUX;
  return 0;
}
end();


private int msg_OpenTx (int jd, vpivci_t vpivci, uint16 llc_flag, uint16 snap)
begin();
{
  vcid_t vcid;
  debug ("jd %d, vpivci %d\n", jd, vpivci);

  if (!(jetdesc[jd].state & JD_STATE_CAN_BIND_WRITE))
    return EACCES;

  if ((vcid = ll_OpenTx (UNIT, CELL_PROCESSING_AAL5, mailbox,
                         SCHEDULE_STATIC, vpivci)) < 0) {
    warn ("can't open tx: error %d\n", vcid);
    return EPERM;
  }
  debug ("opened tx vpivci %d as vcid %d\n", vpivci, vcid);

  jetdesc[jd].tx.msg.vcid = vcid;
  jetdesc[jd].tx.msg.flag = llc_flag;
  jetdesc[jd].tx.msg.header = snap;
  jetdesc[jd].state &= ~JD_STATE_CAN_BIND_WRITE;
  jetdesc[jd].state |=  JD_STATE_CAN_WRITE | JD_STATE_WRITE_VC;

  return 0;
}
end();


private int msg_BindTxIPVC (int jd, vpivci_t vpivci, uint16 snap)
begin();
{
  debug ("jd %d, vpivci %d\n", jd, vpivci);

  if (!(jetdesc[jd].state & JD_STATE_CAN_BIND_WRITE))
    return EACCES;

  jetdesc[jd].tx.snap.vpivci = vpivci;
  jetdesc[jd].tx.snap.header = snap;
  jetdesc[jd].state  &= ~JD_STATE_CAN_BIND_WRITE;
  jetdesc[jd].state  |=  JD_STATE_CAN_WRITE | JD_STATE_WRITE_SNAP_DEMUX;

  return 0;
}
end();

private int msg_OpenSkip (int jd) 
begin();
{
  warn ("SKIP jd is %d\n", jd);
    
  SKIP_JD = jd;
  if (!(jetdesc[jd].state & JD_STATE_CAN_BIND_READ))
    return EACCES;

  jetdesc[jd].rx.msg.head = -1;
  jetdesc[jd].rx.msg.tail = -1;
  jetdesc[jd].state &= ~JD_STATE_CAN_BIND_READ;
  jetdesc[jd].state |= JD_STATE_CAN_READ | JD_STATE_READ_SKIP;
  return 0;
}
end();

private int msg_Fionread (int jd, int *nbytes)
begin();
{
  int bdIndex;
  assert (nbytes != NULL);

  WalkMailbox();
    
  if (!(jetdesc[jd].state & JD_STATE_CAN_READ))
    return EACCES;
    
  if ((bdIndex = jetdesc[jd].rx.msg.head) != -1) {
    debug ("ready to read %d bytes\n", bufferdesc[bdIndex].length);
    *nbytes = bufferdesc[bdIndex].length;
  } else {
    debug ("nothing to read\n");
    *nbytes = 0;
  }
  return 0;
}
end();


/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

private void WalkMailbox (void)
begin();
{
  static int insideWalker = FALSE;
  int dummy = (dummy,assert (insideWalker++ == 0));
  int nextIndex = mailboxHead->message.nextIndex;
  debug ("nextIndex is %d\n", nextIndex);

  if (nextIndex != 0) {
    vr_t *vrp;
    int bdIndex;

    AppendAppropriateFreelist_init();

    /* get rid of dangling VR */
        
    vrp = mailboxHead;
    bdIndex = vrp-vrBase;
    DumpVr ("Dangling VR", vrp);

    if (bufferdesc[bdIndex].consumerDone) {
      debug ("Dangling VR has been processed by the user\n");
      AppendAppropriateFreelist(bdIndex);
    } else {
      debug ("Dangling VR still needs to be processed by the user\n");
      bufferdesc[bdIndex].mailboxDone = TRUE;
    }

    /* process rest of mailbox */
         
    do {
      vr_t hostVr;
      int index = nextIndex;
      vrp = ll_vrBase[UNIT]+index;
      hostVr = *vrp;      /* burst out of SRAM */
      bdIndex = vrp-vrBase;
      nextIndex = hostVr.message.nextIndex;

      debug ("Encountered vr index 0x%x (buffer descriptor %d), "
             "nextIndex 0x%x\n", index, bdIndex, nextIndex);

      DumpVr ("the next one", vrp);

      if (hostVr.message.rxTx == VR_MESSAGE_RX) {
        int jd = RxVcidToJd (hostVr.message.vcid);
        int length=PAGESIZE-hostVr.message.remain;
        int difference=length-hostVr.message.pduLength;

        debug ("length is %d, pduLength is %d\n", length,
               hostVr.message.pduLength);
              
        if (hostVr.message.crcError  ||
            !hostVr.message.endOfPdu ||
            (difference<0) ||
            (difference>55) ||
            (jd<0)) {
          debug ("VR 0x%x(#%d) BAD:crc=%d,eop=%d,vci=%d,jd=%d,"
                 "P-R=%d\n",
                 index, bdIndex,
                 hostVr.message.crcError,
                 hostVr.message.endOfPdu,
                 hostVr.message.vcid, jd, difference);
                
          goto RECYCLE_BUFFER;
        }
              
        bufferdesc[bdIndex].offset       = 0;
        bufferdesc[bdIndex].length       = hostVr.message.pduLength;
        bufferdesc[bdIndex].consumerDone = FALSE;
        bufferdesc[bdIndex].mailboxDone  = (nextIndex != 0);
              
        AppendConsumerRxlist_one(jd, bdIndex);
      } else {
      RECYCLE_BUFFER:
        if (nextIndex != 0)
          AppendAppropriateFreelist(bdIndex);
        else
          bufferdesc[bdIndex].consumerDone = TRUE;
      }
    } while (nextIndex != 0);
    mailboxHead = vrp;
    AppendAppropriateFreelist_done();
  }
  --insideWalker;
}
end ();


private void MailboxInterruptHandler (int unit)
begin();
{
  selwakeup (&selinfo);
  wakeup (&mailboxEvent);
}
end();


public void ip_jetmsg_rx_interface (int unit, struct mbuf *mbuf,
                                    vpivci_t vpivci, int snapHeader)
begin ();
{
  int mbufLength;
  assert (mbuf != NULL);
  assert (unit == UNIT);

  if ((mbufLength = CountMbuf (mbuf)) > PAGESIZE)
    warn ("This mbuf has length %d: too big!\n", mbufLength);
  else if (snapRxFreelistHead == -1)
    warn ("No free snap buffers.\n");
  else {
    int jd;
    warn ("vpivci is %d, snap header is %d\n", vpivci, snapHeader);

    /* use trashy linear search to find out who, if anyone, is waiting for
     * data on this snap header */

    for (jd=0; jd<numjds; jd++)
      if ((jetdesc[jd].state & JD_STATE_READ_SNAP_DEMUX) &&
          (jetdesc[jd].rx.snap.vpivci == vpivci) &&
          (jetdesc[jd].rx.snap.header == snapHeader)) {
        int bdIndex;
        debug ("jd %d is waiting for this packet\n", jd);

        bdIndex = snapRxFreelistHead;
        if ((snapRxFreelistHead = bufferdesc[bdIndex].next) == -1)
          snapRxFreelistTail = -1;
        debug ("dequeued buffer %d\n", bdIndex);
                
        CopyMbufToBuffer (mbuf, BDINDEX_TO_DATABUF(bdIndex),
                          mbufLength);
        m_freem(mbuf);

        bufferdesc[bdIndex].offset       = 0;
        bufferdesc[bdIndex].length       = mbufLength;
        bufferdesc[bdIndex].consumerDone = FALSE;
        bufferdesc[bdIndex].mailboxDone  = TRUE;
        AppendConsumerRxlist_one (jd, bdIndex);
                
        selwakeup (&selinfo);
        wakeup (&snapEvent);

        break;
      }

    if (jd == numjds)
      warn ("Snap packet discarded--no one wants it!\n");
  }
}
end();

public void skip_jetmsg_rx_interface (char *buf, int len)
begin();
{
  if (len > PAGESIZE)
    warn ("This buffer has length %d: too big!\n", len);
  else if (snapRxFreelistHead == -1)
    warn ("No free buffers\n");
  else {
    int jd = SKIP_JD;
    int bdIndex;

    if (jetdesc[jd].state & JD_STATE_READ_SKIP) {
      bdIndex = snapRxFreelistHead;
      if ((snapRxFreelistHead = bufferdesc[bdIndex].next) == -1)
        snapRxFreelistTail = -1;
      warn ("dequeued buffer %d\n", bdIndex);
      bcopy (buf, BDINDEX_TO_DATABUF(bdIndex), len);
      bufferdesc[bdIndex].offset		= 0;
      bufferdesc[bdIndex].length		= len; 
      bufferdesc[bdIndex].consumerDone	= FALSE;
      bufferdesc[bdIndex].mailboxDone	= TRUE;
      AppendConsumerRxlist_one (jd, bdIndex);

      selwakeup (&selinfo);
      wakeup (&snapEvent);

    }
  }
}
end();



/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

/* sort of uncool linear search here.  We'll fix this (someday) */

private int RxVcidToJd (vcid_t vcid)
begin();
{
  int i;
  for (i=0; i<numjds; i++)
    if ((jetdesc[i].state & JD_STATE_CAN_READ) && 
        (jetdesc[i].rx.msg.vcid == vcid))
      return i;
  return -1;
}
end();


private int MbufAppend (struct mbuf **mbufp, void *data, int length)
begin();
{
  struct mbuf *mbuf;
  assert (mbufp != NULL);

  /* get initial mbuf if passed a null pointer */

  if (*mbufp == NULL) {
    debug ("getting initial mbuf\n");
    MGET (*mbufp, M_DONTWAIT, MT_DATA);
    if (*mbufp == NULL)
      return -1;
    if (length >= MINCLSIZE) {
      debug ("making it into cluster mbuf\n");
      MCLGET(*mbufp, M_DONTWAIT);
    }
    (*mbufp)->m_len = 0;
  }
  mbuf = *mbufp;

  while (length > 0) {
    int xferamt;

    while ((xferamt = MIN (length, M_TRAILINGSPACE(mbuf))) == 0) {
      /* this mbuf used up.  Get another one */
      struct mbuf *last = mbuf;

      debug ("getting another mbuf\n");

      MGET (mbuf, M_DONTWAIT, MT_DATA);
      if (mbuf == NULL)
        return -1;
      if (length >= MINCLSIZE) {
        debug ("making it into cluster mbuf\n");
        MCLGET(mbuf, M_DONTWAIT);
      }
      mbuf->m_len = 0;
      last->m_next = mbuf;
    }

    debug ("bcopy(0x%x,0x%x,%d)\n", (uint32)data,
           mtod(mbuf,uint32)+mbuf->m_len, xferamt);
    bcopy (data, mtod(mbuf,void*)+mbuf->m_len, xferamt);

    mbuf->m_len += xferamt;
    length      -= xferamt;
  }

  return 0;
}
end();    


private void CopyMbufToBuffer (struct mbuf *mbuf, void *dest, int length)
begin();
{
  assert (dest != NULL);
  while (length && (mbuf != NULL)) {
    int xferamt = MIN (length, mbuf->m_len);
    debug ("bcopy(0x%x,0x%x,%d)\n", mtod(mbuf,uint32), (uint32)dest,
           xferamt);
    bcopy (mtod(mbuf,char*), dest, xferamt);
    length -= xferamt;
    dest   += xferamt;
    mbuf = mbuf->m_next;
  }
}
end();



/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

private void AppendAppropriateFreelist_one (int bdIndex)
begin();
{
  AppendAppropriateFreelist_init ();
  AppendAppropriateFreelist (bdIndex);
  AppendAppropriateFreelist_done ();
}
end();

private void AppendAppropriateFreelist_init (void)
begin ();
{
  AppendRxFreelist_init   ();
  AppendTxFreelist_init   ();
  AppendSnapFreelist_init ();
}
end ();

private void AppendAppropriateFreelist (int bdIndex)
begin ();
{
  if (IS_RX_BUFFERDESC (bdIndex))
    AppendRxFreelist (bdIndex);
  else if (IS_TX_BUFFERDESC (bdIndex))
    AppendTxFreelist (bdIndex);
  else if (IS_SNAP_BUFFERDESC (bdIndex))
    AppendSnapFreelist (bdIndex);
  else {
    warn ("Illegal buffer index %d\n", bdIndex);
    notreached;
  }
}
end ();

private void AppendAppropriateFreelist_done (void)
begin ();
{
  AppendRxFreelist_done   ();
  AppendTxFreelist_done   ();
  AppendSnapFreelist_done ();
}
end ();



/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

private void AppendTxFreelist (int bdIndex)
begin ();
{
  assert (IS_TX_BUFFERDESC(bdIndex));
  
  bufferdesc[bdIndex].next = -1;
  
  if (txFreelistHead == -1)
    txFreelistHead = bdIndex;
  else
    bufferdesc[txFreelistTail].next = bdIndex;
  txFreelistTail = bdIndex;
  
  { /* debugging stuff */
    vr_t *vrp = vrBase + bdIndex;
    static vr_t hostVr = {{0}};
    hostVr.message.cpcsUuCpi = 0xcafe;
    hostVr.message.vcid      = 0xdead;
    hostVr.message.pduLength = 0xabba;
    hostVr.message.remain    = 0xbeef;
    hostVr.message.address   = 0xbaddbabe;
    
    *vrp = hostVr;
  }
}
end ();


/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

private vr_t *appendRxFreelistPreviousVrp;
private vr_t *appendRxFreelistFirstVrp;
private vr_t  appendRxFreelistPreviousHostVr = {{0}};

#define previousVrp    appendRxFreelistPreviousVrp
#define firstVrp       appendRxFreelistFirstVrp
#define previousHostVr appendRxFreelistPreviousHostVr

private void AppendRxFreelist_one (int bdIndex)
begin();
{
  AppendRxFreelist_init ();
  AppendRxFreelist (bdIndex);
  AppendRxFreelist_done ();
}
end();

private void AppendRxFreelist_init (void)
begin ();
{
#if 0
  previousVrp = NULL;
#endif
}
end ();

private void AppendRxFreelist (int bdIndex)
begin ();
{
#if 0
  vr_t *vrp = vrBase+bdIndex;
  
  assert (IS_RX_BUFFERDESC (bdIndex));
  
  if (previousVrp == NULL)
    firstVrp = vrp;
  else {
    previousHostVr.vr.nextIndex = vrp-ll_vrBase[UNIT];
    *previousVrp = previousHostVr;  /* burst previous VR into SRAM */
  }
  previousVrp = vrp;

  previousHostVr.vr.sarLocation = SAR_HOST;
  previousHostVr.vr.address     = bufferdesc[bdIndex].physAddress;
  previousHostVr.vr.remain      = PAGESIZE;

  { /* debugging stuff */
    previousHostVr.message.vcid      = 0x9cac;
    previousHostVr.message.cpcsUuCpi = 0x7eeb;
    previousHostVr.message.pduLength = 0xbaba;
  }
#else
  vr_t *vrp = vrBase+bdIndex;
  static vr_t hostVr = {{0}};
  assert (IS_RX_BUFFERDESC (bdIndex));
  hostVr.vr.sarLocation = SAR_HOST;
  hostVr.vr.address     = bufferdesc[bdIndex].physAddress;
  hostVr.vr.remain      = PAGESIZE;
  assert (hostVr.vr.nextIndex == 0);
  *vrp = hostVr;
  assert (vrp->vr.nextIndex == 0);
  ll_AppendVrlistToFreelist (UNIT, freelist, vrp, vrp);
#endif
}
end ();

private void AppendRxFreelist_done (void)
begin ();
{
#if 0
  if (previousVrp != NULL) {
    previousHostVr.vr.nextIndex = 0;
    *previousVrp = previousHostVr;  /* burst last VR into SRAM */

    ll_AppendVrlistToFreelist (UNIT, freelist, firstVrp, previousVrp);
  }
#endif
}
end ();

#undef previousVrp
#undef firstVrp
#undef previousHostVr


/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

private void AppendSnapFreelist (int bdIndex)
begin();
{
  assert (IS_SNAP_BUFFERDESC (bdIndex));

  bufferdesc[bdIndex].next = -1;

  if (snapRxFreelistHead == -1)
    snapRxFreelistHead = bdIndex;
  else
    bufferdesc[snapRxFreelistTail].next = bdIndex;
  snapRxFreelistTail = bdIndex;
}
end();



/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

private void AppendConsumerRxlist (int jd, int bdIndex)
begin();
{
  bufferdesc[bdIndex].next = -1;

  if (jetdesc[jd].rx.msg.head == -1)
    jetdesc[jd].rx.msg.head = bdIndex;
  else
    bufferdesc[jetdesc[jd].rx.msg.tail].next = bdIndex;
  jetdesc[jd].rx.msg.tail = bdIndex;
}
end();


/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

private void AppendTx (int vcid, int bdIndex, int length)
begin();
{
  static vr_t hostVr = {{0}};
  vr_t *vrp = vrBase + bdIndex;

  hostVr.vr.sarLocation = SAR_HOST;
  hostVr.vr.address     = bufferdesc[bdIndex].physAddress;
  hostVr.vr.endOfPdu    = 1;
  hostVr.vr.cpcsUuCpi   = 0;
  hostVr.vr.remain      = length;
  hostVr.vr.length      = length;
  hostVr.vr.pduLength   = length;
  hostVr.vr.nextIndex   = 0;

  *vrp = hostVr;  /* burst into SRAM */

  ll_AppendVrThenScheduleDynamic (UNIT, 0, vcid, 4, vrp, vrp);
}
end();
